#ifndef MSM_MACROS_H
#define MSM_MACROS_H

#define ROUNDUP_QUOTIENT(n, k)  (((n)/(k)) + (((n)%(k)) != 0))

enum {
  PERIODIC_NONE = 0,
  PERIODIC_VEC1 = 0x01,
  PERIODIC_VEC2 = 0x02,
  PERIODIC_VEC3 = 0x04,
  PERIODIC_ALL = PERIODIC_VEC1 | PERIODIC_VEC2 | PERIODIC_VEC2
};

enum {
  APPROX_CUBIC = 0, /**< C1 degree 3 poly (numerical Hermite) */
  APPROX_QUINTIC,   /**< C1 degree 5 poly (linear blend of degree 4) */
  APPROX_QUINTIC2,  /**< C2 degree 5 poly */
  APPROX_SEPTIC,    /**< C1 degree 7 poly (linear blend of degree 6) */
  APPROX_SEPTIC3,   /**< C3 degree 7 poly */
  APPROX_NONIC,     /**< C1 degree 9 poly (linear blend of degree 8) */
  APPROX_NONIC4,    /**< C4 degree 9 poly */
  APPROX_BSPLINE,   /**< C2 degree 3 B-spline */
  APPROX_END        /**< (for internal use) */
};

enum {
  SPLIT_TAYLOR2 = 0,  /**< C2 Taylor */
  SPLIT_TAYLOR3,      /**< C3 Taylor */
  SPLIT_TAYLOR4,      /**< C4 Taylor */
  SPLIT_TAYLOR5,      /**< C5 Taylor */
  SPLIT_TAYLOR6,      /**< C6 Taylor */
  SPLIT_TAYLOR7,      /**< C7 Taylor */
  SPLIT_TAYLOR8,      /**< C8 Taylor */
  SPLIT_TAYLOR1,      /**< C1 Taylor */

  SPLIT_SIGMA2_3,     /**< C2, degree 3 (the "perfect" smoothing) */
  SPLIT_SIGMA3_5,     /**< C3, degree 5 */
  SPLIT_SIGMA4_6,     /**< C4, degree 6 */
  SPLIT_SIGMA4_7,     /**< C4, degree 7 */
  SPLIT_SIGMA5_8,     /**< C5, degree 8 */
  SPLIT_SIGMA5_9,     /**< C5, degree 9 */
  SPLIT_SIGMA6_9,     /**< C6, degree 9 */
  SPLIT_SIGMA6_10,    /**< C6, degree 10 */
  SPLIT_SIGMA6_11,    /**< C6, degree 11 */
  SPLIT_SIGMA7_11,    /**< C7, degree 11 */
  SPLIT_SIGMA7_12,    /**< C7, degree 12 */
  SPLIT_SIGMA7_13,    /**< C7, degree 13 */
  SPLIT_SIGMA8_12,    /**< C8, degree 12 */
  SPLIT_SIGMA8_13,    /**< C8, degree 13 */
  SPLIT_SIGMA8_14,    /**< C8, degree 14 */
  SPLIT_SIGMA8_15,    /**< C8, degree 15 */

  SPLIT_SIGMA2_6,     /**< C2, degree 6, even powers of r,
                             chosen to minimize error bound */

  SPLIT_SWITCH1_2,    /**< C2, switching at r/a=1/2 */
  SPLIT_SWITCH3_4,    /**< C2, switching at r/a=3/4 */
  SPLIT_SWITCH7_8,    /**< C2, switching at r/a=7/8 */

  SPLIT_END           /**< (for internal use) */
};

/** SPOLY() calculates the polynomial part of the
 * normalized smoothing of 1/r.
 *
 * Returns g(R), where R=r/a, and (d/dR)g(R).
 *
 *   pg - float*, points to variable to receive g(R)
 *   pdg - float*, points to variable to receive (d/dR)g(R)
 *   ra - (r/a), assumed to be between 0 and 1
 *   split - identify the type of smoothing used to split the potential
 */
#undef  SPOLY
#define SPOLY(pg, pdg, ra, split) \
  do { \
    double _r = ra;  /* _r=(r/a) */ \
    double _s = _r*_r;  /* _s=(r/a)^2 */ \
    double _g = 0, _dg = 0; \
    /* ASSERT(0 <= _s && _s <= 1); */ \
    switch (split) { \
      /* from section 5.1 of thesis */ \
      case SPLIT_TAYLOR2: \
        _g = 1 + (_s-1)*(-1./2 + (_s-1)*(3./8));                               \
        _dg = (2*_r)*(-1./2 + (_s-1)*(3./4)); \
        break;                                                                 \
      case SPLIT_TAYLOR3: \
        _g = 1 + (_s-1)*(-1./2 + (_s-1)*(3./8 + (_s-1)*(-5./16)));             \
        _dg = (2*_r)*(-1./2 + (_s-1)*(3./4 + (_s-1)*(-15./16))); \
        break;                                                                 \
      case SPLIT_TAYLOR4: \
        _g = 1 + (_s-1)*(-1./2 + (_s-1)*(3./8 + (_s-1)*(-5./16                 \
                + (_s-1)*(35./128))));                                         \
        _dg = (2*_r)*(-1./2 + (_s-1)*(3./4 + (_s-1)*(-15./16 \
                + (_s-1)*(35./32))));      \
        break;                                                                 \
      case SPLIT_TAYLOR5: \
        _g = 1 + (_s-1)*(-1./2 + (_s-1)*(3./8 + (_s-1)*(-5./16                 \
                + (_s-1)*(35./128 + (_s-1)*(-63./256)))));                     \
        _dg = (2*_r)*(-1./2 + (_s-1)*(3./4 + (_s-1)*(-15./16 + (_s-1)*(35./32  \
                + (_s-1)*(-315./256)))));                                      \
        break;                                                                 \
      case SPLIT_TAYLOR6: \
        _g = 1 + (_s-1)*(-1./2 + (_s-1)*(3./8 + (_s-1)*(-5./16                 \
                + (_s-1)*(35./128 + (_s-1)*(-63./256                           \
                    + (_s-1)*(231./1024))))));                                 \
        _dg = (2*_r)*(-1./2 + (_s-1)*(3./4 + (_s-1)*(-15./16 + (_s-1)*(35./32  \
                + (_s-1)*(-315./256 + (_s-1)*(693./512))))));                  \
        break;                                                                 \
      case SPLIT_TAYLOR7: \
        _g = 1 + (_s-1)*(-1./2 + (_s-1)*(3./8 + (_s-1)*(-5./16                 \
                + (_s-1)*(35./128 + (_s-1)*(-63./256                           \
                    + (_s-1)*(231./1024 + (_s-1)*(-429./2048)))))));           \
        _dg = (2*_r)*(-1./2 + (_s-1)*(3./4 + (_s-1)*(-15./16 + (_s-1)*(35./32  \
                + (_s-1)*(-315./256 + (_s-1)*(693./512                         \
                    + (_s-1)*(-3003./2048)))))));                              \
        break;                                                                 \
      case SPLIT_TAYLOR8: \
        _g = 1 + (_s-1)*(-1./2 + (_s-1)*(3./8 + (_s-1)*(-5./16                 \
                + (_s-1)*(35./128 + (_s-1)*(-63./256                           \
                    + (_s-1)*(231./1024 + (_s-1)*(-429./2048                   \
                        + (_s-1)*(6435./32768))))))));                         \
        _dg = (2*_r)*(-1./2 + (_s-1)*(3./4 + (_s-1)*(-15./16 + (_s-1)*(35./32  \
                + (_s-1)*(-315./256 + (_s-1)*(693./512                         \
                    + (_s-1)*(-3003./2048 + (_s-1)*(6435./4096))))))));        \
        break;                                                                 \
      case SPLIT_TAYLOR1: \
        _g = 1 + (_s-1)*(-1./2); \
        _dg = (2*_r)*(-1./2); \
        break; \
      /* from section 5.2 of thesis */ \
      case SPLIT_SIGMA2_3:  /* the "perfect" smoothing */ \
        _g = 2 + _s*(-2 + _r);                                                \
        _dg = _r*(-4 + _r*3);                                                  \
        break;                                                                 \
      case SPLIT_SIGMA3_5: \
        _g = 9./4 + _s*(-5./2 + _s*(9./4 - _r));                             \
        _dg = _r*(-5 + _s*(9 + _r*(-5)));                                     \
        break;                                                                 \
      case SPLIT_SIGMA4_6: \
        _g = 21./8 + _s*(-35./8 + _s*(63./8 + _r*(-7 + _r*(15./8))));        \
        _dg = _r*(-35./4 + _s*(63./2 + _r*(-35 + _r*(45./4))));               \
        break;                                                                 \
      case SPLIT_SIGMA4_7: \
        _g = 5./2 + _s*(-7./2 + _s*(7./2 + _s*(-5./2 + _r)));               \
        _dg = _r*(-7 + _s*(14 + _s*(-15 + _r*(7))));                         \
        break;                                                                 \
      case SPLIT_SIGMA5_8: \
        _g = 45./16 + _s*(-21./4 + _s*(63./8 + _s*(-45./4                   \
                + _r*(9 + _r*(-35./16)))));                                    \
        _dg = _r*(-21./2 + _s*(63./2 + _s*(-135./2                           \
                + _r*(63 + _r*(-35./2)))));                                    \
        break;                                                                 \
      case SPLIT_SIGMA5_9: \
        _g = 175./64 + _s*(-75./16 + _s*(189./32 + _s*(-75./16              \
                + _s*(175./64 - _r))));                                       \
        _dg = _r*(-75./8 + _s*(189./8 + _s*(-225./8 + _s*(175./8            \
                  + _r*(-9)))));                                               \
        break;                                                                 \
      case SPLIT_SIGMA6_9: \
        _g = 25./8 + _s*(-15./2 + _s*(63./4 + _s*(-75./2                    \
                + _r*(45 + _r*(-175./8 + _r*4)))));                            \
        _dg = _r*(-15 + _s*(63 + _s*(-225                                    \
                + _r*(315 + _r*(-175 + _r*36)))));                             \
        break;                                                                 \
      case SPLIT_SIGMA6_10: \
        _g = 385./128 + _s*(-825./128 + _s*(693./64 + _s*(-825./64          \
                + _s*(1925./128 + _r*(-11 + _r*(315./128))))));               \
        _dg = _r*(-825./64 + _s*(693./16 + _s*(-2475./32                     \
                + _s*(1925./16 + _r*(-99 + _r*(1575./64))))));                \
        break;                                                                 \
      case SPLIT_SIGMA6_11: \
        _g = 189./64 + _s*(-385./64 + _s*(297./32 + _s*(-297./32            \
                + _s*(385./64 + _s*(-189./64 + _r)))));                      \
        _dg = _r*(-385./32 + _s*(297./8 + _s*(-891./16 + _s*(385./8         \
                  + _s*(-945./32 + _r*(11))))));                              \
        break;                                                                 \
      case SPLIT_SIGMA7_11: \
        _g = 105./32 + _s*(-275./32 + _s*(297./16 + _s*(-495./16            \
                + _s*(1925./32 + _r*(-66 + _r*(945./32 + _r*(-5)))))));       \
        _dg = _r*(-275./16 + _s*(297./4 + _s*(-1485./8                       \
                + _s*(1925./4 + _r*(-594 + _r*(4725./16 + _r*(-55)))))));     \
        break;                                                                 \
      case SPLIT_SIGMA7_12: \
        _g = 819./256 + _s*(-1001./128 + _s*(3861./256                       \
              + _s*(-1287./64 + _s*(5005./256 + _s*(-2457./128              \
                    + _r*(13 + _r*(-693./256)))))));                           \
        _dg = _r*(-1001./64 + _s*(3861./64 + _s*(-3861./32                   \
                + _s*(5005./32 + _s*(-12285./64 + _r*(143                    \
                      + _r*(-2079./64)))))));                                  \
        break;                                                                 \
      case SPLIT_SIGMA7_13: \
        _g = 1617./512 + _s*(-1911./256 + _s*(7007./512 + _s*(-2145./128    \
                + _s*(7007./512 + _s*(-1911./256 + _s*(1617./512 - _r))))));\
        _dg = _r*(-1911./128 + _s*(7007./128 + _s*(-6435./64 + _s*(7007./64 \
                  + _s*(-9555./128 + _s*(4851./128 + _r*(-13)))))));         \
        break;                                                                 \
      case SPLIT_SIGMA8_12: \
        _g = 455./128 + _s*(-715./64 + _s*(3861./128 + _s*(-2145./32        \
                + _s*(25025./128 + _r*(-286 + _r*(12285./64 + _r*(-65         \
                        + _r*(1155./128))))))));                               \
        _dg = _r*(-715./32 + _s*(3861./32 + _s*(-6435./16                    \
                + _s*(25025./16 + _r*(-2574 + _r*(61425./32 + _r*(-715        \
                        + _r*(3465./32))))))));                                \
        break;                                                                 \
      case SPLIT_SIGMA8_13: \
        _g = 441./128 + _s*(-637./64 + _s*(3003./128                         \
              + _s*(-1287./32 + _s*(7007./128 + _s*(-5733./64               \
                    + _r*(91 + _r*(-4851./128 + _r*(6))))))));                 \
        _dg = _r*(-637./32 + _s*(3003./32 + _s*(-3861./16                    \
                + _s*(7007./16 + _s*(-28665./32 + _r*(1001                   \
                      + _r*(-14553./32 + _r*(78))))))));                       \
        break;                                                                 \
      case SPLIT_SIGMA8_14: \
        _g = 3465./1024 + _s*(-9555./1024 + _s*(21021./1024                  \
              + _s*(-32175./1024 + _s*(35035./1024 + _s*(-28665./1024       \
                    + _s*(24255./1024 + _r*(-15 + _r*(3003./1024))))))));     \
        _dg = _r*(-9555./512 + _s*(21021./256 + _s*(-96525./512              \
                + _s*(35035./128 + _s*(-143325./512 + _s*(72765./256        \
                      + _r*(-195 + _r*(21021./512))))))));                     \
        break;                                                                 \
      case SPLIT_SIGMA8_15: \
        _g = 429./128 + _s*(-1155./128 + _s*(2457./128 + _s*(-3575./128     \
                + _s*(3575./128 + _s*(-2457./128 + _s*(1155./128            \
                      + _s*(-429./128 + _r)))))));                            \
        _dg = _r*(-1155./64 + _s*(2457./32 + _s*(-10725./64                  \
                + _s*(3575./16 + _s*(-12285./64 + _s*(3465./32              \
                      + _s*(-3003./64 + _r*(15))))))));                       \
        break;                                                                 \
      /* from section 5.3 of thesis */ \
      case SPLIT_SIGMA2_6: \
        _g = (31./16) + _s*(-23./16 + _s*(9./16 + _s*(-1./16))); \
        _dg = (2*_r)*(-23./16 + _s*(9./8 + _s*(-3./16))); \
        break; \
      /* from section 5.4 of thesis */ \
      case SPLIT_SWITCH1_2:                                             \
        if (_r > 1./2) {                                                       \
          _g = 5./3 + _r + _s*(-3 + _r*(4./3));                               \
          _dg = 1 + _r*(-6 + _r*(4));                                          \
        }                                                                      \
        else {                                                                 \
          _g = 11./6 - _s;                                                    \
          _dg = _r*(-2);                                                       \
        }                                                                      \
        break;                                                                 \
      case SPLIT_SWITCH3_4:                                             \
        if (_r > 3./4) {                                                       \
          _g = 5./7 + _r*(27./7 + _r*(-41./7 + _r*(16./7)));                   \
          _dg = 27./7 + _r*(-82./7 + _r*(48./7));                              \
        }                                                                      \
        else {                                                                 \
          _g = 47./28 + _s*(-5./7);                                           \
          _dg = _r*(-10./7);                                                   \
        }                                                                      \
        break;                                                                 \
      case SPLIT_SWITCH7_8:                                             \
        if (_r > 7./8) {                                                       \
          _g = -19./15 + _r*(49./5 + _r*(-59./5 + _r*(64./15)));               \
          _dg = 49./5 + _r*(-118./5 + _r*(64./5));                             \
        }                                                                      \
        else {                                                                 \
          _g = 191./120 + _s*(-3./5);                                         \
          _dg = _r*(-6./5);                                                    \
        }                                                                      \
        break;                                                                 \
      default: \
        return /* NL_MSM_ERROR_SUPPORT */; \
    } \
    *(pg) = _g; \
    *(pdg) = _dg; \
  } while (0)
  /* closing ';' from use as function call */


/** Calculate the stencil of basis function values of Phi.
 * phi - stencil array (up to size MAX_POLY_DEGREE+1)
 * delta - normalized distance of atom from lowest grid point in stencil
 * approx - APPROX enum value from msm.h
 */
#define STENCIL_1D(_phi, _delta, _approx) \
  do { \
    float *phi = _phi; \
    float t = _delta; \
    switch (_approx) { \
      case APPROX_CUBIC: \
        phi[0] = 0.5f * (1 - t) * (2 - t) * (2 - t); \
        t--; \
        phi[1] = (1 - t) * (1 + t - 1.5f * t * t); \
        t--; \
        phi[2] = (1 + t) * (1 - t - 1.5f * t * t); \
        t--; \
        phi[3] = 0.5f * (1 + t) * (2 + t) * (2 + t); \
        break; \
      case APPROX_QUINTIC: \
        phi[0] = (1.f/24) * (1-t) * (2-t) * (3-t) * (3-t) * (4-t); \
        t--; \
        phi[1] = (1-t) * (2-t) * (3-t) * ((1.f/6) + t * (0.375f - (5.f/24)*t));\
        t--; \
        phi[2] = (1-t*t) * (2-t) * (0.5f + t * (0.25f - (5.f/12)*t)); \
        t--; \
        phi[3] = (1-t*t) * (2+t) * (0.5f - t * (0.25f + (5.f/12)*t)); \
        t--; \
        phi[4] = (1+t) * (2+t) * (3+t) * ((1.f/6) - t * (0.375f + (5.f/24)*t));\
        t--; \
        phi[5] = (1.f/24) * (1+t) * (2+t) * (3+t) * (3+t) * (4+t); \
        break; \
      case APPROX_QUINTIC2: \
        phi[0] = (1.f/24) * (3-t) * (3-t) * (3-t) * (t-2) * (5*t-8); \
        t--; \
        phi[1] = (-1.f/24) * (2-t) * (t-1) * (-48+t*(153+t*(-114+t*25))); \
        t--; \
        phi[2] = (1.f/12) * (1-t) * (12+t*(12+t*(-3+t*(-38+t*25)))); \
        t--; \
        phi[3] = (1.f/12) * (1+t) * (12+t*(-12+t*(-3+t*(38+t*25)))); \
        t--; \
        phi[4] = (-1.f/24) * (2+t) * (t+1) * (48+t*(153+t*(114+t*25))); \
        t--; \
        phi[5] = (1.f/24) * (3+t) * (3+t) * (3+t) * (t+2) * (5*t+8); \
        break; \
      case APPROX_SEPTIC: \
        phi[0] = (-1.f/720)*(t-1)*(t-2)*(t-3)*(t-4)*(t-4)*(t-5)*(t-6); \
        t--; \
        phi[1] = (1.f/720)*(t-1)*(t-2)*(t-3)*(t-4)*(t-5)*(-6+t*(-20+7*t)); \
        t--; \
        phi[2] = (-1.f/240)*(t*t-1)*(t-2)*(t-3)*(t-4)*(-10+t*(-12+7*t)); \
        t--; \
        phi[3] = (1.f/144)*(t*t-1)*(t*t-4)*(t-3)*(-12+t*(-4+7*t)); \
        t--; \
        phi[4] = (-1.f/144)*(t*t-1)*(t*t-4)*(t+3)*(-12+t*(4+7*t)); \
        t--; \
        phi[5] = (1.f/240)*(t*t-1)*(t+2)*(t+3)*(t+4)*(-10+t*(12+7*t)); \
        t--; \
        phi[6] = (-1.f/720)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)*(-6+t*(20+7*t)); \
        t--; \
        phi[7] = (1.f/720)*(t+1)*(t+2)*(t+3)*(t+4)*(t+4)*(t+5)*(t+6); \
        break; \
      case APPROX_SEPTIC3: \
        phi[0] = (3632.f/5) + t*((-7456.f/5) + t*((58786.f/45) + t*(-633 \
                + t*((26383.f/144) + t*((-22807.f/720) + t*((727.f/240) \
                      + t*(-89.f/720))))))); \
        t--; \
        phi[1] = -440 + t*((25949.f/20) + t*((-117131.f/72) + t*((2247.f/2) \
                + t*((-66437.f/144) + t*((81109.f/720) + t*((-727.f/48) \
                      + t*(623.f/720))))))); \
        t--; \
        phi[2] = (138.f/5) + t*((-8617.f/60) + t*((12873.f/40) + t*((-791.f/2) \
                + t*((4557.f/16) + t*((-9583.f/80) + t*((2181.f/80) \
                      + t*(-623.f/240))))))); \
        t--; \
        phi[3] = 1 + t*t*((-49.f/36) + t*t*((-959.f/144) + t*((2569.f/144) \
                + t*((-727.f/48) + t*(623.f/144))))); \
        t--; \
        phi[4] = 1 + t*t*((-49.f/36) + t*t*((-959.f/144) + t*((-2569.f/144) \
                + t*((-727.f/48) + t*(-623.f/144))))); \
        t--; \
        phi[5] = (138.f/5) + t*((8617.f/60) + t*((12873.f/40) + t*((791.f/2) \
                + t*((4557.f/16) + t*((9583.f/80) + t*((2181.f/80) \
                      + t*(623.f/240))))))); \
        t--; \
        phi[6] = -440 + t*((-25949.f/20) + t*((-117131.f/72) + t*((-2247.f/2) \
                + t*((-66437.f/144) + t*((-81109.f/720) + t*((-727.f/48) \
                      + t*(-623.f/720))))))); \
        t--; \
        phi[7] = (3632.f/5) + t*((7456.f/5) + t*((58786.f/45) + t*(633 \
                + t*((26383.f/144) + t*((22807.f/720) + t*((727.f/240) \
                      + t*(89.f/720))))))); \
        break; \
      case APPROX_NONIC: \
        phi[0] = (-1.f/40320)*(t-8)*(t-7)*(t-6)*(t-5)*(t-5)*(t-4)*(t-3)* \
          (t-2)*(t-1); \
        t--; \
        phi[1] = (1.f/40320)*(t-7)*(t-6)*(t-5)*(t-4)*(t-3)*(t-2)*(t-1)* \
          (-8+t*(-35+9*t)); \
        t--; \
        phi[2] = (-1.f/10080)*(t-6)*(t-5)*(t-4)*(t-3)*(t-2)*(t-1)*(t+1)* \
          (-14+t*(-25+9*t)); \
        t--; \
        phi[3] = (1.f/1440)*(t-5)*(t-4)*(t-3)*(t-2)*(t-1)*(t+1)*(t+2)* \
          (-6+t*(-5+3*t)); \
        t--; \
        phi[4] = (-1.f/2880)*(t-4)*(t-3)*(t-2)*(t-1)*(t+1)*(t+2)*(t+3)* \
          (-20+t*(-5+9*t)); \
        t--; \
        phi[5] = (1.f/2880)*(t-3)*(t-2)*(t-1)*(t+1)*(t+2)*(t+3)*(t+4)* \
          (-20+t*(5+9*t)); \
        t--; \
        phi[6] = (-1.f/1440)*(t-2)*(t-1)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)* \
          (-6+t*(5+3*t)); \
        t--; \
        phi[7] = (1.f/10080)*(t-1)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)*(t+6)* \
          (-14+t*(25+9*t)); \
        t--; \
        phi[8] = (-1.f/40320)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)*(t+6)*(t+7)* \
          (-8+t*(35+9*t)); \
        t--; \
        phi[9] = (1.f/40320)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)*(t+5)*(t+6)* \
          (t+7)*(t+8); \
        break; \
      case APPROX_NONIC4: \
        phi[0] = 439375.f/7+t*(-64188125.f/504+t*(231125375.f/2016 \
              +t*(-17306975.f/288+t*(7761805.f/384+t*(-2895587.f/640 \
                    +t*(129391.f/192+t*(-259715.f/4032+t*(28909.f/8064 \
                          +t*(-3569.f/40320))))))))); \
        t--; \
        phi[1] = -56375+t*(8314091.f/56+t*(-49901303.f/288+t*(3763529.f/32 \
                +t*(-19648027.f/384+t*(9469163.f/640+t*(-545977.f/192 \
                      +t*(156927.f/448+t*(-28909.f/1152 \
                          +t*(3569.f/4480))))))))); \
        t--; \
        phi[2] = 68776.f/7+t*(-1038011.f/28+t*(31157515.f/504+t*(-956669.f/16 \
                +t*(3548009.f/96+t*(-2422263.f/160+t*(197255.f/48 \
                      +t*(-19959.f/28+t*(144545.f/2016 \
                          +t*(-3569.f/1120))))))))); \
        t--; \
        phi[3] = -154+t*(12757.f/12+t*(-230123.f/72+t*(264481.f/48 \
                +t*(-576499.f/96+t*(686147.f/160+t*(-96277.f/48 \
                      +t*(14221.f/24+t*(-28909.f/288+t*(3569.f/480))))))))); \
        t--; \
        phi[4] = 1+t*t*(-205.f/144+t*t*(91.f/192+t*(-6181.f/320 \
                +t*(6337.f/96+t*(-2745.f/32+t*(28909.f/576 \
                      +t*(-3569.f/320))))))); \
        t--; \
        phi[5] = 1+t*t*(-205.f/144+t*t*(91.f/192+t*(6181.f/320 \
                +t*(6337.f/96+t*(2745.f/32+t*(28909.f/576 \
                      +t*(3569.f/320))))))); \
        t--; \
        phi[6] = -154+t*(-12757.f/12+t*(-230123.f/72+t*(-264481.f/48 \
                +t*(-576499.f/96+t*(-686147.f/160+t*(-96277.f/48 \
                      +t*(-14221.f/24+t*(-28909.f/288+t*(-3569.f/480))))))))); \
        t--; \
        phi[7] = 68776.f/7+t*(1038011.f/28+t*(31157515.f/504+t*(956669.f/16 \
                +t*(3548009.f/96+t*(2422263.f/160+t*(197255.f/48 \
                      +t*(19959.f/28+t*(144545.f/2016+t*(3569.f/1120))))))))); \
        t--; \
        phi[8] = -56375+t*(-8314091.f/56+t*(-49901303.f/288+t*(-3763529.f/32 \
                +t*(-19648027.f/384+t*(-9469163.f/640+t*(-545977.f/192 \
                      +t*(-156927.f/448+t*(-28909.f/1152 \
                          +t*(-3569.f/4480))))))))); \
        t--; \
        phi[9] = 439375.f/7+t*(64188125.f/504+t*(231125375.f/2016 \
              +t*(17306975.f/288+t*(7761805.f/384+t*(2895587.f/640 \
                    +t*(129391.f/192+t*(259715.f/4032+t*(28909.f/8064 \
                          +t*(3569.f/40320))))))))); \
        break; \
      case APPROX_BSPLINE: \
        phi[0] = (1.f/6) * (2-t) * (2-t) * (2-t); \
        t--; \
        phi[1] = (2.f/3) + t*t*(-1 + 0.5f*t); \
        t--; \
        phi[2] = (2.f/3) + t*t*(-1 - 0.5f*t); \
        t--; \
        phi[3] = (1.f/6) * (2+t) * (2+t) * (2+t); \
        break; \
      default: \
        return /* XXX NL_MSM_ERROR_SUPPORT */; \
    } \
  } while (0)
  /* closing ';' from use as function call */


/** Calculate the stencil of basis function and derivatives of (1/h)Phi.
 * dphi - stencil array (up to size MAX_POLY_DEGREE+1)
 * phi - stencil array (up to size MAX_POLY_DEGREE+1)
 * h_1 - 1/h, h is the grid spacing
 * delta - normalized distance of atom from lowest grid point in stencil
 * approx - APPROX enum value from msm.h
 */
#define D_STENCIL_1D(_dphi, _phi, _h_1, _delta, _approx) \
  do { \
    float *dphi = _dphi; \
    float *phi = _phi; \
    float h_1 = _h_1; \
    float t = _delta; \
    switch (_approx) { \
      case APPROX_CUBIC: \
        phi[0] = 0.5f * (1 - t) * (2 - t) * (2 - t); \
        dphi[0] = (1.5f * t - 2) * (2 - t) * h_1; \
        t--; \
        phi[1] = (1 - t) * (1 + t - 1.5f * t * t); \
        dphi[1] = (-5 + 4.5f * t) * t * h_1; \
        t--; \
        phi[2] = (1 + t) * (1 - t - 1.5f * t * t); \
        dphi[2] = (-5 - 4.5f * t) * t * h_1; \
        t--; \
        phi[3] = 0.5f * (1 + t) * (2 + t) * (2 + t); \
        dphi[3] = (1.5f * t + 2) * (2 + t) * h_1; \
        break; \
      case APPROX_QUINTIC: \
        phi[0] = (1.f/24) * (1-t) * (2-t) * (3-t) * (3-t) * (4-t); \
        dphi[0] = ((-1.f/24) * ((3-t) * (3-t) * (14 + t * (-14 + 3*t)) \
              + 2 * (1-t) * (2-t) * (3-t) * (4-t))) * h_1; \
        t--; \
        phi[1] = (1-t) * (2-t) * (3-t) * ((1.f/6) + t * (0.375f - (5.f/24)*t));\
        dphi[1] = (-((1.f/6) + t * (0.375f - (5.f/24)*t)) * (11 + t * (-12 + 3*t))\
            + (1-t) * (2-t) * (3-t) * (0.375f - (5.f/12)*t)) * h_1; \
        t--; \
        phi[2] = (1-t*t) * (2-t) * (0.5f + t * (0.25f - (5.f/12)*t)); \
        dphi[2] = (-(0.5f + t * (0.25f - (5.f/12)*t)) * (1 + t * (4 - 3*t)) \
            + (1-t*t) * (2-t) * (0.25f - (5.f/6)*t)) * h_1; \
        t--; \
        phi[3] = (1-t*t) * (2+t) * (0.5f - t * (0.25f + (5.f/12)*t)); \
        dphi[3] = ((0.5f + t * (-0.25f - (5.f/12)*t)) * (1 + t * (-4 - 3*t)) \
            - (1-t*t) * (2+t) * (0.25f + (5.f/6)*t)) * h_1; \
        t--; \
        phi[4] = (1+t) * (2+t) * (3+t) * ((1.f/6) - t * (0.375f + (5.f/24)*t));\
        dphi[4] = (((1.f/6) + t * (-0.375f - (5.f/24)*t)) * (11 + t * (12 + 3*t)) \
            - (1+t) * (2+t) * (3+t) * (0.375f + (5.f/12)*t)) * h_1; \
        t--; \
        phi[5] = (1.f/24) * (1+t) * (2+t) * (3+t) * (3+t) * (4+t); \
        dphi[5] = ((1.f/24) * ((3+t) * (3+t) * (14 + t * (14 + 3*t)) \
              + 2 * (1+t) * (2+t) * (3+t) * (4+t))) * h_1; \
        break; \
      case APPROX_QUINTIC2: \
        phi[0] = (1.f/24) * (3-t) * (3-t) * (3-t) * (t-2) * (5*t-8); \
        dphi[0] = ((1.f/24) * (3-t) * (3-t) * ((3-t)*(5*t-8) - 3*(t-2)*(5*t-8) \
              + 5*(t-2)*(3-t))) * h_1; \
        t--; \
        phi[1] = (-1.f/24) * (2-t) * (t-1) * (-48+t*(153+t*(-114+t*25))); \
        dphi[1] = ((-1.f/24) * ((2-t)*(-48+t*(153+t*(-114+t*25))) - (t-1)* \
              (-48+t*(153+t*(-114+t*25))) \
              + (2-t)*(t-1)*(153+t*(-228+t*75)))) * h_1; \
        t--; \
        phi[2] = (1.f/12) * (1-t) * (12+t*(12+t*(-3+t*(-38+t*25)))); \
        dphi[2] = ((1.f/12) * (-(12+t*(12+t*(-3+t*(-38+t*25)))) \
              + (1-t)*(12+t*(-6+t*(-114+t*100))))) * h_1; \
        t--; \
        phi[3] = (1.f/12) * (1+t) * (12+t*(-12+t*(-3+t*(38+t*25)))); \
        dphi[3] = ((1.f/12) * ((12+t*(-12+t*(-3+t*(38+t*25)))) \
              + (1+t)*(-12+t*(-6+t*(114+t*100))))) * h_1; \
        t--; \
        phi[4] = (-1.f/24) * (2+t) * (t+1) * (48+t*(153+t*(114+t*25))); \
        dphi[4] = ((-1.f/24) * ((2+t)*(48+t*(153+t*(114+t*25))) + (t+1)* \
              (48+t*(153+t*(114+t*25))) \
              + (2+t)*(t+1)*(153+t*(228+t*75)))) * h_1; \
        t--; \
        phi[5] = (1.f/24) * (3+t) * (3+t) * (3+t) * (t+2) * (5*t+8); \
        dphi[5] = ((1.f/24) * (3+t) * (3+t) * ((3+t)*(5*t+8) + 3*(t+2)*(5*t+8) \
              + 5*(t+2)*(3+t))) * h_1; \
        break; \
      case APPROX_SEPTIC: \
        phi[0] = (-1.f/720)*(t-1)*(t-2)*(t-3)*(t-4)*(t-4)*(t-5)*(t-6); \
        dphi[0] = (-1.f/720)*(t-4)*(-1944+t*(3644+t*(-2512+t*(807 \
                  +t*(-122+t*7))))) * h_1; \
        t--; \
        phi[1] = (1.f/720)*(t-1)*(t-2)*(t-3)*(t-4)*(t-5)*(-6+t*(-20+7*t)); \
        dphi[1] = (1.f/720)*(756+t*(-9940+t*(17724+t*(-12740+t*(4445 \
                    +t*(-750+t*49)))))) * h_1; \
        t--; \
        phi[2] = (-1.f/240)*(t*t-1)*(t-2)*(t-3)*(t-4)*(-10+t*(-12+7*t)); \
        dphi[2] = (-1.f/240)*(-28+t*(1260+t*(-756+t*(-1260+t*(1365 \
                    +t*(-450+t*49)))))) * h_1; \
        t--; \
        phi[3] = (1.f/144)*(t*t-1)*(t*t-4)*(t-3)*(-12+t*(-4+7*t)); \
        dphi[3] = (1.f/144)*t*(-560+t*(84+t*(644+t*(-175+t*(-150+t*49))))) * \
          h_1; \
        t--; \
        phi[4] = (-1.f/144)*(t*t-1)*(t*t-4)*(t+3)*(-12+t*(4+7*t)); \
        dphi[4] = (-1.f/144)*t*(560+t*(84+t*(-644+t*(-175+t*(150+t*49))))) * \
          h_1; \
        t--; \
        phi[5] = (1.f/240)*(t*t-1)*(t+2)*(t+3)*(t+4)*(-10+t*(12+7*t)); \
        dphi[5] = (1.f/240)*(-28+t*(-1260+t*(-756+t*(1260+t*(1365 \
                    +t*(450+t*49)))))) * h_1; \
        t--; \
        phi[6] = (-1.f/720)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)*(-6+t*(20+7*t)); \
        dphi[6] = (-1.f/720)*(756+t*(9940+t*(17724+t*(12740+t*(4445 \
                    +t*(750+t*49)))))) * h_1; \
        t--; \
        phi[7] = (1.f/720)*(t+1)*(t+2)*(t+3)*(t+4)*(t+4)*(t+5)*(t+6); \
        dphi[7] = (1.f/720)*(t+4)*(1944+t*(3644+t*(2512+t*(807 \
                  +t*(122+t*7))))) * h_1; \
        break; \
      case APPROX_SEPTIC3: \
        phi[0] = (3632.f/5) + t*((-7456.f/5) + t*((58786.f/45) + t*(-633 \
                + t*((26383.f/144) + t*((-22807.f/720) + t*((727.f/240) \
                      + t*(-89.f/720))))))); \
        dphi[0] = ((-7456.f/5) + t*((117572.f/45) + t*(-1899 + t*((26383.f/36) \
                  + t*((-22807.f/144) + t*((727.f/40) + t*(-623.f/720)))))))*h_1; \
        t--; \
        phi[1] = -440 + t*((25949.f/20) + t*((-117131.f/72) + t*((2247.f/2) \
                + t*((-66437.f/144) + t*((81109.f/720) + t*((-727.f/48) \
                      + t*(623.f/720))))))); \
        dphi[1] = ((25949.f/20) + t*((-117131.f/36) + t*((6741.f/2) \
                + t*((-66437.f/36) + t*((81109.f/144) + t*((-727.f/8) \
                      + t*(4361.f/720))))))) * h_1; \
        t--; \
        phi[2] = (138.f/5) + t*((-8617.f/60) + t*((12873.f/40) + t*((-791.f/2) \
                + t*((4557.f/16) + t*((-9583.f/80) + t*((2181.f/80) \
                      + t*(-623.f/240))))))); \
        dphi[2] = ((-8617.f/60) + t*((12873.f/20) + t*((-2373.f/2) + t*((4557.f/4) \
                  + t*((-9583.f/16) + t*((6543.f/40) + t*(-4361.f/240)))))))*h_1; \
        t--; \
        phi[3] = 1 + t*t*((-49.f/36) + t*t*((-959.f/144) + t*((2569.f/144) \
                + t*((-727.f/48) + t*(623.f/144))))); \
        dphi[3] = (t*((-49.f/18) + t*t*((-959.f/36) + t*((12845.f/144) \
                  + t*((-727.f/8) + t*(4361.f/144)))))) * h_1; \
        t--; \
        phi[4] = 1 + t*t*((-49.f/36) + t*t*((-959.f/144) + t*((-2569.f/144) \
                + t*((-727.f/48) + t*(-623.f/144))))); \
        dphi[4] = (t*((-49.f/18) + t*t*((-959.f/36) + t*((-12845.f/144) \
                  + t*((-727.f/8) + t*(-4361.f/144)))))) * h_1; \
        t--; \
        phi[5] = (138.f/5) + t*((8617.f/60) + t*((12873.f/40) + t*((791.f/2) \
                + t*((4557.f/16) + t*((9583.f/80) + t*((2181.f/80) \
                      + t*(623.f/240))))))); \
        dphi[5] = ((8617.f/60) + t*((12873.f/20) + t*((2373.f/2) + t*((4557.f/4) \
                  + t*((9583.f/16) + t*((6543.f/40) + t*(4361.f/240)))))))*h_1; \
        t--; \
        phi[6] = -440 + t*((-25949.f/20) + t*((-117131.f/72) + t*((-2247.f/2) \
                + t*((-66437.f/144) + t*((-81109.f/720) + t*((-727.f/48) \
                      + t*(-623.f/720))))))); \
        dphi[6] = ((-25949.f/20) + t*((-117131.f/36) + t*((-6741.f/2) \
                + t*((-66437.f/36) + t*((-81109.f/144) + t*((-727.f/8) \
                      + t*(-4361.f/720))))))) * h_1; \
        t--; \
        phi[7] = (3632.f/5) + t*((7456.f/5) + t*((58786.f/45) + t*(633 \
                + t*((26383.f/144) + t*((22807.f/720) + t*((727.f/240) \
                      + t*(89.f/720))))))); \
        dphi[7] = ((7456.f/5) + t*((117572.f/45) + t*(1899 + t*((26383.f/36) \
                  + t*((22807.f/144) + t*((727.f/40) + t*(623.f/720)))))))*h_1; \
        break; \
      case APPROX_NONIC: \
        phi[0] = (-1.f/40320)*(t-8)*(t-7)*(t-6)*(t-5)*(t-5)*(t-4)*(t-3)* \
          (t-2)*(t-1); \
        dphi[0] = (-1.f/40320)*(t-5)*(-117648+t*(256552+t*(-221416 \
                +t*(99340+t*(-25261+t*(3667+t*(-283+t*9)))))))*h_1; \
        t--; \
        phi[1] = (1.f/40320)*(t-7)*(t-6)*(t-5)*(t-4)*(t-3)*(t-2)*(t-1)* \
          (-8+t*(-35+9*t)); \
        dphi[1] = (1.f/40320)*(71856+t*(-795368+t*(1569240+t*(-1357692 \
                  +t*(634725+t*(-172116+t*(27090+t*(-2296+t*81))))))))*h_1; \
        t--; \
        phi[2] = (-1.f/10080)*(t-6)*(t-5)*(t-4)*(t-3)*(t-2)*(t-1)*(t+1)* \
          (-14+t*(-25+9*t)); \
        dphi[2] = (1.f/10080)*(3384+t*(-69080+t*(55026 \
                +t*(62580+t*(-99225+t*(51660+t*(-13104+t*(1640 \
                          +t*(-81)))))))))*h_1; \
        t--; \
        phi[3] = (1.f/1440)*(t-5)*(t-4)*(t-3)*(t-2)*(t-1)*(t+1)*(t+2)* \
          (-6+t*(-5+3*t)); \
        dphi[3] = (1.f/1440)*(72+t*(-6344+t*(2070 \
                +t*(7644+t*(-4725+t*(-828+t*(1260+t*(-328+t*27))))))))*h_1; \
        t--; \
        phi[4] = (-1.f/2880)*(t-4)*(t-3)*(t-2)*(t-1)*(t+1)*(t+2)*(t+3)* \
          (-20+t*(-5+9*t)); \
        dphi[4] = (-1.f/2880)*t*(10792+t*(-972+t*(-12516 \
                +t*(2205+t*(3924+t*(-882+t*(-328+t*81)))))))*h_1; \
        t--; \
        phi[5] = (1.f/2880)*(t-3)*(t-2)*(t-1)*(t+1)*(t+2)*(t+3)*(t+4)* \
          (-20+t*(5+9*t)); \
        dphi[5] = (1.f/2880)*t*(-10792+t*(-972+t*(12516 \
                +t*(2205+t*(-3924+t*(-882+t*(328+t*81)))))))*h_1; \
        t--; \
        phi[6] = (-1.f/1440)*(t-2)*(t-1)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)* \
          (-6+t*(5+3*t)); \
        dphi[6] = (1.f/1440)*(-72+t*(-6344+t*(-2070 \
                +t*(7644+t*(4725+t*(-828+t*(-1260+t*(-328+t*(-27)))))))))*h_1; \
        t--; \
        phi[7] = (1.f/10080)*(t-1)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)*(t+6)* \
          (-14+t*(25+9*t)); \
        dphi[7] = (1.f/10080)*(-3384+t*(-69080+t*(-55026 \
                +t*(62580+t*(99225+t*(51660+t*(13104+t*(1640+t*81))))))))*h_1; \
        t--; \
        phi[8] = (-1.f/40320)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)*(t+6)*(t+7)* \
          (-8+t*(35+9*t)); \
        dphi[8] = (-1.f/40320)*(71856+t*(795368+t*(1569240 \
                +t*(1357692+t*(634725+t*(172116+t*(27090+t*(2296 \
                          +t*81))))))))*h_1; \
        t--; \
        phi[9] = (1.f/40320)*(t+1)*(t+2)*(t+3)*(t+4)*(t+5)*(t+5)*(t+6)* \
          (t+7)*(t+8); \
        dphi[9] = (1.f/40320)*(t+5)*(117648+t*(256552+t*(221416 \
                +t*(99340+t*(25261+t*(3667+t*(283+t*9)))))))*h_1; \
        break; \
      case APPROX_NONIC4: \
        phi[0] = 439375.f/7+t*(-64188125.f/504+t*(231125375.f/2016 \
              +t*(-17306975.f/288+t*(7761805.f/384+t*(-2895587.f/640 \
                    +t*(129391.f/192+t*(-259715.f/4032+t*(28909.f/8064 \
                          +t*(-3569.f/40320))))))))); \
        dphi[0] = (-64188125.f/504+t*(231125375.f/1008 \
              +t*(-17306975.f/96+t*(7761805.f/96+t*(-2895587.f/128 \
                    +t*(129391.f/32+t*(-259715.f/576+t*(28909.f/1008 \
                          +t*(-3569.f/4480)))))))))*h_1; \
        t--; \
        phi[1] = -56375+t*(8314091.f/56+t*(-49901303.f/288+t*(3763529.f/32 \
                +t*(-19648027.f/384+t*(9469163.f/640+t*(-545977.f/192 \
                      +t*(156927.f/448+t*(-28909.f/1152 \
                          +t*(3569.f/4480))))))))); \
        dphi[1] = (8314091.f/56+t*(-49901303.f/144+t*(11290587.f/32 \
                +t*(-19648027.f/96+t*(9469163.f/128+t*(-545977.f/32 \
                      +t*(156927.f/64+t*(-28909.f/144 \
                          +t*(32121.f/4480)))))))))*h_1; \
        t--; \
        phi[2] = 68776.f/7+t*(-1038011.f/28+t*(31157515.f/504+t*(-956669.f/16 \
                +t*(3548009.f/96+t*(-2422263.f/160+t*(197255.f/48 \
                      +t*(-19959.f/28+t*(144545.f/2016 \
                          +t*(-3569.f/1120))))))))); \
        dphi[2] = (-1038011.f/28+t*(31157515.f/252+t*(-2870007.f/16 \
                +t*(3548009.f/24+t*(-2422263.f/32+t*(197255.f/8 \
                      +t*(-19959.f/4+t*(144545.f/252 \
                          +t*(-32121.f/1120)))))))))*h_1; \
        t--; \
        phi[3] = -154+t*(12757.f/12+t*(-230123.f/72+t*(264481.f/48 \
                +t*(-576499.f/96+t*(686147.f/160+t*(-96277.f/48 \
                      +t*(14221.f/24+t*(-28909.f/288+t*(3569.f/480))))))))); \
        dphi[3] = (12757.f/12+t*(-230123.f/36+t*(264481.f/16 \
                +t*(-576499.f/24+t*(686147.f/32+t*(-96277.f/8 \
                      +t*(99547.f/24+t*(-28909.f/36 \
                          +t*(10707.f/160)))))))))*h_1; \
        t--; \
        phi[4] = 1+t*t*(-205.f/144+t*t*(91.f/192+t*(-6181.f/320 \
                +t*(6337.f/96+t*(-2745.f/32+t*(28909.f/576 \
                      +t*(-3569.f/320))))))); \
        dphi[4] = t*(-205.f/72+t*t*(91.f/48+t*(-6181.f/64 \
                +t*(6337.f/16+t*(-19215.f/32+t*(28909.f/72 \
                      +t*(-32121.f/320)))))))*h_1; \
        t--; \
        phi[5] = 1+t*t*(-205.f/144+t*t*(91.f/192+t*(6181.f/320 \
                +t*(6337.f/96+t*(2745.f/32+t*(28909.f/576 \
                      +t*(3569.f/320))))))); \
        dphi[5] = t*(-205.f/72+t*t*(91.f/48+t*(6181.f/64 \
                +t*(6337.f/16+t*(19215.f/32+t*(28909.f/72 \
                      +t*(32121.f/320)))))))*h_1; \
        t--; \
        phi[6] = -154+t*(-12757.f/12+t*(-230123.f/72+t*(-264481.f/48 \
                +t*(-576499.f/96+t*(-686147.f/160+t*(-96277.f/48 \
                      +t*(-14221.f/24+t*(-28909.f/288+t*(-3569.f/480))))))))); \
        dphi[6] = (-12757.f/12+t*(-230123.f/36+t*(-264481.f/16 \
                +t*(-576499.f/24+t*(-686147.f/32+t*(-96277.f/8 \
                      +t*(-99547.f/24+t*(-28909.f/36 \
                          +t*(-10707.f/160)))))))))*h_1; \
        t--; \
        phi[7] = 68776.f/7+t*(1038011.f/28+t*(31157515.f/504+t*(956669.f/16 \
                +t*(3548009.f/96+t*(2422263.f/160+t*(197255.f/48 \
                      +t*(19959.f/28+t*(144545.f/2016+t*(3569.f/1120))))))))); \
        dphi[7] = (1038011.f/28+t*(31157515.f/252+t*(2870007.f/16 \
                +t*(3548009.f/24+t*(2422263.f/32+t*(197255.f/8 \
                      +t*(19959.f/4+t*(144545.f/252 \
                          +t*(32121.f/1120)))))))))*h_1; \
        t--; \
        phi[8] = -56375+t*(-8314091.f/56+t*(-49901303.f/288+t*(-3763529.f/32 \
                +t*(-19648027.f/384+t*(-9469163.f/640+t*(-545977.f/192 \
                      +t*(-156927.f/448+t*(-28909.f/1152 \
                          +t*(-3569.f/4480))))))))); \
        dphi[8] = (-8314091.f/56+t*(-49901303.f/144+t*(-11290587.f/32 \
                +t*(-19648027.f/96+t*(-9469163.f/128+t*(-545977.f/32 \
                      +t*(-156927.f/64+t*(-28909.f/144 \
                          +t*(-32121.f/4480)))))))))*h_1; \
        t--; \
        phi[9] = 439375.f/7+t*(64188125.f/504+t*(231125375.f/2016 \
              +t*(17306975.f/288+t*(7761805.f/384+t*(2895587.f/640 \
                    +t*(129391.f/192+t*(259715.f/4032+t*(28909.f/8064 \
                          +t*(3569.f/40320))))))))); \
        dphi[9] = (64188125.f/504+t*(231125375.f/1008 \
              +t*(17306975.f/96+t*(7761805.f/96+t*(2895587.f/128 \
                    +t*(129391.f/32+t*(259715.f/576+t*(28909.f/1008 \
                          +t*(3569.f/4480)))))))))*h_1; \
        break; \
      case APPROX_BSPLINE: \
        phi[0] = (1.f/6) * (2-t) * (2-t) * (2-t); \
        dphi[0] = -0.5f * (2-t) * (2-t) * h_1; \
        t--; \
        phi[1] = (2.f/3) + t*t*(-1 + 0.5f*t); \
        dphi[1] = t*(-2 + 1.5f*t) * h_1; \
        t--; \
        phi[2] = (2.f/3) + t*t*(-1 - 0.5f*t); \
        dphi[2] = t*(-2 - 1.5f*t) * h_1; \
        t--; \
        phi[3] = (1.f/6) * (2+t) * (2+t) * (2+t); \
        dphi[3] = 0.5f * (2+t) * (2+t) * h_1; \
        break; \
      default: \
        return /* XXX NL_MSM_ERROR_SUPPORT */; \
    } \
  } while (0)
  /* closing ';' from use as function call */


#endif // MSM_MACROS_H
