/* ------------------------------------------- */
/*
 *   Author: James Phillips
 *           Department of Physics
 *           Marquette University
 *           Milwaukee, WI 53233
 *
 *   Purpose: Explorer module which crops and
 *            expands global dataset lattices.
 *
 *   Written for the David A. Yuen research group
 *   at the Minnesota Supercomputer Institute and
 *   Department of Geology and Geophysics, 
 *   University of Minnesota.  Christmas, 1992.
 *
 */
/* ------------------------------------------- */

#include <cx/DataTypes.h>
#include <cx/DataAccess.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#define MIN(X,Y) ((X)<(Y)?(X):(Y))
#define MAX(X,Y) ((X)>(Y)?(X):(Y))

#ifdef __cplusplus
	extern "C" {
#endif

int Unif_To_Perimeter(int dim, cxCoord **coordParam);

void  swap(long *a, long *b)
{
   long temp;

   temp = *a;
   *a = *b;
   *b = temp;
}

void partition_uniform (
   float xmin,
   float xmax,
   float dxmin,
   float dxmax,
   long dim,
   long *part,
   long *pnparts,
   long *pnewdim,
   float *xMin,
   float *xMax ) {

   long nparts;
   long ndown;
   long newdim;
   long i,j;
   float dxMin, dxMax;
   float step;
   float fpart[100];
   long tpart[100];

   nparts = 0;
   ndown = 0;

   dxMin = MIN(dxmin,dxmax);
   dxMax = MAX(dxmin,dxmax);
   step = (dxMax-dxMin)/(dim-1);

   while (dxMin>xmin) dxMin-=360.0, dxMax-=360.0, ndown++;

   if (dxMax<xmin) dxMin+=360.0, dxMax+=360.0, ndown--;

   fpart[0] = xmin + ndown*360.0;

   while (dxMax<xmax) {
      fpart[2*nparts+1] = dxMax + (ndown-nparts)*360.0;
      dxMax += 360.0; dxMin += 360.0; nparts++;
      fpart[2*nparts] = dxMin + (ndown-nparts)*360.0;
   }

   if (dxMin>xmax) nparts--;

   fpart[2*nparts+1] = xmax + (ndown-nparts)*360.0;

   for(i=0;i<nparts;i++)
      if(fpart[2*i+1]==(fpart[2*i+2]+360.0)) fpart[2*i+1]-=step*0.99;

   dxMin = MIN(dxmin,dxmax);
   dxMax = MAX(dxmin,dxmax);

   for(i=0;i<=nparts;i++) 
   if(dxmin>dxmax) {
      tpart[2*i]=floor((fpart[2*i]-dxMin)/step+0.01);
      tpart[2*i+1]=ceil((fpart[2*i+1]-dxMin)/step-0.01);
   } else {
      tpart[2*i]=ceil((fpart[2*i]-dxMin)/step-0.01);
      tpart[2*i+1]=floor((fpart[2*i+1]-dxMin)/step+0.01);
   }

   for(i=0;i<=nparts;i++)
      if(((dxmax>dxmin)&&(tpart[2*i]>tpart[2*i+1])) ||
         ((dxmax<dxmin)&&(tpart[2*i]<tpart[2*i+1]))) {
         for(j=i;j<nparts;tpart[2*j]=  tpart[2*j+2],
                          tpart[2*j+1]=tpart[2*j+3],j++);
         nparts--;
      }

   for(i=0,newdim=0;i<=nparts;newdim+=abs(tpart[2*i+1]-tpart[2*i])+1,i++);

   if(dxmin>dxmax)
      for(i=0;i<=nparts;i++) {
         part[2*(nparts-i)+1]=(dim-1)-tpart[2*i];
         part[2*(nparts-1)]=  (dim-1)-tpart[2*i+1];
      }
   else
      for(i=0;i<2*(nparts+1);i++)
         part[i]=tpart[i];

   if(dxmin>dxmax) {
      *xMin = dxmin + part[0]*(dxmax-dxmin)/(dim-1) - (ndown-nparts)*360.0;
      *xMax = dxmin + part[2*nparts+1]*(dxmax-dxmin)/(dim-1) - ndown*360.0;
   } else {
      *xMin = dxmin + part[0]*(dxmax-dxmin)/(dim-1) - ndown*360.0;
      *xMax = dxmin + part[2*nparts+1]*(dxmax-dxmin)/(dim-1)
                                           - (ndown-nparts)*360.0;
   }

   *pnparts = nparts + 1;
   *pnewdim = newdim;

   return;

}

int partition_perimeter (
   float xmin,
   float xmax,
   float *coord,
   long dim,
   long *part,
   long *pnparts,
   long *pnewdim,
   float **pnewcoord ) {

   long nparts;
   long ndown;
   long newdim;
   float *newcoord;
   long i,j,k;
   float dxMin, dxMax;
   float step;
   float fpart[100];
   long tpart[100];
   int normal;

   nparts = 0, ndown = 0;

   dxMin = MIN(coord[0],coord[dim-1]);
   dxMax = MAX(coord[0],coord[dim-1]);

   normal = coord[dim-1]>coord[0];

   while (dxMin>xmin) dxMin-=360.0, dxMax-=360.0, ndown++;

   if (dxMax<xmin) dxMin+=360.0, dxMax+=360.0, ndown--;

   fpart[0] = xmin + ndown*360.0;

   while (dxMax<xmax) {
      fpart[2*nparts+1] = dxMax + (ndown-nparts)*360.0;
      dxMax += 360.0; dxMin += 360.0; nparts++;
      fpart[2*nparts] = dxMin + (ndown-nparts)*360.0;
   }

   if (dxMin>xmax) nparts--;

   fpart[2*nparts+1] = xmax + (ndown-nparts)*360.0;

   for(i=0;i<nparts;i++)
      if(fpart[2*i+1]==(fpart[2*i+2]+360.0)) 
         fpart[2*i+1]-=(normal?(coord[dim-1]-coord[dim-2])*0.99:
                               (coord[0]-coord[1])*0.99);

   dxMin = MIN(coord[0],coord[dim-1]);
   dxMax = MAX(coord[0],coord[dim-1]);

   for(i=0;i<=nparts;i++) {
      if(normal) for(j=0;coord[j]<fpart[2*i]&&j<(dim-1);j++);
      else for(j=dim-1;coord[j]<fpart[2*i]&&j>0;j--);
      tpart[2*i]=j;
      if(normal) for(j=dim-1;coord[j]>fpart[2*i+1]&&j>0;j--);
      else for(j=0;coord[j]>fpart[2*i+1]&&j<(dim-1);j++);
      tpart[2*i+1]=j;
   }

   for(i=0;i<=nparts;i++)
      if((normal&&(tpart[2*i]>tpart[2*i+1])) ||
         (!normal&&(tpart[2*i]<tpart[2*i+1]))) {
         for(j=i;j<nparts;tpart[2*j]=  tpart[2*j+2],
                          tpart[2*j+1]=tpart[2*j+3],j++);
         nparts--;
      }

   for(i=0,newdim=0;i<=nparts;newdim+=abs(tpart[2*i+1]-tpart[2*i])+1,i++);

   newcoord = *pnewcoord = (float*)malloc(newdim*sizeof(float));
   if(!newcoord) return 1;

   if(normal) {
      for(i=0;i<=nparts;i++) {
         part[2*i]=tpart[2*i];
         part[2*i+1]=tpart[2*i+1];
         for(j=tpart[2*i],k=0;j<=tpart[2*i+1];j++,k++)
            newcoord[k]=coord[j]-(ndown-i)*360;
         newcoord+=tpart[2*i+1]-tpart[2*i]+1;
      }
   }
   else {
      newcoord+=newdim;
      for(i=nparts;i>=0;i--) {
         newcoord-=tpart[2*i]-tpart[2*i+1]+1;
         part[2*(nparts-i)+1]=tpart[2*i];
         part[2*(nparts-i)]=tpart[2*i+1];
         for(j=tpart[2*i+1],k=0;j<=tpart[2*i];j++,k++)
            newcoord[k]=coord[j]-(ndown-i)*360;
      }
   }

   *pnparts = nparts + 1;
   *pnewdim = newdim;

   return 0;

}

  void 	worldcrop  (
    double 	  xMin, 
    double 	  xMax, 
    double 	  yMin, 
    double 	  yMax, 
    double 	  zMin, 
    double 	  zMax, 
    cxLattice    *    lat1In, 
    cxLattice    *  * lat1Out,
    cxLattice    *    lat2In, 
    cxLattice    *  * lat2Out,
    cxLattice    *    lat3In,
    cxLattice    *  * lat3Out)
{

   long nCoord;
   cxData *inData;
   char *indata;
   char *indata2;
   cxData *outData;
   char *outdata;
   char *od;
   cxCoord *inCoord;
   float *incoord;
   cxCoord *outCoord;
   float *outcoord;
   float *newcoord;
   long m,n,p;
   long *dims;
   long newdim;
   float outxMin,outxMax;
   long nparts;
   long part[100];
   long outdims[3];
   long datasize;
   long nDim, nDataVar, nCoordVar;
   cxPrimType primType;
   cxCoordType coordType;
   int i,j,k;
   int iMin,iMax,jMin,jMax,kMin,kMax;
   int imin,imax,jmin,jmax,kmin,kmax;
   float *xcoord,*ycoord,*zcoord;
   float step;
   int killcoord;

   cxLattice *inLats[3];
   cxLattice *outLats[3];
   int pchanged;

   *lat1Out = NULL;
   *lat2Out = NULL;
   *lat3Out = NULL;

   if ( xMin > xMax || yMin > yMax || zMin > zMax ) return;

   pchanged = (
      cxInputDataChanged(cxInputPortOpen("x Min")) ||
      cxInputDataChanged(cxInputPortOpen("x Max")) ||
      cxInputDataChanged(cxInputPortOpen("y Min")) ||
      cxInputDataChanged(cxInputPortOpen("y Max")) ||
      cxInputDataChanged(cxInputPortOpen("z Min")) ||
      cxInputDataChanged(cxInputPortOpen("z Max")) );

   if ( cxInputDataChanged(cxInputPortOpen("Lattice 1")) || pchanged )
      inLats[0] = lat1In; else inLats[0] = NULL;
   if ( cxInputDataChanged(cxInputPortOpen("Lattice 2")) || pchanged )
      inLats[1] = lat2In; else inLats[1] = NULL;
   if ( cxInputDataChanged(cxInputPortOpen("Lattice 3")) || pchanged )
      inLats[2] = lat3In; else inLats[2] = NULL;

   /* Copy lattices. */

   for(n=0;n<3;n++) if(inLats[n]) {

      cxLatPtrGet(inLats[n],&inData,&indata,&inCoord,&incoord);
      cxLatDescGet(inLats[n],&nDim,&dims,
                   NULL,&nDataVar,&primType,
                   NULL,&nCoordVar,&coordType);
      datasize = nDataVar * cxDataPrimSize(inData);
      killcoord = 0;

      if ( coordType == cx_coord_curvilinear || nCoordVar != nDim ) return;
      if ( coordType == cx_coord_uniform ) {
         step = (incoord[1]-incoord[0])/(dims[0]-1);
         if ( !( incoord[1] == incoord[0] + 360.0 ||
                 incoord[1] + step == incoord[0] + 360.0 ||
                 incoord[1] == incoord[0] - 360.0 ||
                 incoord[1] + step == incoord[0] - 360.0 
                 ) ) {
            Unif_To_Perimeter(nDim, &inCoord);
            if(cxDataAllocErrorGet()) {
               for(j=0;j<n;j++) if(outLats[j]) cxDataRefDec(outLats[j]);
               return;
            } 
            incoord = cxCoordValsGet(inCoord);
            coordType = cx_coord_perimeter;
            killcoord = 1;
         }
      }
      if ( coordType == cx_coord_uniform && nDim == 2 ) {
         partition_uniform(xMin,xMax,incoord[0],incoord[1],dims[0],
                           part,&nparts,&newdim,&outxMin,&outxMax);
         jmin = (dims[1]-1)*(yMin-incoord[2])/(incoord[3]-incoord[2]);
         jmax = (dims[1]-1)*(yMax-incoord[2])/(incoord[3]-incoord[2]);
         if(jmin>jmax) {
         jMin = jmax = MIN(MAX(jmax,0),dims[1]-1);
         jMax = jmin = MIN(MAX(jmin,0),dims[1]-1);
         } else {
         jMax = jmax = MIN(MAX(jmax,0),dims[1]-1);
         jMin = jmin = MIN(MAX(jmin,0),dims[1]-1);
         }
         outdims[0] = newdim;
         outdims[1] = jMax - jMin + 1;
         outLats[n] = cxLatNew(nDim, outdims, nDataVar, primType,
                               nCoordVar, coordType);
         if(cxDataAllocErrorGet()) {
            for(j=0;j<n;j++) if(outLats[j]) cxDataRefDec(outLats[j]);
            return;
         } 
         cxLatPtrGet(outLats[n],&outData,&outdata,&outCoord,&outcoord);
         for(j=jMin,indata+=jMin*dims[0]*datasize;j<=jMax;
             j++,indata+=dims[0]*datasize)
            for(i=0;i<nparts;
                outdata+=(part[2*i+1]-part[2*i]+1)*datasize,i++)
               memcpy(outdata,indata+part[2*i]*datasize,
                 (part[2*i+1]-part[2*i]+1)*datasize);
         outcoord[0] = outxMin;
         outcoord[1] = outxMax;
         outcoord[2] = incoord[2] + jMin*(incoord[3]-incoord[2])/(dims[1]-1);
         outcoord[3] = incoord[2] + jMax*(incoord[3]-incoord[2])/(dims[1]-1);
      }
      if ( coordType == cx_coord_uniform && nDim == 3 ) {
         partition_uniform(xMin,xMax,incoord[0],incoord[1],dims[0],
                           part,&nparts,&newdim,&outxMin,&outxMax);
         jmin = (dims[1]-1)*(yMin-incoord[2])/(incoord[3]-incoord[2]);
         jmax = (dims[1]-1)*(yMax-incoord[2])/(incoord[3]-incoord[2]);
         kmin = (dims[2]-1)*(zMin-incoord[4])/(incoord[5]-incoord[4]);
         kmax = (dims[2]-1)*(zMax-incoord[4])/(incoord[5]-incoord[4]);
         if(jmin>jmax) {
         jMin = jmax = MIN(MAX(jmax,0),dims[1]-1);
         jMax = jmin = MIN(MAX(jmin,0),dims[1]-1);
         } else {
         jMax = jmax = MIN(MAX(jmax,0),dims[1]-1);
         jMin = jmin = MIN(MAX(jmin,0),dims[1]-1);
         }
         if(kmin>kmax) {
         kMin = kmax = MIN(MAX(kmax,0),dims[2]-1);
         kMax = kmin = MIN(MAX(kmin,0),dims[2]-1);
         } else {
         kMax = kmax = MIN(MAX(kmax,0),dims[2]-1);
         kMin = kmin = MIN(MAX(kmin,0),dims[2]-1);
         }
         outdims[0] = newdim;
         outdims[1] = jMax - jMin + 1;
         outdims[2] = kMax - kMin + 1;
         outLats[n] = cxLatNew(nDim, outdims, nDataVar, primType,
                               nCoordVar, coordType);
         if(cxDataAllocErrorGet()) {
            for(j=0;j<n;j++) if(outLats[j]) cxDataRefDec(outLats[j]);
            return;
         } 
         cxLatPtrGet(outLats[n],&outData,&outdata,&outCoord,&outcoord);
         for(k=kMin,indata+=kMin*dims[1]*dims[0]*datasize;k<=kMax;
             k++,indata+=dims[1]*dims[0]*datasize)
            for(j=jMin,indata2=indata+jMin*dims[0]*datasize;j<=jMax;
                j++,indata2+=dims[0]*datasize)
               for(i=0;i<nparts;
                   outdata+=(part[2*i+1]-part[2*i]+1)*datasize,i++)
                  memcpy(outdata,indata2+part[2*i]*datasize,
                    (part[2*i+1]-part[2*i]+1)*datasize);
         outcoord[0] = outxMin;
         outcoord[1] = outxMax;
         outcoord[2] = incoord[2] + jMin*(incoord[3]-incoord[2])/(dims[1]-1);
         outcoord[3] = incoord[2] + jMax*(incoord[3]-incoord[2])/(dims[1]-1);
         outcoord[4] = incoord[4] + kMin*(incoord[5]-incoord[4])/(dims[2]-1);
         outcoord[5] = incoord[4] + kMax*(incoord[5]-incoord[4])/(dims[2]-1);
      }
      if ( coordType == cx_coord_perimeter && nDim == 2 ) {
         xcoord = incoord;
         ycoord = xcoord + dims[0];
         if(partition_perimeter(xMin,xMax,xcoord,dims[0],
                           part,&nparts,&newdim,&newcoord )) {
            for(j=0;j<n;j++) if(outLats[j]) cxDataRefDec(outLats[j]);
            return;
         }
         for(jMin=j=0;j<dims[1];
            jMin=(fabs(ycoord[jMin]-yMin)<fabs(ycoord[j]-yMin)?jMin:j),j++);
         for(jMax=j=0;j<dims[1];
            jMax=(fabs(ycoord[jMax]-yMax)<fabs(ycoord[j]-yMax)?jMax:j),j++);
         if ( jMin > jMax ) swap(&jMin,&jMax);
         outdims[0] = newdim;
         outdims[1] = jMax - jMin + 1;
         outLats[n] = cxLatNew(nDim, outdims, nDataVar, primType,
                               nCoordVar, coordType);
         if(cxDataAllocErrorGet()) {
            for(j=0;j<n;j++) if(outLats[j]) cxDataRefDec(outLats[j]);
            if ( killcoord ) cxDataRefDec(inCoord);
            return;
         } 
         cxLatPtrGet(outLats[n],&outData,&outdata,&outCoord,&outcoord);
         for(j=jMin,indata+=jMin*dims[0]*datasize;j<=jMax;
             j++,indata+=dims[0]*datasize)
            for(i=0;i<nparts;
                outdata+=(part[2*i+1]-part[2*i]+1)*datasize,i++)
               memcpy(outdata,indata+part[2*i]*datasize,
                 (part[2*i+1]-part[2*i]+1)*datasize);
         memcpy(outcoord,newcoord,newdim*sizeof(float));
         free(newcoord);
         for(j=0,m=newdim;j+jMin<=jMax;outcoord[m]=ycoord[j+jMin],j++,m++);
      }
      if ( coordType == cx_coord_perimeter && nDim == 3 ) {
         xcoord = incoord;
         ycoord = xcoord + dims[0];
         zcoord = ycoord + dims[1];
         if(partition_perimeter(xMin,xMax,xcoord,dims[0],
                           part,&nparts,&newdim,&newcoord )) {
            for(j=0;j<n;j++) if(outLats[j]) cxDataRefDec(outLats[j]);
            return;
         }
         for(jMin=j=0;j<dims[1];
            jMin=(fabs(ycoord[jMin]-yMin)<fabs(ycoord[j]-yMin)?jMin:j),j++);
         for(jMax=j=0;j<dims[1];
            jMax=(fabs(ycoord[jMax]-yMax)<fabs(ycoord[j]-yMax)?jMax:j),j++);
         for(kMin=j=0;j<dims[2];
            kMin=(fabs(zcoord[kMin]-zMin)<fabs(zcoord[j]-zMin)?kMin:j),j++);
         for(kMax=j=0;j<dims[2];
            kMax=(fabs(zcoord[kMax]-zMax)<fabs(zcoord[j]-zMax)?kMax:j),j++);
         if ( jMin > jMax ) swap(&jMin,&jMax);
         if ( kMin > kMax ) swap(&kMin,&kMax);
         outdims[0] = newdim;
         outdims[1] = jMax - jMin + 1;
         outdims[2] = kMax - kMin + 1;
         outLats[n] = cxLatNew(nDim, outdims, nDataVar, primType,
                               nCoordVar, coordType);
         if(cxDataAllocErrorGet()) {
            for(j=0;j<n;j++) if(outLats[j]) cxDataRefDec(outLats[j]);
            if ( killcoord ) cxDataRefDec(inCoord);
            return;
         } 
         cxLatPtrGet(outLats[n],&outData,&outdata,&outCoord,&outcoord);
         for(k=kMin,indata+=kMin*dims[1]*dims[0]*datasize;k<=kMax;
             k++,indata+=dims[1]*dims[0]*datasize)
            for(j=jMin,indata2=indata+jMin*dims[0]*datasize;j<=jMax;
                j++,indata2+=dims[0]*datasize)
               for(i=0;i<nparts;
                   outdata+=(part[2*i+1]-part[2*i]+1)*datasize,i++)
                  memcpy(outdata,indata2+part[2*i]*datasize,
                    (part[2*i+1]-part[2*i]+1)*datasize);
         memcpy(outcoord,newcoord,newdim*sizeof(float));
         free(newcoord);
         for(j=0,m=newdim;j+jMin<=jMax;outcoord[m]=ycoord[j+jMin],j++,m++);
         for(j=0;j+kMin<=kMax;outcoord[m]=zcoord[j+kMin],j++,m++);
      }
      if ( killcoord ) cxDataRefDec(inCoord);

   } else outLats[n] = NULL;

   *lat1Out = outLats[0];
   *lat2Out = outLats[1];
   *lat3Out = outLats[2];

   return;

}

int Unif_To_Perimeter(int dim, cxCoord **coordParam)
{

   cxCoord *coordIn;
   cxCoord *coordOut;
   float *cIn;
   float *cOut;
   int i,j;

   coordIn = *coordParam;

   coordOut = cxCoordNew(dim,coordIn->dims,dim,cx_coord_perimeter);
   if(cxDataAllocErrorGet()) return 1;
   cIn = cxCoordValsGet(coordIn);
   cOut = cxCoordValsGet(coordOut);

   for(i=0;i<dim;i++) {
      for(j=0;j<coordIn->dims[i];j++) {
         cOut[0] = (cIn[1]-cIn[0])*((float)j)/
               ((float)(coordIn->dims[i]-1)) + cIn[0];
         cOut++;
      }
      cIn += 2;
   }

   *coordParam = coordOut;
   return 0;

}

#ifdef __cplusplus
}
#endif
/* ------------------------------------------- */
