############################################################################
#cr                                                                       
#cr            (C) Copyright 1995-2003 The Board of Trustees of the            
#cr                        University of Illinois                         
#cr                         All Rights Reserved                           
#cr                                                                       
############################################################################

############################################################################
# RCS INFORMATION:
#
# 	$RCSfile: vmdinit.tcl,v $
# 	$Author: johns $	$Locker:  $		$State: Exp $
#	$Revision: 1.71 $	$Date: 2003/06/09 22:49:36 $
#
############################################################################
# DESCRIPTION:
#   install the "core" vmd/tcl procedures and variables
#
############################################################################

# This is part of the VMD installation.
# For more information about VMD, see http://www.ks.uiuc.edu/Research/vmd

proc lassign { listvar args } {
  set i 0
  foreach var $args {
    upvar $var x
    set x [lindex $listvar $i] 
    incr i
  }
  return [lrange $listvar $i end]
}

proc lvarpush { listvar var } {
  upvar $listvar x
  set x [list $var $x]
}

proc lvarpop { var } {
  upvar $var x
  set retval [lindex $x 0]
  set x [lrange $x 1 end]
  return $retval
}

# This proc exists for backwards compatibility
proc vmd_calculate_structure { molid } {
  mol ssrecalc $molid
}

# rename the Tcl commands that VMD overrides
# VMD 1.1 and Tk have some conflicts in their names.  The VMD
# commands with this problem are: label, menu and scale
# Luckily, the 1st word in all these for Tk require a .path.name,
# and VMD does not, so we can write a wrapper to distinguish them.
if {[info commands label] == "label"} {
    rename label vmd_tk_label
}
proc label {args} {
    lassign $args arg1
    if {[string first . $arg1] >= 0} {
	return [eval "vmd_tk_label $args"]
    } else {
	return [eval "vmd_label $args"]
    }
}
if {[info commands menu] == "menu"} {
    rename menu vmd_tk_menu
}
proc menu {args} {
    lassign $args arg1
    if {[string first . $arg1] >= 0} {
	return [eval "vmd_tk_menu $args"]
    } else {
	return [eval "vmd_menu $args"]
    }
}
if {[info commands scale] == "scale"} {
    rename scale vmd_tk_scale
}
proc scale {args} {
    lassign $args arg1
    if {[string first . $arg1] >= 0} {
	return [eval "vmd_tk_scale $args"]
    } else {
	return [eval "vmd_scale $args"]
    }
}

proc help {args} {
    return [eval "vmd_help $args"]
}

# default directory locations
if {![info exists env(VMDDIR)]} {
    set env(VMDDIR) /usr/local/lib/vmd
}
if {![info exists env(TMPDIR)]} {
    set env(TMPDIR) /usr/tmp
}

# Define the default HTML viewer, if it hasn't been defined already; 
# this is used by openURL.tcl
if { ![info exists env(VMDHTMLVIEWER)] } {
  switch [vmdinfo arch] {
    WIN32 {
      set env(VMDHTMLVIEWER) "explorer %s"  
    }
    MACOSX {
      set env(VMDHTMLVIEWER) "/usr/bin/open %s"
    }
    default {
      # If you wish, uncomment one of the other netscape options
      # for different window behavior.
      # Open in the current window (default)
      #set env(VMDHTMLVIEWER) "netscape -remote openURL(%s)" 
      # Open in a new window
      set env(VMDHTMLVIEWER) "netscape -remote openURL(%s,new-window)" 
      # Open a new instance of netscape
      # set env(VMDHTMLVIEWER) "netscape %s" 
    }
  }
}


# set the autoloading functions
lappend auto_path $env(VMDDIR)/scripts/vmd

# initialize the vmd_mouse_mode and vmd_mouse_submode variables
set vmd_mouse_mode rotate
set vmd_mouse_submode -1

# some things are NOT autoloaded since that overrides the 
# definitions overridden (for performance reasons) in C++

source $env(VMDDIR)/scripts/vmd/vectors.tcl
source $env(VMDDIR)/scripts/vmd/atomselect.tcl
source $env(VMDDIR)/scripts/vmd/atomselmacros.tcl
source $env(VMDDIR)/scripts/vmd/graphlabels.tcl

# whereas this is needed to set the hotkeys
source $env(VMDDIR)/scripts/vmd/hotkeys.tcl

#  everything else is safe to autoload

# the index is made with:
# auto_mkindex . \
#     {{draw,save_state,openURL,www}.tcl}
#
# The following should not be indexed:
#  {atomselect,hotkeys,vectors,vmdinit}.tcl
#

# Running Tk
if [info exists tk_version] {
    # don't use the toplevel menu
    wm withdraw .
    # but give it a name, just in case
    wm title . "VMD Tk window"
}

# Load a file over http or ftp.  Used by "mol urlload"
proc vmd_mol_urlload {url filename} {
  if { [string equal [string range $url 0 6] http:// ] } {
     vmdhttpcopy $url $filename
  } elseif { [string equal [string range $url 0 5] ftp:// ] } {
     package require ftp
     set base [string range $url 0 5]
     set rest [string range $url 6 end]
     set parts [split $rest /]
     set nparts [llength $parts]
     set site [lindex $parts 0]
     set path [join [lrange $parts 1 [expr $nparts - 2]] /]
     set fname [lindex $parts end]

     set handle [ftp::Open $site ftp ftp]
     ftp::Type $handle binary
     ftp::Cd $handle $path
     ftp::Get $handle $fname $filename
     ftp::Close $handle
   }
   return $filename
}


#######################################
# This proc allows Tk windows to integrate with the rest of VMD's GUI.
# See the VMDTkMenu class for details.  

proc vmd_tkmenu_cb { win op args } {
  switch $op {
    create {
        lassign $args name
        wm protocol $win WM_DELETE_WINDOW "menu $name off"
        # These are used to tell VMD about the status of extension menus,
        # so that "menu $name status" gets the correct answer, but it causes
        # bad things to happen on OS X due to spurious events being generated
        # by Tk.  We should be able to get rid of these bindings and make
        # "menu status" work by a different mechanism; for now it's not 
        # critical that "menu $name status" always give the right answer.
        #bind $win <Map> "+menu $name on"
        #bind $win <Unmap> "+menu $name off"

	# For some window managers (KDE's, anyway), the Tk menu doesn't 
	# remember its screen position when withdraw until after the first
	# call to wm geometry.  So, we do it here.
	# Don't do this on the Mac, though, or the menus will end
	# up under the apple menubar.
	if { [string compare [vmdinfo arch] MACOSX] } {
	  wm geometry $win +[winfo x $win]+[winfo y $win]
	}
      }
    on { wm deiconify $win }
    off { wm withdraw $win }
    loc { return [list [winfo x $win] [winfo y $win]] }
    move {
        lassign $args x y
        wm geometry $win +$x+$y
    }
    remove { wm protocol $win WM_DELETE_WINDOW "" }
  }
  return
}


#######################################
# add a pause command
proc pause {} { 
    gets stdin 
}

#######################################
# Add the RMSD fit routines
proc vmdrmsd { args } {
  global env
  source [file join $env(VMDDIR) scripts vmd rmsd.tcl]
  return
}


#######################################
# Add the plugins directories (plugins/$ARCH/tcl and plugins/noarch/tcl) to
# the Tcl package search path.
lappend auto_path [file join $env(VMDDIR) plugins [vmdinfo arch] tcl]
lappend auto_path [file join $env(VMDDIR) plugins noarch tcl]

# Add plugin directories (polugins/$ARCH/python and plugins/noarch/python) to
# the Python package search path.  The path list is semicolon-delimited on
# Windows, and colon-delimited everywhere else.
# Put everything a function that we'll delete later
proc tmpfunc {} {
  global env
  global tcl_platform
  set delim :
  if { [string match $tcl_platform(platform) windows] } {
    set delim {;}
  }
  set archpath [file join $env(VMDDIR) plugins [vmdinfo arch] python]
  set noarchpath [file join $env(VMDDIR) plugins noarch python]
  if { ! [info exists env(PYTHONPATH)] } {
    set env(PYTHONPATH) $archpath
  } else {
    append env(PYTHONPATH) $delim $archpath
  }
  append env(PYTHONPATH) $delim $noarchpath
}
tmpfunc
rename tmpfunc ""

# define a convenience function for scanning plugins
proc vmd_plugin_scandirectory { dir pattern } {
  # check that the directory exists
  if { ! [file isdirectory $dir] } {
    puts "Plugin directory '$dir'\ndoes not exist!"
    return
  }
  set num 0
  foreach lib [glob -directory $dir -nocomplain $pattern] {
    if { [catch {plugin dlopen $lib} result] } {
      puts "Unable to dlopen '$lib': \n$result"
    } else {
      incr num $result
    }
  }
  if { $num } {
    plugin update
    puts "Found $num plugins or data handlers in directory\n\t'$dir'."
  }
  return
}

# Load the molecule file reader plugins
vmd_plugin_scandirectory [file join $env(VMDDIR) plugins [vmdinfo arch] molfile] *.so

#######################################
# Make Fltk the default file chooser, except for Windows, where the Tk
# file chooser is more native-like.
if ![info exists env(VMDFILECHOOSER)] { 
  switch [vmdinfo arch] {
    MACOSX -
    WIN32 {
      set env(VMDFILECHOOSER) TK 
    }
    default {
      set env(VMDFILECHOOSER) FLTK 
    }
  }
}

#######################################
# BioCoRE features
# Force load of BioCoRE startup code
source $env(VMDDIR)/scripts/vmd/biocore.tcl

# Check for BioCoRE startup
check_biocore

# if tk is present, load in the tcl commands for the Sequence window
# and draw the windows (in withdrawn state).  We have to load it here,
# and not in the vmd_load_extension_packages proc, because Sequence
# goes and gets secondary structure whenever it finds any molecules are
# available, even if the user has never opened the window or selected
# a molecule.  This needs to be fixed so that zoomseq can be loaded
# along with the other packages.
set rc [catch { \
  if [info exists tk_version] { \
    package require zoomseq
  } \
} msg]
if { $rc } {
  puts "The VMD sequence viewer could not be initialized properly:\n$msg"
}



# These packages create new menu items in the VMD menubar, and therefore
# should be loaded automatically by VMD.  They should, however, not be
# loaded until after the .vmdrc file is read, so that users (and VMD
# maintainers) can customize the Tcl search paths and insert their own 
# packages.
proc vmd_load_extension_packages {} {
global tk_version
if { [info exists tk_version] } {

  switch [vmdinfo arch] {
    AIX4 -
    IRIX6_64 -
    WIN32 {
      puts "The AutoIMD package is not supported by this platform\n"
    }
    default {
      if { [catch "package require autoimd" msg] } {
        puts "AutoIMD package could not be loaded:\n$msg"
      } elseif { [catch {menu tk register autoimd autoimd_menu_tk_register} msg] } {
        puts "AutoIMD Control Panel could not be started:\n$msg"
      } 
    }
  }
  
  if { [catch "package require imdmenu" msg] } {
    puts "IMD menu package could not be loaded:\n$msg"
  } elseif { [catch {menu tk register imd imd_menu_tk_register} msg] } { 
    puts "IMD menu could not be started:\n$msg"
  }
  
  if { [catch "package require contactmap" msg] } {
    puts "Contact Map package could not be loaded:\n$msg"
  } elseif { [catch {menu tk register contactmap contactmap} msg] } { 
    puts "Contact Map could not be started:\n$msg"
  }
  
  if { [catch "package require ramaplot" msg] } {
    puts "RamaPlot package could not be loaded:\n$msg"
  } elseif { [catch ramaplot msg] } { 
    puts "RamaPlot could not be started:\n$msg"
  } else {
    menu ramaplot off
  }
  
  if { [catch "package require solvate" msg] } {
    puts "Solvate package could not be loaded:\n$msg"
  } elseif { [catch {menu tk register solvate solvategui} msg] } {
    puts "Solvate GUI package could not be initialized properly:\n$msg."
  }
  
  if { [catch "package require timeline" msg] } {
    puts "Timeline package could not be loaded:\n$msg"
  } elseif { [catch {menu tk register timeline timeline} msg] } { 
    puts "Timeline package could not be started:\n$msg"
  }
  
  if { [catch {package require vmdtkcon} msg] } {
    puts "VMD TkCon package could not be loaded:\n$msg"
  } elseif { [catch {menu tk register "tkcon" vmdtkcon} msg] } { 
    puts "VMD TkCon could not be started:\n$msg"
  }

  if { [catch "package require vmdmovie" msg] } {
    puts "VMD Movie Generator package could not be loaded:\n$msg"
  } elseif { [catch {menu tk register vmdmovie vmdmovie_tk_cb}] } {
    puts "VMD Movie Generator could not be initialized properly."
  } else {
  }

}
} 
