# to make the package index, run this:
# pkg_mkIndex /home/pgrayson/src/plugins/namdrun/ *.tcl

package provide namdrun 0.1


namespace eval NAMDRun {
  variable dqs_queue_specifier ""
  variable queuingSystem ""
  variable namd_name "/usr/local/bin/namd2"
}


# Tells NAMDRun to use a particular DQS queue
proc NAMDRun::useDQSDestination { destination_identifier_list } {
  variable dqs_queue_specifier
  set dqs_queue_specifier "-q $destination_identifier_list"
}


# Tells NAMDRun to use a particular DQS resource list
proc NAMDRun::useDQSResourceList { resource_list } {
  variable dqs_queue_specifier
  set dqs_queue_specifier "-l $resource_list"
}


# Tells NAMDRun the name of the namd program; eg "/usr/local/bin/namd2"
# This will be ignored by any queuing system (eg namd plugin) that
# doesn't need it. 
proc NAMDRun::useNAMDName { name } {
  variable namd_name
  set namd_name $name
}


# Submits a NAMD job to the chosen queuing system.
# All file names are given relative to <simdir> - the directory in
# which NAMD will be run.
#  <namd_opts> - special options to pass to NAMD
#  <jobname>   - the name of the job
#  <confname>  - the  name of the configuration file
#  <logname>   - the  name of the logfile
# Returns a jobspec, which can be passed to "abortJob"
proc NAMDRun::submitJob { simdir namd_opts jobname confname logname } {
  variable queuingSystem
    
  switch -exact $queuingSystem {
    "DQS" { 
      set jobspec [submitJobDQS $simdir $namd_opts $jobname $confname $logname]
    }
    "local" {
      set jobspec [submitJobLocal $simdir $namd_opts $jobname $confname $logname]
    }
    "none" {
      set jobspec "none 0"
      # Don't submit anything
    }
    default { 
      error "Unknown queuing system: \"$queuingSystem\"." 
    }
  }
    
  puts "submitted job: $jobspec"
  return "$jobspec"
}


proc NAMDRun::submitJobLocal { simdir namd_opts jobname confname logname } {
  variable namd_name
    
  set olddir [pwd]
    
  set ret [catch {
    cd $simdir
    exec rm -f $logname
    set procid [exec $namd_name $namd_opts $confname >& $logname &]
  } var]

  # be sure to return to the old directory before passing on any error
  cd $olddir

  if {$ret} { error $var }

  return "local $procid"
}


proc NAMDRun::submitJobDQS { simdir namd_opts jobname confname logname } {
  variable dqs_queue_specifier
  variable namd_name

  if { [llength $dqs_queue_specifier] == 0 } {
    error "No DQS queue has been specified for NAMDRun."
  }

  exec rm -f "$simdir/$logname";
  set outputtext [eval "exec echo \"cd $simdir; $namd_name $namd_opts $confname\" | qsub -N $jobname $dqs_queue_specifier -o $simdir/$logname -j y"]
    
  scan "$outputtext" "your job %d has been submitted" procid  
  return "DQS $procid"
}



# Specifies what queuing system to use to run NAMD.
# Possible values: DQS
proc NAMDRun::useQueuingSystem { system } {
  variable queuingSystem
  
  switch -exact $system {
    "none" { set queuingSystem "none" }
    "local" { set queuingSystem "local" }
    "DQS" { set queuingSystem "DQS" }
    default { error "Unknown queuing system: \"$system\"." }
  }
}


# This aborts a running job using kill, qdel, etc.
proc NAMDRun::abortJob { jobspec } {
  set queue [lindex $jobspec 0]
  set procid [lindex $jobspec 1]
  
  switch -exact $queue { 
    "DQS" { catch {exec qdel $procid & } }
    "local" { catch {exec kill $procid } }
    "none" { ;# Don't do anything }
    default { error "Unknown queuing system: \"$queue\"." }
  }
}
