# to make the package index, run this:
# pkg_mkIndex /home/pgrayson/src/autoimd/ *.tcl

# Author: Paul Grayson
# Maintainer: Jordi Cohen

# This package requires a VMD version >= 1.8a29
# Call "autoimd" to create the AutoIMD window

package require namdrun
package require psfgen
package provide autoimd 1.3


namespace eval AutoIMD {
  variable BASEDIR [file join "$env(VMDDIR)" plugins noarch tcl autoimd1.3]
    
  # Define variables
  variable hosts
		
  set  backgroundmol  -1
  set  imdmol         -1   ;# molecule ID of the live IMD simulation  
  set  startmol       top  ;# molecule ID for the initial coordinates 
  set  scratchdir     ""
  set  moltenseltext  ""
  set  fixedseltext   "within 8 of imdmolten"
  variable namdoutputrate      ;# fraction of steps sent by NAMD
  
  set  imdstate       "uninitialized"
  variable hostname
  variable imdport
  variable imdcurrentport
  variable imdjobspec
  variable parfiles
  variable presubmitscript
  
  set  nprocessors    1
  set  namd_opts      "+netpoll"
  variable psffile
  variable pdbfile

  set imdrep {
    mol representation Bonds 0.300000 6.000000
    mol color Name
    mol selection "imdbonds and noh"
    mol material Opaque
    mol addrep $imdmol

    mol representation VDW 1.000000 8.000000
    mol color ColorID 4
    mol selection "imdions"
    mol material Opaque
    mol addrep $imdmol

    mol representation Bonds 0.300000 6.000000
    mol color Name
    mol selection "imdwater"
    mol material Opaque
    mol addrep $imdmol
  }
  variable currentserversettings
  variable serversettings
#  set serversettings(XXX) {} ;#temporary fix for last minute glitch before release

  variable maxprocs
  variable connectiontimeout  ;# seconds till stops attempting to connect
  
  set  sim_mode        "simulate" ;# allows different types of NAMD jobs
  set  minimizesteps   100        ;# no. of timesteps to minimize
  set  runsteps        1000000    ;# no. of timesteps to run
  set  temperature     310        ;# T used for the sim
          
  set  namdtmplfile [file join "$BASEDIR" template.namd]
  
  variable statustext
  variable w
}


# the GUI
proc AutoIMD::startGUI { } {
  variable currentserversettings
  variable w
  global env
  
  # make the initial window
  set w [toplevel ".autoimd"]
  wm title $w "AutoIMD Controls"
  wm resizable $w no no

  #	initialize "currentserversettings" variable:
  if ![info exists currentserversettings] {
    set searchid [array startsearch AutoIMD::serversettings]
    set currentserversettings [array nextelement AutoIMD::serversettings $searchid]  
    array donesearch AutoIMD::serversettings $searchid
  }
   
  set textfont [font create -family helvetica -size 10]
  set labelfont [font create -family helvetica -size 10]
  set buttonfont [font create -family helvetica -weight bold -size 10]
  set labelwidth 9
    
  ##############################
  # Create menubar
  
  frame $w.menubar -relief raised -bd 2
  pack $w.menubar -fill x -side top
  
  menubutton $w.menubar.file -text "File" -menu $w.menubar.file.menu
  menubutton $w.menubar.settings -text "Settings" -menu $w.menubar.settings.menu
  pack $w.menubar.file $w.menubar.settings -side left
  
  # File menu
  menu $w.menubar.file.menu -tearoff no
  $w.menubar.file.menu add command -label "Save Full PDB As..." -command "AutoIMD::dialog_savepdb"
  $w.menubar.file.menu add separator
  $w.menubar.file.menu add command -label "Show Simulation Controller" -command "menu imd on"
  $w.menubar.file.menu add command -label "Show Tool Controller" -command "menu tool on"
    $w.menubar.file.menu add separator
  $w.menubar.file.menu add command -label "Close Window" -command "wm withdraw $w"

  # Settings menu
  menu $w.menubar.settings.menu -tearoff no
  $w.menubar.settings.menu add command -label "Simulation Parameters..." -command "AutoIMD::dialog_simsettings"
  $w.menubar.settings.menu add separator
  $w.menubar.settings.menu add radiobutton -label "Minimization Mode" -variable AutoIMD::sim_mode -value "minimize"
  $w.menubar.settings.menu add radiobutton -label "Simulation Mode" -variable AutoIMD::sim_mode -value "simulate"
  
  # Help menu
  menubutton $w.menubar.help -text "Help" -menu $w.menubar.help.menu
  pack $w.menubar.help -side right 
  menu $w.menubar.help.menu -tearoff no
  $w.menubar.help.menu add command -label "AutoIMD Help..." -command "vmd_open_url [string trimright [vmdinfo www] /]/plugins/autoimd"
              
  ##############################
  # Create Main Window  

  frame $w.win ;# Main contents
  pack $w.win -padx 4 -pady 4
			
  # make a frame for the manipulation selection
  frame $w.win.maniplabel
  label $w.win.maniplabel.label -text "1. Type in the atom selections for imd (top molecule):" -font $labelfont -justify right
  pack $w.win.maniplabel.label -anchor nw
  
  frame $w.win.manip
  label $w.win.manip.label -text "imdmolten:" -font $labelfont -width $labelwidth -justify left -anchor nw
  entry $w.win.manip.entry -width 30 -relief sunken -bd 1 -textvariable AutoIMD::moltenseltext -font $textfont
  pack  $w.win.manip.label -side left
  pack $w.win.manip.entry -side left -fill x -expand yes
  
  frame $w.win.fixed
  label $w.win.fixed.label -text "imdfixed:" -font $labelfont -width $labelwidth -justify left -anchor nw
#  label $w.win.fixed.label2 -text "same residue as" -font $labelfont -justify left -anchor nw
  entry $w.win.fixed.entry -width 30 -relief sunken -bd 1 -textvariable AutoIMD::fixedseltext -font $textfont
  pack $w.win.fixed.label -side left ;#$w.win.fixed.label2 
  pack  $w.win.fixed.entry  -side left -fill x -expand yes
  
  # make a frame for the simulation options
  frame $w.win.procslabel
  label $w.win.procslabel.label -text "2. Choose where and how to run the simulation:" -font $labelfont -justify right
  pack $w.win.procslabel.label -anchor nw
  
  frame $w.win.procs
  frame $w.win.procs.server -width 20  ;# XXX DOESN'T WORK AS INTENDED
  set servermenu [eval tk_optionMenu $w.win.procs.server.servermenu AutoIMD::currentserversettings [array names AutoIMD::serversettings]]
  set i 0
  foreach config [array names AutoIMD::serversettings] {
    $servermenu entryconfigure $i -command "AutoIMD::loadsettings \"$config\""
    incr i
  }
  pack $w.win.procs.server.servermenu -expand yes -fill x ;# XXX DOESN'T WORK AS INTENDED
  
  label $w.win.procs.usingtxt -text "using " -font $textfont
  entry $w.win.procs.scale -textvariable AutoIMD::nprocessors -width 3 -relief sunken -bd 1 -font $textfont
  label $w.win.procs.slash -text "/" -font $textfont
  label $w.win.procs.maxprocs -textvariable AutoIMD::maxprocs -font $textfont
  label $w.win.procs.proctxt -text "processors" -font $textfont
  pack $w.win.procs.server $w.win.procs.usingtxt $w.win.procs.scale $w.win.procs.slash $w.win.procs.maxprocs $w.win.procs.proctxt -side left
		
  # make a frame to put the basic controls in
  frame $w.win.controlslabel
  label $w.win.controlslabel.label -text "3. Submit your job:" -font $labelfont -justify right
  pack $w.win.controlslabel.label -anchor nw
  
  frame $w.win.controls
  button $w.win.controls.submit -text "Submit" -width 7 -font $buttonfont -command {AutoIMD::submit "$AutoIMD::moltenseltext"}
  button $w.win.controls.connect -text "Connect" -command "AutoIMD::autoconnect" \
      -font $buttonfont
  button $w.win.controls.pause -text "Pause" -width 7 -command "imd pause toggle" -font $buttonfont
  button $w.win.controls.finish  -text "Finish" -width 7  -command "AutoIMD::finish" -font $buttonfont
  button $w.win.controls.discard -text "Discard" -width 7 -command "AutoIMD::finish -nosave" -font $buttonfont
  pack $w.win.controls.submit $w.win.controls.connect $w.win.controls.pause $w.win.controls.finish $w.win.controls.discard -side left

  # add a status bar indicating what is being done
  frame $w.win.status
  label $w.win.status.label -text "status:" -font $labelfont
  label $w.win.status.text -width 30 -font $textfont -anchor nw -textvariable AutoIMD::statustext
  pack $w.win.status.label -side left
  pack $w.win.status.text -side left -fill x -expand yes
  
  # pack the frames
  set padx 10
  set pady 3
  pack $w.win.maniplabel -fill x -expand yes -anchor nw -pady $pady
  pack $w.win.manip -fill x -expand yes -pady $pady -padx $padx
  pack $w.win.fixed -fill x -expand yes -pady $pady -padx $padx 
  pack $w.win.procslabel -fill x -expand yes -anchor nw -pady $pady
  pack $w.win.procs -padx $padx -pady $pady -anchor nw
  pack $w.win.controlslabel -fill x -expand yes -anchor nw -pady $pady
  pack $w.win.controls -padx $padx -pady $pady -anchor nw
  pack $w.win.status -fill x -expand yes -anchor nw -pady $pady
}


# Must be run before everything else
proc AutoIMD::startup {} {
  global env

  # If already initialized, just deiconify and return
  if { [winfo exists ".autoimd"] } {
    wm deiconify ".autoimd"
    raise ".autoimd"
    return
  }
  
  atomselect macro imdmolten "none"
  atomselect macro imdexclude "all"
    
  atomselect macro imdfixed "none"
  atomselect macro imdwater "none"
  atomselect macro imdions  "none"
  atomselect macro imdbonds "none" 
  
  startGUI
  
  setstate "uninitialized"
  loadsettings $AutoIMD::currentserversettings
}




proc AutoIMD::loadsettings {settingsname} {
  if { "$AutoIMD::imdstate" != "uninitialized"} {
    error "Cannot change setup while simulation is running!"
  }
  # Reset server-dependent variables
  set AutoIMD::presubmitscript {}
  set AutoIMD::maxprocs 1
  set AutoIMD::hosts ""
  set AutoIMD::imdport "random"
  set AutoIMD::connectiontimeout 15
  set AutoIMD::namdoutputrate 2
        
  eval $AutoIMD::serversettings($settingsname)  
  set AutoIMD::currentserversettings $settingsname
}

  
  
# keep track of the state
# uninitialized, initialized, submitted, ready, connected
proc AutoIMD::setstate { newstate } {
  variable imdstate
  variable w

  $w.win.controls.submit configure -state disabled
  $w.win.controls.connect configure -state disabled
  $w.win.controls.pause configure -state disabled
  $w.win.controls.finish configure -state disabled
  $w.win.controls.discard configure -state disabled
  $w.win.procs.server.servermenu configure -state disabled
  		
  if { $newstate == "uninitialized" } {
    AutoIMD::showstatus "Waiting for user input"
    set imdstate $newstate
    $w.win.controls.submit configure -state normal
    $w.win.procs.server.servermenu configure -state normal
  } elseif { $newstate == "waitforsubmit" } { ;# waiting to be submitted
    AutoIMD::showstatus  "Ready to submit"
    set imdstate $newstate
    $w.win.controls.submit configure -state active
  } elseif { $newstate == "ready" } {  ;#ready to be connected
    AutoIMD::showstatus "Ready to connect"
    set imdstate $newstate
    $w.win.controls.connect configure -state normal
    $w.win.controls.discard configure -state normal
  } elseif { $newstate == "connected" } {
    AutoIMD::showstatus "Connected to simulation"
    set imdstate $newstate
    $w.win.controls.pause configure -state normal
    $w.win.controls.finish configure -state normal
    $w.win.controls.discard configure -state normal
  } else {
    error "Unknown state: $newstate"
  }
  
  #update display:
 # update idletasks  
}



# save the reps that the user set
proc AutoIMD::get_reps { {mol top} } {
  set ret ""
  for {set i 0} {$i < [molinfo $mol get numreps]} {incr i} {
    append ret "mol representation [molinfo $mol get "\"rep $i\""]\n"
    append ret "mol color [molinfo $mol get "\"color $i\""]\n"
    append ret "mol selection [molinfo $mol get "\"selection $i\""]\n"
    append ret "mol material [molinfo $mol get "\"material $i\""]\n"
    append ret "mol addrep top\n\n"
  }
  return $ret;
}



# delete all the reps of mol <mol>
proc AutoIMD::delreps { mol } {
  while { [molinfo $mol get numreps] > 0 } {
    mol delrep 0 $mol
  }
}



proc AutoIMD::savestructure { } {
  variable scratchdir
  variable psffile
  variable pdbfile
  variable backgroundmol

  AutoIMD::showstatus "Loading structure and coordinates..."
	
  resetpsf
  readpsf $psffile
  
  AutoIMD::showstatus "Generating reduced psf/pdb files..."
  set sel [atomselect $backgroundmol "imdexclude"]
  set deletelist [$sel get {segname resid}]
  $sel delete
  
  set previtem {null 0}
  foreach item $deletelist {
    if {$item == $previtem} continue
    set previtem $item
    delatom [lindex $item 0] [lindex $item 1]
  }

  writepsf [file join "$scratchdir" autoimd.psf]
  
  set imdsel [atomselect $backgroundmol "not imdexclude"]
  set savebeta [$imdsel get beta] ;# save old values
  $imdsel set beta 0
  set imdfixedsel [atomselect $backgroundmol "imdfixed"] 
  $imdfixedsel set beta 1
  $imdsel writepdb [file join "$scratchdir" autoimd.pdb]
  $imdsel set beta $savebeta ;# restore old values
  
  $imdsel delete
  $imdfixedsel delete
}



proc AutoIMD::generate_namd_conf { } {
  variable imdcurrentport

  AutoIMD::showstatus "Generating NAMD config file..."
	
  if {"$AutoIMD::imdport" == "random"} {
    set imdcurrentport [expr int(rand()*5000+2000)]
  } else {
    set imdcurrentport $AutoIMD::imdport
  }
  
  set shortparfiles {}
  foreach filepath $AutoIMD::parfiles {
    lappend shortparfiles [file tail $filepath]
  }
      
  # open up the conf file for writing
  set conffile [open [file join "$AutoIMD::scratchdir" autoimd.namd] w]
  
  puts $conffile "set temperature $AutoIMD::temperature" 
  puts $conffile "set paramfile_list \{$shortparfiles\}" 
  puts $conffile "set imdport $imdcurrentport" 
  puts $conffile "set sim_mode $AutoIMD::sim_mode" 
  puts $conffile "set namdoutputrate $AutoIMD::namdoutputrate" 
  
  if { "$AutoIMD::sim_mode" == "minimize" }  {
    puts $conffile "set minimizesteps 1000000" 
  } else { 
    puts $conffile "set minimizesteps $AutoIMD::minimizesteps" 
  }
  puts $conffile "set runsteps $AutoIMD::runsteps" 
          
  # copy over the predefined NAMD config. template
  set templatefile [open "$AutoIMD::namdtmplfile" r]
  puts $conffile [read "$templatefile"]
  close $templatefile

  close $conffile
}



proc AutoIMD::load_imd_molecule { } {
  variable scratchdir
  variable imdmol
  variable backgroundmol

  AutoIMD::showstatus "Reloading the view..."
		
  # now load the new molecule
  set viewpoint [molinfo $backgroundmol get { \
      center_matrix rotate_matrix scale_matrix global_matrix}]
  display update off

  mol load psf [file join "$scratchdir" autoimd.psf] pdb [file join "$scratchdir" autoimd.pdb]
  set imdmol [molinfo top]
  mol rename $imdmol "AutoIMD Simulation"

  foreach mol [molinfo list] {
    molinfo $mol set {
      center_matrix rotate_matrix scale_matrix
      global_matrix
    } $viewpoint
  }
  display update on
}



proc AutoIMD::autoconnect { } {
  mol top $AutoIMD::imdmol
  
  AutoIMD::showstatus "Trying to connect..."
  
  set connected "false"
  set timecounter 0
  while {$connected == "false"} {
    incr timecounter
    foreach host [lindex $AutoIMD::hosts 0] {
      catch {
        imd connect "$host" "$AutoIMD::imdcurrentport"
        # The following will only execute if "imd connect" was successful
        set connected "true"
        setstate "connected"
        trace variable ::vmd_timestep($AutoIMD::imdmol) w AutoIMD::tracetimestep 
        break;
      }
    }
    if {$timecounter > $AutoIMD::connectiontimeout} {
      error "Attempt to connnect to the simulation timed out!" -type ok
    }
    if {$connected == "false"} {after 1500}
  }
}



proc AutoIMD::connect { } {
  mol top $AutoIMD::imdmol
  foreach host $AutoIMD::hosts {
    if {! [catch { imd connect $host $AutoIMD::imdcurrentport }]} {
      setstate "connected"  ;# success!
    }
  }
}



proc AutoIMD::add_exclude_to_background { } {
  variable backgroundmol
  for {set i 0} {$i < [molinfo $backgroundmol get numreps]} {incr i} {
    set sel [lindex [molinfo $backgroundmol get "\"selection $i\""] 0]
    if { [lindex $sel [expr [llength $sel]-1]] != "imdmolten" } {
      set newsel "($sel) and not imdmolten"
      mol modselect $i $backgroundmol $newsel
    }
  }
 }
 
  

proc AutoIMD::restore_background {} {
  mol top $AutoIMD::backgroundmol
  
  atomselect macro imdmolten "none"
  atomselect macro imdexclude "all"
    
  atomselect macro imdfixed "none"
  atomselect macro imdwater "none"
  atomselect macro imdions  "none"
  atomselect macro imdbonds "none"    
      
  redrawbackgroundrep
}


proc AutoIMD::redrawbackgroundrep {} {
  variable backgroundmol

  for {set i 0} {$i < [molinfo $backgroundmol get numreps]} {incr i} {
    mol modselect $i $backgroundmol \
        [lindex [molinfo $backgroundmol get "\"selection $i\""] 0]
  }
}


# initialize the system IF necessary
proc AutoIMD::initparams {} {
  variable pdbfile
  variable psffile
  variable imdmol 
  variable backgroundmol 
  variable startmol
  variable scratchdir
  variable imdstate
  variable w
  
  setstate "waitforsubmit"
  AutoIMD::showstatus "Initializing AutoIMD..."

  # Require a scratch directory
  if { ! [file exists $AutoIMD::scratchdir] } {
    set answer [tk_messageBox -message "Your AutoIMD scratch directory \($scratchdir\) does not exist! Do you want AutoIMD to try to create it for you? (Click No if you wish to create it yourself.)" -type yesno -default "yes" -parent $w -icon question]
    if {"$answer" == "yes"} {
      if [catch {exec mkdir -p "$scratchdir"}] {
        setstate "uninitialized"
        error "Could not create the directory \($scratchdir\). Please setup your AutoIMD scratch directory."
      }
    } else {
      setstate "uninitialized"
    }
  }

  set AutoIMD::pdbfile [file join "$AutoIMD::scratchdir" autoimd_final.pdb]
  
  set psffile ""
  foreach filename [lindex [molinfo $startmol get filename] 0] filetype [lindex [molinfo $startmol get filetype] 0] {
  # make sure to get the *last* psf file in the list
    if {![string compare "$filetype" "psf"]} {
      set psffile "$filename"
    }
  }
  # make sure that we have a PSF - we need this for psfgen
  if { "$psffile" == "" } {
    setstate "uninitialized"
    error "You must have a PSF file loaded."
  }
		
  set backgroundmol [molinfo $startmol]

  # Make the initial finishfile.
  set all [atomselect $backgroundmol all]
  if { [ file exists $pdbfile ] } {
    file rename -force "$pdbfile" "$pdbfile.BAK"
  }
  $all writepdb "$pdbfile"
  $all delete

  # setup softlinks to paramfiles
  if {"[vmdinfo arch]" != "WIN32"} {
    foreach filepath $AutoIMD::parfiles {
      set localfilepath "[file join "$AutoIMD::scratchdir" "[file tail $filepath]"]" 
      if {"$filepath" != "$localfilepath"} {
        exec rm -f "$localfilepath"
        exec ln -sf "$filepath" "$localfilepath"
      }
    }

  }
  
  # make the background molecule look right
  add_exclude_to_background
}



proc AutoIMD::submit { seltext } {
  variable imdrep
  variable pdbfile
  variable scratchdir
  variable pdbfile
  variable imdmol
  variable backgroundmol
  variable fixedseltext
    
  initparams
  
  # XXX check that selection has >0 atoms
  
  if [catch {      
    atomselect macro imdmolten "\{$seltext\}"
    
    # check validity of fixed atoms text selection:
    atomselect macro imd_tmp_test "\{$fixedseltext\}"
    atomselect delmacro imd_tmp_test
   }] {
    setstate "uninitialized"
    error "Invalid atom selection: $seltext."
  }
  
  if [catch {  
    # check that molten selection is not empty (to prevent later nastiness):
    set testsel [atomselect $backgroundmol imdmolten]
    if {[$testsel num] == 0} {error "IMD atom selection contains no atoms!"}
    $testsel delete
    
    #The extra words around $fixedseltext are necessary for the proper functioning of AutoIMD.   
    atomselect macro imdfixed "(same residue as (\{$fixedseltext\} or imdmolten)) and not imdmolten"
    atomselect macro imdexclude "not imdfixed and not imdmolten"
    
    atomselect macro imdwater "imdmolten and water"
    atomselect macro imdions  "imdmolten and ion"
    atomselect macro imdbonds "imdmolten and (not water) and (not ion)"       
  } mesg ] {
    setstate "uninitialized"
    error "$mesg"
  }

  # the current coordinates are in pdbfile, so we are ready!
  savestructure
  generate_namd_conf

  if { "$AutoIMD::presubmitscript" != "" } { if [ catch {eval $AutoIMD::presubmitscript} msg] {
    setstate "uninitialized"
    error "Error in \"presubmitscript\".\n$msg."
  }}

  AutoIMD::showstatus "Submitting your job..."

  if [catch { 
    set AutoIMD::imdjobspec [ NAMDRun::submitJob $scratchdir [concat $AutoIMD::namd_opts +p$AutoIMD::nprocessors] \
      "autoimd" "autoimd.namd" "autoimd.log" ]
  } msg] {
    # error! better delete the new molecule before passing it on.
#    mol delete $imdmol  XXX should this still be deleted??
    restore_background
    setstate "uninitialized"
    error $msg
  }
  
  load_imd_molecule  
    
  # set up a nice view
  redrawbackgroundrep
    
  # set up a nice representation
  mol delrep all $imdmol
  eval $imdrep
    
  setstate "ready"
}


# This kills the IMD connection
proc AutoIMD::killsimulation {} {
  variable imdstate

  if { $imdstate == "connected" } {
    catch {imd kill}
  }
  
  NAMDRun::abortJob $AutoIMD::imdjobspec
  
  puts "Please make sure that the simulation has been properly killed!"
}



proc AutoIMD::finish { args } {
  variable pdbfile
  variable imdmol
  variable imdrep
  
  if [ catch {  
  trace vdelete vmd_timestep($imdmol) w AutoIMD::tracetimestep

  # parse arguments  
  set savefile yes ;# update coords of background molecule?
  foreach arg $args {
    if {![string compare "$arg" "-nosave"]} {set savefile no}
  }
  
  catch {killsimulation}
  

  set imdrep [get_reps $imdmol]

  if { "$savefile" == "yes" } {
    if { [ file exists $pdbfile ] } { file rename -force "$pdbfile" "$pdbfile.BAK" }
    writecompletepdb $pdbfile -updatecoords
  }
  
  AutoIMD::showstatus "Cleaning up..."
    
  mol delete $imdmol
  restore_background
  } msg] {
    tk_messageBox -parent $AutoIMD::w -title "AutoIMD Error" -message  "Could not save final coordinates!\n$msg" -type ok
  }
  
  puts "Done with AutoIMD."
  # Must run the following lines even if an error has occured
  setstate "uninitialized"
}


# Saves a copy of the "complete" coordinates to a PDB file
proc AutoIMD::writecompletepdb {pdbfilename args} {
  if [ catch {
  variable imdmol
  variable backgroundmol

  if { "$pdbfilename" == "" } return

  # parse arguments  
  set updatecoords no ;# update coords of background molecule?
  foreach arg $args {
    if {![string compare "$arg" "-updatecoords"]} {set updatecoords yes}
  }
  
  AutoIMD::showstatus "Saving coordinates to file..."
  set updatesel [atomselect $backgroundmol "not imdexclude"]
  if { "$updatecoords" == "no" } {set oldcoords [$updatesel get {x y z}]} ;#make backup of coords
  
  # save the coords to file
  $updatesel set {x y z} [[atomselect $imdmol all] get {x y z}]
  $updatesel delete
  set all [atomselect $backgroundmol all] 
  $all writepdb "$pdbfilename"
  $all delete
  
  if {"$updatecoords" == "no"} {$updatesel set {x y z} $oldcoords} ;#reinstate the old coordinates
  } msg] {
    tk_messageBox -parent $AutoIMD::w -title "AutoIMD Error" -message "Could not save final coordinates!\n$msg" -type ok
  }
}



# Callback for whenever IMD receives a new frame
proc AutoIMD::tracetimestep {args} {
  variable imdmol
  
  if { "$AutoIMD::sim_mode" == "minimize" }  {
    set AutoIMD::statustext "Minimizing [molinfo $imdmol get timesteps]"
  } else {  
    set timesteps [molinfo $imdmol get timesteps]
    if {$timesteps <= $AutoIMD::minimizesteps} {
      set AutoIMD::statustext "Connected (minimizing: $timesteps)"
    } else {    
      set AutoIMD::statustext "Connected (time: [format "%.2f" [expr 2.*($timesteps - $AutoIMD::minimizesteps)/1000.]] ps)"
    }
  }
}


######################################################################
### USER INTERFACE ELEMENTS AND COMMANDS                           ###
######################################################################

proc AutoIMD::showstatus {messagetext} {
  set AutoIMD::statustext "$messagetext"
  #update display:
  update idletasks
}

proc AutoIMD::dialog_simsettings {} {
  if { [winfo exists ".autoimd_simsettings"] } {
    wm deiconify ".autoimd_simsettings"
    return
  }

  set ws [toplevel ".autoimd_simsettings"]
  wm title $ws "Simulation Parameters"
  wm resizable $ws no no
  
  set textfont [font create -family helvetica -size 10]
  set labelfont [font create -family helvetica -size 10]
  set buttonfont [font create -family helvetica -weight bold -size 9]
  set labelwidth 15
  
  frame $ws.scrdir
  label $ws.scrdir.label -text "scratch directory:" -font $labelfont -width $labelwidth -justify right -anchor ne
  entry $ws.scrdir.entry -textvariable AutoIMD::scratchdir -width 30 -relief sunken -bd 1 -font $textfont
  button $ws.scrdir.browsebutton -text "Browse..." -command "AutoIMD::dialog_getscratchdir $ws" -width 7 -pady 1 -font $buttonfont
  pack $ws.scrdir.label -side left
  pack $ws.scrdir.entry -side left -fill x -expand yes
  pack $ws.scrdir.browsebutton -side right -padx 3

  frame $ws.namdtmpl
  label $ws.namdtmpl.label -text "NAMD config file:" -font $labelfont -width $labelwidth -justify right -anchor ne
  entry $ws.namdtmpl.entry -textvariable AutoIMD::namdtmplfile -width 30 -relief sunken -bd 1 -font $textfont
  button $ws.namdtmpl.browsebutton -text "Browse..." -command "AutoIMD::dialog_getNAMDfile $ws" -width 7 -pady 1 -font $buttonfont
  pack $ws.namdtmpl.label -side left
  pack $ws.namdtmpl.entry -side left -fill x -expand yes
  pack $ws.namdtmpl.browsebutton -side right -padx 3
    
  frame $ws.parfiles
  label $ws.parfiles.label -text "CHARMM params:" -font $labelfont -width $labelwidth -justify right -anchor ne
  entry $ws.parfiles.entry -width 40 -textvariable AutoIMD::parfiles -relief sunken -bd 1 -font $textfont
  button $ws.parfiles.browsebutton -text "Add..." -command "AutoIMD::dialog_getCHARMMparam $ws" -width 7 -pady 1 -font $buttonfont
  pack $ws.parfiles.label -side left
  pack $ws.parfiles.entry -side left -fill x -expand yes
  pack $ws.parfiles.browsebutton -side right -padx 3
  
  frame $ws.min
  label $ws.min.label -text "minimization:" -font $labelfont -width $labelwidth -justify right -anchor ne
  entry $ws.min.entry -textvariable AutoIMD::minimizesteps -width 10 -relief sunken -bd 1 -font $textfont
  label $ws.min.units -text "steps" -font $labelfont -justify right
  pack $ws.min.label $ws.min.entry $ws.min.units -side left
    
  frame $ws.temp
  label $ws.temp.label -text "temperature:" -font $labelfont -width $labelwidth -justify right -anchor ne
  entry $ws.temp.entry -textvariable AutoIMD::temperature -width 10 -relief sunken -bd 1 -font $textfont
  label $ws.temp.units -text "K" -font $labelfont -justify right
  pack $ws.temp.label $ws.temp.entry $ws.temp.units -side left
  
  frame $ws.okbutton
  button $ws.okbutton.okbutton -text "OK" -command "wm withdraw $ws" -width 10 -pady 1 -font $buttonfont
  pack $ws.okbutton.okbutton -side bottom
     
  pack $ws.scrdir $ws.namdtmpl $ws.parfiles -padx 4 -pady 4 -fill x -expand yes 
  pack $ws.min $ws.temp -padx 4 -pady 4 -anchor nw
  pack $ws.okbutton -pady 4
}


proc AutoIMD::dialog_getscratchdir {parent} {
  set newscratchdir [tk_chooseDirectory -title "Choose the scratch directory" -mustexist true -initialdir $AutoIMD::scratchdir -parent $parent ]
  if {[string length $newscratchdir] > 0} {set AutoIMD::scratchdir $newscratchdir}
}

proc AutoIMD::dialog_savepdb {} {
  set filename [tk_getSaveFile -title "Write to PDB" -parent $AutoIMD::w -filetypes {{"PDB files" {".pdb"}} {"All files" {"*"}}}]
  if {[string length $filename] > 0} {
    if {"$AutoIMD::imdstate" == "connected"} {
      writecompletepdb "$filename"
    } elseif {"$AutoIMD::backgroundmol" != -1} {
      set sel [atomselect $AutoIMD::backgroundmol all] 
      $sel writepdb "$filename"
      $sel delete
    } else {
      set sel [atomselect top all]
      $sel writepdb "$filename"
      $sel delete
    }
  }
}

proc AutoIMD::dialog_getNAMDfile {parent} {
  set filename [tk_getOpenFile -title "Choose a NAMD config file" -parent $parent -filetypes {{"NAMD config files" {"*.namd" "*.conf"}} {"All files files" {"*"}}}]
  if {[string length $filename] > 0} {set AutoIMD::namdtmplfile $filename}
}

proc AutoIMD::dialog_getCHARMMparam {parent} {
  set filename [tk_getOpenFile -title "Choose a CHARMM parameter file" -parent $parent -filetypes {{"All files files" {"*"}} {"CHARMM Parameters" {"par*.inp"}} }]
  if {[string length $filename] > 0} {lappend AutoIMD::parfiles $filename}
}





######################################################################
### API                                                            ###
######################################################################



# To be used to reset server list
proc AutoIMD::api_clearservers {args} {
  if {[llength $args] != 0} {
    puts "usage: autoimd clearservers"
    return
  }
  
  if [info exists AutoIMD::serversettings] {unset AutoIMD::serversettings}
  if ![winfo exists ".autoimd"] return
  $AutoIMD::w.win.procs.server.servermenu.menu delete 0 last
}


# To be used to add a new server configuration
proc AutoIMD::api_addserver {args} {
  variable w
  set rebuildmenu true
  
  if {[llength $args] != 2} {
    puts "usage: autoimd addserver <name> <script>"
    return
  }
  set servername [lindex $args 0] 
  set serverscript [lindex $args 1]
  
  # don't add new menu item if server already exists
  foreach name [array names AutoIMD::serversettings] {
    if {"$servername" == "$name"} {set rebuildmenu false}
  }
      
  set "AutoIMD::serversettings($servername)" $serverscript
  
  if ![winfo exists ".autoimd"] return
  if {"$rebuildmenu" != "true"} return
  
  $w.win.procs.server.servermenu.menu add radiobutton -label "$servername" -variable AutoIMD::currentserversettings -value "$servername" -command "AutoIMD::loadsettings \"$servername\""
  
  set i 0
  foreach config [
  ] {
    $w.win.procs.server.servermenu entryconfigure $i -command "AutoIMD::loadsettings \"$config\""
    incr i
  }
}


# Sets the current active server
proc AutoIMD::api_setserver {args} {
  if {[llength $args] != 1} {
    puts "usage: autoimd setserver \<name\>"
    return
  }
  set AutoIMD::currentserversettings [lindex $args 0]
}



######################################################################
### USER CALLS                                                     ###
######################################################################

# run this first!
proc autoimd {args} {
  global env
  
  if {[llength $args] == 0} { # if run without any arguments
      puts "usage: autoimd <command> options..."
      puts "  showwindow"
      puts "  clearservers"
      puts "  addserver \<name\> \<script\>"
      puts "  setserver \<name\>"
      puts "  set \<var\> \<value\>"
  }

  set command [lindex $args 0]
  switch $command {
    showwindow   { AutoIMD::startup }
    addserver    { eval AutoIMD::api_addserver [lrange $args 1 end] }
    resetservers { eval AutoIMD::api_clearservers [lrange $args 1 end] }
    setserver    { eval AutoIMD::api_setserver [lrange $args 1 end] }
    set          { set "AutoIMD::[lindex $args 1]" [lrange $args 2 end] }
  }
}


proc autoimd_menu_tk_register {} {
  AutoIMD::startup
  return $AutoIMD::w  
}


# This line loads the default settings when the package is loaded
source [file join "$AutoIMD::BASEDIR" autoimd-local.tcl]
