/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: utilities.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.23 $	$Date: 96/03/23 05:22:36 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * General utility routines and definitions.
 *
 ***************************************************************************/

#include <iostream.h>
#include <strings.h>
#include <ctype.h>
#include <sys/time.h>
#include <unistd.h>
#include <pwd.h>
#include <malloc.h>
#include <math.h>
#include "utilities.h"

#include "DisplayDevice.h"
extern DisplayDevice *display;
#include "MoleculeList.h"
extern MoleculeList *moleculeList;

#ifdef VMDTCL
// used for routines to get/set Tcl variables (so far, only set pick values)
#include <tcl.h>
#include "UIText.h"
extern UIText *uiText;
#endif

// given an argc, argv pair, take all the arguments from the Nth one on
// and combine them into a single string with spaces separating words.  This
// allocates space for the string, which must be freed by the user.
char *combine_arguments(int argc, char **argv, int n) {
  char *newstr = NULL;

  if(argc > 0 && n < argc && n >= 0) {
    int i, sl = 0;
    // find out the length of the words we must combine
    for(i=n; i < argc; i++)
      sl += strlen(argv[i]);

    // combine the words together
    if(sl) {
      newstr = new char[sl + 8 + argc - n];	// extra buffer added
      *newstr = '\0';
      for(i=n; i < argc; i++) {
        if(i != n)
          strcat(newstr," ");
        strcat(newstr, argv[i]);
      }
    }
  }

  // return the string, or NULL if a problem occurred
  return newstr;
}


// duplicate a string using c++ new call
char *stringdup(const char *s) {
  char *rs;

  if(!s)
    return NULL;

  rs = new char[strlen(s) + 1];
  strcpy(rs,s);

  return rs;
}


// convert a string to upper case
char *stringtoupper(char *s) {
  register int i;

  if(s) {
    for(i=(strlen(s) - 1); i >= 0; i--)
      s[i] = toupper(s[i]);
  }

  return s;
}


// do upper-case comparison
int strupcmp(const char *a, const char *b) {
  char *ua, *ub;
  int retval;

  ua = stringtoupper(stringdup(a));
  ub = stringtoupper(stringdup(b));

  retval = strcmp(ua,ub);

  delete [] ub;
  delete [] ua;

  return retval;
}


// do upper-case comparison, up to n characters
int strupncmp(const char *a, const char *b, int n) {
#ifdef ARCH_AIX3
   while (n-- > 0) {
      if (toupper(*a) != toupper(*b)) {
	 return toupper(*b) - toupper(*a);
      }
      if (*a == 0) return 0;
      a++; b++;
   }
   return 0;
#else
   return strncasecmp(a, b, n);
#endif
}


// break a file name up into path + name, returning both in the specified
//	character pointers.  This creates storage for the new strings
//	by allocating space for them.
void breakup_filename(char *full, char **path, char **name) {
  char *namestrt;
  int pathlen;

  if(full == NULL) {
    *path = *name = NULL;
    return;
  } else if (strlen(full) == 0) {
    *path = new char[1];
    *name = new char[1];
    (*path)[0] = (*name)[0] = '\0';
    return;
  }

  // find start of final file name
  if((namestrt = strrchr(full,'/')) != NULL && strlen(namestrt) > 0) {
    namestrt++;
  } else {
    namestrt = full;
  }

  // make a copy of the name
  *name = stringdup(namestrt);

  // make a copy of the path
  pathlen = strlen(full) - strlen(*name);
  *path = new char[pathlen + 1];
  strncpy(*path,full,pathlen);
  (*path)[pathlen] = '\0';
} 


// break a configuration line up into tokens.  Returns tokenized string
// if the line is a valid one, i.e. the 2nd word is '=' and there is at 
// least one more word after the equal.
char *command_tokenize(char *newcmd, int *argc, char *argv[]) {
  char *cmd, *eqloc;

  // make a new string with a ' = ' instead of '='
  if(!(eqloc = strchr(newcmd,'=')))
    return NULL;
    
  cmd = new char[strlen(newcmd) + 3];
  strncpy(cmd,newcmd,(eqloc - newcmd));
  strcpy(cmd + (eqloc - newcmd)," = ");
  strcat(cmd,eqloc + 1);

  *argc = 0;

  argv[*argc] = strtok(cmd," ,;\t\n");
  if (argv[*argc] == NULL) {
    delete [] cmd;
    return NULL;
  }

  // see if the first token starts with '#'
  if(!strncmp(argv[0],"#",1)) {
    delete [] cmd;
    return NULL;
  }

  (*argc)++;

  // break up the rest of the string
  while ((argv[*argc] = strtok(NULL," ,;\t\n")) != NULL)
    (*argc)++;

  // make sure the 2nd word is "=", and there are 3 words or more
  if(*argc < 3 || strcmp(argv[1],"=")) {
    delete [] cmd;
    return NULL;
  }

  return cmd;
}


// break a configuration line up into tokens.
char *str_tokenize(char *newcmd, int *argc, char *argv[]) {
  char *cmd;

  cmd = stringdup(newcmd);
  *argc = 0;

  // initialize tokenizing calls
  argv[*argc] = strtok(cmd," ,;\t\n");

  // loop through words until end-of-string, or comment character, found
  while(argv[*argc] != NULL) {
    // see if the token starts with '#'
    if(argv[*argc][0] == '#') {
      delete [] cmd;
      return argv[0];
    } else {
      (*argc)++;		// another token in list
    }
    
    // scan for next token
    argv[*argc] = strtok(NULL," ,;\t\n");
  }

  return (*argc > 0 ? argv[0] : NULL);
}


// get the time of day from the system clock, and store it (in seconds)
double time_of_day(void) {
  struct timeval tm;
  struct timezone tz;

  gettimeofday(&tm, &tz);
  return((double)(tm.tv_sec) + (double)(tm.tv_usec)/1000000.0);
}


// return the username of the currently logged-on user
char *username(void) {
  struct passwd *pwentry;

  if((pwentry = getpwuid(getuid())) != NULL)
    return stringdup(pwentry->pw_name);
  else
    return NULL;
}


// return mem usage info (print it to stderr)
void meminfo(char *msg) {
  struct mallinfo mi;
  int i = 0;

  if(i) {
    mi = mallinfo();

    cerr << "Malloc info: ";
    if(msg)
      cerr << msg;
    cerr << "\n  Total space in arena: " << mi.arena
         << "\n  Total Ordinary blocks: " << mi.ordblks
         << "    Small blocks: " << mi.smblks
         << "\n  Space used: Ordinary = " << mi.uordblks
         << ", Small = " << mi.usmblks
         << "\n  Space free: Ordinary = " << mi.fordblks
         << ", Small = " << mi.fsmblks << endl;
  }
}

//  turn the cursor into the WAIT cursor
void start_wait(void) {
  if (display) {
    display->start_wait();
  }
}

// turn the WAIT cursor off
void stop_wait(void) {
  if (display) {
    display->stop_wait();
  }
}

// take three 3-vectors and compute x2 cross x3; with the results
// in x1.  x1 must point to different memory than x2 or x3
// This returns a pointer to x1
float * cross_prod(float *x1, float *x2, float *x3)
{
  x1[0] =  x2[1]*x3[2] - x3[1]*x2[2];
  x1[1] = -x2[0]*x3[2] + x3[0]*x2[2];
  x1[2] =  x2[0]*x3[1] - x3[0]*x2[1];
  return x1;
}

// normalize a vector, and return a pointer to it
// Warning:  it changes the value of the vector!!
float * normalize(float *vect)
{
  float len = vect[0]*vect[0] + vect[1]*vect[1] + vect[2]*vect[2];
  if (len != 1) {
    len = sqrtf(len);
    vect[0] /= len;
    vect[1] /= len;
    vect[2] /= len;
  }
  return vect;
}


// find and return the norm of a 3-vector
float norm(float *vect)
{
  return sqrtf(vect[0]*vect[0] + vect[1]*vect[1] + vect[2]*vect[2]);
}


// compute the angle (in degrees) between two vectors a & b
extern float angle(float *a, float *b) {
  register float amag = sqrtf(a[0] * a[0] + a[1] * a[1] + a[2] * a[2]);
  register float bmag = sqrtf(b[0] * b[0] + b[1] * b[1] + b[2] * b[2]);
  register float dotprot = a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
  if (amag == 0 || bmag == 0) {
    return 180;
  }
  return (57.2958 * acosf(dotprot / (amag * bmag)));
}

// compute the distance between points a & b
extern float distance(float *a, float *b) {
  return sqrtf(distance2(a,b));
}

// compute the squared distance between points a & b
extern float distance2(float *a, float *b) {
  float c[3];
  subtract(c,a,b);
  return (c[0]*c[0] + c[1]*c[1] + c[2]*c[2]);
}

/// conversion from a string to a boolean
// returns TRUE == 1, FALSE == 0, and unknown == -1
extern int str2bool(const char *s)
{
  if (!strncasecmp("true", s, 5)) return 1;  // perhaps I went a bit overboard
  if (!strncasecmp("yes", s, 4)) return 1;
  if (!strncasecmp("yep", s, 4)) return 1;
  if (!strncasecmp("si", s, 3)) return 1;
  if (!strncasecmp("oui", s, 4)) return 1;
  if (!strncasecmp("1", s, 2)) return 1;
  if (!strncasecmp("on", s, 3)) return 1;
  if (!strncasecmp("yeah", s, 5)) return 1;
  if (!strncasecmp("ja", s, 3)) return 1;
  if (!strncasecmp("t", s, 2)) return 1;
  
  if (!strncasecmp("false", s, 6)) return 0; 
  if (!strncasecmp("no", s, 3)) return 0;
  if (!strncasecmp("nope", s, 5)) return 0;
  if (!strncasecmp("bo", s, 3)) return 0;
  if (!strncasecmp("non", s, 4)) return 0;
  if (!strncasecmp("0", s, 2)) return 0;
  if (!strncasecmp("off", s, 4)) return 0;
  if (!strncasecmp("nah", s, 4)) return 0;
  if (!strncasecmp("nine", s, 5)) return 0;
  if (!strncasecmp("f", s, 2)) return 0;
  
  return -1;
}

// If not using Tcl, these do nothing
#ifdef VMDTCL
// save the text of a selection to the Tcl variable "vmd_pick_selection"
// The format is of the form "index %d %d %d .... %d" for each atom picked.
// (If nothing is picked, the text is "none")
void set_pick_selection(int , int num, int *atoms)
{
  if (uiText && uiText -> tclInterp) {
    if (!num) { set_pick_selection(); return;}
    char s[20];
    // create the text equivalent to the command
    Tcl_SetVar(uiText->tclInterp, "vmd_pick_selection", "index",
	       TCL_GLOBAL_ONLY);
    for (int i=0; i<num; i++) {
      sprintf(s, " %d", atoms[i]);
      Tcl_SetVar(uiText->tclInterp, "vmd_pick_selection", s,
		 TCL_GLOBAL_ONLY | TCL_APPEND_VALUE);
    }
  }
}
// make the selection invalid
void set_pick_selection(void) {
  Tcl_SetVar(uiText->tclInterp, "vmd_pick_selection", "none",
	     TCL_GLOBAL_ONLY);
}

// set "pick_value" to the value of the pick, if there was one
void set_pick_value(int newval)
{
  char s[20];
  sprintf(s, "%d", newval);
  set_pick_value(s);
}
void set_pick_value(double newval)
{
  char s[20];
  sprintf(s, "%f", newval);
  set_pick_value(s);
}
void set_pick_value(char *newval)
{
  Tcl_SetVar(uiText->tclInterp, "vmd_pick_value", newval, TCL_GLOBAL_ONLY);
}
#else
// do-nothing stubs
void set_pick_selection(int , int, int *){}
void set_pick_selection(void){}
void set_pick_value(int ) {}
void set_pick_value(double) {}
void set_pick_value(char *) {}
#endif


// These contain references to the previously picked atom
static int vmd_picked_mol = -1;
static int vmd_picked_atomid = -1;
static int vmd_picked_atom_announce = 0; // if 1, tell VMD a change happend
void set_picked_atom(int mol, int atomid)
{
  vmd_picked_mol = mol;       // always put info here so others can reference
  vmd_picked_atomid = atomid; // even w/o Tcl
  vmd_picked_atom_announce = 1;
}
void get_picked_atom(int *mol, int *atomid)
{
  int i;
  if (vmd_picked_mol >= 0 && vmd_picked_atomid >= 0 &&
      (i = moleculeList -> mol_index_from_id(vmd_picked_mol)) >= 0 &&
      (moleculeList -> molecule(i)->nAtoms > vmd_picked_atomid) ) {
    *mol = vmd_picked_mol;
    *atomid = vmd_picked_atomid;
    return;
  } else {
    *mol = -1;
    *atomid = -1;
  }
}


// Used by Global.C to tell Tcl the value of the newly
// picked mol / atom.  Must delay until then so the label
// appears FIRST, then Tcl can take action.
void announce_picked_atom(void) {
#ifdef VMDTCL
  if (vmd_picked_atom_announce) {
    // set the TCL variable
    char s[20];
    sprintf(s, "%d", vmd_picked_mol);
    Tcl_SetVar(uiText->tclInterp, "vmd_pick_mol", s, TCL_GLOBAL_ONLY);
    sprintf(s, "%d", vmd_picked_atomid);
    Tcl_SetVar(uiText->tclInterp, "vmd_pick_atom", s, TCL_GLOBAL_ONLY);
    vmd_picked_atom_announce = 0;
  }
#endif
}


