/***************************************************************************
 *cr
 *cr		(C) Copyright 1995 The Board of Trustees of the
 *cr			    University of Illinois
 *cr			     All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: TokenDisplayDevice.C,v $
 *	$Author: dalke $		$Locker:  $		   $State: Exp $
 *	$Revision: 1.3 $	      $Date: 96/03/24 07:07:20 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * FileRenderer type for listing all tokens in the display list
 *
 ***************************************************************************/
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "TokenDisplayDevice.h"
#include "Stack.h"
#include "Matrix4.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"

// Utility function to encode a point into a string for use in printf()s
// Up to 8 points may be used in an expression before overwriting
//  previously generated strings.
static const char *pt(float *p) {
    static const int nbuf = 8;
    static char buf[nbuf][50];
    static int bufptr = 0;

    bufptr = (bufptr + 1) % nbuf;   // use next buffer in round-robin fashion
    sprintf(buf[bufptr], "%g %g %g", p[0], p[1], p[2]);
    return buf[bufptr];
}

///////////////////////// constructor and destructor

// constructor ... initialize some variables
TokenDisplayDevice::TokenDisplayDevice(char *nm) : FileRenderer(nm) {

    MSGDEBUG(1,"Creating TokenDisplayDevice ..." << sendmsg);

    strcpy(renderCommand,"true");
    strcpy(fname,"token.txt");
}

// destructor
TokenDisplayDevice::~TokenDisplayDevice(void) { }

///////////////////////// protected nonvirtual routines

// draw a point
void TokenDisplayDevice::prpoint(const char *tok, float *p) {
    fprintf(ofp, "%-16s%s\n", tok, pt(p));
}

// draw a sphere
void TokenDisplayDevice::prsphere(const char *tok, float *p, float r) {
    fprintf(ofp, "%-16s%s\t%g\n", tok, pt(p), r);
}

// draw a line (cylinder) from a to b
void TokenDisplayDevice::prline(const char *tok, float *a, float*b) {
    fprintf(ofp, "%-16s%s\t%s\n", tok, pt(a), pt(b));
}

// draw a cylinder (lists endpoints, resolution, and radius)
void TokenDisplayDevice::prcylinder(const char *tok, float *a, float *b,
				  int res, float radius) {
    fprintf(ofp, "%-16s%s\t%s\t%d %g\n", tok, pt(a), pt(b), res, radius);
}

// draw a precomputed cylinder (lists edges normals & vertices for each edge)
void TokenDisplayDevice::prcylinder(const char *tok, int num, float *edges) {
    fprintf(ofp, "%-16s%d\n", tok, num);
    for (int i = 0; i < num; i++) {
	fprintf(ofp, "\t\t%s\t%s\t%s\n", pt(edges), pt(edges+3), pt(edges+6));
	edges += 9;
    }
}

// draw a cone
void TokenDisplayDevice::prcone(const char *tok, float *a, float *b,
			      int res, float radius) {
    fprintf(ofp, "%-16s%s\t%s\t%d %g\n", tok, pt(a), pt(b), res, radius);
}

// draw a triangle
void TokenDisplayDevice::prtriangle(const char *tok, float *a, float *b, float *c,
				  float *n1, float *n2, float *n3) {
    fprintf(ofp, "%-16s%s\t%s\n", tok, pt(a), pt(n1));
    fprintf(ofp, "\t\t%s\t%s\n", pt(b), pt(n2));
    fprintf(ofp, "\t\t%s\t%s\n", pt(c), pt(n3));
}

// draw a square
void TokenDisplayDevice::prsquare(const char *tok, float *norm, float *a, float *b, float *c, float *d) {
    fprintf(ofp, "%-16s%s\n", tok, pt(norm));
    fprintf(ofp, "\t\t%s\t%s\n", pt(a), pt(b));
    fprintf(ofp, "\t\t%s\t%s\n", pt(c), pt(d));
}

// matrix stack operation
void TokenDisplayDevice::prmatop(const char *tok, float *mat) {
    fprintf(ofp, "%-16s\n", tok);
    for (int row = 0; row < 4; row++) {
	fprintf(ofp, "\t\t%g %g %g %g\n",
	    mat[0], mat[1], mat[2], mat[3]);
	mat += 4;
    }
}

///////////////////// public virtual routines

// initialize the file for output
int TokenDisplayDevice::prepare3D(int) {
    if (!(ofp = fopen(fname,"w"))) {
	msgErr << "Cannot open file " << fname << " for Token output" << sendmsg;
	Initialized = FALSE;
    } else {
	Initialized = TRUE;
	ofpname = stringdup(fname);

	// file for Token raytracer.

	fprintf(ofp, "# Token dump of VMD Scene\n");

	fprintf(ofp, "# View\n");
	fprintf(ofp, "eye %s\n", pt(eyePos));
	fprintf(ofp, "at %s\n", pt(eyeDir));
	fprintf(ofp, "up %s\n", "0 1 0");

	fprintf(ofp, "# Light Definitions\n");
	for (int i = 0; i < DISP_LIGHTS; i++) {
	    if (lightDefined[i] && lightOn[i]) {
		fprintf(ofp, "light color %s\n", pt(lightColor[i]));
		fprintf(ofp, "light direction %s\n", pt(lightPos[i]));
	   }
       }

       fprintf(ofp, "background color %s\n", pt(backColor));

       fprintf(ofp, "# Token list\n");
    }
    return Initialized;
}


// clean up after yourself
void TokenDisplayDevice::update(int) {
    if (Initialized) {
	fprintf(ofp, "# End of tokens\n");
	fclose(ofp);
	msgInfo << "Token file generation finished" << sendmsg;
    }
}

// render the command list to the file in order.
void TokenDisplayDevice::render(void *cmdlist) {
    char *cmdptr = (char *)cmdlist;
    int tok, cmdsize;

    MSGDEBUG(3, "TokenDisplayDevice: rendering command list." << sendmsg);

    if (!cmdlist) return;

    // scan through the list, getting each command and executing it, until
    // the end of commands token is found
    dataBlock = NULL;
    while((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
	float *fp;
	int *ip, n;

	if (tok == DLINKLIST) {
	    cmdptr += sizeof(int);	 // step forward one (no length here)
	    cmdptr = *((char **) cmdptr);// and go to the next link
	    tok = ((int *)cmdptr)[0];	 // this is guaranteed to be neither
	}				// a DLINKLIST nor DLASTCOMMAND

	cmdsize = ((int *)cmdptr)[1];
	cmdptr += 2*sizeof(int);

	fp = (float *)cmdptr;
	ip = (int *)cmdptr;

	switch (tok) {
	    case DPOINT:
		prpoint("point", fp);
		break;
	    case DPOINT_I:
		fp = dataBlock + ip[0];
		prpoint("point_i", fp);
		break;
	    case DLINE:
		prline("line", fp, fp+3);
		break;
	    case DLINE_I:
		prline("line_i", dataBlock + ip[0], dataBlock + ip[1]);
		break;
	    case DSPHERE:
		prsphere("sphere", fp, fp[3]);
		break;
	    case DSPHERE_I:
		prsphere("sphere_i", dataBlock + int(fp[0]), fp[1]);
		break;
	    case DTRIANGLE:
		prtriangle("triangle", fp, fp+3, fp+6, fp+9, fp+12, fp+15);
		break;
	    case DTRIANGLE_I:
		prtriangle("triangle_i",
			 dataBlock + ip[1],
			 dataBlock + ip[2],
			 dataBlock + ip[3],
			 dataBlock + ip[0],
			 dataBlock + ip[0],
			 dataBlock + ip[0]);
		break;
	    case DSQUARE:
		prsquare("square", fp, fp+3, fp+6, fp+9, fp+12);
		break;
	    case DSQUARE_I:
		prsquare("square_i",
		       dataBlock + ip[0],
		       dataBlock + ip[1],
		       dataBlock + ip[2],
		       dataBlock + ip[3],
		       dataBlock + ip[4]);
		break;
	    case DCYLINDER:
#ifdef USE_SLOW_CYLINDERS
		prcylinder("cylinder", fp, fp+3, int(fp[7]), fp[6]);
#else
		prcylinder("cylinder", int(fp[7]), fp+8);
#endif
		break;
	    case DCYLINDER_I:
		prcylinder("cylinder",
			 dataBlock + int(fp[0]),
			 dataBlock + int(fp[1]),
			 int(fp[3]),
			 fp[2] );
		break;
	    case DCONE:
		// draw a cone of given radius and resolution
		prcone("cone", fp, fp+3, int(fp[7]), fp[6]);
		break;
	    case DCONE_I:
		prcone("cone_i",
		     dataBlock + int(fp[0]),
		     dataBlock + int(fp[1]),
		     int(fp[3]),
		     fp[2] );
		break;
	    case DTEXTPOS:
		fprintf(ofp, "textpos\t\t%s\n", pt(fp));
		break;
	    case DTEXTPOS_I:
		fprintf(ofp, "textpos_i\t%s\n", pt(dataBlock + ip[0]));
		break;
	    case DTEXT:
		fprintf(ofp, "text\t\t%s\n", cmdptr);
		break;
	    case DCOLORINDEX:
		// set the current color to the given color index ... assumes the
		// color has already been defined
		n = ip[0];

		fprintf(ofp, "colorindex\t%d\n", n);
		break;
	    case DCOLORRGB:
		fprintf(ofp, "colorrgb\t%s\n", pt(fp));
		break;
	    case DPICKPOINT:
		fprintf(ofp, "pickpoint\t%s %d\n", pt(fp), ip[3]);
		break;
	    case DPICKPOINT_I:
		fprintf(ofp, "pickpoint_i\t%s %d\n",
		    pt(dataBlock + ip[0]), ip[1]);
		break;
	    case DPICKLINE:
		fprintf(ofp, "pickline\n");
		break;
	    case DPICKLINE_I:
		fprintf(ofp, "pickline_i\n");
		break;
	    case DPICKBOX:
		fprintf(ofp, "pickbox\n");
		break;
	    case DPICKBOX_I:
		fprintf(ofp, "pickbox_i\n");
		break;
	    case DLIGHTONOFF:
		fprintf(ofp, "lightonoff\t%d %d\n", ip[0], ip[1]);
		break;
	    case DMATERIALS:
		fprintf(ofp, "materials\t%d\n", ip[0]);
		break;
	    case DSPHERERES:
		fprintf(ofp, "sphereres\t%d\n", ip[0]);
		break;
	    case DSPHERETYPE:
		fprintf(ofp, "spheretype\t%d\n", ip[0]);
		break;
	    case DLINESTYLE:
		fprintf(ofp, "linestyle\t%d\n", ip[0]);
		break;
	    case DLINEWIDTH:
		fprintf(ofp, "linewidth\t%d\n", ip[0]);
		break;
	    case DPUSH:
		fprintf(ofp, "push\n");
		break;
	    case DPOP:
		fprintf(ofp, "pop\n");
		break;
	    case DLOAD:
		prmatop("loadmat", fp);
		break;
	    case DMULT:
		prmatop("multmat", fp);
		break;
	    case DTRANS:
		fprintf(ofp, "trans\t\t%s\n", pt(fp));
		break;
	    case DSCALE:
		fprintf(ofp, "scale\t\t%s\n", pt(fp));
		break;
	    case DROT:
		fprintf(ofp, "rot\t\t%g %c\n", fp[0], 'x' + int(fp[1]));
		break;
	    case DCOLORDEF:
		// define a new color
		fprintf(ofp, "colordef\t%d\t", int(fp[0]));
		for (n = 0; n < COLOR_ITEMS; n++)
		    fprintf(ofp, "%g ", fp[n+1]);
		putc('\n', ofp);
		break;
	    case DLIGHTDEF:
		fprintf(ofp, "lightdef\t%d\t%s\t%s\n",
		    int(fp[0]), pt(fp+1), pt(fp+4));
		break;
	    case DCLEAR:
		fprintf(ofp, "clear\n");
		break;
	    case DMOREDATA:
	    case DNEWDATA:
		// set the current drawing data block
#ifdef VMDCAVE
		dataBlock = (float *)cmdptr;	// point to current position in list
#else
		dataBlock = ((float **)cmdptr)[0];  // point to given memory loc
#endif
		break;
	    default:
		// command not found, help!
		fprintf(ofp, "unknown token %d\n", tok);
	}

	// update position in array
	cmdptr += cmdsize;
    }
}

