/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Stage.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.7 $	$Date: 95/05/11 23:57:08 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A Displayable3D object which consists of a set of Stage, which may be
 * drawn anywhere on the screen, at any size.
 *
 ***************************************************************************/

#include "Stage.h"
#include "DisplayDevice.h"
#include "Scene.h"
#include "DispCmds.h"
#include "ColorList.h"
#include "Inform.h"
#include "utilities.h"


// string descriptions of stage locations
static char *stageloc[Stage::STAGEPOS_TOTAL] = {
	"Off", "Origin", "Bottom", "Top", "Left", "Right", "Behind" };


// default colors
#define STAGEEVENCOL	REGBLUE
#define STAGEODDCOL	REGGREEN


Stage::Stage(Scene *sc) : Displayable3D(MULT, "Stage", sc, 32) {

  // indicate we don't yet have a color object to use
  colorCat = (-1);

  // Displayable characteristics
  rot_off();
  scale_off();
  glob_trans_off();
  cent_trans_off();
  
  // put stage in lower part of image by default
  Panels = 0;
  panels(STAGE_PANELS);		// really sets the panels, and panel size
  location(STAGE_LOWER);	// position the stage
}


//////////////////////////  protected virtual routines  ////////////////////

// do action when a new color list is provided
// This creates a color category for use by the Stage, with the colors for
// the components.  These colors can then be edited by the user.
void Stage::do_use_colors(void) {
  // add new category (or just get it's index if it exists)
  colorCat = colorList->add_color_category("Stage");
  
  // add components, and their default colors
  (colorList->color_category(colorCat))->add_name("Even", STAGEEVENCOL);
  (colorList->color_category(colorCat))->add_name("Odd", STAGEODDCOL);
  
  // indicate we need to recreate the command list
  need_update = TRUE;
}

// do action due to the fact that a color for the given ColorList for
// the specified category has changed
void Stage::do_color_changed(ColorList *changelist, int clr) {
  if(changelist == colorList && clr == colorCat) {
    // color changed for us, recreate command list
    need_update = TRUE;
  }
}

//////////////////////////  public virtual routines  ////////////////////

// create the drawing command list
void Stage::create_cmdlist(void) {
  int i, j, k, odd;
  float c[4][3], corner1[3];
  int usecolors[2];

  MSGDEBUG(2,"Stage: creating command list ..." << sendmsg);

  corner1[0] = corner1[2] = -1.0;  corner1[1] = 0.0;
  c[0][1] = c[1][1] = c[2][1] = c[3][1] = 0.0;

  // find which colors to use
  if(colorCat >= 0) {
    usecolors[0] = (colorList->color_category(colorCat))->data("Even");
    usecolors[1] = (colorList->color_category(colorCat))->data("Odd");
  } else {
    usecolors[0] = STAGEEVENCOL;
    usecolors[1] = STAGEODDCOL;
  }

  // do reset first
  reset_disp_list();

  // turn off material characteristics
  cmdMaterials.putdata(FALSE, this);

  // draw odd/even squares separately
  for(k=0; k < 2; k++) {
    odd = (k == 1);
    
    // set color in checkerboard: Blue (even) or Green (odd)
    if(odd)
      cmdColor.putdata(REGCOLOR(usecolors[0]),this);
    else
      cmdColor.putdata(REGCOLOR(usecolors[1]),this);
    
    // start in lower-left corner
    c[0][0] = c[3][0] = corner1[0];
    c[1][0] = c[2][0] = corner1[0] + inc;

    for(i=0; i < Panels; i++) {
      c[0][2] = c[1][2] = corner1[2];
      c[2][2] = c[3][2] = corner1[2] + inc;
      for(j=0; j < Panels; j++) {
        if(!odd)
          cmdSquare.putdata(c[2], c[1], c[0], this);
	odd = !odd;
        c[0][2] += inc;  c[1][2] += inc;  c[2][2] += inc;  c[3][2] += inc;
      }
      c[0][0] += inc;  c[1][0] += inc;  c[2][0] += inc;  c[3][0] += inc;
      if(Panels % 2 == 0)
        odd = !odd;
    }
  }
}


// set stage display mode; return success
int Stage::location(int ap) {

  if(ap != stagePos) {
    stagePos = ap;
    if(ap == NO_STAGE) {
      off();
    } else {
      on();
      need_update = TRUE;
    }
  }

  return TRUE;
}


// return descripton of location
char *Stage::loc_description(int ap) {
  return stageloc[ap];
}


// routine to prepare the displayable object; must set the origin properly
void Stage::prepare(DisplayDevice *display) {
  float rot_amount = 0.0, strans[3];
  char rot_axes = 'z';
  
  strans[0] = strans[1] = strans[2] = 0.0;
  
  // move the stage to its proper position
  if(need_update && stagePos != NO_STAGE) {
    if(stagePos == STAGE_ORIGIN) {
      ;
    } else if(stagePos == STAGE_LOWER) {
      strans[1] = -1.0;
    } else if(stagePos == STAGE_UPPER) {
      strans[1] = 1.0;
    } else if(stagePos == STAGE_LEFT) {
      strans[0] = -1.0;
      rot_amount = -90.0;
    } else if(stagePos == STAGE_RIGHT) {
      strans[0] = 1.0;
      rot_amount = 90.0;
    } else if(stagePos == STAGE_BEHIND) {
      strans[2] = -1.0;
      rot_axes = 'x';
      rot_amount = 90.0;
    } else {
      msgErr << "Stage: Illegal stage location " << stagePos << " specified."
             << sendmsg;
      stagePos = STAGE_ORIGIN;
      prepare(display);
      return;
    }

    // update the current transformation
    MSGDEBUG(2,"Stage: updating position ..." << sendmsg);
    need_update = FALSE;
    
    // (re)create the command list
    create_cmdlist();

    // reset tranformation
    glob_trans_on();
    rot_on();
    reset_transformation();
    set_glob_trans(strans[0], strans[1], strans[2]);
    add_rot(rot_amount, rot_axes);
    rot_off();
    glob_trans_off();
  }
}

