/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Scene.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.14 $	$Date: 96/03/24 07:18:53 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Scene object, which maintains a list of Displayable objects and
 * draws them to a DisplayDevice.
 * Each Scene has a list of Displayable objects, and also a list of display
 * commands.  The command lists are used to draw the objects, the Displayable
 * objects to prepare and update objects for drawing.
 *
 ***************************************************************************/

#include "Scene.h"
#include "Global.h"
#include "DisplayDevice.h"
#include "Inform.h"
#include "DispCmds.h"
#include "utilities.h"
#include "FileRenderList.h"
#include "FileRenderer.h"

///////////////////////////  constructor 
Scene::Scene(void) {

  MSGDEBUG(1,"Creating a Scene ..." << sendmsg);

  numCmdLists2D = numCmdLists3D = 0;
  numDisplayable2D = numDisplayable3D = 0;
  Rocking = rockOnce = FALSE;
  rockSteps = currRockStep = 0;
  rockAmount = 0.0;
  rockAxis = 'y';
}

///////////////////////////  destructor
Scene::~Scene(void) {
  // does nothing
}

///////////////////////////  public routines 

// 'rock' the scene, by moving it through a given arc (possibly just a
// continuous circle), with a specified number of steps.
// There are two ways:
//	a) doOnce == TRUE: the rocking is done once, from a --> b
//	b) doOnce == FALSE: the rocking is done until told to stop,
//		from (a+b)/2 --> b --> a --> b .....
// Note that if steps < 0, the rocking is continuous, that is no ending point
// is specified so the rotation is continually in one direction.  In this
// case doOnce means nothing, and is just automatically used as if it were
// FALSE.
void Scene::start_rocking(float a, char ax, int steps, int doOnce) {

  // set rocking parameters
  Rocking = TRUE;
  rockSteps = steps;		// if < 0, continuous
  rockAmount = a;
  rockAxis = ((ax >= 'x' && ax <= 'z') ? ax : 'y');

  // when currRockStep == rockSteps, flip rockAmount or stop
  rockOnce = (doOnce && steps >= 0);
  currRockStep = (rockOnce ? 0 : (int)( ((float)steps)/2.0 ));
}


// clear the entire Scene, i.e. remove all displayable objects
void Scene::clear(void) {
  // go through list of Displayables, ask them all to unregister with this
  // Scene object
  while(num_displayable2D() > 0)
    (displayable2D(0))->unRegister(this);
  while(num_displayable3D() > 0)
    (displayable3D(0))->unRegister(this);
}


// do action due to the fact that a color for the given ColorList for
// the specified category has changed
void Scene::color_changed(ColorList *changelist, int clr) {
  int i;

  // just call this routine for all the registered displayables
  int num3D = num_displayable3D();
  for(i=0; i < num3D; i++)
    (displayable3D(i))->color_changed(changelist, clr);

  int num2D = num_displayable2D();
  for(i=0; i < num2D; i++)
    (displayable2D(i))->color_changed(changelist, clr);
}


// prepare all registered Displayables
void Scene::prepare(DisplayDevice *display) {
  int i;

  if (!display)
    return;

  // rock the scene, if necessary
  if(Rocking) {
    add_rot(rockAmount, rockAxis);
    if(rockSteps >= 0 && ++currRockStep >= rockSteps) {
      currRockStep = 0;
      rockAmount *= -1.0;		// reverse direction of rocking
      if(rockOnce)
        stop_rocking();			// rocked once; now quit
    }
  }

  // prepare all registered Displayables
  int num3D = num_displayable3D();
  MSGDEBUG(3,"Scene: preparing " << num3D << " 3D objects ..." << sendmsg);
  for(i=0; i < num3D; i++)
    (displayable3D(i))->draw_prepare(display);

  int num2D = num_displayable2D();
  MSGDEBUG(3,"Scene: preparing " << num2D << " 2D objects ..." << sendmsg);
  for(i=0; i < num2D; i++)
    (displayable2D(i))->draw_prepare(display);
}


// a displayable calls this when it is done preparing.  Arguments are
// the pointer to the command list the displayable used, and the dimension
// of the displayable.
void Scene::done_preparing(char *, int) {
  // currently does nothing
}


// draws the scene to the given DisplayDevice
// this is the only Scene which tells the display to do graphics commands
void Scene::draw(DisplayDevice *display) {
  int i;
  char *cmdlist;

  if(!display)
    return;

  // draw 3D objects first ... prepare the display
  display->prepare3D(TRUE);

  // draw left eye first, if stereo
  if(display->stereo_mode())
    display->left();
  else
    display->normal();
      
  // draw all the objects for left eye position
  MSGDEBUG(3,"Scene: drawing (L) " << num_disp_list3D() << " 3D objects ...");
  MSGDEBUG(3,sendmsg);
  for(i=0; i < num_disp_list3D(); i++) {
    cmdlist = disp_list3D(i);
    if( ((int *)cmdlist)[1] )
      display->render((void *)((int *)cmdlist + 3));
  }
    
  // now draw right eye, if in stereo
  if(display->stereo_mode()) {
    display->right();
    MSGDEBUG(3,"Scene: drawing (R)" << num_disp_list3D() << " 3D objects ...");
    MSGDEBUG(3,sendmsg);
    for(i=0; i < num_disp_list3D(); i++) {
      cmdlist = disp_list3D(i);
      if( ((int *)cmdlist)[1] )
        display->render((void *)((int *)cmdlist + 3));
    }
  }

  // draw 2D objects last
  if(display->has_2D() && num_disp_list2D() > 0) {
    display->prepare2D(FALSE);
    display->normal();
      
    MSGDEBUG(3,"Scene: drawing " << num_disp_list2D() << " 2D objects ...");
    MSGDEBUG(3,sendmsg);
    // draw all the objects for left eye position
    for(i=0; i < num_disp_list2D(); i++) {
      cmdlist = disp_list2D(i);
      if( ((int *)cmdlist)[1] )
        display->render((void *)((int *)cmdlist + 3));
    }
  }
  
  // update the display
  display->update(TRUE);
}

// draw the scene to a file in a given format, trying to match the
// view of the given DisplayDevice as closely as possible
// returns TRUE if successful, FALSE if not
int Scene::filedraw(char *method, char *filename, DisplayDevice *display) {
  char *cmdlist;

  FileRenderer *render = fileRenderList->find(method);
  if (!render) { 
    msgErr << "I don't know how to do the output format ";
    msgErr << method << "." << sendmsg;
    if (fileRenderList->num() == 0) {
      msgInfo << "In fact, there are no possible output formats." << sendmsg;
    } else {
      msgInfo << "The valid output formats are:" << sendmsg;
      for (int i=0; i<fileRenderList->num(); i++) {
        msgInfo << i << "  " << fileRenderList->name(i) << sendmsg;
      }
    return FALSE;
    }
  }
  render -> set_filename(filename);

  // Copy all relevant info from the current main DisplayDevice
  (*((DisplayDevice *)render)) = (*display);

  // now prepare the output for drawing, and draw the scene to the file
  if (!render -> prepare3D(TRUE)) {  // returns FALSE if not able to open file
    return FALSE;
  }

  for(int i=0; i < num_disp_list3D(); i++) {
    cmdlist = disp_list3D(i);
    if( ((int *)cmdlist)[1] )
      render -> render((void *)((int *)cmdlist + 3));
  }
  render -> update(TRUE);
  // rendering was successful
  return TRUE;
}

///////////////////  routines to affect all Displayables in the Scene 

// just change the transformation to the one given
void Scene::load_transformation(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->load_transformation(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->load_transformation(m);
} 

// reset to identity the transformation
void Scene::reset_transformation(void) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->reset_transformation();
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->reset_transformation();
}


void Scene::set_scale(float s) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_scale(s);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_scale(s);
}

void Scene::mult_scale(float s) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->mult_scale(s);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->mult_scale(s);
}

void Scene::set_scale(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_scale(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_scale(m);
}

void Scene::mult_scale(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->mult_scale(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->mult_scale(m);
}

void Scene::add_rot(float x, char axis) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_rot(x, axis);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_rot(x, axis);
}

void Scene::add_rot(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_rot(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_rot(m);
}

void Scene::set_rot(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_rot(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_rot(m);
}

void Scene::set_glob_trans(float x, float y, float z) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_glob_trans(x, y, z);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_glob_trans(x, y, z);
}

void Scene::set_glob_trans(Matrix4& m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_glob_trans(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_glob_trans(m);
}

void Scene::add_glob_trans(float x, float y, float z) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_glob_trans(x, y, z);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_glob_trans(x, y, z);
}

void Scene::add_glob_trans(Matrix4& m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_glob_trans(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_glob_trans(m);
}

void Scene::set_cent_trans(float x, float y, float z) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_cent_trans(x, y, z);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_cent_trans(x, y, z);
}

void Scene::set_cent_trans(Matrix4& m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_cent_trans(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_cent_trans(m);
}

void Scene::add_cent_trans(float x, float y, float z) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_cent_trans(x, y, z);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_cent_trans(x, y, z);
}

void Scene::add_cent_trans(Matrix4& m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_cent_trans(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_cent_trans(m);
}

