/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: ResizeArray.c,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 95/05/11 23:42:39 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Automatically-adjusting single-dim array template.
 *
 ***************************************************************************/

#include <string.h>
#include "ResizeArray.h"


////////////////////  constructor
template<class T>
ResizeArray<T>::ResizeArray(int s, float rsf) {
  // save factor by which to increase array if necessary
  resizeFactor = rsf;
  
  // initialize values for current array size and max array size
  currSize = 0;
  sz = (s > 0 ? s : 10);
  
  // allocate space for maximum size of array
  data = new T[sz];
}


////////////////////  destructor
template<class T>
ResizeArray<T>::~ResizeArray(void) {
  // free up the current array, if necessary
  if(data)
    delete [] data;
}
  

///////////////////
///////////////////  public routines
///////////////////


// return the nth item; assume N is >= 0
template<class T>
T& ResizeArray<T>::item(int N) {

  // check and see if this is attempting to access an item larger than
  // the array.  If so, extend the max size of the array by resizeFactor,
  // and return the value at the Nth position (which will be basically
  // random).
  if(N >= sz) {			// extend size of array if necessary
  
    // calc new size of array
    int newsize = (int)((float)N * resizeFactor + 0.5);

    // allocate new space, and copy over previous copy.  We only need to
    // copy over the first currSize elements, since currSize is the max
    // index that has been accessed (read OR write).  Then delete old
    // storage.
    T *newdata = new T[newsize];
    if(data) {
      memcpy((void *)newdata, (void *)data, currSize * sizeof(T));
      delete [] data;
    }
    
    // save new values
    data = newdata;
    sz = newsize;
  }
  
  // remember what the maximum index reached so far is
  if(N >= currSize)
    currSize = N + 1;
    
  // return element at Nth position
  return data[N];
}


// insert new item after the specified index.  Return index of new item.
// if specified index is < 0, item put at beginning of list.
// if specified index is >= num(), item put at end of list.
template<class T>
int ResizeArray<T>::insert_after(int n, const T& val) {
  int thissize = currSize;
  int newindx = n + 1;
  if(n < 0)
    newindx = 0;
  else if(n >= currSize)
    newindx = currSize;

  for(int i=thissize; i > newindx; i--)
    item(i) = item(i-1);
  data[newindx] = val;
    
  return newindx;
}


// insert new item before the specified index.  Return index of new item.
// if specified index is < 0, item put at beginning of list.
// if specified index is >= num(), item put at end of list.
template<class T>
int ResizeArray<T>::insert_before(int n, const T& val) {
  int thissize = currSize;
  int newindx = n;
  if(n < 0)
    newindx = 0;
  else if(n >= currSize)
    newindx = currSize;
    
  for(int i=thissize; i > newindx; i--)
    item(i) = item(i-1);
  data[newindx] = val;

  return newindx;
}


// remove the Mth ... Nth items: move all items lower in the
// array up to fill the empty slots.
// If both arguments are < 0, removes ALL items.
// If the second argument is < 0, just removes item m
template<class T>
void ResizeArray<T>::remove(int m, int n) {
  if(m < currSize && n < currSize) {
    if((m < 0 && n < 0) || (m >= 0 && n < 0) || (m > 0 && n > 0 && m <= n)) {
      int N = n, M = (m >= 0 ? m : 0);
      if(m < 0 && n < 0)
	N = (currSize - 1);
      else if(n < 0)
        N = M;
      else
	N = n;
      int removed = N - M + 1;
      for(int i=N+1; i < currSize; i++)
        data[i - removed] = data[i];
      currSize -= removed;
    }
  }
}


// scan the array until the first item that matches in the array is
// found.  Return the index if found, (-1) otherwise.
template<class T>
int ResizeArray<T>::find(const T& val) {
  int i, retval = (-1);
  for(i=0; i < currSize; i++) {
    if(data[i] == val) {
      retval = i;
      break;
    }
  }
  return retval;
}

