/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Remote.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.10 $	$Date: 1995/11/01 08:10:48 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Remote object, which maintains a connection to a remote computer that
 * provides data and structures.
 *
 ***************************************************************************/
#ifndef REMOTE_H
#define REMOTE_H

#include <rapp.h>
// mdcomm's Bool conflicts with
//      /usr/include/X11/Xlib.h:#define Bool int
#ifdef Bool
#undef Bool
#endif
#include "mdcomm.h"

class MoleculeRemote;

// some X include file #defines Status as int ... grrrrr...
#undef Status

class Remote {

// make the MoleculeRemote, which uses this class, a friend
friend class MoleculeRemote;

public:
  // settings which can be modified for a running sim
  enum Settings { TRANSFER_RATE, SAVEFRAME_RATE, TOTAL_SETTINGS };

  // status level of this object
  enum Status { NO_CONNECT, SELECTING_APP, EDITING_PARAMS, CONNECTED,
  	 CONNECT_ERROR, READ_ERROR, TOTAL_STATUS };

private:
  // have we run mdc_init yet?
  static int run_mdc_init;

protected:
  // are we trying to start a new app (T), or connecting to a running one?
  int startingNewApp;

  // are we currently editing parameters?
  int editingParams;

  // have we made a connection?
  int madeConnection;

  // current connection status description
  Status remoteStatus;
  
  // most recent error value
  int returnValue;
  
  // name of user on remote machine (if NULL, not yet set)
  char *username;
  
  // name of remote computer (if NULL, not yet set)
  char *computer;
  
  // name of consumer program (the complete path)
  char *consumerProg;

  // which application we're working with (either from 'proglist' list,
  // or 'joblist' list, depending on whether we're running a new program,
  // or attaching to an already-running one)
  int application;
  
  // possible applications to run
  struct rapp_proglist *proglist;
  int availApps;
  
  // available jobs running on the remote machine, to which we can connect
  struct rapp_joblist *joblist;
  int availJobs;

  // option list (if NULL, not yet set)
  struct rapp_optlist *optlist;
  int availOptions;
  
  // connection structure (if NULL, not yet set)
  rapp_client_handle_t consumer;
  
  // structure returned in connection (if NULL, not yet set)
  VmdStaticData staticData;
  
  // 'state' of the connection (used for shared memory access)
  VmdDynData dynamicData;
  
  //
  // modifiable parameters
  //

  // rate that new frames are transferred here from remote connection
  int transferRate;
  
  // frequency with which we save a frame ... every Nth frame received from
  // the remote connection.  If < 1, do not save any.
  int saveFrameRate;

  // return index of for given setting, -1 if error
  int setting_index(char *);

public:
  // constructor/destructor: arguments are username, hostname.
  Remote(char *, char *);
  ~Remote(void);
  
  //
  // general information
  //

  // return username to use on remote machine
  char *user(void) { return username; }
  
  // return host for remote connection
  char *host(void) { return computer; }
  
  // return application for remote connection
  char *app(void);

  // current status of connection (not the same as the error status string)
  char *status(void);

  // name of molecule to simulate (taken from STR keyword, if available)
  char *mol_name(void);

  // are the parameters being edited currently?
  int editing_parameters(void) { return editingParams; }

  // are we running a simulation properly?
  int running_simulation(void) { return madeConnection; }

  // return the static data structure for this object
  VmdStaticData *static_data(void) { return &staticData; }

  // return the dynamic data structure for this object
  VmdDynData *dynamic_data(void) { return &dynamicData; }

  // current save frame rate
  int save_frame_rate(void) { return saveFrameRate; }

  //
  // routines to change remote object state, i.e. make connection, etc.
  //
  
  // NOTE: either 'get_parameters' or 'attach_to_job' should be run, but ONLY
  // ONE of them (they are mutually-exclusive).

  // get parameters, for the Nth available app.  Return success.
  int get_parameters(int);
  
  // attach to the Nth running job.  Return success.
  int attach_to_job(int);
  
  // if a new simulation is being run, and get_parameters has been called with
  // the parameters all set up, this makes a connection and runs the sim.
  // return success.
  int start_new_simulation(void);

  // check for a new timestep; return TRUE if available.  To actually get
  // the data, should call get_next_ts.
  int next_ts_available();
  
  // gets timestep data, if available.  returns pointer to dynamic
  // data structure if successful, NULL otherwise.
  VmdDynData *get_next_ts(void);
  
  // signal a new timestep can be sent.  Should be called after get_next_ts
  // has returned true, and the data has been fetched properly.
  void done_with_ts(void);
  
  // close: finishes the connection
  // if arg = TRUE, and an application is running, this will disconnect from
  // the job but let it keep running.  Otherwise, this will kill the remote
  // application.
  void remote_close(int);
  
  //
  // routines to query/change the status of running simulation
  // strings with setting name, and type (i, s, f) are declared as
  // external strings at the bottom of this file.
  //

  // return number of settings available
  int avail_settings(void) { return TOTAL_SETTINGS; }
  
  // change the value of the specified setting.  return success.
  int change_setting(char *, char *);

  // write the value of the specified setting into the given string storage.
  // return success.
  int get_setting(char *, char *);
  int get_setting(int, char *);
  
  //
  // routines to work with parameter data files
  //

  // process list of options of the form "<keyword> = <value>" in a file.
  // return success.
  int read_param_file(FILE *);
  
  // write current parameter list to file, in form "<keyword> = <value>"
  // return success.
  int write_param_file(FILE *);

  //
  // error status routines
  //

  // query latest return status
  int error_status(void) { return returnValue; }
  
  // query return status description
  char *error_status_desc(int st);
  char *error_status_desc(void);

  //
  // query about available already-running jobs
  //

  // number of jobs to which we can connect
  int running_jobs(void) { return availJobs; }
  
  // return pid and uid of the Nth job
  void running_job_ids(int, int &, int &);
  
  // return name of Nth available job's program
  char *running_job_name(int);

  //
  // query about available apps on the remote computer
  //

  // number of apps on the remote machine
  int avail_apps(void) { return availApps; }

  // name of Nth available app
  char *avail_app_name(int);

  //
  // query about / change parameters
  //

  // number of options in optlist
  int options(void) { return availOptions; }

  // complete string description of Nth option, including keyword, value,
  // required, and type
  char *option_string(int);

  // type of nth option; if < 0, error
  int option_type(int n) {
    int retval = (-1);
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->type;
    return retval;
  }
  
  // keyword of nth option; if NULL, error
  char *option_keyword(int n) {
    char *retval = NULL;
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->keyword;
    return retval;
  }
  
  // text description of nth option
  char *option_desc(int n) {
    char *retval = NULL;
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->desc;
    return retval;
  }
  
  // is nth option required?; if < 0, error
  int option_req(int n) {
    int retval = (-1);
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->required;
    return retval;
  }
  
  // default value of nth option; if NULL, error
  char *option_default(int n) {
    char *retval = NULL;
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->def;
    return retval;
  }
  
  // change value of nth option; return success
  int set_option(int n, char *val) {
    int retval = FALSE;
    if(optlist && val && n >= 0 && n < optlist->count)
      retval = (rapp_set_opt(optlist, option_keyword(n), val) == 0);
    return retval;
  }
  
  // change value of option with given keyword; return success
  int set_option(char *key, char *val) {
    int retval = FALSE;
    if(optlist && key && val)
      retval = (rapp_set_opt(optlist, key, val) == 0);
    return retval;
  }

  // For sending information to the Remote simulation
  // I guess technically these should be friends of MoleculeRemote ...
public:
  void send_new_forces(int num, int *indicies, float *fx, 
		       float *fy, float *fz);
  void send_new_coords(int num, int *indicies, float *fx, 
		       float *fy, float *fz);
};


// string descriptions, and types, for available simulation settings
extern char *remoteSettingName[Remote::TOTAL_SETTINGS];
extern char *remoteSettingKeyword[Remote::TOTAL_SETTINGS];
extern char  remoteSettingType[Remote::TOTAL_SETTINGS];

#endif

