/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: R3dDisplayDevice.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.16 $	$Date: 96/03/24 07:07:20 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * The R3dDisplayDevice implements routines needed to render to a file 
 * in raster3d format
 *
 ***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "R3dDisplayDevice.h"
#include "Stack.h"
#include "Matrix4.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"

#define DEFAULT_RADIUS 0.002
#define DASH_LENGTH 0.02

///////////////////////// constructor and destructor

// constructor ... initialize some variables
R3dDisplayDevice::R3dDisplayDevice(char *nm) : FileRenderer(nm) {
	
    MSGDEBUG(1,"Creating R3dDisplayDevice ..." << sendmsg);
    
    // There are two Raster3D output types.  One goes directly to
    // Ethan Merritt's "render" program, the other to "tkRaster3D"
    // written by Hillary S. Gilson, hillary@indigo15.carb.nist.gov .
    if (!strcmp(nm, "tkRaster3D")) {
      strcpy(renderCommand, " tkRaster3D -r3d %s &");
      strcpy(fname, "plot.r3d");
    } else {
      strcpy(renderCommand," render < %s -sgi %s.rgb; ipaste %s.rgb");
      strcpy(fname,"plot.r3d");
    }
}
               
//destructor
R3dDisplayDevice::~R3dDisplayDevice(void) { }

///////////////////////// protected nonvirtual routines

// draw a point
void R3dDisplayDevice::point(float spdata[]) {
  float vec[3];

  // transform the world coordinates
  (transMat.top()).multpoint3d(spdata, vec);
   
  // draw the sphere
  
  fprintf(r3d_file, "2 \n");  // sphere
  fprintf(r3d_file, "%7f %7f %7f ", vec[0], vec[1], vec[2]); // center of sphere
  fprintf(r3d_file, "%7f ", float(lineWidth)*DEFAULT_RADIUS ); // the radius of the sphere
  fprintf(r3d_file, "%3.2f %3.2f %3.2f \n", currentColor[0], currentColor[1],
  currentColor[2]);
}

// draw a sphere
void R3dDisplayDevice::sphere(float spdata[]) {
  
  float vec[3];
  float radius;
    
  // transform the world coordinates
  (transMat.top()).multpoint3d(spdata, vec);
  radius = scale_radius(spdata[3]);
   
  // draw the sphere
  
  fprintf(r3d_file, "2 \n");  // sphere
  fprintf(r3d_file, "%7f %7f %7f ", vec[0], vec[1], vec[2]); // center of sphere
  fprintf(r3d_file, "%7f ", radius); // the radius of the sphere
  fprintf(r3d_file, "%3.2f %3.2f %3.2f \n", currentColor[0], currentColor[1],
  currentColor[2]);
  
}

// draw a line (cylinder) from a to b
void R3dDisplayDevice::line(float *a, float*b) {
    int i, j, test;
    float dirvec[3], unitdirvec[3];
    float from[3], to[3], tmp1[3], tmp2[3];
    float len;
    
    if(lineStyle == ::SOLIDLINE ) {
  
        // transform the world coordinates
        (transMat.top()).multpoint3d(a, from);
        (transMat.top()).multpoint3d(b, to);
    
        // draw the cylinder
        fprintf(r3d_file, "5 \n"); // flat-ended cylinder
        fprintf(r3d_file, "%7f %7f %7f ", from[0], from[1], from[2]); // first point
        fprintf(r3d_file, "%7f ", float(lineWidth)*DEFAULT_RADIUS); // radius 1
        fprintf(r3d_file, "%7f %7f %7f ", to[0], to[1], to[2]); // second point
        fprintf(r3d_file, "%7f ", float(lineWidth)*DEFAULT_RADIUS); // radius 2
        fprintf(r3d_file, "%3.2f %3.2f %3.2f \n", currentColor[0], currentColor[1],
        currentColor[2]);
        
    } else if (lineStyle == ::DASHEDLINE ) {
        
         // transform the world coordinates
        (transMat.top()).multpoint3d(a, tmp1);
        (transMat.top()).multpoint3d(b, tmp2);

        // how to create a dashed line
        for(i=0;i<3;i++) {
            dirvec[i] = tmp2[i] - tmp1[i];  // vector from a to b
        }
        len = sqrtf( dirvec[0]*dirvec[0] + dirvec[1]*dirvec[1] + dirvec[2]*dirvec[2] );
        for(i=0;i<3;i++) {
            unitdirvec[i] = dirvec[i] / sqrtf(len);  // unit vector pointing from a to b
        }
           
        test = 1;
        i = 0;

        while( test == 1 ) {
            for(j=0;j<3;j++) {
                from[j] = tmp1[j] + (2*i)*DASH_LENGTH*unitdirvec[j];
                to[j] = tmp1[j] + (2*i + 1)*DASH_LENGTH*unitdirvec[j];
            }
            if( fabsf(tmp1[0] - to[0]) >= fabsf(dirvec[0]) ) {
                for(j=0;j<3;j++) {
                    to[j] = tmp2[j];
                }
                test = 0;
            }
    
            // draw the cylinder
            fprintf(r3d_file, "5 \n"); // flat-ended cylinder
            fprintf(r3d_file, "%7f %7f %7f ", from[0], from[1], from[2]); // first point
            fprintf(r3d_file, "%7f ", float(lineWidth)*DEFAULT_RADIUS); // radius 1
            fprintf(r3d_file, "%7f %7f %7f ", to[0], to[1], to[2]); // second point
            fprintf(r3d_file, "%7f ", float(lineWidth)*DEFAULT_RADIUS); // radius 2
            fprintf(r3d_file, "%3.2f %3.2f %3.2f \n", currentColor[0], currentColor[1],
            currentColor[2]);
            
            i++;
        }
    } else {
        msgErr << "R3dDisplayDevice: Unknown line style " << lineStyle << sendmsg;
    }

}

// draw a cylinder
void R3dDisplayDevice::cylinder(float *a, float *b, float r) {

  float vec1[3], vec2[3];
  float radius;
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
  radius = scale_radius(r);
    
  // draw the cylinder
  
  fprintf(r3d_file, "5 \n"); // flat-ended cylinder
  fprintf(r3d_file, "%7f %7f %7f ", vec1[0], vec1[1], vec1[2]); // first point
  fprintf(r3d_file, "%7f ", radius); // radius 1
  fprintf(r3d_file, "%7f %7f %7f ", vec2[0], vec2[1], vec2[2]); // second point
  fprintf(r3d_file, "%7f ", radius); // radius 2
  fprintf(r3d_file, "%3.2f %3.2f %3.2f \n", currentColor[0], currentColor[1],
  currentColor[2]);

}

// draw a cone
void R3dDisplayDevice::cone(float *a, float *b, float r) {

  register int i, h;
  register float hresinc, rodrad, length, theta, m, n;
  float axis[3], lenaxis[3], perp[3], perp2[3];
  float norms1[4], norms2[4], norms3[4], vert1[4], vert2[4];
  
  // need to do some preprcessing
  lenaxis[0] = a[0] - b[0];
  lenaxis[1] = a[1] - b[1];
  lenaxis[2] = a[2] - b[2];
  
  m = lenaxis[0]*lenaxis[0] + lenaxis[1]*lenaxis[1] + lenaxis[2]*lenaxis[2];
  if (m <= 0.0)
    return;
  length = sqrtf(m);
  axis[0] = lenaxis[0] / length;
  axis[1] = lenaxis[1] / length;
  axis[2] = lenaxis[2] / length;
  
  // dot product with unit axis vector.
  if ((ABS(axis[0]) < ABS(axis[1])) &&
     (ABS(axis[0]) < ABS(axis[2])))
    i = 1;
  else if ((ABS(axis[1]) < ABS(axis[2])))
    i = 1;
  else 
    i = 2;
    
  perp[i] = 0;
  perp[(i+1)%3] = axis[(i+2)%3];
  perp[(i+2)%3] = -axis[(i+1)%3];
  m = sqrtf(perp[0]*perp[0] + perp[1]*perp[1] + perp[2]*perp[2]);
  if (m > 0.0 ) {
    perp[0] /= m;
    perp[1] /= m;
    perp[2] /= m;
  }
  
  // get cross product, perp2 will be normalized automagically.
  perp2[0] = axis[1]*perp[2] - axis[2]*perp[1];
  perp2[1] = axis[2]*perp[0] - axis[0]*perp[1];
  perp2[2] = axis[0]*perp[1] - axis[1]*perp[0];
  
  // do the polygons
  rodrad = r;
  hresinc = TWOPI / 20;
  theta = 0.0;
  for (h=0; h < 20; h++) {
  
    if (h==0) {
      m = cos(theta);
      n = sin(theta);
        norms1[0] = m*perp[0] + n*perp2[0];
        norms1[1] = m*perp[1] + n*perp2[1];
        norms1[2] = m*perp[2] + n*perp2[2];
      } else {
	norms1[0] = norms2[0];
	norms1[1] = norms2[1];
	norms1[2] = norms2[2];
      }

    theta += hresinc;
    m = cos(theta);
    n = sin(theta);
    norms2[0] = m*perp[0] + n*perp2[0];
    norms2[1] = m*perp[1] + n*perp2[1];
    norms2[2] = m*perp[2] + n*perp2[2];

    norms3[0] = 0.5 * (norms1[0] + norms2[0]);
    norms3[1] = 0.5 * (norms1[1] + norms2[1]);
    norms3[2] = 0.5 * (norms1[2] + norms2[2]);

    if(h==0) {
      vert1[0] = a[0] + rodrad * norms1[0];
      vert1[1] = a[1] + rodrad * norms1[1];
      vert1[2] = a[2] + rodrad * norms1[2];
    } else {
      vert1[0] = vert2[0];
      vert1[1] = vert2[1];
      vert1[2] = vert2[2];
    }

    vert2[0] = a[0] + rodrad * norms2[0];
    vert2[1] = a[1] + rodrad * norms2[1];
    vert2[2] = a[2] + rodrad * norms2[2];

    // side of cone
   
    triangle(&vert1[0], &vert2[0], &b[0], &norms1[0], &norms1[0], &norms1[0]);
    
    // bottom disk of cone
    
    triangle(&vert1[0], &vert2[0], &a[0], &axis[0], &axis[0], &axis[0]);
   
  }


}

// draw a triangle
void R3dDisplayDevice::triangle(float *a, float *b, float *c, float *n1, 
float *n2, float *n3) {

  float vec1[3], vec2[3], vec3[3];
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
  (transMat.top()).multpoint3d(c, vec3);

  // draw the triangle
  
  fprintf(r3d_file, "1 \n"); // triangle
  fprintf(r3d_file, "%7f %7f %7f ", vec1[0], vec1[1], vec1[2]); // point one
  fprintf(r3d_file, "%7f %7f %7f ", vec2[0], vec2[1], vec2[2]); // point two
  fprintf(r3d_file, "%7f %7f %7f ", vec3[0], vec3[1], vec3[2]); // point three
  fprintf(r3d_file, "%3.2f %3.2f %3.2f \n", currentColor[0], currentColor[1],
  currentColor[2]);
  
  // this data type is new as of r3d 2.1.5
  // draw the explicit normals

  fprintf(r3d_file, "7 \n"); // explicit normals
  fprintf(r3d_file, "%7f %7f %7f ", n1[0], n1[1], n1[2]); // normal one
  fprintf(r3d_file, "%7f %7f %7f ", n2[0], n2[1], n2[2]); // normal two
  fprintf(r3d_file, "%7f %7f %7f ", n3[0], n3[1], n3[2]); // normal three
  fprintf(r3d_file, "%3.2f %3.2f %3.2f \n", currentColor[0], currentColor[1],
  currentColor[2]);
  
}

// draw a square
void R3dDisplayDevice::square(float *norm, float *a, float *b, float *c, float *d) {
  
  // draw as two triangles
  triangle(a, b, c, norm, norm, norm);
  triangle(a, c, d, norm, norm, norm);
  
}

///////////////////// public virtual routines

// initialize the file for output
int R3dDisplayDevice::prepare3D(int bar) {
  long tilex, tiley;

  bar = bar;

  if(!(r3d_file = fopen(fname,"w"))) {
    msgErr << "Cannot open file " << fname << " for Raster3d output" << sendmsg;
    Initialized = FALSE;
  } else {
  
    Initialized = TRUE;
    r3d_filename = stringdup(fname);
    
    // write header to r3d file (title less than 80 characters, aiieee!)
    // (can we say "FORTRAN" ?)
    // apparently, no such thing as comments, either.
    
    // this is going to be bad if there are multiple command
    // lists.  Have to check for existence of file and 
    // if it exists, open to append, and avoid entering the header.
    // 
    // Better yet, open a new file with "fname.2, fname.3, etc."
    
    fprintf(r3d_file, "r3d input script : %s ", r3d_filename);
    fprintf(r3d_file, "for user %s\n",cuserid(NULL));
    
    // This figures out a image roughly the same resolution
    // as the VMD window.  I use 14 instead of 21 because I am using
    // the 3x3 -> 2x2 anti-aliasing scheme.  See the Raster3D 
    // documentation for more explaination.
    
    tilex = xSize / 14;
    if (xSize%14 != 0) {
        tilex++;
    }
    tiley = ySize / 14;
    if (ySize%14 != 0) {
        tiley++;
    }
    
    fprintf(r3d_file, "%d %d          tiles in x,y \n", tilex, tiley);
    fprintf(r3d_file, "21 21          computing pixels per tile \n");
    
    // anti-aliasing is on, in all cases.
    
    fprintf(r3d_file, "3              alti-aliasing level 3; 3x3 -> 2x2 \n");
    
    // background color is black until I figure out a good way to set it
    // interactively
    
    fprintf(r3d_file, "%3.2f %3.2f %3.2f          background color\n", 
    backColor[0], backColor[1], backColor[2]);
    
    // shadows on
    
    fprintf(r3d_file, "T              shadows on \n");
    
    // Phong power for specular highlights.  Smaller value -> Larger spot.
    // light values, too.
    
    // smaller phong value equals larger spots.
    fprintf(r3d_file, "20             Phong power \n");
    // primary = 1-secondary
    fprintf(r3d_file, "0.25           secondary light contribution \n");
    fprintf(r3d_file, "0.10           ambient light contribution \n");
    fprintf(r3d_file, "0.50           specular reflection component \n");
    
    // Eyepos will be fun to mess with.  Eyepos = 4 gives a perspective
    // corresponding to a viewing distance 4 times the narrow dimension 
    // of the described scene.
    
    fprintf(r3d_file, "2.0              Eye position \n");
    
    // Primary light source position.  Only this light casts shadows.
    // Either use the first light or use the default light position.
    
    if (lightDefined[1]) { 
        fprintf(r3d_file, 
        "%7.6f %7.6f %7.6f          Main light source position\n",
        lightPos[0][0], lightPos[0][1], lightPos[0][2]);
    } else {
        fprintf(r3d_file, "-0.2 0.2 1          Main light source position \n");
    }
    
    // Global transformation matrix for objects.  This will not
    // be used for the objects in this implementation
    
    fprintf(r3d_file, "1 0 0 0        Global xform matrix \n");
    fprintf(r3d_file, "0 1 0 0 \n");
    fprintf(r3d_file, "0 0 1 0 \n");
    fprintf(r3d_file, "0 0 0 2.2 \n"); // the 2.2 make Raster3d's image
    // the same size as VMD's screen image.  The factor was found by
    // hand and really depends on the screen W and D values
    
    // Input mode is always 3
    
    fprintf(r3d_file, "3 \n");
    
    // three stars for some reason
    
    fprintf(r3d_file, "* \n* \n* \n");
    
    // and that's it for the header.  next comes free format 
    // triangle, sphere, and cylinder descriptors
    
  }
  return Initialized;
}

    
// clean up after yourself
void R3dDisplayDevice::update(int foo) {

  foo = foo;
  
  if(Initialized) {
    fclose(r3d_file);
    if(r3d_filename) delete [] r3d_filename;
    msgInfo << "Raster3D file generation finished" << sendmsg;
  }
}

