/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: PickModeAtomMove.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 96/03/23 05:12:00 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * PickMode is the base class for all objects which perform some form
 * of processing of object selection via a pointer.  When a pointer is used
 * to select an object, the pointer is in a certain pick mode.  A PickList
 * object maintains a list of classes derived from this PickMode; virtual
 * routines in this class are called once an item has been selected, to
 * process and use the motion of the mouse at the starting, during, and end
 * of the picking process.
 *
 * PickModeAtomMove is used to allow the user to change the coordinates of
 * an atom with the pointer.
 *
 ***************************************************************************/

#include <math.h>
#include <string.h>
#include "PickModeAtomMove.h"
#include "Pickable.h"
#include "DisplayDevice.h"
#include "Molecule.h"
#include "MoleculeList.h"
#include "Inform.h"
#include "utilities.h"

#ifdef USE_GL_MOUSE_HACK
#include <gl/gl.h>     // use these for now
#include <gl/device.h>
#endif

//////////////////////////// constructor  ////////////////////////////
PickModeAtomMove::PickModeAtomMove(char *nm, MoleculeList *mlist,
				   AtomMoveMethod movemethod) {
  moveName = stringdup(nm);
  moveMethod = movemethod;
  pQuery = NULL;
  molList = mlist;

  MSGDEBUG(1,"Created new PickModeAtomMove " << moveName << sendmsg);
}


//////////////////////////// destructor  ////////////////////////////
PickModeAtomMove::~PickModeAtomMove(void) {
  delete [] moveName;
}


///////////////////// private routines  ////////////////////////

// For the given Pickable, determine if it is a remote mol or a representation
// of a molecule, and return the proper pointer if it is (NULL otherwise)
Molecule *PickModeAtomMove::check_pickable(Pickable *pobj) {

  // search through each remote molecule ...
  int mnum = molList->num();
  for(int i=0; i < mnum; i++) {

    // and check each molecule's reps to see if it matches
    Molecule *mol = molList->molecule(i);
    if(pobj == mol)
      return mol;

    int repnum = mol->components();
    for(int j=0; j < repnum; j++)
      if(pobj == mol->component(j))
	return mol;
  }

  // if here, nothing found
  return NULL;
}


///////////////////// public virtual routines  ////////////////////////

// called when a pick is begun:
//	args = display to use, obj picked, button, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
void PickModeAtomMove::pick_start(DisplayDevice *d,
			Pickable *p, int , int atm, int dim, float *pos) {


  if((pQuery = check_pickable(p)) != NULL) {

    // selected an atom in a molecule ... start process now
    moveAtom = atm;

    Timestep *ts = pQuery->current();
    if(!(moveAtom < 0 || ts == NULL)) {
      // get the current pointer position in 3 space
      float *atomPos = ts->pos + moveAtom * 3;
      float *newpos;
      float mouseObjPos[3];
      if(dim == 2) {
        float atomWorldPos[3];

        // convert the atom position to world coordinates
        (pQuery->tm).multpoint3d(atomPos, atomWorldPos);

        // find the 3D position (in world coords) of the mouse
        d->find_3D_from_2D(atomWorldPos, pos, mouseObjPos);

        // indicate the mouse world pos is the position to convert back to
        // molecule object coordinates
        newpos = mouseObjPos;

      } else {

        // for 3D pointer, just use the pointer position as-is
        newpos = pos;
      }
      // save the position for use next time
      copy(lastPos, newpos);
    } 
  }
}


// called when a pick moves:
//	args = display to use, obj picked, button, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
void PickModeAtomMove::pick_move(DisplayDevice *d,
			Pickable *, int btn, int, int dim, float *pos) {

  // move the selected atom if we are currently picking an object
  if(pQuery != NULL) {

    // get the current timestep
    Timestep *ts = pQuery->current();
    if(moveAtom < 0 || ts == NULL)
      return;

    // find the new position of the atom, from the pointer
    float *atomPos = ts->pos + moveAtom * 3;
    float *newpos;
    float mouseObjPos[3];
    if(dim == 2) {
      float atomWorldPos[3];

      // convert the atom position to world coordinates
      (pQuery->tm).multpoint3d(atomPos, atomWorldPos);

      // find the 3D position (in world coords) of the mouse
      d->find_3D_from_2D(atomWorldPos, pos, mouseObjPos);

      // indicate the mouse world pos is the position to convert back to
      // molecule object coordinates
      newpos = mouseObjPos;

    } else {

      // for 3D pointer, just use the pointer position as-is
      newpos = pos;
    }

    // convert the new position back to molecule object coordinates
    Matrix4 tminv(pQuery->tm);
    tminv.inverse();
    float moveAmount[3];
    tminv.multpoint3d(newpos, moveAmount);

    // find the position change vector = mouse - atom
    subtract(moveAmount, moveAmount, atomPos);

    // now, based on the picking method, move either an atom, or the
    // residue, or the fragment
    if(moveMethod == MOVE_ATOM) {
      add(atomPos, atomPos, moveAmount);

    } else if(moveMethod == MOVE_RESIDUE) {      //////////// RESIDUE
      // if the shift key is pressed, do rotations
      if (d -> shift_state() & DisplayDevice::SHIFT) {
        // center is at the picked atom
        Matrix4 trans;
        trans.translate(-atomPos[0], -atomPos[1], -atomPos[2]);
        // rotation is determined by a rotation about x, then y;
        // or z if the middle button was used
        Matrix4 rotx, roty, rotz;
        if (dim == 2) {
          if (btn == MouseEvent::B_MIDDLE ||
	      d -> shift_state() & DisplayDevice::ALT) {
            rotz.rot( (newpos[0] - lastPos[0])*150, 'z');
          } else {
            roty.rot( (newpos[0] - lastPos[0])*150, 'y');
            rotx.rot( (-newpos[1] + lastPos[1])*150, 'x');
          }
        } else {
          rotx.rot( (newpos[0] - lastPos[0])*150, 'x');
          roty.rot( (newpos[1] - lastPos[1])*150, 'y');
          rotz.rot( (newpos[2] - lastPos[2])*150, 'z');
        }
    
        // return to original coordinates
        Matrix4 transinv;
        transinv.translate(atomPos[0], atomPos[1], atomPos[2]);
        
        // compute the total transformation matrix
	Matrix4 rotinv(pQuery->rotm);
	rotinv.inverse();
        Matrix4 transmat;
        transmat.multmatrix(transinv);
	transmat.multmatrix(rotinv);
        transmat.multmatrix(rotz);
        transmat.multmatrix(roty);
        transmat.multmatrix(rotx);
	transmat.multmatrix(pQuery->rotm);
        transmat.multmatrix(trans);
  
        // matrix generated; apply it to the atoms in the residue
        Residue *res = pQuery->atom_residue(moveAtom);
        if (res) {
          float tmp[4];
          tmp[3] = 1.0;
          int natm = (res->atoms).num();
          for (int n=0; n<natm; n++) {
            float *ap = ts->pos+3*(res->atoms)[n];
            copy(tmp, ap);
            transmat.multpoint4d(tmp,tmp);
            copy(ap, tmp);
          }
        }
      } else  // do a translation
	{
        // take the difference between the curr and prev world coordinates
        float old_moveAmount[3];
        tminv.multpoint3d(lastPos, old_moveAmount);
        subtract(old_moveAmount, old_moveAmount, atomPos);
        subtract(moveAmount, moveAmount, old_moveAmount);
        Residue *res = pQuery->atom_residue(moveAtom);
        if(res) {
	  int natm = (res->atoms).num();
	  for(int n = 0; n < natm; n++) {
	    float *ap = ts->pos + 3 * (res->atoms)[n];
	    add(ap, ap, moveAmount);
	  }
        }
      }
    } else if(moveMethod == MOVE_FRAGMENT) {      /// FRAGMENT
      // if the shift key is pressed, do rotations
      if (d -> shift_state() & DisplayDevice::SHIFT) {
        // center is at the picked atom
        Matrix4 trans;
        trans.translate(-atomPos[0], -atomPos[1], -atomPos[2]);
        // rotation is determined by a rotation about x, then y;
        // or z if the middle button was used
        Matrix4 rotx, roty, rotz;
        if (dim == 2) {
          if (btn == MouseEvent::B_MIDDLE ||
	      d -> shift_state() & DisplayDevice::ALT) {
            rotz.rot( (newpos[0] - lastPos[0])*150, 'z');
          } else {
            roty.rot( (newpos[0] - lastPos[0])*150, 'y');
            rotx.rot( (-newpos[1] + lastPos[1])*150, 'x');
          }
        } else {
          rotx.rot( (newpos[0] - lastPos[0])*150, 'x');
          roty.rot( (newpos[1] - lastPos[1])*150, 'y');
          rotz.rot( (newpos[2] - lastPos[2])*150, 'z');
        }
	
        // return to original coordinates
        Matrix4 transinv;
        transinv.translate(atomPos[0], atomPos[1], atomPos[2]);
        
        // compute the total transformation matrix
	Matrix4 rotinv(pQuery->rotm);
	rotinv.inverse();
        Matrix4 transmat;
        transmat.multmatrix(transinv);
	transmat.multmatrix(rotinv);
        transmat.multmatrix(rotz);
        transmat.multmatrix(roty);
        transmat.multmatrix(rotx);
	transmat.multmatrix(pQuery->rotm);
        transmat.multmatrix(trans);
	
        // matrix generated; apply it to the atoms in the residue
        Fragment *frag = pQuery->atom_fragment(moveAtom);
        if (frag) {
	  int nres = frag->num();
          float tmp[4];
	  tmp[3] = 1.0;
	  for (int r=0; r<nres; r++) {
	    Residue *res = pQuery->residue((*frag)[r]);
	    int natm = (res->atoms).num();
	    for(int n = 0; n < natm; n++) {
	      float *ap = ts->pos + 3 * (res->atoms)[n];
	      copy(tmp, ap);
	      transmat.multpoint4d(tmp,tmp);
	      copy(ap, tmp);
	    }
	  }	    
        }
      } else  // do a translation
	{
	  // take the difference between the curr and prev world coordinates
	  float old_moveAmount[3];
	  tminv.multpoint3d(lastPos, old_moveAmount);
	  subtract(old_moveAmount, old_moveAmount, atomPos);
	  subtract(moveAmount, moveAmount, old_moveAmount);
	  Fragment *frag = pQuery->atom_fragment(moveAtom);
	  if(frag) {
	    int nres = frag->num();
	    for(int r = 0; r < nres; r++) {
	      Residue *res = pQuery->residue((*frag)[r]);
	      int natm = (res->atoms).num();
	      for(int n = 0; n < natm; n++) {
		float *ap = ts->pos + 3 * (res->atoms)[n];
		add(ap, ap, moveAmount);
	      }
	    }
	  }
	}
    }

    // save the (now) old coordinate
    copy(lastPos, newpos);
        
    // tell the molecule that something has changed
    pQuery->force_recalc();
  }
}


// called when a pick ends:
//	args = display to use, obj picked, button, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
// Here, who cares what the button was
void PickModeAtomMove::pick_end(DisplayDevice *,
			Pickable *, int, int, int, float *) {
  // nothing else to do but note we're not moving anything now
  pQuery = NULL;
}


