/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: PickList.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 95/11/11 23:52:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The PickList object, which maintains a list of Pickable objects and
 * has the ability to find and deal with items picked by a pointer device.
 *
 * Each Scene is derived from PickList.
 *
 * Each PickList has a list of possible picking modes; these are created by
 * the request of Pickable objects, which then get notified when an
 * object is picked.  Only Pickables which are interested in the specified
 * mode are told of the picking result.
 *
 * For each pick mode, there is a corresponding PickMode object; PickMode
 * is a base class for the particular type of picking mode requested.
 * Pickable objects may request to have a new pick mode added to this list,
 * and if they do so they must provide an instance of the PickMode object
 * they require.  If the mode exists, the objects may replace the mode or
 * may just use the current one.  When modes are removed, the instance is
 * deleted.
 *
 * Only one picking operation may be in effect at a time.  A picking operation
 * consists of these steps:
 *	1. A pick is started by a pointer, by queueing a CmdPickStart command.
 *	2. pick_start is called, which determines if something is selected.
 *		If so, returns tag of item, and sets internal flags to
 *		indicate what is being picked, and how.
 *		2.1 Each Pickable which is interested in the current mode is
 *			told when the pick starts, and all later steps.
 *		2.2 The object for the current pick mode is also told of
 *			the start of the pick.
 *	3. pick_move is called whenever the pointer moves, by queueing the
 *		CmdPickMove command.  This continues until the picking is
 *		finished.  The PickMode object and interested Pickables are
 *		also told of the move.
 *	4. pick_end is called by queueing CmdPickEnd; this behaves similarly
 *		the pick_move.  When finished, the internal flags are reset
 *		to indicate that picking is finished, and a new one can begin.
 * NOTE: multiple concurrent picks could be implemented later by providing
 * an object which maintains separate sets of the internal flags and variables,
 * for each picking device.  This would also require another argument to the
 * pick_* routines, to indicate which picking device is being used.  Also,
 * PickMode and Pickable object involved in another picking operation would
 * have to be excluded from being informed of other picking operations.
 ***************************************************************************/
#ifndef PICKLIST_H
#define PICKLIST_H

#include "ResizeArray.h"
#include "NameList.h"
class Pickable;
class PickMode;
class DisplayDevice;


class PickList {

private:
  // current picking state: button, mode, last position
  int currPickMode, currPickDim, currPickTag;
  PickMode *currPickModePtr;
  Pickable *currPickable;

protected:
  // list of Pickable objects which contain pickable objects and which
  // wish to be notified of successful picks
  ResizeArray<Pickable *> pickableObjs;
  
  // flags for whether the Pickable objects in the associated array are
  // interested in the current pick mode
  ResizeArray<int> pickableInterested;

  // list of PickModes, which currently may only be added (or replaced).
  NameList<PickMode *> pickModes;
  
public:
  // constructor: no information needed
  PickList(void);
  
  // destructor: delete all pickModes that have been added
  ~PickList(void);
  
  //
  // routines for registering Pickable objects
  //
  
  // see if the given Pickable is in the current list of ones
  int has_pickable(Pickable *p) { return (pickableObjs.find(p) >= 0); }
  
  // adds a Pickable to the current list, if it is not already in the list
  // Return TRUE if the item is being added for the first time.
  int add_pickable(Pickable *);

  // remove the given pickable from the list; return TRUE if it was in the list
  int remove_pickable(Pickable *);
  
  // return number of pickable objects
  int num_pickable(void) { return pickableObjs.num(); }

  // return the Nth pickable
  Pickable *pickable(int n) { return pickableObjs[n]; }

  //
  // routines for adding/replacing pick modes
  //
  
  // return if a pick mode of the given name exists already
  int pick_mode_num(char *name) { return pickModes.typecode(name); }
  
  // add a new pick mode, with the given name.  Return index of this mode.
  // If a mode of the name exists already, delete the old one and replace it
  // with the new one provided.
  int add_pick_mode(char *, PickMode *);

  // return number of pick modes
  int num_pick_modes(void) { return pickModes.num(); }

  // return the Nth pick mode object
  PickMode *pick_mode(int n) { return pickModes.data(n); }

  // return the name of the Nth pick mode
  char *pick_mode_name(int n) { return pickModes.name(n); }

  //
  // routines to handle starting a pick, moving during a pick,
  // ending of a pick
  //

  // return whether something is being picked at present.
  int picking(void) { return (currPickMode >= 0); }
  
  // return the currently active pickable, i.e. the one most recently
  // returned when a pick was checked.
  Pickable *picked_pickable(void) { return currPickable; }

  // return the tag used for the most recently picked object
  int picked_tag(void) { return currPickTag; }

  // using the given display, this checks to see if any object is under
  // the given pointer position.  This does not set any internal picking
  // flags, it just checks to see if something has been picked.  Returns
  // a pointer to the Pickable object selected if successful, or NULL if
  // nothing is picked.  If successful, returns 'tag' of item in final 
  // argument.
  // arguments: display device, dim, position, returned tag
  Pickable *pick_check(DisplayDevice *, int, float *, int &);

  // called when a pick is begun: display device, button, mode, dim, pos
  // returns 'tag' of closest object, or (-1) if nothing is picked.
  // When a pick is started, the internal flags for this object are set,
  // and no other item can be picked until pick_end is called.
  // For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
  // For 3D version: x,y,z are transformed position of pointer
  int pick_start(DisplayDevice *, int, int, int, float *);
  
  // called when a pick moves: display device, button, mode, dim, pos
  // Returns TRUE if a pick is currently active, FALSE otherwise.
  // For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
  // For 3D version: x,y,z are transformed position of pointer
  int pick_move(DisplayDevice *, int, int, int, float *);
  
  // called when a pick ends: display device, button, mode, dim, pos
  // Returns TRUE if a pick is currently active, FALSE otherwise.
  // For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
  // For 3D version: x,y,z are transformed position of pointer
  int pick_end(DisplayDevice *, int, int, int, float *);

};

#endif

