/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: OpenGLRenderer.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $	$Date: 96/02/01 01:34:02 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that use OpenGL for rendering.
 * Will render drawing commands into a window.
 * This is not the complete definition,
 * however, of a DisplayDevice; something must provide routines to open
 * windows, reshape, clear, set perspective, etc.  This object contains the
 * code to render a display command list.
 *
 ***************************************************************************/
#ifndef OPENGLRENDERER_H
#define OPENGLRENDERER_H

#include "DisplayDevice.h"
#include <GL/gl.h>
#include <GL/glu.h>

// forward declaration of classes here
class OpenGLRenderer;


// the class definition
class OpenGLRenderer : public DisplayDevice {

private:
  // pointer to data block (should be changed to allow for multiple blocks)
  float *dataBlock;

  // do we need to turn antialiasing and depth-cueing on when materials
  // are turned off?
  int need_aaon, need_cueon;

  // quadric objects for drawing spheres, cylinders, and disks
  GLUquadricObj *pointsQuadric;
  GLUquadricObj *objQuadric;
  GLUquadricObj *sphereQuadric;

protected:
  // font info to use for our display ... MUST BE SET BY DERIVED CLASS
  int haveFontData;
  GLuint fontListBase;

  // total size of the screen, in pixels ... MUST BE SET BY DERIVED CLASS
  int screenX, screenY;

  // routines to perform various OGL-specific graphics operations
  void set_line_width(int);
  void set_line_style(int);
  void set_sphere_res(int);
  void set_sphere_mode(int);
  void triangle(float *x1, float *x2, float *x3,
		float *n1, float *n2, float *n3);  // triangle with normals
  void square(float *norm, float *, float *, float *, float *); 
  void cylinder(float *, float *, int, float, float);  // slow cylinder version
  
  //
  // protected virtual routines
  //
  
  // virtual routines to deal with light sources at device level
  // all return success of operation
  virtual int do_define_light(int n, float *color, float *position);
  virtual int do_activate_light(int n, int turnon);

  // virtual routines to deal with materials at device level
  // all return success of operation
  virtual int do_define_material(int n, float *data);
  virtual int do_activate_material(int n, int turnon);

public:
  // constructor/destructor
  OpenGLRenderer(char *);
  virtual ~OpenGLRenderer(void);

  //
  // virtual routines to affect the devices transformation matrix
  //
  
  virtual void push(void);		// push device's curr state on stack
  virtual void pop(void);		// pop device's curr state from stack
  virtual void loadmatrix(Matrix4 &);	// replace trans matrix w. given one
  virtual void multmatrix(Matrix4 &);	// multiply trans matrix w. given one

  //
  // virtual routines to find characteristics of display itself
  //

  // return absolute 2D screen coordinates, given 2D or 3D world coordinates.
  virtual void abs_screen_loc_3D(float *, long *);
  virtual void abs_screen_loc_2D(float *, long *);

  // size of physical display object
  virtual void screen_size_mm(long &, long &);
  virtual void screen_size_pixel(long &, long &);

  // Given a 3D point (pos A),
  // and a 2D rel screen pos point (for pos B), computes the 3D point
  // which goes with the second 2D point at pos B.  Result returned in B3D.
  virtual void find_3D_from_2D(float *A3D, float *B2D, float *B3D);

  //
  // virtual functions to turn on/off depth cuing and antialiasing
  //
  
  virtual void aa_on(void);
  virtual void aa_off(void);
  virtual void cueing_on(void);
  virtual void cueing_off(void);

  //
  // virtual routines for preparing to draw, drawing, and finishing drawing
  //
  
  virtual void render(void *);		// process list of draw cmds

};

#endif

/***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	OpenGLRenderer.h,v $
 * Revision 1.2  96/02/01  01:34:02  01:34:02  billh (Bill Humphrey)
 * Fixed problem with finding the origin of the window;
 * added character text display and cursor shape changing capabilities.
 * 
 * Revision 1.1  1995/10/19 16:08:17  billh
 * Initial revision
 *
 * Revision 1.9  1995/08/15  21:33:20  dalke
 * added support for AIX3
 *
 * Revision 1.8  1995/06/07  02:28:23  billh
 * Moved cursor and button virtual functions to GLDisplayDevice
 * (from GLRenderer).
 *
 * Revision 1.7  95/05/11  22:49:06  billh
 * Added virtual functions to queue and check events, and to get/modify
 * the cursor position, shape, etc.  All graphics/windowing-specific
 * functions have now been moved to the DisplayDevice.
 * 
 * Revision 1.6  95/04/11  09:46:51  dalke
 * new formate for the DTRIANGLE that allows different normals
 * for the triangle verticies.
 * 
 * Revision 1.5  1995/03/24  18:49:38  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 *
 * Revision 1.4  1995/03/10  07:49:33  dalke
 * added fast cylinders by precomputing in the DispCmd
 * This makes things about 2.5 times faster for pure cylinder
 * drawing.
 *
 * Revision 1.3  1995/02/22  04:07:05  billh
 * Added virtual routine find_3D_from_2D, which takes a 3D point at pos A,
 * and 2D rel screen position at B, and returns the 3D point corresponding
 * to the 2D point at B in the same plane as point A.
 *
 * Revision 1.2  94/10/04  22:07:55  billh
 * Took out sphere library usage if __NPGL__ is defined, since the npgl lib
 * does not have a version of the sphere drawing routines.  Spheres are
 * just drawn as points for npgl rendering.
 * 
 * Revision 1.1  94/09/23  06:01:39  billh
 * Initial revision
 * 
 ***************************************************************************/
