/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Mouse.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.27 $	$Date: 96/05/14 04:59:09 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Mouse UI object, which maintains the current state of the 
 * mouse, including what it is currently used for, and how much it has moved
 * from one measurement to the next.  This also deals with any pop-up or
 * pull-down menus available by using the mouse, as well as picking objects.
 *
 * A three-button mouse is assumed here, with the following usage:
 *	1) Buttons 1 and 2 : manipulation and picking.
 *	2) Button 3 (right): pop-up menu
 *
 * This is the general base class definition; specific versions for each
 * windowing/graphics system must be supplied.
 *
 ***************************************************************************/

#include <ctype.h>
#include "Mouse.h"
#include "MouseEvent.h"
#include "PopupMenu.h"
#include "DisplayDevice.h"
#include "Pickable.h"
#include "Scene.h"
#include "LightList.h"
#include "UIList.h"
#include "UIText.h"
#include "MolAction.h"
#include "Molecule.h"
#include "GeometryList.h"
#include "CmdDisplay.h"
#include "CmdPick.h"
#include "CmdTrans.h"
#include "Inform.h"
#include "Global.h"

#ifdef VMDTCL
#include "tcl.h"
#endif

// string names for the move modes
static char *mm_names[5] = {"Rotate","Translate","Scale","Lights","Pick"};

// bitmap for different cursors
// translation cursor
unsigned short transCursor[32] = { 0x0000, 0x0080, 0x01C0, 0x03E0,
                                   0x0080, 0x0080, 0x1084, 0x3086,
                                   0x7FFF, 0x3086, 0x1804, 0x0080,
                                   0x0080, 0x03E0, 0x01C0, 0x0080,

                                   0x0000, 0x0140, 0x0220, 0x0410,
                                   0x0770, 0x194C, 0x294A, 0x4F79,
                                   0x0000, 0x4F79, 0x294A, 0x194C,
                                   0x0770, 0x0410, 0x0220, 0x0140 };

// scaling cursor
unsigned short scaleCursor[32] = { 0x0000, 0x000E, 0x0006, 0x000A,
                                   0x0010, 0x0020, 0x00E0, 0x01C0,
                                   0x0790, 0x4F00, 0x7F00, 0x7E00,
                                   0x7C00, 0x7C00, 0x7E00, 0x0000,

                                   0x001F, 0x0011, 0x0019, 0x0035,
                                   0x006F, 0x01D8, 0x0310, 0x0E30,
                                   0xF860, 0xB0C0, 0x8080, 0x8180,
                                   0x8300, 0x8300, 0x8100, 0xFF00 };

// picking cursor
unsigned short pickCursor[32] = { 0x0180, 0x0180, 0x0180, 0x0180,
                                  0x0180, 0x0180, 0x0180, 0xFFFF,
                                  0xFFFF, 0x0180, 0x0180, 0x0180,
                                  0x0180, 0x0180, 0x0180, 0x0180,

                                  0x0000, 0x0000, 0x0000, 0x0000,
                                  0x0000, 0x0000, 0x0000, 0x0000,
                                  0x0000, 0x0000, 0x0000, 0x0000,
                                  0x0000, 0x0000, 0x0000, 0x0000 };

// wait cursor (an hourglass)
unsigned short waitCursor[32] = { 0x1FF0, 0x1FF0, 0x0820, 0x0820,
                                  0x0820, 0x0C60, 0x06C0, 0x0100,
                                  0x0100, 0x06C0, 0x0C60, 0x0820,
                                  0x0820, 0x0820, 0x1FF0, 0x1FF0,

                                  0x0000, 0x0000, 0x0000, 0x0000,
                                  0x0000, 0x0000, 0x0000, 0x0000,
                                  0x0000, 0x0000, 0x0000, 0x0000,
                                  0x0000, 0x0000, 0x0000, 0x0000 };

// commands we are interested in
static int numCmds = 1;
static int cmdList[1] = { Command::WIN_EVENT };


// constructor
Mouse::Mouse(UIList *uil, CommandQueue *cq, DisplayDevice *d)
	: UIObject("Mouse", uil, cq) {

  MSGDEBUG(1, "Creating Mouse ..." << sendmsg);

  // tell the display to queue events, if necessary ... only if there is
  // no GUI available
#ifndef VMDGUI
  display->queue_events();
#endif

  // record which commands we want
  for(int i=0; i < numCmds; command_wanted(cmdList[i++]));
  
  // save the display device to be used by the mouse
  dispDev = d;

  // don't need to do this, but I like setting values
  old_detail_level = d -> detail_level();

  // set the default translation and rotation increments
  transInc = 0.002;
  rotInc = 1.0/15.0;
  scaleInc = 0.0002;
  RotVelScale = 0.4;
  currX = currY = oldX = oldY = 0;
  stop_rotation();
  moveObj = 0;
  moveMode = ROTATION;
  activePickMode = (-1);
  pickInProgress = FALSE;
  isTracking = FALSE;
  activeButton = MouseEvent::B_NONE;
  mouseMoved = FALSE;

  // define the cursors
  dispDev->change_cursor_shape(TRANS_CURSOR, transCursor, 8, 8);
  dispDev->change_cursor_shape(SCALE_CURSOR, scaleCursor, 0, 15);
  dispDev->change_cursor_shape(PICK_CURSOR , pickCursor, 8, 8);
  dispDev->change_cursor_shape(WAIT_CURSOR , waitCursor, 8, 8);
  dispDev->set_cursor();

  // create initial version of keystroke commands=
  create_key_commands();

  // create user pop-up menus
  create_user_menu("User Commands");
  mainUserMenu = userMenuList.data("User Commands");
}


// destructor
Mouse::~Mouse(void) {
  move_mode(ROTATION);
}


////////////////////////////  protected routines  /////////////////

// strings used to create label menu items
static char *label_cmds[] = { "Show", "Hide", "Delete" };

// routine which creates the initial mouse pop-up menu
// The 1st argument is an optional Pickable object which may have a
// popup menu available for it; in that case  it is added to the main
// menu displayed.  If the 1st argument is null, this just creates the
// regular popup menu.  The second argument is the tag of the picked item.
PopupMenu *Mouse::create_popup_menu(Pickable *pickedObj, int tag) {
  register int i, j, k;
  PopupMenu *sm1, *sm2, *sm3;
  char mbuf[128], mbuf2[64];

  // create the main-level menu
  PopupMenu *mouseMenu = new PopupMenu("Main Menu");

  // if a displayable was given, and it has a specialized popup menu,
  // add it first
  if(pickedObj) {
    PopupMenu *dpm = pickedObj->create_popup_menu(tag);

    // only need to add this if a specialized menu is available
    if(dpm) {
      mouseMenu->add_submenu(dpm, TRUE, FALSE, FALSE);
      mouseMenu->add_separator();
    }
  }

  // create sub-menu for setting the mouse mode
  sm1 = new PopupMenu("Mouse Mode");
  sm1->add_item(move_mode_name(ROTATION), "mouse mode 0 0", TRUE, TRUE,
		curr_move_mode() == ROTATION);
  sm1->add_item(move_mode_name(TRANSLATION), "mouse mode 1 0", TRUE, TRUE,
		curr_move_mode() == TRANSLATION);
  sm1->add_item(move_mode_name(SCALING), "mouse mode 2 0", TRUE, TRUE,
		curr_move_mode() == SCALING);

  sm2 = new PopupMenu("Move Light");
  lights->reset();
  i = 0;
  while(lights->is_current()) {
    sprintf(mbuf,"Light %1d",i);
    sprintf(mbuf2,"mouse mode 3 %d",i);
    sm2->add_item(mbuf, mbuf2, TRUE, TRUE,
			 (lights->current())->light_displayed());
    lights->next();
    i++;
  }
  sm1->add_submenu(sm2, TRUE, TRUE, curr_move_mode() == LIGHT);

  mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);

  // create sub-menu for setting the picking mode
  if(scene) {
    sm1 = new PopupMenu("Pick Item Mode");
    for(i=0; i < scene->num_pick_modes(); i++) {
      sprintf(mbuf2,"mouse mode 4 %d",i);
      sm1->add_item(scene->pick_mode_name(i), mbuf2, TRUE, TRUE,
			curr_pick_mode() == i);
    }
    mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);
  }

  // create sub-menu for setting the stereo mode
  sm1 = new PopupMenu("Display Modes");
  //  add the two viewing projections
  sm1->add_item("Perspective", "display projection perspective", TRUE, 
		TRUE, dispDev -> projection() == DisplayDevice::PERSPECTIVE);
  sm1->add_item("Orthographic", "display projection orthographic", TRUE, 
		TRUE, dispDev -> projection() == DisplayDevice::ORTHOGRAPHIC);
  sm1->add_separator();
  for(i=0; i < dispDev->num_stereo_modes(); i++) {
    sprintf(mbuf,"display stereo %s", dispDev->stereo_name(i));
    sm1->add_item( i == 0 ? "Stereo off" : dispDev->stereo_name(i), 
		   mbuf, TRUE, TRUE, dispDev->stereo_mode() == i);
  }
  mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);

#ifdef VMDTCL
  // the best fit alignment routines
  sm1 = new PopupMenu("Fit");
  // The information if I should just print the RMSD or actually do
  // the alignment is stored in the variable "vmd_fit_do_align"
  {
    int do_align = 1;  // default is to do the alignment
    char *s = Tcl_GetVar(uiText -> tclInterp, "vmd_fit_do_align",
			 TCL_GLOBAL_ONLY);
    if (!s || Tcl_GetBoolean(uiText -> tclInterp, s, &do_align) != TCL_OK) {
      do_align = 1;
    }
    sm1 -> add_item("Do RMSD Fit", "vmd_fit_do_align_true", TRUE, TRUE, 
		    do_align);
    sm1 -> add_item("Print RMSD", "vmd_fit_do_align_false", TRUE, TRUE, 
		    !do_align);
  }
  sm2 = new PopupMenu("Two Molecules");
  sm1 -> add_separator();
  {
    // is this on?  What is the mode?
    int is_fitting = 0, fit_mode = 0;
    char *s;
    s = Tcl_GetVar(uiText -> tclInterp, "vmd_fit_started",
		   TCL_GLOBAL_ONLY);
    if (!s || Tcl_GetBoolean(uiText -> tclInterp, s, 
			     &is_fitting) != TCL_OK) {
      is_fitting = FALSE;
    }
    s = Tcl_GetVar(uiText -> tclInterp, "vmd_fit_mode",
		   TCL_GLOBAL_ONLY);
    if (!s || Tcl_GetInt(uiText -> tclInterp, s,
			     &fit_mode) != TCL_OK) {
      fit_mode = 0;
    }
    sm2 -> add_item("All Atoms", "vmd_fit_molecule_all", 
		    TRUE, TRUE, is_fitting && fit_mode == 0);
    sm2 -> add_item("Heavy Atoms", "vmd_fit_molecule_heavy", 
		    TRUE, TRUE, is_fitting && fit_mode == 1);
    sm2 -> add_item("Backbone Atoms", "vmd_fit_molecule_backbone", 
		    TRUE, TRUE, is_fitting && fit_mode == 2);
    sm2 -> add_item("CA Atoms", "vmd_fit_molecule_ca", 
		    TRUE, TRUE, is_fitting && fit_mode == 3);
    sm2 -> add_item("Picked Atoms", "vmd_fit_molecule_pick", 
		    TRUE, TRUE, is_fitting && fit_mode == 4);
    sm1 -> add_submenu(sm2, TRUE, FALSE);

    sm2 = new PopupMenu("Two Fragments");
    sm2 -> add_item("All Atoms", "vmd_fit_fragment_all", 
		    TRUE, TRUE, is_fitting && fit_mode == 5);
    sm2 -> add_item("Heavy Atoms", "vmd_fit_fragment_heavy", 
		    TRUE, TRUE, is_fitting && fit_mode == 6);
    sm2 -> add_item("Backbone Atoms", "vmd_fit_fragment_backbone", 
		    TRUE, TRUE, is_fitting && fit_mode == 7);
    sm2 -> add_item("CA Atoms", "vmd_fit_fragment_ca", 
		    TRUE, TRUE, is_fitting && fit_mode == 8);
    sm2 -> add_item("Picked Atoms", "vmd_fit_fragment_pick", 
		    TRUE, TRUE, is_fitting && fit_mode == 9);
    sm1 -> add_submenu(sm2, TRUE, FALSE);
    mouseMenu->add_submenu(sm1, TRUE, FALSE);
  }
#endif

  mouseMenu->add_separator();

  // create sub-menu for changing labels
  sm1 = new PopupMenu("Labels");

  for(i=0; i < sizeof(label_cmds) / sizeof(char *); i++) {
    sm2 = new PopupMenu(label_cmds[i]);

    // for each category, add a similar menu
    for(j=0; j < geometryList->num_lists(); j++) {

      // first add an "all" command
      sm3 = new PopupMenu(geometryList->geom_list_name(j));
      sprintf(mbuf,"label %s %s all", sm2->name(), sm3->name());
      sm3->add_item("All", mbuf, TRUE, FALSE, FALSE);

      // now add an entry for each label
      GeomListPtr glist = geometryList->geom_list(j);
      int gnum = glist->num();
      for(k=0; k < gnum; k++) {
	Geometry *g = (*glist)[k];
	sprintf(mbuf, "label %s %s %d", sm2->name(), sm3->name(), k);
	sm3->add_item(g->name(), mbuf, TRUE, TRUE, g->on());
      }

      // finally add this menu to the parent one
      sm2->add_submenu(sm3, TRUE, FALSE, FALSE);
    }

    // finally add this menu to the parent one
    sm1->add_submenu(sm2, TRUE, FALSE, FALSE);
  }
  // delete everything
  sprintf(mbuf, "label delete Atoms; label delete Bonds; "
	  "label delete Angles; label delete Dihedrals");
  sm1->add_item("Delete all", mbuf, TRUE, FALSE, FALSE);
  

  mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);
  mouseMenu->add_separator();

  // create sub-menu for animating the scene
  sm1 = new PopupMenu("Animate Scene");
  sm1->add_item("play forward", "animate forward", TRUE, FALSE, FALSE);
  sm1->add_item("play reverse", "animate reverse", TRUE, FALSE, FALSE);
  sm1->add_item("step 1 forward", "animate next", TRUE, FALSE, FALSE);
  sm1->add_item("step 1 reverse", "animate prev", TRUE, FALSE, FALSE);
  sm1->add_item("pause", "animate pause", TRUE, FALSE, FALSE);
  sm1->add_item("goto start", "animate goto start", TRUE, FALSE, FALSE);
  sm1->add_item("goto end", "animate goto end", TRUE, FALSE, FALSE);

  sm2 = new PopupMenu("Style");
  for(i=0; i < Animation::TOTAL_STYLES; i++) {
    sprintf(mbuf,"animate style %s", animationStyleName[i]);
    sm2->add_item(animationStyleName[i], mbuf, TRUE, FALSE, FALSE);
  }
  sm1->add_submenu(sm2, TRUE, FALSE, FALSE);
  mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);

  // create sub-menu for spinning or rocking the scene
  sm1 = new PopupMenu("Spin/Rock Scene");
  sm1->add_item("Stop Rotation", "rotate stop", TRUE, FALSE, FALSE);
  sm1->add_item("Spin X", "rock x by 1.0 -1", TRUE, FALSE, FALSE);
  sm1->add_item("Spin Y", "rock y by 1.0 -1", TRUE, FALSE, FALSE);
  sm1->add_item("Spin Z", "rock z by 1.0 -1", TRUE, FALSE, FALSE);
  sm1->add_item("Rock X", "rock x by 1.0 70", TRUE, FALSE, FALSE);
  sm1->add_item("Rock Y", "rock y by 1.0 70", TRUE, FALSE, FALSE);
  sm1->add_item("Rock Z", "rock z by 1.0 70", TRUE, FALSE, FALSE);
  mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);

  // add other command to affect the current transformation
  mouseMenu->add_item("Stop Rotation", "rotate stop", TRUE, FALSE, FALSE);
  mouseMenu->add_item("Reset View", "display resetview", TRUE, FALSE, FALSE);
  mouseMenu->add_separator();

  // create sub-menu to turn on/off forms
  sm1 = new PopupMenu("Show Form");
  sm2 = new PopupMenu("Hide Form");
  int m_added = 0;
  for (int mn = 0; mn < uiList->num(); mn++) {
    UIObject *menu = uiList->item(mn);
    if(menu->is_menu()) {
      ++m_added;
      sprintf(mbuf, "menu %s on", menu->name);
      sprintf(mbuf2,"menu %s off",menu->name);
      sm1->add_item(menu->name, mbuf,  TRUE, TRUE, menu->is_on());
      sm2->add_item(menu->name, mbuf2, TRUE, TRUE, menu->is_on());
    }
  }
  mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);
  mouseMenu->add_submenu(sm2, TRUE, FALSE, FALSE);
  mouseMenu->add_separator();

  // create submenu for user-defined menus ... this contains a list of
  // N submenus or commands; each submenu can contain a list of commands
  // as well.
  if(userMenuList.num() > 0) {
    for (i=0; i < userMenuList.num(); i++) {
      NameListStringPtr usermenu = userMenuList.data(i);
      if(usermenu && usermenu->num() > 0) {
        sm1 = new PopupMenu(userMenuList.name(i));
        for(j=0; j < usermenu->num(); j++) {
          char *usertxtcmd = usermenu->data(j);
          if(usertxtcmd)
            sm1->add_item(usermenu->name(j), usertxtcmd, TRUE, FALSE, FALSE);
          else
            sm1->add_separator();
        }
        mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);
      }
    }
    mouseMenu->add_separator();
  }

  // add help menu commands
  mouseMenu->add_item("General Help", "help", TRUE, FALSE, FALSE);
  int help_items = uiText->num_commands();
  if(help_items > 0) {
    sm1 = new PopupMenu("Specific Help");
    for(char ch='a'; ch <= 'z'; ch++) {
      int ch_added = 0;
      for(i=0; i < help_items; i++) {
        char *helpword = uiText->word(i);
        if(ch == tolower(*helpword)) {
          if(!ch_added) {
            char newname[2];
            newname[0] = ch;  newname[1] = '\0';
            sm2 = new PopupMenu(newname);
            sm1->add_submenu(sm2, TRUE, FALSE, FALSE);
          }
          ch_added++;
          sprintf(mbuf, "help %s", helpword);
          sm2->add_item(helpword, mbuf, TRUE, FALSE, FALSE);
        }
      }
    }
    mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);
  }
  mouseMenu->add_separator();

  // add command to quit
  mouseMenu -> add_item("Play File", "vmd_play_file", TRUE, FALSE, FALSE);
  mouseMenu -> add_item("Save Config", "vmd_save_state", TRUE, FALSE, FALSE);
  sm1 = new PopupMenu("Quit");
  sm1->add_item("Yes, Really", "quit", TRUE, FALSE, FALSE);
  sm1->add_item("Yes, and Save", "vmd_save_state_q", TRUE, FALSE, FALSE);
  sm1->add_item("No, Don't", "", TRUE, FALSE, FALSE);
  mouseMenu->add_submenu(sm1, TRUE, FALSE, FALSE);

  // return the completed menu
  return mouseMenu;
}


// routine which creates the initial keyboard shortcut commands
void Mouse::create_key_commands(void) {
  add_user_key_command('r', "mouse mode 0 0");
  add_user_key_command('R', "mouse mode 0 0");
  add_user_key_command('t', "mouse mode 1 0");
  add_user_key_command('T', "mouse mode 1 0");
  add_user_key_command('s', "mouse mode 2 0");
  add_user_key_command('S', "mouse mode 2 0");
  add_user_key_command('0', "mouse mode 4 0");
  add_user_key_command('c', "mouse mode 4 1");
  add_user_key_command('1', "mouse mode 4 2");  // atom
  add_user_key_command('2', "mouse mode 4 3");  // bond
  add_user_key_command('3', "mouse mode 4 4");  // angle
  add_user_key_command('4', "mouse mode 4 5");  // dihedral
  add_user_key_command('5', "mouse mode 4 6");  //  move atom
  add_user_key_command('6', "mouse mode 4 7");  //  move residue
  add_user_key_command('7', "mouse mode 4 8");  //  move fragment
  add_user_key_command('%', "mouse mode 4 9");  // force atom
  add_user_key_command('^', "mouse mode 4 10"); // force residue
  add_user_key_command('&', "mouse mode 4 11"); // force fragment
  
  add_user_key_command('x', "rock x by 1.0 -1");
  add_user_key_command('y', "rock y by 1.0 -1");
  add_user_key_command('z', "rock z by 1.0 -1");
  add_user_key_command('X', "rock x by 1.0 70");
  add_user_key_command('Y', "rock Y by 1.0 70");
  add_user_key_command('Z', "rock Z by 1.0 70");
  add_user_key_command('+', "animate next");
  add_user_key_command('f', "animate next");
  add_user_key_command('F', "animate next");
  add_user_key_command('-', "animate prev");
  add_user_key_command('b', "animate prev");
  add_user_key_command('B', "animate prev");
  add_user_key_command('.', "animate forward");
  add_user_key_command('>', "animate forward");
  add_user_key_command(',', "animate reverse");
  add_user_key_command('<', "animate reverse");
  add_user_key_command('/', "animate pause");
  add_user_key_command('?', "animate pause");
}


// routine which actually adds a new menu separator
int Mouse::do_add_user_menu_separator(NameListStringPtr menu) {
  char nmbuf[16];

  if(!menu)
    return FALSE;

  // make a strange, unique name that no sane person will normally add
  sprintf(nmbuf, "##%06d##", menu->num() + 1);

  // add the name, but with a NULL text command string, which signals a
  // separator
  int exists = menu->add_name(nmbuf, NULL);

  // return whether we could add the name
  return (exists >= 0);
}


// add a new item to the user-defined list, at the main level (as opposed
// to adding a new submenu)
int Mouse::do_add_user_menu_item(NameListStringPtr menu,
					char *text, char *txtcmd) {

  // make sure the menu item does not already exist ...
  int oldnum = menu->num();
  int exists = menu->add_name(text, NULL);

  if(exists < 0)
    // an error, could not add name
    return FALSE;

  // if it does exist, delete the old one
  if(exists >= 0 && oldnum == menu->num()) {
    char *cmd = menu->data(exists);
    if(cmd)
      delete [] cmd;
  }

  // add the new text command
  menu->set_data(exists, stringdup(txtcmd));

  return TRUE;
}


// stop rotation of object
void Mouse::stop_rotation(void) {
  xRotVel = yRotVel = zRotVel = 0.0;
}

// true if mouse is tracking an object or motion (while a button is down)
int Mouse::is_tracking(void) const {
    return isTracking;
}

// the semantics of the UI, handled for non-polling interfaces like Tk.
//  This is a prototype for
//  reimplementation of act_on_command().
void Mouse::handle_mouse_event(const MouseEvent &ev) {

  MSGDEBUG(3,"Mouse: handling mouse event " << ev.type() << sendmsg);

  dispDev->record_mouse_event(ev);

  if (ev.type() == MouseEvent::BUTTON_DOWN) {
    // ignore right button
    if (ev.button() == MouseEvent::B_RIGHT)
      return;
    // ignore other buttons while tracking
    if (isTracking)
      return;

    // initialize variables at start of mouse movement
    mouseMoved = FALSE;
    isTracking = TRUE;
    activeButton = ev.button();
    oldX = currX = x();
    oldY = currY = y();
    xRotVel = yRotVel = zRotVel = 0.0;
    MSGDEBUG(4, "handle_mouse_event: mouse " << activeButton);
    MSGDEBUG(4, "down, tracking at " << x() << " " << y() << sendmsg);

    // check for a picked item if we are in a picking mode
    if(curr_move_mode() == PICKING && ! picking()) {
      // the first two are 'query' and 'center', the 2nd is atom, 3rd is bond
      pickInProgress = 2 + ev.button(); 
      float mx = (float) currX;
      float my = (float) currY;
      dispDev->rel_screen_pos(mx, my);

      MSGDEBUG(4, "\tpick_start button " << ev.button() << " mode ");
      MSGDEBUG(4, curr_pick_mode() << " at " << mx << " " << my << sendmsg);
      // if picking an object fails, assume we are rotating the object
      if(! runcommand(new CmdPickStart(ev.button(),curr_pick_mode(),
                            mx,my,id())))
        pickInProgress = 0;
    }
  } else if (ev.type() == MouseEvent::MOTION) {
    if (!isTracking)
      return;

    MSGDEBUG(4, "handle_mouse_event: mouse " << activeButton << " motion at ");
    MSGDEBUG(4, x() << " " << y() << sendmsg);
    // update scene by checking if the mouse has moved any
    if(mouse_moved(activeButton == MouseEvent::B_LEFT,
                   activeButton == MouseEvent::B_MIDDLE)) {
      if (!mouseMoved) {
	// first time, so save the old value ...
	old_detail_level = dispDev -> detail_level();
	// .. and use the alternate detail level
	dispDev -> detail_level( dispDev -> alt_detail_level());
      }
      mouseMoved = TRUE;
    }

    // redraw scene, since main event loop will just call us back again
    VMDupdate(FALSE);

  } else if (ev.type() == MouseEvent::BUTTON_UP) {
    // ignore it unless it's a release of the initially selected button
    if (ev.button() != activeButton)
      return;

    MSGDEBUG(4, "handle_mouse_event: mouse " << activeButton);
    MSGDEBUG(4, " up, tracking at " << x() << " " << y() << sendmsg);

    if (mouseMoved) {
      // return the detail level to its original value
      dispDev -> detail_level(old_detail_level);
    }

    // we're done moving the mouse while the button is down; check if
    // we need to do anything more
    if(picking() || !mouseMoved) {
      int button_used = ev.button();
      float mx = (float) currX;
      float my = (float) currY;
      dispDev->rel_screen_pos(mx, my);

      if(picking()) {
        // must finish the picking process
        pickInProgress = 0;
        int m = curr_pick_mode();
        MSGDEBUG(4, "\tpick_end button " << button_used << " mode " << m);
        MSGDEBUG(4, " at " << mx << " " << my << sendmsg);
        runcommand(new CmdPickEnd(button_used, m, mx, my, id()));
      } else {
        // if the mouse was not ever moved, we do a simple pick operation
        pickInProgress = 0;
	// the first two are 'query' and 'center', the 2nd is atom, 3rd is bond
        int m = 2 + ev.button();  // pick mode
        MSGDEBUG(4, "\tpick start/end button " << button_used << " mode " 
		 << m - 1);
        MSGDEBUG(4, " at " << mx << " " << my << sendmsg);
        if(runcommand(new CmdPickStart(button_used, m , mx, my, id())))
          runcommand(new CmdPickEnd(button_used, m , mx, my, id()));
      }
    }

    isTracking = FALSE;
  }
}


// return the current x or y position of the mouse, or the state of a button
int Mouse::x(void) { return dispDev->x(); }
int Mouse::y(void) { return dispDev->y(); }
int Mouse::button_down(int b) { return dispDev->button_down(b); }


// set the mouse move mode to the given state; return success
int Mouse::move_mode(MoveMode mm, int mobj) {

  MSGDEBUG(2,"Mouse: setting mouse mode to " << mm << ", mobj " << mobj);
  MSGDEBUG(2,sendmsg);

  // we cannot change the mouse mode if an active pick is going on
  if(picking())
    return FALSE;

  // set active picking mode if necessary
  if(mm == PICKING)
    // if we are changing to a picking mode, mobj is the new mode to use
    activePickMode = mobj;
  else
    // not a picking mode; reset setting
    activePickMode = (-1);

  // stop rotating no matter what move we've changed to
  stop_rotation();

  // signal we are/are not moving a light around
  if(moveMode == LIGHT) {
    // turn off old light
    addcommand(new CmdDisplayLightHL(moveObj, FALSE, id()));
  }

  // special actions based on the new mode
  if(mm == LIGHT) {
    // turn on new light number mobj
    moveObj = mobj;
    addcommand(new CmdDisplayLightHL(moveObj, TRUE, id()));
  }

  // change the mouse mode now
  moveMode = mm;
#ifdef VMDTCL
  // Tell TCL the new mouse mode
  // Set the Tcl variable "vmd_mouse_mode" to the correct string
  // and set the variable "vmd_mouse_submode" to the mobj number
  {
    char *s = "rotate";  // the default
    switch(moveMode) {
    default:        // when in doubt ..
    case ROTATION:
      break;
    case TRANSLATION:
      s = "translate";
      break;
    case SCALING:
      s = "scale";
      break;
    case LIGHT:
      s = "light";
      break;
    case PICKING:
      s = "pick";
      break;
    case USER:
      s = "user";
      break;
    }
    // and set the Tcl variable
    Tcl_SetVar(uiText -> tclInterp, "vmd_mouse_mode", s, TCL_GLOBAL_ONLY);
    char temps[20];
    sprintf(temps, "%d", mobj);
    Tcl_SetVar(uiText -> tclInterp, "vmd_mouse_submode", temps, 
	       TCL_GLOBAL_ONLY);
  }
#endif

  // command was successful ... change the cursor style
  if(mm == LIGHT)
    dispDev->set_cursor();
  else if(mm == PICKING || mm == USER)
    dispDev->set_cursor(PICK_CURSOR);
  else
    dispDev->set_cursor(mm);

  // signal success
  return TRUE;
}

  
// string name for the given move mode
char *Mouse::move_mode_name(MoveMode mm) {
  return mm_names[mm];
}


////////// virtual routines to set the user-defined menu items  //////////

// create a new submenu in the main user-defined list (it will be initially
// empty, but can be added to later).  Return success of creation.
int Mouse::create_user_menu(char *label) {

  // make sure the menu does not already exist ...
  int oldnum = userMenuList.num();
  int exists = userMenuList.add_name(label, NULL);

  // if it does exist, delete all the items in the old one
  if(exists >= 0 && oldnum == userMenuList.num()) {
    NameListStringPtr currMenu = userMenuList.data(exists);
    int n = currMenu->num();
    for(int i=0; i < n; i++) {
      char *cmd = currMenu->data(i);
      if(cmd) {
        delete [] cmd;
        currMenu->set_data(i, NULL);
      }
    }

  } else if(exists >= 0) {
    // number of items changed, so we must have added a new one.
    // create a new NameList, and put it in the data value for the new menu.
    userMenuList.set_data(exists, new NameList<char *>);

  } else
    // there was a problem, return an error
    return FALSE;

  return TRUE;
}


// add a new item to the user-defined list, at the main level (as opposed
// to adding a new submenu)
int Mouse::add_user_menu_item(char *text, char *txtcmd) {

  // return whether we could add the name
  return (do_add_user_menu_item(mainUserMenu, text, txtcmd));
}


// add a separator to the end of the main user menu
int Mouse::add_user_menu_separator(void) {

  // return whether we could add the name
  return (do_add_user_menu_separator(mainUserMenu));
}


// add a new item to the submenu with the given name
int Mouse::add_user_submenu_item(char *submenu, char *text,
		char *txtcmd) {

  // make sure the submenu does already exist ...
  int exists = userMenuList.typecode(submenu);

  // if it does, add the item to it
  if(exists >= 0)
    return (do_add_user_menu_item(userMenuList.data(exists), text, txtcmd));
  else
    return FALSE;
}


// add a separator to the end of the specified user submenu
int Mouse::add_user_submenu_separator(char *submenu) {

  // make sure the submenu does already exist ...
  int exists = userMenuList.typecode(submenu);

  // if it does, add the item to it
  if(exists >= 0)
    return (do_add_user_menu_separator(userMenuList.data(exists)));
  else
    return FALSE;
}


// print out the user menu commands to the given Inform object. Return success
int Mouse::print_user_menu(Inform *m) {

  *m << "User-Defined Menu Commands:" << sendmsg;
  *m << "---------------------------" << sendmsg;
  for (int i=0; i < userMenuList.num(); i++) {
    NameListStringPtr menu = userMenuList.data(i);
    if(menu) {
      *m << userMenuList.name(i) << " ==> Submenu:" << sendmsg;
      if(menu->num() > 0) {
        for(int j=0; j < menu->num(); j++) {
          *m << "   ";
          char *txtcmd = menu->data(j);
          if(txtcmd)
            *m << menu->name(j) << " = " << txtcmd;
          else
            *m << "----- separator -----";
          *m << sendmsg;
        }
      } else
        *m << "   (empty)" << sendmsg;
    }
  }
  
  return TRUE;
}


/////////////////////// virtual routines to add key cmds  /////////////

//
// routines to set the user-defined single-key shortcuts commands.
// All return success.
//
  
// bind a new text command to a given key ... replaces old command if nec.
int Mouse::add_user_key_command(char c, const char *txtcmd) {
  char key[2];
  int indx;

  if(!txtcmd)
    return FALSE;

  key[0] = c;  key[1] = '\0';
  indx = userKeys.typecode(key);

  if(indx < 0)
    userKeys.add_name(key, stringdup(txtcmd));
  else {
    delete [] userKeys.data(indx);
    userKeys.set_data(indx, stringdup(txtcmd));
  }

  return TRUE;
}


// print out the user keyboard shortcut commands.  Return success.
int Mouse::print_user_keys(Inform *m) {

  *m << "Keyboard shortcuts:" << sendmsg;
  *m << "-------------------" << sendmsg;
  for(int i=0; i < userKeys.num(); i++)
    *m << "'" << userKeys.name(i) << "' : " << userKeys.data(i) << sendmsg;

  return TRUE;
}


// do action when the mouse is moved
// arg: which buttons are currently being pressed
// return: whether the mouse moved any
int Mouse::mouse_moved(int b1Down, int b2Down) {
  int dx, dy, mouseMoved;

  // get current mouse position
  currX = x();
  currY = y();
  
  // and calculate distance mouse has moved
  dx = 5 * (currX - oldX);
  dy = -5 * (currY - oldY);
  xRotVel = yRotVel = zRotVel = 0.0;
  mouseMoved = (dx != 0 || dy != 0);
  if(!mouseMoved)
    return FALSE;		// can quit this early if nothing happened

  // and then save the settings
  oldX = currX;
  oldY = currY;

  // check if we are picking something; if so, generate pick-move command
  if(picking()) {
    float mx = (float) currX;
    float my = (float) currY;
    display->rel_screen_pos(mx, my);
    if(mx >= 0.0 && mx <= 1.0 && my >= 0.0 && my <= 1.0)
      // -2 comes from the Query and Center
      runcommand(new CmdPickMove(picking()-2,curr_pick_mode(),mx,my,id()));
    return TRUE;
  }

  // Otherwise, if a button is pressed, check how far the mouse moved,
  // and transform the view accordingly.

  // check for button 1 action
  if(b1Down) {
    if(curr_move_mode() == ROTATION || curr_move_mode() == LIGHT
	|| curr_move_mode() == PICKING || curr_move_mode() == USER) {
      xRotVel = rotInc * (float)dy;
      yRotVel = rotInc * (float)dx;
      if(curr_move_mode() == ROTATION || curr_move_mode() == PICKING
	 || curr_move_mode() == USER) {
	if(xRotVel != 0.0) {
          runcommand(new CmdRotate(xRotVel, 'x', CmdRotate::BY, id()));
          xRotVel *= RotVelScale;
	}
	if(yRotVel != 0.0) {
          runcommand(new CmdRotate(yRotVel, 'y', CmdRotate::BY, id()));
          yRotVel *= RotVelScale;
	}
      } else {			// rotate particular light
	if(xRotVel != 0.0) {
	  runcommand(new CmdDisplayLightRot(moveObj, xRotVel, 'x', id()));
          xRotVel *= RotVelScale;
	}
	if(yRotVel != 0.0) {
	  runcommand(new CmdDisplayLightRot(moveObj, yRotVel, 'y', id()));
          yRotVel *= RotVelScale;
	}
      }

    } else if(curr_move_mode() == TRANSLATION && mouseMoved) {
      runcommand(new CmdTranslate(transInc*(float)dx, -transInc*(float)dy,
        0.0, CmdTranslate::BY, id()));

    } else if(curr_move_mode() == SCALING && dx != 0) {
      float scf = scaling + scaleInc * (float)dx;
      if(scf < 0.0)
        scf = 0.0;
      runcommand(new CmdScale(scf, CmdScale::BY, id()));
    }
  }
  
  // check for button 2 action
  if(b2Down) {
    if(curr_move_mode() == ROTATION || curr_move_mode() == LIGHT
	|| curr_move_mode() == PICKING || curr_move_mode() == USER) {
      zRotVel = rotInc * (float)dx;
      if(curr_move_mode() == ROTATION || curr_move_mode() == PICKING ||
	 curr_move_mode() == USER) {
	if(zRotVel != 0.0) {
          runcommand(new CmdRotate(zRotVel, 'z', CmdRotate::BY, id()));
          zRotVel *= RotVelScale;
	}
      } else {
	if(zRotVel != 0.0) {
	  runcommand(new CmdDisplayLightRot(moveObj, zRotVel, 'z', id()));
          zRotVel *= RotVelScale;
	}
      }

    } else if(curr_move_mode() == TRANSLATION && dx != 0) {
      runcommand(new CmdTranslate(0.0, 0.0, transInc*(float)dx,
	CmdTranslate::BY, id()));

    } else if(curr_move_mode() == SCALING && dx != 0) {
      float scf = scaling + 10.0 * scaleInc * (float)dx;
      if(scf < 0.0)
        scf = 0.0;
      runcommand(new CmdScale(scf, CmdScale::BY, id()));
    }
  }

  // return the fact that the mouse has moved
  return TRUE;
}


/////////////////////// virtual routines for UI init/display  /////////////
   
// reset the mouse to original settings
void Mouse::reset(void) {
  scaling = 1.0;
  stop_rotation();
  currX = oldX = x();
  currY = oldY = y();
}


// update the display due to a command being executed.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int Mouse::act_on_command(int type, Command *cmd, int) {

  MSGDEBUG(3,"Mouse: acting on command " << type << sendmsg);

  // check if window event
  if(type == Command::WIN_EVENT) {
    long dev = ((WinEvent *)cmd)->eventCode;
    long val = ((WinEvent *)cmd)->val;

    if(dev == DisplayDevice::WIN_LEFT ||
		dev == DisplayDevice::WIN_MIDDLE) {	// left/middle button
      if(val == 1) {		// make sure the button was pressed DOWN

        // initialze variables at start of mouse movement
	int mouseMoved = FALSE;
        oldX = currX = x();
        oldY = currY = y();
	xRotVel = yRotVel = zRotVel = 0.0;

        // check for a picked item if we are in a picking mode
        if(curr_move_mode() == PICKING && ! picking()) {
	  // the first two are 'query' and 'center', 
	  // the 2nd is atom, 3rd is bond
	  pickInProgress = 2 + (dev == DisplayDevice::WIN_LEFT ?
			DisplayDevice::B_LEFT : DisplayDevice::B_MIDDLE);
	  float mx = (float) currX;
	  float my = (float) currY;
	  dispDev->rel_screen_pos(mx, my);

          // if picking an object fails, assume we are rotating the object
	  if(! runcommand(new CmdPickStart(pickInProgress-2,curr_pick_mode(),
	      			mx,my,id())))
            pickInProgress = 0;
        }

        // have the mouse take control until the button is released
        while(TRUE) {
          long newdev, newval;

          // check for a new mouse event, and if it is the same button,
          // finish the loop
          if(dispDev->read_event(newdev, newval))
            if(newdev == dev && newval == 0)
              break;		// button released

          // update scene by checking if the mouse has moved any
          if(mouse_moved(dev == DisplayDevice::WIN_LEFT,
			 dev == DisplayDevice::WIN_MIDDLE)) {
	    if (!mouseMoved) {
	      // first time, so save the old value ...
	      old_detail_level = dispDev -> detail_level();
	      // .. and use the alternate detail level
	      dispDev -> detail_level( dispDev -> alt_detail_level());
	    }
            mouseMoved = TRUE;
	  }

          // finally, redraw the scene, but WITHOUT checking for other events
          VMDupdate(FALSE);
        }

	// if the detail changed, return it to its original value
	if (mouseMoved == TRUE) {
	  dispDev -> detail_level( old_detail_level);
	}

        // we're done moving the mouse while the button is down; check if
        // we need to do anything more
        if(picking() || !mouseMoved) {
	  int button_used = (dev == DisplayDevice::WIN_LEFT ?
			DisplayDevice::B_LEFT : DisplayDevice::B_MIDDLE);
	  float mx = (float) currX;
	  float my = (float) currY;
	  dispDev->rel_screen_pos(mx, my);

          if(picking()) {
            // must finish the picking process
	    pickInProgress = 0;
            int m = curr_pick_mode();
	    runcommand(new CmdPickEnd(button_used, m, mx, my, id()));

          } else {
            // if the mouse was not ever moved, we do a simple pick operation
	    pickInProgress = 0;
	    // do either "Atom" or "Bond" picking
            int m = (dev == DisplayDevice::WIN_LEFT ? 2 : 3);

	    if(runcommand(new CmdPickStart(button_used, m, mx, my, id())))
	      runcommand(new CmdPickEnd(button_used, m, mx, my, id()));
          }
        }
      }

    } else if(dev == DisplayDevice::WIN_RIGHT) {		// pop-up menu
      if (val == 1) {
	// check to see if something is under the pointer
	Pickable *pickedObj = NULL;
	int tag = 0;
	// because this takes time, you must press the shift key to
	// do the lookup.  Maybe it should be the other way?
	if ((dispDev -> shift_state() & DisplayDevice::SHIFT)) {
	  float mxy[2];
	  mxy[0] = (float) x();
	  mxy[1] = (float) y();
	  dispDev->rel_screen_pos(mxy[0], mxy[1]);
	  pickedObj = scene->pick_check(dispDev, 2, mxy, tag);
	}

        // create the pop-up menu, and post it
	PopupMenu *mouseMenu = create_popup_menu(pickedObj, tag);
        PopupMenuItem *menuval = mouseMenu->activate(dispDev);

        // execute command if necessary
        if(menuval && menuval->return_code() >= 0) {
          runcommand(new TextEvent(menuval->text_command(), FALSE, id()));
        }

	// now delete the pop-up menu
        dispDev->menu_delete(mouseMenu);
        delete mouseMenu;
      }

    } else if(dev == DisplayDevice::WIN_KEYBD) {	// keyboard pressed
      char key[2];
      key[0] = (char)val;  key[1] = '\0';
      int indx = userKeys.typecode(key);
      if(indx >= 0)
        runcommand(new TextEvent(userKeys.data(indx), FALSE, id()));

    } else if (dev == DisplayDevice::WIN_REDRAW) {
      runcommand(new CmdDisplayReshape(id()));

    } else if (dev == DisplayDevice::WIN_INPUTCHANGE) {
      ;		// who cares, just return

    } else			// unrecognized device
      return FALSE;

  } else			// unrecognized command
    return FALSE;
  
  // if here, command we can use was found
  return TRUE;
}


// check for an event, and queue it if found.  Return TRUE if an event
// was generated.
int Mouse::check_event(void) {
  int retval = FALSE;

  // if no GUI being used, we must check for events ourselves
#ifndef VMDGUI
  long dev, val;
  if((retval = dispDev->read_event(dev, val)))
    runcommand(new WinEvent(dev, val, id()));
#endif

  // apply ang velocity, if necessary
  if(xRotVel != 0.0 || yRotVel != 0.0 || zRotVel != 0.0) {
    if(curr_move_mode() != LIGHT) {		// (possibly) rotate scene
      if(xRotVel != 0.0)
        runcommand(new CmdRotate(xRotVel, 'x', CmdRotate::BY, id()));
      if(yRotVel != 0.0)
        runcommand(new CmdRotate(yRotVel, 'y', CmdRotate::BY, id()));
      if(zRotVel != 0.0)
        runcommand(new CmdRotate(zRotVel, 'z', CmdRotate::BY, id()));
    } else {				// (possibly) rotate particular light
      if(xRotVel != 0.0)
        runcommand(new CmdDisplayLightRot(moveObj, xRotVel, 'x', id()));
      if(yRotVel != 0.0)
        runcommand(new CmdDisplayLightRot(moveObj, yRotVel, 'y', id()));
      if(zRotVel != 0.0)
        runcommand(new CmdDisplayLightRot(moveObj, zRotVel, 'z', id()));
    }
  }

  return retval;
}


//////////////////////////////////////////////////////////////////////
/////////////////////// window event definition  /////////////////////
//////////////////////////////////////////////////////////////////////

//////////  General window event (other than a GUI event)
// constructor; must specify what the event type code and value are
WinEvent::WinEvent(long d, long v, int newUIid)
	: Command(Command::WIN_EVENT, newUIid) {
  eventCode = d;
  val = v;
}

