/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeRemote.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.21 $	$Date: 1996/03/20 17:55:39 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A MoleculeRemote is inherited from Molecule and contains a Remote;
 * it is a Molecule which is representing a remote simulation, from which
 * it gets parameters and coordinates.
 *
 ***************************************************************************/

#include "MoleculeRemote.h"
#include "Remote.h"
#include "Atom.h"
#include "DrawPatch.h"
#include "DrawForce.h"
#include "Inform.h"
#include "utilities.h"
#include "SymbolTable.h"

#ifdef VMDTCL
// used to tell Tcl a new timestep has arrived for the given molecule
#include <tcl.h>
#include "UIText.h"
extern UIText *uiText;
#endif

extern MoleculeList *moleculeList;
extern SymbolTable atomSelParser;

////////////////////////////  constructor  

MoleculeRemote::MoleculeRemote(Remote *newrem, Scene *sc)
	: Molecule(newrem->host(), sc) {
  do_construct(newrem);
}

MoleculeRemote::MoleculeRemote(Remote *newrem, Displayable *dp)
	: Molecule(newrem->host(), dp) {
  do_construct(newrem);
}


// do main constructor tasks
void MoleculeRemote::do_construct(Remote *newrem) {
  rem = newrem;
  strName = strPath = NULL;
  Step = 0;
  hasPatchDisplay = FALSE;
  drawPatch = NULL;
  drawForce = NULL;
  framesReceived = 0;
  MSGDEBUG(1,"Creating MoleculeRemote, host='" << rem->host() << "', app='");
  MSGDEBUG(1,rem->app() << "' ..." << sendmsg);
}


/////////////////////////  destructor  
MoleculeRemote::~MoleculeRemote(void) {
  MSGDEBUG(1,"Deleting MoleculeRemote ..." << sendmsg);
  
  if(strPath)  delete [] strPath;
  if(strName)  delete [] strName;
  
  // delete patch display, if possible
  if(drawPatch)
    delete drawPatch;

  // delete force display, if possible
  if(drawForce)
    delete drawForce;

  // close remote connection
  if(rem)  delete rem;
}


/////////////////////  public routines  
// return the path of this molecule
char *MoleculeRemote::file_path(void) {
  if(!strPath)
    return rem->host();
  else
    return strPath;
}


// return the name of this molecule
char *MoleculeRemote::file_name(void) {
  if(!strName)
    return rem->app();
  else
    return strName;
}


// create the molecule from the data gained over the network.  return success.
int MoleculeRemote::create(void) {
  MSGDEBUG(2,"MoleculeRemote: creating structure ..." << sendmsg);
  
  if(nAtoms != 0) {
    msgErr << "Attempt to create molecule after it has already been created.";
    msgErr << sendmsg;
    return FALSE;
  }

  // make sure the remote simulation is connected ...
  if(!rem || ! rem->running_simulation())
    return FALSE;
    
  // we're here OK, so connection is good.  Get structure and create molecule.
  msgInfo << "Retrieving molecular structure from " << rem->host() << " ...";
  msgInfo << sendmsg;
  if(!(createFromRemote(rem)))
    return FALSE;

  // save the file name
  char *mname, *molname = rem->mol_name();
  if(molname) {
    mname = strrchr(molname, '/');
    if(!mname)
      mname = molname;
    else
      mname++;
  } else {
    mname = rem->host();
  }
  set_name(mname);

  // do anything else the parent class needs
  int retval = Molecule::create();
  
  // create patch representation object
  drawPatch = new DrawPatch(this);

  // create force representation object
  drawForce = new DrawForce(this);
  drawForce->use_colors(colorList);

  return retval;
}


// read in a molecule structure from the remote connection.
// Return TRUE if molecule was successfully read, FALSE if there is an error.
int MoleculeRemote::createFromRemote(Remote *rm) {
  int i;
  char namesbuf[8], rnamesbuf[8], segnamesbuf[8], atypesbuf[8], ridstrbuf[8];
  char *names, *rnames, *segnames, *atypes, *ridstr;
  float newpos[ATOMCOORDS], newdata[ATOMEXTRA_DYNAMIC + ATOMEXTRA_STATIC];

  // access molecular structure struct
  if(!rm)
    return FALSE;
  
  VmdStaticData *mds = rm->static_data();
  int natoms = mds->numAtoms;
  int nbonds = mds->numBonds;
  
  MSGDEBUG(1,"MoleculeRemote: Creating structure from network (");
  MSGDEBUG(1,rem->host() << ") ..."  << sendmsg);

  // initialize data arrays that are used to create new atoms
  for(i=0; i < ATOMCOORDS; newpos[i++] = 0.0);
  namesbuf[mds->nameLen] = '\0';
  atypesbuf[mds->nameLen] = '\0';
  rnamesbuf[mds->nameLen] = '\0';
  segnamesbuf[mds->nameLen] = '\0';
  ridstrbuf[mds->nameLen] = '\0';

  MSGDEBUG(1,"                Remote structure with:" << sendmsg);
  MSGDEBUG(1,"                  " << natoms << " atoms" << sendmsg);
  MSGDEBUG(1,"                  " << nbonds << " bonds" << sendmsg);
  MSGDEBUG(1,"                  " << mds->numAtomNames << " atom names"<<sendmsg);
  MSGDEBUG(1,"                  " << mds->numAtomTypes << " atom types"<<sendmsg);

  // read in the atom structure info first
  int *typePos = mds->atomTypeIndexes;
  int *namePos = mds->atomNameIndexes;
  char *resIdPos = mds->resIds;
  char *resNamePos = mds->resNames;
  char *segNamePos = mds->segIds;
  float *massPos = mds->mass;
  float *chargePos = mds->charge;
  float *radiusPos = mds->radii;
  float *occupPos = mds->occupancy;
  float *betaPos = mds->beta;

  MSGDEBUG(2,"   Creating " << natoms << " atoms ..." << sendmsg);

  init_atoms(natoms);
  char chain[2];
  chain[1] = 0;
  for(i = 0; i < natoms; i++) {
    names = namesbuf;
    atypes = atypesbuf;
    rnames = rnamesbuf;
    ridstr = ridstrbuf;
    segnames = segnamesbuf;

    // copy names into buffers, and increment pointers into mdcomm lists
    strncpy(names, mds->atomNames + (mds->nameLen * *(namePos++)),
    	mds->nameLen);
    strncpy(atypes, mds->atomTypes + (mds->nameLen * *(typePos++)),
        mds->nameLen);
    strncpy(rnames, resNamePos, mds->nameLen);
    strncpy(ridstr, resIdPos, mds->nameLen);
    strncpy(segnames, segNamePos, mds->nameLen);
    resNamePos += mds->nameLen;
    resIdPos += mds->nameLen;
    segNamePos += mds->nameLen;

    // save integer and float data
    newdata[ATOMCHARGE] = *(chargePos++);
    newdata[ATOMMASS] = *(massPos++);
    newdata[ATOMRAD] = *(radiusPos++);
    newdata[ATOMBETA] = *(betaPos++);
    newdata[ATOMOCCUP] = *(occupPos++);

    MSGDEBUG(1, "New atom " << i << ": name='" << names << "', mass=" 
		<< newdata[ATOMMASS] << sendmsg);

    chain[0] = segnames[0];  // no chain ids across the net...
    add_atom(names, atypes, rnames, ridstr, chain, segnames, newpos, newdata);
  }
  
  // now read bond data
  MSGDEBUG(2,"   Creating " << nbonds << " bonds ..." << sendmsg);

  // add bonds to current structure
  int *bondPos = mds->bonds;
  for(i=0; i < nbonds; i++) {
    add_bond(*bondPos, *(bondPos + 1));
    bondPos += 2;
  }

  MSGDEBUG(2,"Finished creating structure from remote connection." << sendmsg);

  return TRUE;
}


// function used by Displayable to prepare the Displayable object; here
//      it checks if new data is available from the remote machine
void MoleculeRemote::prepare(DisplayDevice *display) {
  Timestep *newts;
  MolAtom *atm;
  VmdDynData *mdd;

  // make sure we're connected
  if(rem && nAtoms > 0) {

    MSGDEBUG(3,"Checking for new timestep from " << rem->host() << sendmsg);

    // check for new timestep
    if((mdd = rem->get_next_ts()) != NULL) {

      MSGDEBUG(2,"New timestep " << *(mdd->timestep));
      MSGDEBUG(2," available from " << rem->host() << sendmsg);

      // new step available - read it and add it to animation list
      // see if we need a new frame, or should replace the old one
      int saveFrame = (num() < 1 || framesReceived < 1 ||
      	(rem->save_frame_rate() > 0 
	        && (framesReceived % rem->save_frame_rate() == 0)));

      if(!saveFrame) {
        MSGDEBUG(3,"  Replacing frame " << num() - 1 << sendmsg);
        newts = item(num() - 1);
      } else {
        MSGDEBUG(3,"  Adding new frame " << num() << sendmsg);
        newts = new Timestep(nAtoms, 0.0);
      }

      // save new step number, and elapsed time from start of simulation
      Step = *(mdd->timestep);
      newts->dt = *(mdd->elapsed_time);

      // put coordinates into new timestep
      float *pos = newts->pos;
      float *tsx = mdd->X;
      float *tsy = mdd->Y;
      float *tsz = mdd->Z;
      for(int i=0; i < nAtoms; i++) {
	if(num() == 0) {
	  // this is the first frame, put the positions in the MolAtom objects
	  atm = atom(i);
	  atm->pos[0] = *tsx;
	  atm->pos[1] = *tsy;
	  atm->pos[2] = *tsz;
	}
        *(pos++) = *(tsx++);
        *(pos++) = *(tsy++);
        *(pos++) = *(tsz++);
      }
      
      // put energies into new timestep
      newts->energy[TSE_BOND] = *(mdd->Ebond);
      newts->energy[TSE_ANGLE] = *(mdd->Eangle);
      newts->energy[TSE_DIHE] = *(mdd->Edihe);
      newts->energy[TSE_IMPR] = *(mdd->Eimpr);
      newts->energy[TSE_VDW] = *(mdd->Evdw);
      newts->energy[TSE_COUL] = *(mdd->Eelec);
      newts->energy[TSE_HBOND] = *(mdd->Ehbo);
      newts->energy[TSE_TEMP] = *(mdd->T);
      newts->energy[TSE_PE] = *(mdd->Epot);
      newts->energy[TSE_TOTAL] = *(mdd->Etot);
      newts->energy[TSE_KE] = *(mdd->Etot) - *(mdd->Epot);

      // save patch data, if available
      if(*(mdd->numPatches) > 0) {
        float *pdata = newts->create_patch_storage(*(mdd->numPatches));
        float *px = mdd->pXOrigins;
	float *py = mdd->pYOrigins;
	float *pz = mdd->pZOrigins;
	float *pxlen = mdd->patchLength;
	float *pylen = mdd->patchWidth;
	float *pzlen = mdd->patchHeight;
	float *pload = mdd->patchLoads;
	float *pnode = mdd->patchNode;
	float *patoms = mdd->patchAtomNums;
        for(int np=0; np < newts->numPatches; np++) {
	  pdata[TSP_X] = *(px++);
	  pdata[TSP_Y] = *(py++);
	  pdata[TSP_Z] = *(pz++);
	  pdata[TSP_XLEN] = *(pxlen++);
	  pdata[TSP_YLEN] = *(pylen++);
	  pdata[TSP_ZLEN] = *(pzlen++);
	  pdata[TSP_LOAD] = *(pload++);
	  pdata[TSP_NODE] = *(pnode++);
	  pdata[TSP_ATOMS] = *(patoms++);
	  pdata += TSPATCHDATA;
	}
      } else {
        float *pdata = newts->create_patch_storage(1);
        pdata[TSP_X] = pdata[TSP_Y] = pdata[TSP_Z] = 0.0;
        pdata[TSP_XLEN] = pdata[TSP_YLEN] = pdata[TSP_ZLEN] = 0.0;
        pdata[TSP_LOAD] = 1.0;
        pdata[TSP_ATOMS] = (float)nAtoms;
        pdata[TSP_NODE] = 0.0;
      }

      if(newts->numPatches > 0)
        hasPatchDisplay = TRUE;

      // signal that a new timestep can be sent
      rem->done_with_ts();

      // now append the new frame, if necessary
      int new_frame_number;
      if(saveFrame) {
        new_frame_number = append_frame(newts);	// append the new frame
      } else {
        newts->init();			// recalc values for the timestep
	curr_frame_changed();		// signal the current frame changed
	new_frame_number = num() - 1;
      }
      framesReceived++;

      // change patch loc and size JUST FOR THE CASE WHERE NO PATCH DATA
      // GIVEN IN MDCOMM DATA
      // this makes the single patch the size of the current frame
      if(*(mdd->numPatches) < 1) {
        float *pdata = newts->patchData;
	pdata[TSP_X] = newts->minpos[0];
	pdata[TSP_Y] = newts->minpos[1];
	pdata[TSP_Z] = newts->minpos[2];
	pdata[TSP_XLEN] = newts->maxpos[0] - pdata[TSP_X];
	pdata[TSP_YLEN] = newts->maxpos[1] - pdata[TSP_Y];
	pdata[TSP_ZLEN] = newts->maxpos[2] - pdata[TSP_Z];
	newts->minpatch[TSP_X] = pdata[TSP_X];
	newts->minpatch[TSP_Y] = pdata[TSP_Y];
	newts->minpatch[TSP_Z] = pdata[TSP_Z];
	newts->maxpatch[TSP_X] = pdata[TSP_X] + pdata[TSP_XLEN];
	newts->maxpatch[TSP_Y] = pdata[TSP_Y] + pdata[TSP_YLEN];
	newts->maxpatch[TSP_Z] = pdata[TSP_Z] + pdata[TSP_ZLEN];
	newts->minpatch[TSP_XLEN] = newts->maxpatch[TSP_XLEN] =pdata[TSP_XLEN];
	newts->minpatch[TSP_YLEN] = newts->maxpatch[TSP_YLEN] =pdata[TSP_YLEN];
	newts->minpatch[TSP_ZLEN] = newts->maxpatch[TSP_ZLEN] =pdata[TSP_ZLEN];
      }

#ifdef VMDTCL
      // Tell Tcl that a new timestep has arrived
      {
	char mol[10];
	char n[10];
	sprintf(mol, "%d", id());
	sprintf(n, "%d", new_frame_number);
	Tcl_SetVar2(uiText->tclInterp, "vmd_timestep", mol, n,
		    TCL_GLOBAL_ONLY);
      }
#endif
    }
  } // if (rem && nAtoms > 0)
      
  // do any other preparations necessary
  Molecule::prepare(display);

  // Check for information to send to the Remote
  if (rem && nAtoms > 0) {
    // send_new_coords();
    send_new_forces();
  }
}


// get the method for displaying patches
int MoleculeRemote::patch_display_method(void) {
  if(drawPatch)
    return drawPatch->display_method();
  else
    return DrawPatch::BYNONE;
}
  
  
// change the method for displaying patches
int MoleculeRemote::patch_display_method(int newval) {
  if(drawPatch)
    return drawPatch->display_method(newval);
  else
    return DrawPatch::BYNONE;
}


// These send messages to the Remote simulation
// They are all called by "prepare"
void MoleculeRemote::send_new_forces(void) 
{
  // have any of the forces changed since the last time I looked?
  // Do this by getting the value of "changed_userforces"
  // from the AtomSelection data base
  AtomSel atomSel(moleculeList);
  if (atomSel.change("changed_userforces != 0") == AtomSel::NO_PARSE) {
    msgErr << "MoleculeRemote::send_new_forces: " 
      "Does the 'changed_userforces' data field exist?" << sendmsg;
    return;
  }
  atomSel.find(this);
  
  // scan to see if anything is selected
  int count = 0;
  for (int i=0; i<nAtoms; i++) {
    if (atomSel.on[i]) {
      count++;
    }
  }
  if (count) {  // there are new forces, so send them
    // get the new forces
    int *ind = new int[nAtoms];
    double *fx = new double[3*nAtoms];
    double *fy = fx + nAtoms;
    double *fz = fy + nAtoms;

    atomSel.use();
    atomSelParser.extract_keyword_info("index", nAtoms, ind, atomSel.on);
    atomSelParser.extract_keyword_info("ufx", nAtoms, fx, atomSel.on);
    atomSelParser.extract_keyword_info("ufy", nAtoms, fy, atomSel.on);
    atomSelParser.extract_keyword_info("ufz", nAtoms, fz, atomSel.on);

    // and send them to MDComm
    int *tmp = new int[3*count];
    float *forces = new float[3*count];
    int j = 0;
    for (int i=0; i<nAtoms; i++) {
      if (atomSel.on[i]) {
	tmp[j] = ind[i];
	forces[j]         = fx[i];
	forces[j+count]   = fy[i];
	forces[j+2*count] = fz[i];
	j++;
      }
    }
    rem -> send_new_forces(count, tmp, forces, forces+count, forces+2*count);

    // and reset the changed_userforces
    for (i=0; i<nAtoms; i++) {
      ind[i] = 0;
    }
    atomSelParser.set_keyword_info("changed_userforces", nAtoms, 
				   ind, atomSel.on);
    delete [] forces;
    delete [] tmp;
    delete [] fx;
    delete [] ind;

  }
}
void MoleculeRemote::send_new_coords(void) 
{
  // have any of the coords changed since the last time I looked?
  // Yep, so send to the Remote simulation
}


