/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile$
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.1 $	$Date: 96/02/21 16:46:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Manages a list of graphics objects.  They can be queried and modified.
 *  This is for use by the text interface (and perhaps others).
 *
 ***************************************************************************/

#include "MoleculeGraphics.h"
#include "DispCmds.h"
#include "ColorList.h"
#include "Global.h"

static DispCmdTriangle triangle;
static DispCmdMaterials materials;
static DispCmdCylinder cylinder;
static DispCmdPoint point;
static DispCmdLine line;
static DispCmdCone cone;
static DispCmdColorIndex color;
static DispCmdLineType linetype;
static DispCmdLineWidth linewidth;
static DispCmdSphere sphere;
static DispCmdSphereRes sph_res;
#define TEXTLEN 10
static DispCmdText text("          ");
static DispCmdTextPos textpos;

static char *graphics_info;
static int graphics_info_size;

// This helps manage the graphics_info* values
// an object is created at the start (done by the compiler) which makes
// sure the string is long enough.  This is only needed by 'text' since
// that is of indeterminate length
class GraphicsInfoHelper {
public:
  GraphicsInfoHelper(void) {
    graphics_info = new char[250];
    graphics_info_size = 250;
  }
  ~GraphicsInfoHelper(void) {
    delete [] graphics_info;
  }
  void append(char *s){
    if (strlen(s) + strlen(graphics_info) > graphics_info_size) {
      graphics_info_size =strlen(graphics_info) + strlen(s) + 10; 
      char *t = new char [graphics_info_size];
      strcpy(t, graphics_info);
      delete [] graphics_info;
      graphics_info = t;
    }
    strcat(graphics_info, s);
  }
};
static GraphicsInfoHelper graphics_info_helper;


void MoleculeGraphics::create_cmdlist(void) {
  reset_disp_list();
  // set the default values
  materials.putdata(1, this); // material characteristics are on
  color.putdata(0, this);     // use the first color by default (blue)
  int last_res = 6;
  sph_res.putdata(last_res, this); // sphere resolution

  int last_line = ::SOLIDLINE;       // default for lines
  linetype.putdata(last_line, this); // solid and
  int last_width = 1;
  linewidth.putdata(last_width, this);  //  of width 1

  // go down the list and draw things
  int num = num_elements();
  ShapeClass *shape;
  for (int i=0; i<num; i++) {
    shape = &(shapes[i]);
    switch (shape->shape) {
    case NONE: {
      break;
    }
    case POINT: {
      point.putdata(shape->data+0, this);
      break;
    }
    case LINE: {
      int style = int(shape->data[6]);
      int width = int(shape->data[7]);
      if (style != last_line) {
	linetype.putdata(style, this);
	last_line = style;
      }
      if (width != last_width) {
	linewidth.putdata(width, this);
	last_width = width;
      }
      line.putdata(shape->data+0, shape->data+3, this);
      break;
    }
    case TRIANGLE: {
      triangle.putdata(shape->data+0, shape->data+3, shape->data+6, this);
      break;
    }
    case TRINORM: {
      triangle.putdata(shape->data+0, shape->data+3 , shape->data+6, 
		       shape->data+9, shape->data+12, shape->data+15, this);
      break;
    }
    case CYLINDER: {
      cylinder.putdata(shape->data+0, shape->data+3, shape->data[6], 
		       int(shape->data[7]), int (shape->data[8]), this);
      break;
    }
    case CONE: {
      cone.putdata(shape->data+0, shape->data+3, shape->data[6], 
		       int(shape->data[7]), this);
      break;
    }
    case TEXT: {
      textpos.putdata(shape->data+0, this);
      // because of limitations in the DispCmdText, I have to break the
      // data into TEXTLEN blocks
      char *s = (char *) (shape -> data + 3);
      while (strlen(s) > TEXTLEN) {
	char c = s[TEXTLEN];
	s[TEXTLEN] = 0;
	text.putdata(s, this);
	s[TEXTLEN] = c;
	s += TEXTLEN;
      }
      text.putdata(s, this); // finish it off
      break;
    }
    case SPHERE: {
      int res = shape->data[4];
      if (res != last_res) {
	sph_res.putdata(res,this);
	last_res = res;
      }
      sphere.putdata(shape->data+0, shape->data[3], this);
      break;
    }
    case MATERIALS: {
      materials.putdata(shape->data[0], this);
      break;
    }
    case COLOR: {
      color.putdata(int(shape->data[0]), this);
      break;
    }
    default:
      msgErr << "Sorry, can't draw that" << sendmsg;
    }
  }
  needRegenerate = 0;
  
}

// resets the {next,max}_{id,index} values after something is added
// returns the value of the new element
int MoleculeGraphics::added(void)
{
  needRegenerate = 1;
  next_index = shapes.num();
  int retval = next_id;
  if (next_id == max_id) { // this was a new shape
    max_id++;
  } 
  next_id = max_id;
  return retval;
}


int MoleculeGraphics::add_triangle(float *x1, float *x2, float *x3)
{
  // save the points
  float *data = new float[9];
  copy(data+0, x1);
  copy(data+3, x2);
  copy(data+6, x3);
  
  // new one goes at next_id
  shapes[next_index].assign(TRIANGLE, data, next_id);
  return added();
}

int MoleculeGraphics::add_trinorm(float *x1, float *x2, float *x3,
				  float *n1, float *n2, float *n3)
{
  // save the points
  float *data = new float[18];
  copy(data+0, x1);
  copy(data+3, x2);
  copy(data+6, x3);
  copy(data+9, n1);
  copy(data+12, n2);
  copy(data+15, n3);
  
  // new one goes at next_id
  shapes[next_index].assign(TRINORM, data, next_id);
  return added();
}

int MoleculeGraphics::add_point(float *x)
{
  // save the point
  float *data = new float[3];
  copy(data+0, x);
  shapes[next_index].assign(POINT, data, next_id);
  return added();
}

int MoleculeGraphics::add_line(float *x1, float *x2, int style, int width)
{
  // save the endpoints
  float *data = new float[8];
  copy(data+0, x1);
  copy(data+3, x2);
  data[6] = float(style) + 0.1;
  data[7] = float(width) + 0.1;
  shapes[next_index].assign(LINE, data, next_id);
  return added();
}

int MoleculeGraphics::add_cylinder(float *x1, float *x2, float rad,
				   int n, int filled)
{
  // save the points
  float *data = new float[9];
  copy(data+0, x1);
  copy(data+3, x2);
  data[6] = rad;
  data[7] = float(n) + 0.1;
  data[8] = float(filled) + 0.1;
  
  // new one goes at next_id
  shapes[next_index].assign(CYLINDER, data, next_id);
  return added();
}

int MoleculeGraphics::add_cone(float *x1, float *x2, float rad, int n)
{
  // save the points
  float *data = new float[8];
  copy(data+0, x1);
  copy(data+3, x2);
  data[6] = rad;
  data[7] = float(n) + 0.1;
  
  // new one goes at next_id
  shapes[next_index].assign(CONE, data, next_id);
  return added();
}

int MoleculeGraphics::add_sphere(float *x, float rad, int n)
{
  // save the points
  float *data = new float[5];
  copy(data+0, x);
  data[3] = rad;
  data[4] = float(n) + 0.1;
  
  // new one goes at next_id
  shapes[next_index].assign(SPHERE, data, next_id);
  return added();
}


int MoleculeGraphics::add_text(float *x, char *text)
{
  // this is somewhat nasty as I save 'char *' data to a 
  // 'double *' array.  I also make sure I have enough space w/ the +2
  int sz = 3 + ((strlen(text) + 1) * sizeof(char))/
			 sizeof(float) + 2;
  float *data = new float[sz];
  copy(data+0, x);
  strcpy( (char *) (data+3), text);

  shapes[next_index].assign(TEXT, data, next_id);
  return added();
}

int MoleculeGraphics::use_materials(int yes_no)
{
  float *data = new float;
  data[0] = yes_no;
  shapes[next_index].assign(MATERIALS, data, next_id);
  return added();
}
// given RGB, finds the "nearest" index and uses that
int MoleculeGraphics::use_color(float r, float g, float b)
{
  return use_color(::colors -> nearest_index(r, g, b));
}
// do this based on the index
int MoleculeGraphics::use_color(int index)
{
  float *data = new float;
  data[0] = float(index) + 0.1; // just to be on the safe side for rounding
  shapes[next_index].assign(COLOR, data, next_id);
  return added();
}

// return the index in the array, or -1 if it doesn't exist
int MoleculeGraphics::index_id(int find_id)
{
  // the values in the array are numerically increasing, so I can do
  // a binary search.
  int max_loc = num_elements()-1;
  int min_loc = 0;
  if (max_loc < min_loc) {
    return -1;
  }
  int loc = (max_loc + min_loc) / 2;
  int id = shapes[loc].id;
  while (id != find_id && min_loc < max_loc) {
    if (id < find_id) {
      min_loc = loc+1;
    } else {
      max_loc = loc-1;
    }
    loc = (max_loc + min_loc) / 2;
    if (loc < 0) break;
    id = shapes[loc].id;
  }
  // and make sure it is for real
  if (id == find_id && shapes[loc].shape != NONE) {
    return loc;
  }
  return -1; // not found
}

// delete just the data components
void MoleculeGraphics::delete_data(void) {
  int num = num_elements();
  ShapeClass *shape;
  for (int i=0; i<num; i++) {
    shape = &(shapes[i]);
    if (shape->data) {
      delete [] shape->data;
      shape->data = NULL;
    }
  }
}
void MoleculeGraphics::delete_data(int start, int stop) 
{
  ShapeClass *shape;
  for (int i=start; i<=stop; i++) {
    shape = &(shapes[i]);
    if (shape->data) {
      delete [] shape->data;
      shape->data = NULL;
    }
  }
}
// delete everything
void MoleculeGraphics::delete_all(void)
{
  delete_data();         // first get rid of the 'data' terms of shapes
  shapes.remove(-1, -1); // then reset the resize array
  delete_count = 0;      // and reset the internal variables
  next_index = 0;
  next_id = 0;
  max_id = 0;
  needRegenerate = 1;
}

// delete given the id
void MoleculeGraphics::delete_id(int id)
{
  int index = index_id(id);
  if (index < 0) return;
  shapes[index].assign(NONE, NULL);
  delete_count++;
  if (delete_count > 1/* && 
      float(delete_count)/float(num_elements()) > 0.2*/) {
    // clear out the deleted elements
    int i, j=0, n = num_elements();
    // moving from i to j
    for (i=0; i<n; i++) {
      if (shapes[i].shape != NONE) {
	if (i != j) {
	  shapes[j] = shapes[i];
	}
	j++;
      }
    }
    // and cleaning out the rest w/o deleting any 'data'
    i=j;
    while (i<n) {
      shapes[i].shape = NONE;
      shapes[i].data = NULL;
      i++;
    }
    // now call the main remove
    delete_data(j, n-1);
    shapes.remove(j, n-1);
    delete_count = 0;
  }
  needRegenerate = 1;
  // delete overrides a replace
  next_id = max_id;
  next_index = num_elements();
}

// have the next added shape replace the given element
// returns index
int MoleculeGraphics::replace_id(int id)
{
  int index = index_id(id);
  if (index < 0) return -1;
  // if one was already assigned to be replaced, and we want to
  // replace another, increase the delete count
  if (next_id != max_id) {
    delete_count++;
  }
  // do the replacement
  shapes[index].assign(NONE, NULL);
  next_id = id;
  next_index = index;
  return index;
}
  

const MoleculeGraphics::ShapeClass& MoleculeGraphics::element(int n)
{
  static ShapeClass tmp;
  if (n>=0 && n<num_elements()) {
    return shapes[n];
  }
  return tmp;
}
const char *MoleculeGraphics::info_id(int id)
{
  int index = index_id(id);
  if (index < 0) return NULL;
  ShapeClass *shape;
  shape = &(shapes[index]);
  if (!shape->data) {
    msgErr << "NULL data for a shape in MoleculeGraphics.C" << sendmsg;
    return "";
  }
  switch (shape->shape) {
  case NONE: {
    return NONE;
  }
  case POINT: {
    sprintf(graphics_info, "point {%f %f %f}",
	    shape->data[0], shape->data[1], shape->data[2]);
    return graphics_info;
  }
  case LINE: {
    sprintf(graphics_info, "line {%f %f %f} {%f %f %f} style %s width %d",
	    shape->data[0], shape->data[1], shape->data[2], 
	    shape->data[3], shape->data[4], shape->data[5],
	    shape->data[6] < 0.5 ? "solid" : "dashed",
	    int(shape->data[7]));
    return graphics_info;
  }
  case TRIANGLE: {
    sprintf(graphics_info, "triangle {%f %f %f} {%f %f %f} {%f %f %f}",
	    shape->data[0], shape->data[1], shape->data[2], 
	    shape->data[3], shape->data[4], shape->data[5], 
	    shape->data[6], shape->data[7], shape->data[8]);
    return graphics_info;
  }
  case TRINORM: {
    sprintf(graphics_info, "trinorm {%f %f %f} {%f %f %f} {%f %f %f}"
	    "{%f %f %f} {%f %f %f} {%f %f %f}",
	    shape->data[0], shape->data[1], shape->data[2], 
	    shape->data[3], shape->data[4], shape->data[5], 
	    shape->data[6], shape->data[7], shape->data[8],
	    shape->data[9], shape->data[10], shape->data[11], 
	    shape->data[12], shape->data[13], shape->data[14], 
	    shape->data[15], shape->data[16], shape->data[17]);
    return graphics_info;
  }
  case CYLINDER: {
    sprintf(graphics_info, "cylinder {%f %f %f} {%f %f %f} "
	    "radius %f resolution %d filled %d",
	    shape->data[0], shape->data[1], shape->data[2], 
	    shape->data[3], shape->data[4], shape->data[5], 
	    shape->data[6], int(shape->data[7]), int(shape->data[8]));
    return graphics_info;
  }
  case CONE: {
    sprintf(graphics_info, "cone {%f %f %f} {%f %f %f} "
	    "radius %f resolution %d",
	    shape->data[0], shape->data[1], shape->data[2], 
	    shape->data[3], shape->data[4], shape->data[5], 
	    shape->data[6], int(shape->data[7]));
    return graphics_info;
  }
  case SPHERE: {
    sprintf(graphics_info, "sphere {%f %f %f} radius %f resolution %d",
	    shape->data[0], shape->data[1], shape->data[2], 
	    shape->data[3], int(shape->data[4]));
    return graphics_info;
  }
  case TEXT: {
    sprintf(graphics_info, "text {%f %f %f} {",
	    shape->data[0], shape->data[1], shape->data[2]);
    graphics_info_helper.append((char *) (shape->data+3));
    graphics_info_helper.append("}");
    return graphics_info;
  }

  case MATERIALS: {
    sprintf(graphics_info, "materials %d", int(shape->data[0]));
    return graphics_info;
  }
  case COLOR: {
    sprintf(graphics_info, "color %d", int(shape->data[0]));
    return graphics_info;
  }
  default:
    return "";
  }
}


// return the center of volume and scaling factor
#define CHECK_RANGE(v)     \
{                          \
  if (!found_one) {        \
    found_one = 1;         \
    minx = maxx = (v)[0];  \
    miny = maxy = (v)[1];  \
    minz = maxz = (v)[2];  \
  } else {                 \
    if (minx > (v)[0]) minx = (v)[0]; else if (maxx < (v)[0]) maxx = (v)[0]; \
    if (miny > (v)[1]) miny = (v)[1]; else if (maxy < (v)[1]) maxy = (v)[1]; \
    if (minz > (v)[2]) minz = (v)[2]; else if (maxz < (v)[2]) maxz = (v)[2]; \
  }                        \
}

void MoleculeGraphics::find_sizes(void)
{
  float minx, maxx;
  float miny, maxy;
  float minz, maxz;
  int found_one = 0;
  // go down the list and draw things
  int num = num_elements();
  ShapeClass *shape;
  for (int i=0; i<num; i++) {
    shape = &(shapes[i]);
    switch (shape->shape) {
    case NONE: {
      break;
    }
    case POINT: {
      CHECK_RANGE(shape->data+0);
      break;
    }
    case LINE: {
      CHECK_RANGE(shape->data+0);
      CHECK_RANGE(shape->data+3);
      break;
    }
    case TRIANGLE: {
      CHECK_RANGE(shape->data+0);
      CHECK_RANGE(shape->data+3);
      CHECK_RANGE(shape->data+6);
      break;
    }
    case TRINORM: {
      CHECK_RANGE(shape->data+0);
      CHECK_RANGE(shape->data+3);
      CHECK_RANGE(shape->data+6);
      break;
    }
    case CYLINDER: {
      CHECK_RANGE(shape->data+0);
      CHECK_RANGE(shape->data+3);
      break;
    }
    case CONE: {
      CHECK_RANGE(shape->data+0);
      CHECK_RANGE(shape->data+3);
      break;
    }
    case SPHERE: { // I suppose I should include +/- radius ...
      CHECK_RANGE(shape->data+0);
      break;
    }
    case TEXT: { // I suppose I should include the string length ...
      CHECK_RANGE(shape->data+0);
      break;
    }
    default:
      break;
    }
  }

  // compute the values for center of volume center and scale
  if (!found_one) {
    cov_pos[0] = cov_pos[1] = cov_pos[2];
    cov_scale = 0.1;
  } else {
    cov_pos[0] = (minx + maxx) / 2.0;
    cov_pos[1] = (miny + maxy) / 2.0;
    cov_pos[2] = (minz + maxz) / 2.0;
    float dx = maxx - minx;
    float dy = maxy - miny;
    float dz = maxz - minz;
    // a bit of sanity check (eg, suppose there is only one point)
    if (dx == 0 && dy == 0 && dz == 0) dx = 10;
    if (dx > dy) {
      if (dx > dz) {
	cov_scale = 2.0 / dx;
      } else {
	cov_scale = 2.0 / dz;
      }
    } else {
      if (dy > dz) {
	cov_scale = 2.0 / dy;
      } else {
	cov_scale = 2.0 / dz;
      }
    }
  }
}

