/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Light.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.8 $	$Date: 95/05/11 23:02:00 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A Displayable3D object which represents a light source at infinity, along
 * some line from the origin, of some color.
 *
 ***************************************************************************/

#include "Light.h"
#include "Inform.h"
#include "Displayable.h"
#include "DispCmds.h"
#include "DisplayDevice.h"


// constructor
Light::Light(Scene *sc, int ln, float *c, float *p)
	: Displayable3D(LOAD, "Light", sc, 0), htxt(" ") {
  // initialize variables
  lightOn = FALSE;
  need_define = need_onoff = TRUE;
  need_highlight = highlightOn = FALSE;
  needDefineCleanup = FALSE;

  // set number of this light
  if(ln < 0 || ln >= DISP_LIGHTS)
    num = 0;
  else
    num = ln;
  
  // store color and position
  for(int i=0; i < 3; i++) {
    color[i] = c[i];
    pos[i] = p[i];
  }

  // create command to load identity matrix on dispay's stack
  loadIdentMat = new DispCmdLoad(&ident);

  // Displayable characteristics
  rot_off();
  scale_off();
  glob_trans_off();
  cent_trans_off();
}


// destructor
Light::~Light(void) {
  delete loadIdentMat;
}


// override the on/off commands since the light will always need to be
// taken care of
void Light::off(void) {
  Displayable3D::on();
  lightOn = FALSE; need_onoff = TRUE;
}

void Light::on(void) {
  Displayable3D::on();
  lightOn = need_onoff = TRUE;
}


// extra action added to Displayable's rotating commands, 
// to signal a redef of the
// light is needed, and to update the rotation matrix
void Light::add_rotation(float x, char axis) {
  need_define = TRUE;
  rot_on();
  add_rot(x,axis);
  rot_off();
}

void Light::add_rotation(Matrix4 &m) {
  need_define = TRUE;
  rot_on();
  add_rot(m);
  rot_off();
}

void Light::set_rotation(Matrix4 &m) {
  need_define = TRUE;
  rot_on();
  set_rot(m);
  rot_off();
}


// origin position
static float lightOrigin[3] = { 0.0, 0.0, 0.0 };


// create the command list
void Light::create_cmdlist(void) {
  
  // do reset first
  reset_disp_list();

  // see if we need to define the light source
  if(need_define)
    // put in define light command
    deflight.putdata(num, color, pos, this);
  
  // see if we need to turn the light on or off
  if(need_onoff)
    // put in on/off command
    onofflight.putdata(num, lightOn, this);

  // draw line from origin to light if we are highlighting
  if(highlightOn) {
    cmdLineType.putdata(DASHEDLINE, this);
    cmdLineWidth.putdata(3, this);
    hcol.putdata(color, this);
    hline.putdata(lightOrigin, pos, this);
    htxtpos.putdata(pos, this);
    char txt[2];
    txt[0] = '0' + num;
    txt[1] = '\0';
    htxt.putdata(txt, this);
  }
  
  // set drawing flags
  needDefineCleanup = (need_define || need_onoff || need_highlight);
  need_define = need_onoff = need_highlight = FALSE;
}


// prepare the light for display ... regenerate command list if necessary
void Light::prepare(DisplayDevice *) {
  // if a flag has been set, change the settings
  if(need_define || need_onoff || need_highlight || needDefineCleanup)
    create_cmdlist();
}

