/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: GraphicsFormsObj.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.31 $	$Date: 96/04/28 20:11:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The on-screen menu to control the coloring and display method of molecules.
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "GraphicsFormsObj.h"
#include "CmdMol.h"
#include "CmdMenu.h"
#include "MoleculeList.h"
#include "Molecule.h"
#include "NameList.h"
#include "UIList.h"
#include "SymbolTable.h"
#include "AtomColor.h"
#include "AtomSel.h"
#include "AtomRep.h"
#include "Global.h"      //  atomSelParser is located here

// commands we are interested in
static int numCmds = 8;
static int cmdList[8] = { Command::MOL_NEW,     Command::MOL_DEL,
        Command::MOL_ADDREP,            Command::MOL_MODREP,
        Command::MOL_DELREP,            Command::MOL_MODREPITEM,
        Command::REMOTE_ATTACH,         Command::REMOTE_RUN };


// type of editing the menu is doing currently
#define GREDIT_UNKNOWN		0
#define GREDIT_COLORDRAW	1
#define GREDIT_ATOMSEL		2
#define GREDIT_DEFAULT		GREDIT_COLORDRAW


// collection of objects which are used for the different editing modes
static FL_OBJECT **colordraw_objs[] = { &color_method_box, &draw_method_box,
	&graphics_color, &graphics_cindex, &graphics_trans,
	&graphics_drawstyle, &graphics_bond_rad, &graphics_bond_res,
	&graphics_sphere_rad, &graphics_sphere_res, &graphics_line_thick };

static FL_OBJECT **atomsel_objs[] = { &atomsel_method_box,
	&atomsel_which_list, &atomsel_namelist };

static FL_OBJECT **atomrep_counters[] = { &graphics_sphere_rad,
	&graphics_bond_rad, &graphics_sphere_res, &graphics_bond_res,
        &graphics_line_thick };


/////////////////////////  constructor  
GraphicsFormsObj::GraphicsFormsObj(UIList *uil, CommandQueue *cq,
	int sh,int bd,int *npos) : FormsObj("graphics", uil, cq, bd, npos) {
	
  // set values of common menu elements
  form = graphicsMenu;
  lightButton = graphics_menu_button;
  offButton = graphics_menu_off;
  
  // register which commands we are interested in
  for(int i=0; i < numCmds; command_wanted(cmdList[i++]));
  
  // turn on if required
  if(sh)
    On();

  // indicate we do not yet have a selected molecule
  molID = (-1);
  molRep = (-1);
  dmol = NULL;
  
  // set the current editing mode
  editMode = GREDIT_UNKNOWN;
}


/////////////////////////  private routines  

// fill the given string buffer with a string describing the mth rep of
// the current mol.  Returns a pointer to the string.  If error, string is
// returned empty.
char *GraphicsFormsObj::graphics_rep_string(int m, char *str) {
  DrawMolItem *component;

  // check if no rep available
  if(!dmol || !(component = dmol->component(m))) {
    str[0] = '\0';
  } else {
    // component is the item to get data from
    strncpy(str, (component->atomRep)->cmdStr, 11);
    str[11] = '\0';
    while(strlen(str) < 12)
      strcat(str, " ");
    strncat(str, (component->atomColor)->cmdStr, 11);
    str[23] = '\0';
    while(strlen(str) < 24)
      strcat(str, " ");
    strncat(str, (component->atomSel)->cmdStr, 20);
    str[44] = '\0';
  }
  
  return str;
}



// set the graphics editing mode to the given value
void GraphicsFormsObj::set_editing_mode(int newMode) {

  if(newMode == editMode)
    return;
  editMode = newMode;

  freeze();
    // turn off objects first
    if(newMode != GREDIT_COLORDRAW) {
      int amt = sizeof(colordraw_objs) / sizeof(FL_OBJECT **);
      for(int i=0; i < amt; i++)
        fl_hide_object( *(colordraw_objs[i]) );
    }
    if(newMode != GREDIT_ATOMSEL) {
      int amt = sizeof(atomsel_objs) / sizeof(FL_OBJECT **);
      for(int i=0; i < amt; i++)
        fl_hide_object( *(atomsel_objs[i]) );
    }
    
    // turn on objects last
    if(newMode == GREDIT_COLORDRAW) {
      fl_set_button(graphics_mode_colordraw, TRUE);
      int amt = sizeof(colordraw_objs) / sizeof(FL_OBJECT **);
      for(int i=0; i < amt; i++)
        fl_show_object( *(colordraw_objs[i]) );
      reset_coloring_method(FALSE);
      reset_drawing_method(FALSE);
      reset_selection_method(FALSE);
    } else if(newMode == GREDIT_ATOMSEL) {
      fl_set_button(graphics_mode_atomsel, TRUE);
      int amt = sizeof(atomsel_objs) / sizeof(FL_OBJECT **);
      for(int i=0; i < amt; i++)
        fl_show_object( *(atomsel_objs[i]) );
    }
  unfreeze();
}

// fill the namelist browser for atom selection with the proper names
void GraphicsFormsObj::reset_namelist_browser(void) {
  freeze();
    // clear the name browser
    fl_clear_browser(atomsel_namelist);

    // get current name list selection, and check for a category with names
    int list = fl_get_browser(atomsel_which_list) - 1;
  
    // only if a list has been selected, and is useful one, do we enter names
    // also, only need to fill browser if there is a molecule currently
    if(dmol && list >= 0 && list < atomSelParser.names.num() && molRep >= 0) {
      // tell the atom selection commands to use the current molecule to 
      // get the required information
      dmol->component(molRep)->atomSel->use();

       // get all the info from the item in the atomSelParser
       // first see what it does
       SymbolTableName *info = atomSelParser.names[list];
       // the only one that shows anything are the KEYWORDs
       if (info -> is_a == SymbolTableName::KEYWORD) {
	  // then get the data as a string
	  int num = dmol->nAtoms;
	  GString *data = new GString[num];
	  int *flgs = new int[num];
	  for (int i=0; i<num; i++) {
	     flgs[i] = 1;
	  }
	  // get the sorted (hence the 1) keywords
	  // I special case 'resid' since mostly it is a numeric, but
	  // X-PLOR allows resids with value > 9999 to be like A000
	  if (info->regex->match("resid", 5) != -1) {
	     atomSelParser.extract_keyword_info(list, num, data, flgs, 1,
						SymbolTableName::IS_INT);
	  } else {
	     atomSelParser.extract_keyword_info(list, num, data, flgs, 1);
	  }
	  
	  // now find the unique ones
	  for (i=0; i<num; ) {
	     int j=i+1;
	     while (j<num && data[j] == data[i]) {
		flgs[j] = 0;
		j++;
	     }
	     i = j;
	  }
	  // and show them
	  for (i=0; i<num; i++) {
	     if (flgs[i]) {
		fl_add_browser_line(atomsel_namelist,
				    (char *)(const char *)data[i]);
	     }
	  }
	  delete [] flgs;
	  delete [] data;
       }
    }
  unfreeze();
}


// fill the graphics rep browser with the settings for the current mol
void GraphicsFormsObj::fill_graphics_reps_browser(void) {
  freeze();
    char s[100];
    int N = (dmol ? dmol->components() : 0);
    fl_clear_browser(graphics_reps);
    if(N > 0) {
      for(int i=0; i < N; i++)
        fl_add_browser_line(graphics_reps, (char *) graphics_rep_string(i,s));
	
      // by default, select the first item if no other option, else keep same
      // selection
      if(molRep < 0 || molRep >= N)
        molRep = 0;
      fl_select_browser_line(graphics_reps, molRep + 1);
    } else {
      molRep = (-1);
    }
    reset_edited_rep();
  unfreeze();
}


// reset the form to use the molecule with given ID as the current mol
void GraphicsFormsObj::reset_edited_molecule(int n) {
  freeze();
    fl_clear_choice(graphics_mol);

    if(moleculeList->num() < 1) {
      // no molecules; reset everything
      molID = (-1);
      dmol = NULL;
    } else {
      // select a molecule (possibly the same one as previously chosen)
      int newline = (-1), newID = (-1), topline = (-1), oldline = (-1);
      for(int i=0; i < moleculeList->num(); i++) {
        Molecule *m = moleculeList->molecule(i);
        fl_addto_choice(graphics_mol, m->name);
        if(m->id() == n)
          newline = i;
	if(m->id() == molID)
	  oldline = i;
	if(moleculeList->is_top(m))
	  topline = i;
      }
      if(newline < 0) {
        newline = (topline < 0 ? oldline : topline);
        if(newline < 0)		// STILL no new molecule chosen, so
	  newline = 0;		// just take the first one
      }

      fl_set_choice(graphics_mol, newline + 1);
      newID = (moleculeList->molecule(newline))->id();

      // a new molecule; reset other items
      molID = newID;
      dmol = moleculeList->molecule(newline);
    }

    // reset all the other browsers and options based on the new molecule
    molRep = (-1);			// no rep selected yet
    fill_graphics_reps_browser();	// put text in rep browser

    reset_namelist_browser();		// put text in namelist browser
  unfreeze();
}


// reset the form based on the current selections in the graphics rep browser
void GraphicsFormsObj::reset_edited_rep(void) {

  freeze();
    // scan through the rep browser until a line is found, and then
    // set the settings for that item.
    int N = (dmol ? dmol->components() : 0);
    for(int i=0; i < N; i++) {
      if(fl_isselected_browser_line(graphics_reps, i+1)) {
        molRep = i;
        break;
      }
    }
    
    // now reset all controls based on new molRep value
    reset_coloring_method(TRUE);
    reset_drawing_method(TRUE);
    reset_selection_method(TRUE);
  unfreeze();
}


// reset coloring method controls based on current rep, if any
void GraphicsFormsObj::reset_coloring_method(int setvals) {
  freeze();
    int newmethod = fl_get_choice(graphics_color) - 1;
    if(setvals && molRep >= 0) {
      AtomColor *ac = (dmol->component(molRep))->atomColor;
      newmethod = ac->method();
      fl_set_choice(graphics_color, newmethod + 1);
      fl_set_button(graphics_trans, ac->transparent());
      if(newmethod == AtomColor::COLORID)
        set_input_int(graphics_cindex, ac->color_index(), 3);
    }
    if(editMode == GREDIT_COLORDRAW) {
      if(newmethod == AtomColor::COLORID)
        fl_show_object(graphics_cindex);
      else
        fl_hide_object(graphics_cindex);
    }
  unfreeze();
}


// reset drawing method controls based on current rep, if any
void GraphicsFormsObj::reset_drawing_method(int setvals) {
  freeze();
    AtomRep *ar = NULL;
    int newstyle = fl_get_choice(graphics_drawstyle) - 1;

    // determine if we have an item selected
    if(setvals && molRep >= 0) {
      ar = (dmol->component(molRep))->atomRep;
      newstyle = ar->method();
      fl_set_choice(graphics_drawstyle, newstyle + 1);
    }

    AtomRepParamStruct *arInfo = &(AtomRepInfo[newstyle]);
    for(int i=0; i < arInfo->numdata; i++) {

      // configure the counter properly for this data item
      AtomRepDataInfo *arData = &(arInfo->repdata[i]);
      FL_OBJECT *obj = *(atomrep_counters[i]);
      float usemin = -1000000, usemax = 1000000;
      if(arData->hasmin)
	usemin = arData->min;
      if(arData->hasmax)
	usemax = arData->max;
      fl_set_counter_precision(obj, arData->sigDigits);
      fl_set_counter_step(obj, arData->mininc, arData->maxinc);
      fl_set_counter_bounds(obj, usemin, usemax);
      fl_set_object_label(obj, arData->name);

      // set the counter value if we have a rep selected
      float newval = (ar != NULL ?
		      ar->get_data(arData->index) : arData->defValue);
      fl_set_counter_value(obj, newval);

      // display the counter if necessary
      if(editMode == GREDIT_COLORDRAW)
        fl_show_object(obj);
      else
        fl_hide_object(obj);
    }

    // now hide the remaining counters
    for(int j=arInfo->numdata; j < MAX_ATOMREP_DATA; j++)
      fl_hide_object(*(atomrep_counters[j]));

  unfreeze();
}


// reset atom selection controls based on current rep, if any
void GraphicsFormsObj::reset_selection_method(int setvals) {
  if(molRep < 0 || !setvals)
    return;

  freeze();
    AtomSel *as = (dmol->component(molRep))->atomSel;
    fl_set_input(graphics_sel, as->cmdStr);
  unfreeze();
}


// fill string with text command to set objects to proper color, rep, or sel.
// NOTE: this routine is NOT re-entrant.  But that should not matter here ...
char *GraphicsFormsObj::create_color_cmd(void) {
  static char colorStr[256];
  int colorVal = fl_get_choice(graphics_color) - 1;
  
  if(colorVal >= 0) {
    // create new color command string
    if(colorVal == AtomColor::COLORID)
      sprintf(colorStr, "%s %d", AtomColorName[colorVal],
	atoi(fl_get_input(graphics_cindex)));
    else
      strcpy(colorStr, AtomColorName[colorVal]);
    if(fl_get_button(graphics_trans))
      strcat(colorStr, " transparent");
  } else
    colorStr[0] = '\0';

  return colorStr;
}


// fill string with text command to set objects to proper color, rep, or sel.
// NOTE: this routine is NOT re-entrant.  But that should not matter here ...
char *GraphicsFormsObj::create_rep_cmd(void) {
  static char styleStr[256], tmpstr[256];
  int styleVal = fl_get_choice(graphics_drawstyle) - 1;

  // create new display method string
  if(styleVal >= 0) {
    AtomRepParamStruct *arInfo = &(AtomRepInfo[styleVal]);
    strcpy(styleStr, arInfo->name);

    // tack on parameter values
    for(int i=0; i < arInfo->numdata; i++) {
      sprintf(tmpstr, " %f", fl_get_counter_value(*(atomrep_counters[i])));
      strcat(styleStr, tmpstr);
    }
  } else
    *styleStr = '\0';

  return styleStr;
}


// fill string with text command to set objects to proper color, rep, or sel.
char *GraphicsFormsObj::create_sel_cmd(void) {
  char *selInput = (char *) fl_get_input(graphics_sel);
  
  if(selInput && strlen(selInput) > 0)
    return selInput;
  else
    return NULL;
}


// returns the num of possible selection commands
int GraphicsFormsObj::num_selection_commands(void)
{
   return atomSelParser.names.num();
}

// return the string corresponding to the "nth" selection command
const char *GraphicsFormsObj::selection_command(int index)
{
   static GString my_string;
   if (index < 0 || index >= num_selection_commands()) {
      my_string = "";
   } else {
      my_string = atomSelParser.names[index]->visible;
      // prettify functions; cos -> cos(
      if (atomSelParser.names[index]->is_a == SymbolTableName::FUNCTION) {
	 my_string += "(" ;
      }
   }
   return (const char *)my_string;
}

//////////////////////  protected virtual routines  

// routine to check the main form for use
int GraphicsFormsObj::check(FL_OBJECT *obj) {
//  char selbuf[256], molbuf[16];
   GString selbuf;
   char molbuf[16];

   static FL_OBJECT* prev_obj = NULL;
   static int prev_count = -1;
   if (obj != prev_obj) {
     prev_count = -1;
   }
   prev_obj = obj;

  if(obj == graphics_display_colors) {
    if(uiL->item("color")) {
      int clrid = (uiL->item("color")) -> id();
      addcommand(new CmdMenuShow(clrid, TRUE, id()));
    }

  } else if(obj == graphics_display_molecules) {
    if(uiL->item("mol")) {
      int clrid = (uiL->item("mol")) -> id();
      addcommand(new CmdMenuShow(clrid, TRUE, id()));
    }

  } else if(obj == graphics_mol) {
    Molecule *mol = moleculeList->molecule(fl_get_choice(obj) - 1);
    if(mol)
      reset_edited_molecule(mol->id());
    else
      reset_edited_molecule(-1);

  } else if(obj == graphics_reps) {
    reset_edited_rep();

  } else if(obj == graphics_color || obj == graphics_cindex || 
  	obj == graphics_trans) {
    if(molRep < 0 || !fl_get_button(graphics_auto_change)) {
      if(obj == graphics_color)
        reset_coloring_method(FALSE);
    } else {
      if(obj == graphics_color) {
        freeze();
          int colorVal = fl_get_choice(graphics_color) - 1;
          reset_coloring_method(TRUE);
          fl_set_choice(graphics_color, colorVal + 1);
          reset_coloring_method(FALSE);
        unfreeze();
      }

      // add command to just change the coloring of the current item
      sprintf(molbuf,"%d",molID);
      addcommand(new CmdMolChangeRepItem(molRep, molbuf,
			CmdMolChangeRepItem::COLOR,create_color_cmd(),id()));
    }

  } else if(obj == graphics_drawstyle || obj == graphics_bond_rad ||
	obj == graphics_bond_res || obj == graphics_sphere_rad ||
	obj == graphics_sphere_res || obj == graphics_line_thick) {
    if(molRep < 0 || !fl_get_button(graphics_auto_change)) {
      if(obj == graphics_drawstyle)
        reset_drawing_method(FALSE);
    } else {
      if(obj == graphics_drawstyle) {
        freeze();
          int styleVal = fl_get_choice(graphics_drawstyle) - 1;
          reset_drawing_method(TRUE);
          fl_set_choice(graphics_drawstyle, styleVal + 1);
          reset_drawing_method(FALSE);
        unfreeze();
      }

      // add command to just change the drawing method of the current item
      sprintf(molbuf,"%d",molID);
      addcommand(new CmdMolChangeRepItem(molRep, molbuf,
			CmdMolChangeRepItem::REP, create_rep_cmd(), id()));
    }

  } else if(obj == graphics_sel) {
    if(molRep >= 0 && fl_get_button(graphics_auto_change)) {
      char *s = create_sel_cmd();
      if(s) {
        sprintf(molbuf,"%d",molID);
        addcommand(new CmdMolChangeRepItem(molRep, molbuf,
			CmdMolChangeRepItem::SEL, s, id()));
      } else {
        reset_selection_method(TRUE);
      }
    }

  } else if(obj == graphics_new_selected || obj == graphics_apply_selected) {
    // put in commands to change current settings
    if(molID >= 0 && (obj == graphics_new_selected || molRep >= 0)) {
      char *s = create_sel_cmd();
      if(s) {
        // put in commands to set the 'default' rep settings
        addcommand(new CmdMolColor(create_color_cmd(), id()));
        addcommand(new CmdMolRep(create_rep_cmd(), id()));
        addcommand(new CmdMolSelect(s, id()));
        
        // put in command to add item
        sprintf(molbuf,"%d",molID);
        if(obj == graphics_new_selected)
          addcommand(new CmdMolAddRep(molbuf, id()));
        else
          addcommand(new CmdMolChangeRep(molRep, molbuf, id()));
      } else
        reset_selection_method(TRUE);
    }

  } else if(obj == graphics_clear_changes) {
    // on a double click, clear the whole line
    if (prev_count++ == -1) {  // initially, will be -1
      reset_edited_rep();
    } else {
      prev_count = -1;
      fl_set_input(graphics_sel, "");  // clear the line
    }

  } else if(obj == graphics_delete) {
    if(molID >= 0 && molRep >= 0) {
      sprintf(molbuf,"%d",molID);
      addcommand(new CmdMolDeleteRep(molRep, molbuf, id()));
    }

  } else if(obj == graphics_mode_colordraw) {
    set_editing_mode(GREDIT_COLORDRAW);
    
  } else if(obj == graphics_mode_atomsel) {
    set_editing_mode(GREDIT_ATOMSEL);

  } else if(obj == atomsel_namelist) {
    // append name to end of selection string
     selbuf = fl_get_input(graphics_sel);
     selbuf += " ";
     selbuf += fl_get_browser_line(obj, fl_get_browser(obj));
     fl_set_input(graphics_sel, (char *)(const char *)selbuf);

  } else if(obj == atomsel_which_list) {
    if (prev_count == fl_get_browser(obj)) {  // poor-man's double click
      // append name to end of selection string
      selbuf = fl_get_input(graphics_sel);
      if (selbuf.length() != 0) selbuf += " ";
      selbuf += fl_get_browser_line(obj, fl_get_browser(obj));
      fl_set_input(graphics_sel, (char *)(const char *)selbuf);
      prev_count = -1;
    } else {
      prev_count = fl_get_browser(obj);
    }
    
    // fill the namelist browser with the proper names
    reset_namelist_browser();

  } else
    return FALSE;

  return TRUE;
}


// do form-specific acting on commands.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int GraphicsFormsObj::forms_act_on_command(int type, Command *, int) {

  MSGDEBUG(3,"GraphicsFormsObj: acting on command " << type << sendmsg);

  if(type == Command::MOL_NEW || type == Command::MOL_DEL ||
	type == Command::MOL_ADDREP || type == Command::MOL_MODREP ||
	type == Command::MOL_DELREP || 
	type == Command::REMOTE_ATTACH || type == Command::REMOTE_RUN) {
    freeze();
      if(type == Command::MOL_NEW || type == Command::MOL_DEL ||
		type == Command::REMOTE_RUN) {
        // make the top molecule the active one here
        molID = (-1);
      }
      reset();
      if(type == Command::MOL_ADDREP) {
        // select the last item in the representation browser, if any,
	// so that new reps are automatically selected
	int maxl = fl_get_browser_maxline(graphics_reps);
	if(maxl > 0) {
	  fl_deselect_browser(graphics_reps);
	  fl_select_browser_line(graphics_reps, maxl);
	  reset_edited_rep();
        }
      }
    unfreeze();

  } else if(type == Command::MOL_MODREPITEM) {
    fill_graphics_reps_browser();

  } else
    // unknown command type
    return FALSE;
    
  return TRUE;
}


//////////////////////  public virtual routines  

// initialize the user interface
void GraphicsFormsObj::init(void) {
  int i;

  freeze();
    fl_set_button(graphics_trans, FALSE);
    fl_set_button(graphics_auto_change, TRUE);
    fl_set_browser_fontstyle(graphics_reps, FL_FIXED_STYLE);

    fl_clear_choice(graphics_color);
    for(i=0; i < AtomColor::TOTAL; i++)
      fl_addto_choice(graphics_color, AtomColorName[i]);
    fl_set_choice(graphics_color, DEFAULT_ATOMCOLOR + 1);

    fl_clear_choice(graphics_drawstyle);
    for(i=0; i < AtomRep::TOTAL; i++)
      fl_addto_choice(graphics_drawstyle, AtomRepInfo[i].name);
    fl_set_choice(graphics_drawstyle, DEFAULT_ATOMREP + 1);

    fl_clear_browser(atomsel_which_list);
    for(i=0; i < num_selection_commands(); i++) {
       fl_add_browser_line(atomsel_which_list, (char *) selection_command(i));
    }
   
    fl_clear_browser(atomsel_namelist);

    // do a reset to set initial values
    reset();

    // properly set the editing mode
    set_editing_mode(GREDIT_DEFAULT);
  unfreeze();
}


// reset the user interface
void GraphicsFormsObj::reset(void) {
  reset_edited_molecule(molID);
}

