/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: GLRenderer.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.23 $	$Date: 1996/05/14 04:50:11 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that are GL-specific.  Will render drawing
 * commands into a single GL window.  This is not the complete definition,
 * however, of a DisplayDevice; something must provide routines to open
 * windows, reshape, clear, set perspective, etc.  This object contains the
 * code to render a display command list.
 *
 ***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "GLRenderer.h"
#include "PopupMenu.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"

// array for defining lights
static float lightdata[] = {
	LCOLOR, 0.0, 0.0, 0.0,
	POSITION, 0.0, 0.0, 0.0, 0.0,
	LMNULL };

// array for defining materials
static float matdata[] = {
	AMBIENT,  0.0, 0.0, 0.0,
	DIFFUSE,  0.0, 0.0, 0.0,
	SPECULAR, 0.0, 0.0, 0.0,
	SHININESS, 0.0,
	ALPHA, 0.0,
	LMNULL };


/////////////////////////  constructor and destructor  

// constructor ... initialize some variables
GLRenderer::GLRenderer(char *nm) : DisplayDevice(nm) {
  need_aaon = need_cueon = FALSE;
  dataBlock = NULL;
  mainMenu = NULL;

  MSGDEBUG(1,"Creating GLRenderer ..." << sendmsg);

}

//#define USE_DISKS
//#define USE_NURBS


#ifdef USE_DISKS
// the following code derived from 'circ2.c' by Paul Heckbert 
// <ph@cs.cmu.edu>
#define NMAX 100
#define NSIDES 10
typedef struct {float cos, sin;} circle;
static circle circtab[NMAX];		/* lookup table for circle-drawing */

static void table_init(int nsides) {
    int i;
    float ang;

    if (nsides > NMAX) nsides = NMAX;
    for (i=0; i<nsides; i++) {
	ang = 2.*M_PI*i/nsides;
	circtab[i].cos = cosf(ang);
	circtab[i].sin = sinf(ang);
    }
}

#define VEC_SET(xx,yy,zz,a) ((a)[0]=xx, (a)[1]=yy, (a)[2]=zz)
#define VEC_LEN(a) sqrtf(a[0]*a[0]+a[1]*a[1]+a[2]*a[2])
void table_circle(float *cen, float *n, float r) {
    /* doesn't assume n is a unit vector */
    register int i;
    register circle *tab;
    float u[3], v[3], pt[3];

    if (n[0]*n[0] < .33) VEC_SET(0, n[2], -n[1], u);	/* n X (1,0,0) */
    else VEC_SET(-n[2], 0, n[0], u);			/* n X (0,1,0) */
    normalize(u);
    u[0] *= r; u[1] *= r; u[2] *= r; 
    cross_prod(v, n, u);		/* v is another tangent vector*/
    normalize(v);
    v[0] *= r; v[1] *= r; v[2] *= r; 
    /* u, v, and n are mutually orthogonal, u and v have length r */

    //    texbind(TX_TEXTURE_0, 1);
    bgnpolygon();
    float tx[2];
    n[0] = 0.0;
    n[1] = 0.0;
    n[2] = 1.0;

    for (tab=circtab, i=NSIDES; --i>=0; tab++) {
	pt[0] = cen[0] + tab->cos*u[0] + tab->sin*v[0];
	pt[1] = cen[1] + tab->cos*u[1] + tab->sin*v[1];
	pt[2] = cen[2] + tab->cos*u[2] + tab->sin*v[2];
	n3f(n);
	v3f(pt);
	tx[0] = tab->cos * 8 + 8;
	tx[1] = tab->sin * 16 + 16;
	//	t2f(tx);
    }
    endpolygon();
}

// init the table at startup
class GLRenderer_helper {
public:
  GLRenderer_helper::GLRenderer_helper(){
    table_init(NSIDES);
  }
};
static GLRenderer_helper glrenderer_helper;


static Matrix4 world_mat;
static float world_scale;
void fast_sphdraw(float *pos, float *eyePos)
{

  float sph[3]; // location of the sphere in eye space
  world_mat.multpoint3d(pos, sph);

  // get the normal, which is along the line from the viewer to the sphere
  float norm[3];
  subtract(norm, eyePos, sph);
  table_circle(sph, norm, pos[3] * world_scale);

}
#endif // USE_DISKS

#ifdef USE_NURBS

#define S_NUMKNOTSX	12   /* number of knots in each dimension of surface*/
#define S_NUMKNOTSY     8
#define S_NUMCOORDS	3   /* number of surface coordinates, (x,y,z) */
#define S_ORDER		3

/* number of control points in each dimension of NURBS */
#define S_NUMPOINTSX	(S_NUMKNOTSX - S_ORDER)
#define S_NUMPOINTSY	(S_NUMKNOTSY - S_ORDER)

double	surfknotsx[S_NUMKNOTSX] = {
0, 0, 0,0.25, 0.25, 0.5, 0.5, 0.75, 0.75, 1, 1, 1
	};

double	surfknotsy[S_NUMKNOTSY] = {
               0,               0,               0,             0.5,
             0.5,               1,               1,               1
	};

double ctlpoints[S_NUMPOINTSX][S_NUMPOINTSY * (S_NUMCOORDS+1)] = {
               0,              0,             -1,              1,
               0,              0,      -0.707107,       0.707107,
               0,              0,             -1,              1,
               0,              0,      -0.707107,       0.707107,
               0,              0,             -1,              1,
               0,              0,      -0.707107,       0.707107,
               0,              0,             -1,              1,
               0,              0,      -0.707107,       0.707107,
               0,              0,             -1,              1,
        0.707107,              0,      -0.707107,       0.707107,
             0.5,            0.5,           -0.5,            0.5,
               0,       0.707107,      -0.707107,       0.707107,
            -0.5,            0.5,           -0.5,            0.5,
       -0.707107,              0,      -0.707107,       0.707107,
            -0.5,           -0.5,           -0.5,            0.5,
               0,      -0.707107,      -0.707107,       0.707107,
             0.5,           -0.5,           -0.5,            0.5,
        0.707107,              0,      -0.707107,       0.707107,
               1,              0,              0,              1,
        0.707107,       0.707107,              0,       0.707107,
               0,              1,              0,              1,
       -0.707107,       0.707107,              0,       0.707107,
              -1,              0,              0,              1,
       -0.707107,      -0.707107,              0,       0.707107,
               0,             -1,              0,              1,
        0.707107,      -0.707107,              0,       0.707107,
               1,              0,              0,              1,
        0.707107,              0,       0.707107,       0.707107,
             0.5,            0.5,            0.5,            0.5,
               0,       0.707107,       0.707107,       0.707107,
            -0.5,            0.5,            0.5,            0.5,
       -0.707107,              0,       0.707107,       0.707107,
            -0.5,           -0.5,            0.5,            0.5,
               0,      -0.707107,       0.707107,       0.707107,
             0.5,           -0.5,            0.5,            0.5,
        0.707107,              0,       0.707107,       0.707107,
               0,              0,              1,              1,
               0,              0,       0.707107,       0.707107,
               0,              0,              1,              1,
               0,              0,       0.707107,       0.707107,
               0,              0,              1,              1,
               0,              0,       0.707107,       0.707107,
               0,              0,              1,              1,
               0,              0,       0.707107,       0.707107,
               0,              0,              1,              1

	};

void nurbs_sphdraw(float *pos, float *eyePos)
{
  float sph[3];
  world_mat.multpoint3d(pos,sph);
  float r = pos[3] * world_scale;
  double ctlpoints2[S_NUMPOINTSX][S_NUMPOINTSY * (S_NUMCOORDS+1)];
  for (int i=0; i<S_NUMPOINTSX; i++) {
    for (int j=0; j < S_NUMPOINTSY; j++) {
      ctlpoints2[i][j*4+0] = ctlpoints[i][j*4+0];// * r + sph[0];
      ctlpoints2[i][j*4+1] = ctlpoints[i][j*4+1];// * r + sph[0];
      ctlpoints2[i][j*4+2] = ctlpoints[i][j*4+2];// * r + sph[0];
      ctlpoints2[i][j*4+3] = ctlpoints[i][j*4+3];
    }
  }
  static Object myobj = 0;
  if (myobj == 0) {
    myobj = genobj();
    makeobj(myobj);
    bgnsurface();
      nurbssurface( 
	       sizeof( surfknotsx) / sizeof( double ), surfknotsx, 
	       sizeof( surfknotsy) / sizeof( double ), surfknotsy, 
	       sizeof(double) * (S_NUMCOORDS + 1), 
	       sizeof(double) * S_NUMPOINTSX * (S_NUMCOORDS + 1), 
	       (double *)ctlpoints2, 
	       S_ORDER, S_ORDER, 
	       N_V3DR
	       );
    endsurface();
    closeobj();
  }
  callobj(myobj);
}
#endif

#ifdef USE_NURBS
#define sphdraw(x) nurbs_sphdraw(x, eyePos)
#endif
#ifdef USE_DISKS
#define sphdraw(x) fast_sphdraw(x, eyePos)
#endif

// destructor
GLRenderer::~GLRenderer(void) {

 // delete the current pop-up menu, if any
 menu_delete();

}

/////////////////////////  protected nonvirtual routines  

// change current line width
//  check detail level before coming here
void GLRenderer::set_line_width(int w) {
  if(w > 0) {
#ifdef ARCH_AIX3
    linewidth(w);
#else
    linewidthf((float)w);
#endif
    lineWidth = w;
  }
}

// change current line style ... styles defined in constructor
// the check for detail level should have been done already
void GLRenderer::set_line_style(int s) {
  lineStyle = s;
  if(s == ::SOLIDLINE)
    setlinestyle(1);
  else if(s == ::DASHEDLINE)
    setlinestyle(2);
  else {
    setlinestyle(1);
    lineStyle = ::SOLIDLINE;
  }
}

// change current sphere resolution
void GLRenderer::set_sphere_res(int r) {

  // the NPGL library current does not include the sphere library

#if !defined(__NPGL__) && !defined(ARCH_AIX3)
  if(r >= 1 && r <= SPH_MAXDEPTH && my_detail_level > DETAIL_WIREFRAME) {
    sphmode(SPH_DEPTH,r);
    sphereRes = r;
  }
#else
  sphereRes = r;
#endif
}

// change current sphere type
void GLRenderer::set_sphere_mode(int m) {
  sphereMode = m;

#if !defined(__NPGL__) && !defined(ARCH_AIX3)
  if (my_detail_level > DETAIL_WIREFRAME) {
    if(m == ::SOLIDSPHERE)
      sphmode(SPH_PRIM, SPH_MESH);
    else if(m == ::POINTSPHERE)
      sphmode(SPH_PRIM, SPH_POINT);
    else {
      sphmode(SPH_PRIM, SPH_MESH);
      sphereMode = ::SOLIDSPHERE;
    }
  }
#endif
}

// this routine draws a triangle between three points 
void GLRenderer::triangle( float *p1, float *p2, float *p3,
			   float *n1, float *n2, float *n3) {
  switch (my_detail_level) {
  case DETAIL_FULL:
  case DETAIL_FLAT:
    bgnpolygon();  // nice and simple
    n3f(n1);
    v3f(p1);
    n3f(n2);
    v3f(p2);
    n3f(n3);
    v3f(p3);
    endpolygon();
    break;
  case DETAIL_WIREFRAME:
    bgnline();
      v3f(p1);
      v3f(p2);
      v3f(p3);
      v3f(p1);
    endline();
    break;
  case DETAIL_POINTS:
    bgnpoint();
      v3f((float *)p1);
      v3f((float *)p2);
      v3f((float *)p3);
    endpoint();
    break;
  case DETAIL_NONE:
  default:
    break;
  }
}

void GLRenderer::square( float *perp, float *p1, float *p2, float *p3,
                              float *p4) {
  switch (my_detail_level) {
  case DETAIL_FULL:
  case DETAIL_FLAT:
    bgnpolygon();
      n3f(perp);
      v3f(p1);
      v3f(p2);
      v3f(p3);
      v3f(p4);
    endpolygon();
    break;
  case DETAIL_WIREFRAME:
    bgnline();
      v3f(p1);
      v3f(p2);
      v3f(p3);
      v3f(p4);
    endline();
    break;
  case DETAIL_POINTS:
    bgnpoint();
      v3f(p1);
      v3f(p2);
      v3f(p3);
      v3f(p4);
    endpoint();
    break;
  case DETAIL_NONE:
  default:
    break;
  }
}


// this routine draws a cylinder from end to start, using rod_res panels,
//      and of radius rod_radius
void GLRenderer::cylinder(float *start, float *end, int rod_res,
   float rod_radius, int filled) {
  register int i, h;
  register float hresinc, rodrad, length, theta, m, n;
  float axis[3], lenaxis[3], perp[3], perp2[3];
  float norms1[4], norms2[4], vert1[4], vert2[4], vert3[4], vert4[4];

  // need to do some preprocessing
  lenaxis[0] = start[0] - end[0];
  lenaxis[1] = start[1] - end[1];
  lenaxis[2] = start[2] - end[2];

  m = lenaxis[0]*lenaxis[0] + lenaxis[1]*lenaxis[1] + lenaxis[2]*lenaxis[2];
  if (m <= 0.0)
    return;
  length = sqrtf(m);
  axis[0] = lenaxis[0] / length;
  axis[1] = lenaxis[1] / length;
  axis[2] = lenaxis[2] / length;

  // dot product with unit axis vector
  if ((ABS(axis[0]) < ABS(axis[1])) &&
      (ABS(axis[0]) < ABS(axis[2])))
    i = 0;
  else if ((ABS(axis[1]) < ABS(axis[2])))
    i = 1;
  else
    i = 2;
  perp[i] = 0;
  perp[(i+1)%3] = axis[(i+2)%3];
  perp[(i+2)%3] = -axis[(i+1)%3];
  m = sqrtf(perp[0]*perp[0] + perp[1]*perp[1] + perp[2]*perp[2]);
  if(m > 0.0) {
    perp[0] /= m; 
    perp[1] /= m;
    perp[2] /= m;
  }

  // get cross product; perp2 will be normalized automatically
  perp2[0] = axis[1]*perp[2] - axis[2]*perp[1];
  perp2[1] = axis[2]*perp[0] - axis[0]*perp[2];
  perp2[2] = axis[0]*perp[1] - axis[1]*perp[0];

  // do the polygons
  rodrad = rod_radius;
  hresinc = TWOPI / ((float)rod_res);
  theta = 0.0;


    for (h=0; h < rod_res; h++) {
      if(h==0) {
	m = cos(theta);
	n = sin(theta);
	norms1[0] = m*perp[0] + n*perp2[0];
	norms1[1] = m*perp[1] + n*perp2[1];
	norms1[2] = m*perp[2] + n*perp2[2];
      } else {
	norms1[0] = norms2[0];
	norms1[1] = norms2[1];
	norms1[2] = norms2[2];
      }
      
      theta += hresinc;
      m = cos(theta);
      n = sin(theta);
      norms2[0] = m*perp[0] + n*perp2[0];
      norms2[1] = m*perp[1] + n*perp2[1];
      norms2[2] = m*perp[2] + n*perp2[2];
      
      if(h==0) {
	vert1[0] = end[0] + rodrad * norms1[0];
	vert1[1] = end[1] + rodrad * norms1[1];
	vert1[2] = end[2] + rodrad * norms1[2];
	vert2[0] = vert1[0] + lenaxis[0];
	vert2[1] = vert1[1] + lenaxis[1];
	vert2[2] = vert1[2] + lenaxis[2];
      } else {
	vert1[0] = vert4[0];
	vert1[1] = vert4[1];
	vert1[2] = vert4[2];
	vert2[0] = vert3[0];
	vert2[1] = vert3[1];
	vert2[2] = vert3[2];
      }
      
      vert3[0] = start[0] + rodrad * norms2[0];
      vert3[1] = start[1] + rodrad * norms2[1];
      vert3[2] = start[2] + rodrad * norms2[2];
      vert4[0] = vert3[0] - lenaxis[0];
      vert4[1] = vert3[1] - lenaxis[1];
      vert4[2] = vert3[2] - lenaxis[2];
      
      switch (my_detail_level) {
      case DETAIL_FULL:
      case DETAIL_FLAT:
	bgnpolygon();
          n3f(&norms1[0]);
          v3f(&vert1[0]);
          v3f(&vert2[0]);
      
          n3f(&norms2[0]);
          v3f(&vert3[0]);
          v3f(&vert4[0]);
        endpolygon();
	// UNTESTED CODE -- CHECK IT IF NOT USING FAST_CYLINDERS
	if (filled) { // fill end the ends
	  bgnpolygon();
	    n3f(axis);
	    v3f(&vert1[0]);
	    v3f(&vert3[0]);
	    v3f(start);
          endpolygon();
	  bgnpolygon();     // other side (note, don't need to reverse the
	  n3f(axis);        // axis direction since I v3f in the same dir.)
	    v3f(&vert2[0]);
	    v3f(&vert4[0]);
	    v3f(start);
	  endpolygon();
	}
	break;
      case DETAIL_WIREFRAME:  // doesn't duplicate some lines
	bgnline();
	  v3f(&vert2[0]);
	  v3f(&vert3[0]);
	  v3f(&vert4[0]);
          v3f(&vert1[0]);
	endline();
	break;
      case DETAIL_POINTS:
	bgnpoint();
          v3f(&vert1[0]);
          v3f(&vert2[0]);
	endpoint();
	break;
      case DETAIL_NONE:  // don't do anything
      default:
	break;
      } // end switch
    } // end of for loop
} // end of function

// this routine also draws a cylinder.  However, it assumes that
// the cylinder drawing command has precomputed the data.  This
// uses more memory, but is faster
// the data are: num == number of edges
//  edges = a normal, start, and end edge
#ifndef USE_SLOW_CYLINDERS
void GLRenderer::cylinder(int num, float *edges, int filled)
{
   if (num < 2)
     return;

   int n = num;
   float *start = edges;
   switch (my_detail_level) {
   case DETAIL_FULL:
   case DETAIL_FLAT:
#ifdef DO_NOT_USE_QUAD_STRIPS
     bgnpolygon();
     n3f(edges);
     v3f(edges+6);
     v3f(edges+3);
     while (--n>0) {
       edges += 9;
       n3f(edges);
       v3f(edges+3);
       v3f(edges+6);
       endpolygon();
       bgnpolygon();
       n3f(edges);
       v3f(edges+6);
       v3f(edges+3);
     }
     n3f(start);
     v3f(start+3);
     v3f(start+6);
     endpolygon();
#else
     bgnqstrip();
     while (n-- > 0) {
       n3f(edges);
       v3f(edges+6);
       v3f(edges+3);
       edges += 9;
     }
     n3f(start);  // loop back to the beginning
     v3f(start+6);
     v3f(start+3);
     endqstrip();
#endif
     // and fill in the top and bottom, if needed
     if (filled) {
       float axis[3];
       axis[0] = start[6] - start[3];
       axis[1] = start[7] - start[4];
       axis[2] = start[8] - start[5];
       normalize(axis);
       n = num;            // get one side
       edges = start + 3;
       bgnpolygon();
       n3f(axis);
       while (--n >= 0) { 
	 v3f(edges);
	 edges += 9;
       }
       endpolygon();

       n = num;          // and the other
       edges = start + 6;
       bgnpolygon();
       n3f(axis);       // I'm going the other direction, so 
       while (--n >= 0) { // I can use the same normal
       	 v3f(edges);
       	 edges += 9;
       }
       endpolygon();
     }
     break;
   case DETAIL_WIREFRAME:
     while (--n > 0) {
       bgnline();
       v3f(edges+   6);
       v3f(edges+9 +6);
       v3f(edges+9 +3);
       v3f(edges   +3);
       endline();
       edges += 9;
     }
     bgnline();
     v3f(edges+6);
     v3f(start+6);
     v3f(start+3);
     v3f(edges+3);
     endline();
     break;
   case DETAIL_POINTS:
     bgnpoint();
     while (--n >= 0) {
       v3f(edges+3);
       v3f(edges+6);
       float tmp[3];  // make a few more point to increase the visibility
       tmp[0] = (edges[3] + edges[6]) / 2.0;
       tmp[1] = (edges[4] + edges[7]) / 2.0;
       tmp[2] = (edges[5] + edges[8]) / 2.0;
       v3f(tmp);
       edges += 9;
     }
     endpoint();
     break;
   case DETAIL_NONE:
   default:
     break;
   }
       
}
#endif      
   
// this routine draws a cone with axis from end to start, using rod_res panels,
//      and with base of radius rod_radius
void GLRenderer::cone(float *start, float *end, int rod_res,
   float rod_radius) {
  register int i, h;
  register float hresinc, rodrad, length, theta, m, n;
  float axis[3], lenaxis[3], perp[3], perp2[3];
  float norms1[4], norms2[4], norms3[4], vert1[4], vert2[4];

  // need to do some preprocessing
  lenaxis[0] = start[0] - end[0];
  lenaxis[1] = start[1] - end[1];
  lenaxis[2] = start[2] - end[2];

  m = lenaxis[0]*lenaxis[0] + lenaxis[1]*lenaxis[1] + lenaxis[2]*lenaxis[2];
  if (m <= 0.0)
    return;
  length = sqrtf(m);
  axis[0] = lenaxis[0] / length;
  axis[1] = lenaxis[1] / length;
  axis[2] = lenaxis[2] / length;

  // dot product with unit axis vector
  if ((ABS(axis[0]) < ABS(axis[1])) &&
      (ABS(axis[0]) < ABS(axis[2])))
    i = 0;
  else if ((ABS(axis[1]) < ABS(axis[2])))
    i = 1;
  else
    i = 2;

  perp[i] = 0;
  perp[(i+1)%3] = axis[(i+2)%3];
  perp[(i+2)%3] = -axis[(i+1)%3];
  m = sqrtf(perp[0]*perp[0] + perp[1]*perp[1] + perp[2]*perp[2]);
  if(m > 0.0) {
    perp[0] /= m;
    perp[1] /= m;
    perp[2] /= m;
  }

  // get cross product; perp2 will be normalized automatically
  perp2[0] = axis[1]*perp[2] - axis[2]*perp[1];
  perp2[1] = axis[2]*perp[0] - axis[0]*perp[2];
  perp2[2] = axis[0]*perp[1] - axis[1]*perp[0];

  // do the polygons
  rodrad = rod_radius;
  hresinc = TWOPI / ((float)rod_res);
  theta = 0.0;

  // just draw one point at the vertex
  if (my_detail_level == DETAIL_POINTS) {
    bgnpoint();
      v3f(&start[0]);
      v3f(&end[0]);
    endpoint();
  }

  for (h=0; h < rod_res; h++) {
    if(h==0) {
      m = cos(theta);
      n = sin(theta);
      norms1[0] = m*perp[0] + n*perp2[0];
      norms1[1] = m*perp[1] + n*perp2[1];
      norms1[2] = m*perp[2] + n*perp2[2];
    } else {
      norms1[0] = norms2[0];
      norms1[1] = norms2[1];
      norms1[2] = norms2[2];
    }
    
    theta += hresinc;
    m = cos(theta);
    n = sin(theta);
    norms2[0] = m*perp[0] + n*perp2[0];
    norms2[1] = m*perp[1] + n*perp2[1];
    norms2[2] = m*perp[2] + n*perp2[2];
    
    norms3[0] = 0.5 * (norms1[0] + norms2[0]);
    norms3[1] = 0.5 * (norms1[1] + norms2[1]);
    norms3[2] = 0.5 * (norms1[2] + norms2[2]);
    
    if(h==0) {
      vert1[0] = start[0] + rodrad * norms1[0];
      vert1[1] = start[1] + rodrad * norms1[1];
      vert1[2] = start[2] + rodrad * norms1[2];
    } else {
      vert1[0] = vert2[0];
      vert1[1] = vert2[1];
      vert1[2] = vert2[2];
    }
    
    vert2[0] = start[0] + rodrad * norms2[0];
    vert2[1] = start[1] + rodrad * norms2[1];
    vert2[2] = start[2] + rodrad * norms2[2];
    
    switch (my_detail_level) {
    case DETAIL_FULL:
    case DETAIL_FLAT:
      // side of cone
      bgnpolygon();
        n3f(&norms3[0]);
	v3f(&end[0]);
    
	n3f(&norms2[0]);
	v3f(&vert2[0]);
    
	n3f(&norms1[0]);
	v3f(&vert1[0]);
      endpolygon();
    
      // bottom disk of cone
      bgnpolygon();
        n3f(&axis[0]);
	v3f(&vert1[0]);
	v3f(&vert2[0]);
	v3f(&start[0]);
      endpolygon();
      break;

    case DETAIL_WIREFRAME:
      bgnline();
	v3f(&vert2[0]);
	v3f(&vert1[0]);
      	v3f(&end[0]);
      endline();
      bgnline();
	v3f(&vert1[0]);
	v3f(&start[0]);
      endline();
      break;
    case DETAIL_POINTS:
      bgnpoint();
	v3f(&vert1[0]);
	float tmp[4];
	tmp[0] = (vert1[0] + end[0])/2.0; // make the edges more distinct
	tmp[1] = (vert1[1] + end[1])/2.0; // by adding a few more points
	tmp[2] = (vert1[2] + end[2])/2.0;
	v3f(tmp);
      endpoint();
      break;
    case DETAIL_NONE:
    default:
      break;
    }// end switch
  } // end for loop
}


/////////////////////////  protected virtual routines  

// define a new light source ... return success of operation
int GLRenderer::do_define_light(int n, float *color, float *position) {
  int i;
  
  for(i=0; i < 3; i++)  {
    lightdata[i+1] = color[i];
    lightdata[i+5] = position[i];
  }
  lmdef(DEFLIGHT, n+1, 10, lightdata);

  return TRUE;
}

// activate a given light source ... return success of operation
int GLRenderer::do_activate_light(int n, int turnon) {

  if(turnon)
    lmbind(LIGHT0 + n, n+1);
  else
    lmbind(LIGHT0 + n, 0);

  return TRUE;
}

// define a new material ... return success of operation
int GLRenderer::do_define_material(int n, float *data) {
  int i;
  
  matdata[13] = data[SHININESS_INDEX];
  matdata[15] = data[ALPHA_INDEX];
  for(i=0; i < 3; i++) {
    matdata[1 + i] = data[AMBIENT_INDEX + i];
    matdata[5 + i] = data[DIFFUSE_INDEX + i];
    matdata[9 + i] = data[SPECULAR_INDEX + i];
  }
  lmdef(DEFMATERIAL, n+1, 17, matdata);

  return TRUE;
}

// activate the use of material characteristics ... return success
int GLRenderer::do_activate_material(int n, int turnon) 
{
  if(turnon) {
    if(aa_enabled()) {
      need_aaon = TRUE;
      aa_off();
    }
    if(cueing_enabled()) {
      need_cueon = TRUE;
      cueing_off();
    }
    lmbind(MATERIAL, n+1);
    
  } else {
    
    if(need_aaon && aa_available())
      aa_on();
    if(need_cueon && cueing_available())
      cueing_on();
    need_aaon = need_cueon = FALSE;
    lmbind(MATERIAL, 0);
  }

  return TRUE;
}

/////////////////////////  public virtual routines  

// size of physical display object
void GLRenderer::screen_size_mm(long& x, long& y) {
#ifdef ARCH_AIX3
   x = 0; y = 0;
#else
  x = getgdesc(GD_XMMAX);  y = getgdesc(GD_YMMAX);
#endif
}
  
void GLRenderer::screen_size_pixel(long& x, long& y) {
#ifdef ARCH_AIX3
   x = 100; y=100; // fake it for now
#else
   x = getgdesc(GD_XPMAX);  y = getgdesc(GD_YPMAX);
#endif
}


//
// virtual routines to affect the device's transformation matrix
// These do not use the local transMat matrix, but just use the GL viewing
// matrix directly.
//

void GLRenderer::push(void) {
  ::pushmatrix();
}

void GLRenderer::pop(void) {
  ::popmatrix();
}

void GLRenderer::loadmatrix(Matrix4 &m) {
  Matrix *tmpmat = (Matrix *)(m.mat[0]);
  ::loadmatrix(*tmpmat);
}

void GLRenderer::multmatrix(Matrix4 &m) {
  Matrix *tmpmat = (Matrix *)(m.mat[0]);
  ::multmatrix(*tmpmat);
}

//
// virtual routines to return 2D screen coordinates, given 2D or 3D world
// coordinates.  These assume the proper GL window has focus, etc.
// The coordinates returned are absolute screen coords, relative to the lower
// left corner of the display monitor
//
void GLRenderer::abs_screen_loc_3D(float *loc, long *spos) {
  short x, y;
  cmov((Coord)loc[0], (Coord)loc[1], (Coord)loc[2]);
  getcpos(&x, &y);
  spos[0] = (long)x;
  spos[1] = (long)y;
}

void GLRenderer::abs_screen_loc_2D(float *loc, long *spos) {
  short x, y;
  cmov2((Coord)loc[0], (Coord)loc[1]);
  getcpos(&x, &y);
  spos[0] = (long)x;
  spos[1] = (long)y;
}


// Given a 3D point (pos A),
// and a 2D rel screen pos point (for pos B), computes the 3D point
// which goes with the second 2D point at pos B.  Result returned in B3D.
// NOTE: currently, this algorithm only supports the simple case where the
// eye look direction is along the Z-axis.  A more sophisticated version
// requires finding the plane to which the look direction is normal, which is
// assumed here to be the Z-axis (for simplicity in coding).
void GLRenderer::find_3D_from_2D(float *A3D, float *B2D, float *B3D) {
  Screencoord lvp, rvp, bvp, tvp;	// for current viewport
  Screencoord sx, sy;			// screen position
  Matrix mview, mproj;			// for current modelview and proj
  Object transobj = 1;			// object to use to reverse-map pos
  Coord w1[3], w2[3];			// points on line returned by rev map
  float lsx, lsy;			// used to convert rel screen -> abs

  // get current status of viewport and modelview/proj matricies
  getviewport(&lvp, &rvp, &bvp, &tvp);
  mmode(MPROJECTION);
  getmatrix(mproj);
  mmode(MVIEWING);
  getmatrix(mview);

  // create a display list object which sets these values
  makeobj(transobj);
    viewport(lvp, rvp, bvp, tvp);
    mmode(MPROJECTION);
    ::loadmatrix(mproj);
    mmode(MVIEWING);
    ::loadmatrix(mview);
  closeobj();
  
  // now use this object to map back from the 2D point to 2 3D points (a line)
  lsx = B2D[0];		lsy = B2D[1];
  abs_screen_pos(lsx, lsy);
  sx = (Screencoord)lsx - (Screencoord)xOrig;
  sy = (Screencoord)lsy - (Screencoord)yOrig;
  mapw(transobj, sx, sy, w1, w1+1, w1+2, w2, w2+1, w2+2);
  
  // finally, find the point where line returned as w1..w2 intersects the 
  // given 3D point's plane (this plane is assumed to be parallel to the X-Y
  // plane, i.e., with a normal along the Z-axis.  A more general algorithm
  // would need to find the plane which is normal to the eye look direction,
  // and which contains the given 3D point.)
  
  // special case: w1z = w2z ==> just return given 3D point, since there
  //		is either no solution, or the line is in the given plane
  if(w1[2] == w2[2]) {
    memcpy((void *)B3D, (void *)A3D, 3*sizeof(float));
  } else {
    float relchange = (A3D[2] - w1[2]) / (w2[2] - w1[2]);
    B3D[0] = (w2[0] - w1[0]) * relchange + w1[0];
    B3D[1] = (w2[1] - w1[1]) * relchange + w1[1];
    B3D[2] = A3D[2];
  }
}


//
// antialiasing and depth-cueing
//

// turn on antialiasing effect
void GLRenderer::aa_on(void) {

  if(!aaAvailable) {
    aaEnabled = FALSE;
  } else {
    blendfunction(BF_SA, BF_MSA);
    pntsmooth(SMP_ON);
    linesmooth(SML_ON);
//    polysmooth(PYSM_ON);
    aaEnabled = TRUE;
  }
}

// turn off antialiasing effect
void GLRenderer::aa_off(void) {

  if(aaAvailable) {
    pntsmooth(SMP_OFF);
    linesmooth(SML_OFF);
//    polysmooth(PYSM_OFF);
  }
  aaEnabled = FALSE;
}

// turn on hardware depth-cueing
void GLRenderer::cueing_on(void) {
  if(!cueingAvailable) {
    cueingEnabled = FALSE;
  } else {
    depthcue(TRUE);
    cueingEnabled = TRUE;
  }
}

// turn off hardware depth-cueing
void GLRenderer::cueing_off(void) {
  if(cueingAvailable) {
    depthcue(FALSE);
  }
  cueingEnabled = FALSE;
}


//
// virtual routines to create and activate a popup (possibly pull-down) menu 
//

// given a PopupMenu definition, create it in the window (but do not
// activate it yet).  Return success.
int GLRenderer::menu_create(PopupMenu *pm) {
  char mbuf[128];
  int needSep, itemAdded, totalAdded = 0;
  long addSubMenu;

  // start at the top level, and create items until a submenu is reached; at
  // that point create the submenu
  int n = pm->items();
  if(pm->menuExists)
    freepup(pm->menuID);

  // only create menu if there are items in it
  if(n > 0) {

    // create the new menu
    pm->menuID = newpup();
    pm->menuExists = TRUE;
    sprintf(mbuf,"%s%%t",pm->name());
    addtopup(pm->menuID, mbuf, 0);

    // add all the items
    for(int i=0; i < n; i++) {
      PopupMenuItem *mItem = pm->menu_item(i);

      // make sure this is not a separator
      if(! mItem->is_separator()) {
        needSep = (i < (n-1) && (pm->menu_item(i+1))->is_separator());
        itemAdded = TRUE;
        addSubMenu = 0;

        // check if this is a submenu item; if so, need to construct it first
        if(mItem->is_submenu()) {
          if(menu_create(mItem->sub_menu()))
            // submenu was successfully created; get ID from current mainMenu
            addSubMenu = mainMenu->menuID;
          else
            // submenu was not created for some reason; do not add this entry
            itemAdded = FALSE;
        }  

        // now construct the string describing the menu item
        if(itemAdded) {
          sprintf(mbuf,"%s%%x%2d", mItem->name(), mItem->return_code());
          if(addSubMenu != 0)
            strcat(mbuf,"%m");
          if(needSep)
            strcat(mbuf,"%l");
          addtopup(pm->menuID, mbuf, addSubMenu);

          // modify the item with special characteristics
          if(! mItem->showing())
            setpup(pm->menuID, totalAdded + 1, PUP_GREY);

	  // the GL on our AIX didn't handle these ...
#ifndef ARCH_AIX3
          if(mItem->checked())
            setpup(pm->menuID, totalAdded + 1, PUP_CHECK);
          else if(mItem->show_check())
            setpup(pm->menuID, totalAdded + 1, PUP_BOX);
#endif
	  
          // indicate we have one more item
          totalAdded++;
        }
      }
    }

    // set the current menu to the given one
    if(totalAdded > 0) {
      mainMenu = pm;
      return TRUE;
    } else {
      freepup(pm->menuID);
      pm->menuExists = FALSE;
      return FALSE;
    }

  } else {
    pm->menuExists = FALSE;
    mainMenu = NULL;
    return FALSE;
  }
}


// activate a previously-created menu.  If the windowing system has no
// current menu, or cannot do this operation, returns (-1); otherwise,
// returns the return code for the select (-1 if none selected).
int GLRenderer::menu_activate(void) {

  if(mainMenu && mainMenu->menuExists) {
    // do the menu, get the result
    return (int)(dopup(mainMenu->menuID));
  }

  // if here, could not do the menu selection
  return (-1);
}


// delete the given menu from the display.
// If no argument is given, deletes the current menu (if any).
// Returns success.
int GLRenderer::menu_delete(PopupMenu *pm) {

  // check which menu to delete
  if(!pm || pm == mainMenu) {
    if(!mainMenu)
      return FALSE;
    pm = mainMenu;
    mainMenu = NULL;
  }

  // delete this menu
  if(pm->menuExists) {
    freepup(pm->menuID);
    pm->menuExists = FALSE;
  }

  // go through the menu and delete all submenus first
  int n = pm->items();
  for(int i=0; i < n; i++) {
    PopupMenuItem *mItem = pm->menu_item(i);
    if(mItem->is_submenu())
      menu_delete(mItem->sub_menu());
  }

  // done deleting menu
  return TRUE;
}



//
//*******************  the rendering routine  *********************
//
// This scans the given command list until the end, doing the commands
// in the order they appear
//
void GLRenderer::render(void *cmdList) {
  char *cmdptr = (char *)cmdList;
  int tok, cmdsize;

#ifndef THIS_IS_TEMPORARY
  {
    float texprops[] = {TX_MINFILTER, TX_POINT, TX_MAGFILTER,
			TX_POINT, TX_WRAP, TX_REPEAT, TX_NULL};
    float tevprops[] = {TV_COLOR, 0.75, 0.13, 0.06, 1., TV_BLEND, TV_NULL};
    unsigned long bricks [] = {
      0x00ffffff, 0xffffffff,
      0x00ffffff, 0xffffffff,
      0x00ffffff, 0xffffffff,
      0x00ffffff, 0xffffffff,
      0x00000000, 0x00000000,
      0xffffffff, 0x00ffffff,
      0xffffffff, 0x00ffffff,
      0xffffffff, 0x00ffffff,
      0x00000000, 0x00000000};
    texdef2d(1,1,8,8,bricks,0,texprops);
    tevdef(1,0,tevprops);
    tevbind(TV_ENV0, 1);
  }
#endif


  MSGDEBUG(3, "GLRenderer: rendering command list." << sendmsg);
  
  if(!cmdList)
    return;
  
  
  switch(my_detail_level) {
  case DETAIL_FULL: // full detail and material characteristics
    break;
  case DETAIL_FLAT: // full detail, no material characteristics
    deactivate_materials();
    break;
  case DETAIL_WIREFRAME: // set up the lines for wireframe
    set_line_style(SOLIDLINE);
    set_line_width(2);
    break;
  case DETAIL_POINTS: // points
    set_line_width(2);
    break;
  case DETAIL_NONE: // nothing
  default:
    // Then I'm outta here!!!!
    return;
  }

  // save transformation matrix
  push();

  // scan through the list, getting each command and executing it, until
  // the end of commands token is found
  dataBlock = NULL;
#if defined(USE_DISKS) || defined(USE_NURBS)
  int last_tok = -1;
#endif
  while((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
    if (tok == DLINKLIST) {
       cmdptr += sizeof(int);       // step forward one (no length here)
       cmdptr = *((char **) cmdptr);// and go to the next link
       tok = ((int *)cmdptr)[0];    // this is guarenteed to be neither
    }                               // a DLINKLIST nor DLASTCOMMAND
       
    cmdsize = ((int *)cmdptr)[1];
    cmdptr += 2*sizeof(int);

    MSGDEBUG(4, "GLRenderer: doing command " << tok);
    MSGDEBUG(4, " of size " << cmdsize << sendmsg);

#if defined(USE_DISKS) || defined(USE_NURBS)
    // for fast sphere, see if I need to return the matrix to its
    // correct state
    if (last_tok == DSPHERE) {
      switch (tok) {
      case DSPHERE:
      case DSPHERE_I:
      case DCOLORINDEX:
      case DCOLORRGB:
      case DCOLORDEF:
      case DPICKPOINT:
      case DPICKPOINT_I:
      case DPICKLINE: 
      case DPICKLINE_I:
      case DPICKBOX:
      case DPICKBOX_I:
      case DSPHERERES:
      case DSPHERETYPE:
	break;
      default:
	// return to the correct state
	::popmatrix();
	last_tok = tok;
	break;
      }
    } else switch (tok) {
      case DSPHERE:
      case DSPHERE_I:
	// go into the special state
	Matrix world;
	::getmatrix(world);
	world_scale = sqrtf(world[0][0]*world[0][0]+
			    world[1][0]*world[1][0]+
			    world[2][0]*world[2][0]);
	// world_mat is a static variable
	memcpy(world_mat.mat, world, 16 * sizeof(float));
	::pushmatrix();
	Matrix ident = {{1,0,0,0},{0,1,0,0},{0,0,1,0},{0,0,0,1}};
	::loadmatrix(ident);
	last_tok = DSPHERE;
	break;
    default:
      last_tok = tok;
      break;
    }
#endif
    if(tok == DPOINT) {
      // plot a point at the given position
      // (every render level can draw a point, except NONE, but
      //  NONE never gets here)
      bgnpoint();
      v3f((float *)cmdptr);
      endpoint();

    } else if(tok == DPOINT_I) {
      // plot a point at the given position, using indices into dataBlock
      bgnpoint();
      v3f(dataBlock + ((int *)cmdptr)[0]);
      endpoint();

    } else if(tok == DLINE) {
      // plot a line
      if (my_detail_level > DETAIL_POINTS) {
	bgnline();
	  v3f((float *)cmdptr);
	  v3f(((float *)cmdptr) + 3);
	endline();
      } else {
	bgnpoint();
	  float *ptr = (float *) cmdptr;
	  v3f(ptr);
	  v3f(ptr+3);
	  float tmp[3];
	  tmp[0] = (ptr[0] + ptr[3]) / 2.0; // make line more distinct
	  tmp[1] = (ptr[1] + ptr[4]) / 2.0;
	  tmp[2] = (ptr[2] + ptr[5]) / 2.0;
	  v3f(tmp);
	endpoint();
      }
    } else if(tok == DLINE_I) {
      // plot a line, using indices into dataBlock
      float *ptr1 = dataBlock + ((int *) cmdptr)[0];
      float *ptr2 = dataBlock + ((int *) cmdptr)[1];
      if (my_detail_level > DETAIL_POINTS) {
	bgnline();
	v3f(ptr1);
	v3f(ptr2);
	endline();
      } else {
	bgnpoint();
	  v3f(ptr1);
	  v3f(ptr2);
	  float tmp[3];
	  tmp[0] = (ptr1[0] + ptr2[0]) / 2.0; // make line more distinct
	  tmp[1] = (ptr1[1] + ptr2[1]) / 2.0;
	  tmp[2] = (ptr1[2] + ptr2[2]) / 2.0;
	v3f(tmp);
      }
      
    }  else if(tok == DSPHERE) {
#if !defined( __NPGL__) && !defined(ARCH_AIX3)
      // draw a sphere of given radius
      if (my_detail_level > DETAIL_POINTS) {
	sphdraw((float *)cmdptr);
      } else {
	bgnpoint();
	  v3f((float *)cmdptr);
	endpoint();
      }

#else
      // plot a point at the given position
      bgnpoint();
	v3f((float *)cmdptr);
      endpoint();
#endif

    }  else if(tok == DSPHERE_I) {
      int n1 = (int)(((float *)cmdptr)[0]);
#if !defined( __NPGL__) && !defined(ARCH_AIX3)
      // draw a sphere of given radius, using indices into dataBlock
      if (my_detail_level > DETAIL_WIREFRAME) {
	float spheredata[4];
	spheredata[0] = dataBlock[n1++];
	spheredata[1] = dataBlock[n1++];
	spheredata[2] = dataBlock[n1];
	spheredata[3] = ((float *)cmdptr)[1];
	sphdraw(spheredata);
      } else {
	bgnpoint();
	v3f(dataBlock + n1);
	endpoint();
      }

#else
      // plot a point at the given position
      bgnpoint();
        v3f(dataBlock + n1);
      endpoint();
      }
#endif
     
    }  else if(tok == DTRIANGLE) {
      // draw a triangle between the three points
       triangle(  (float *) cmdptr      , ((float *) cmdptr) + 3,
		 ((float *) cmdptr) +  6, ((float *) cmdptr) + 9,
		 ((float *) cmdptr) + 12, ((float *) cmdptr) + 15
		  );

    }  else if(tok == DTRIANGLE_I) {
      // draw a triangle between the three points, using indices into dataBlock
       triangle( dataBlock + ((int *) cmdptr)[1],
		 dataBlock + ((int *) cmdptr)[2],
		 dataBlock + ((int *) cmdptr)[3],
		 dataBlock + ((int *) cmdptr)[0],
		 dataBlock + ((int *) cmdptr)[0],
		 dataBlock + ((int *) cmdptr)[0] 
		 );
    }  else if(tok == DSQUARE) {
      // draw a square, given the four points
      square( (float *) cmdptr     , ((float *) cmdptr) + 3, 
             ((float *) cmdptr) + 6, ((float *) cmdptr) + 9,
             ((float *) cmdptr) +12 );

    }  else if(tok == DSQUARE_I) {
      // draw a square, given the four points, using indices into dataBlock
      square(dataBlock + ((int *)cmdptr)[0], dataBlock + ((int *)cmdptr)[1],
             dataBlock + ((int *)cmdptr)[2], dataBlock + ((int *)cmdptr)[3],
             dataBlock + ((int *)cmdptr)[4]);

    }  else if(tok == DCYLINDER) {
      // draw a cylinder of given radius and resolution
      // The #define is a tradeoff between speed and memory usage
//#define USE_SLOW_CYLINDERS
#ifdef USE_SLOW_CYLINDERS
      cylinder((float *)cmdptr, ((float *)cmdptr) + 3,
      		(int)(((float *)cmdptr)[7]), ((float *)cmdptr)[6], 
	       int(((float *)cmdptr)[8]) != 0);
#else
      cylinder((int)(((float *)cmdptr)[7]), ((float *)cmdptr) + 9,
	       int(((float *) cmdptr)[8]) != 0);
#endif
    }  else if(tok == DCYLINDER_I) {
      // draw a cylinder of given radius and resolution, using indices
      cylinder(dataBlock +  (int)(((float *)cmdptr)[0]),
      		dataBlock +  (int)(((float *)cmdptr)[1]),
		(int)(((float *)cmdptr)[3]), 
		((float *)cmdptr)[2],
	       int(((float *) cmdptr)[8]) != 0);

    } else if(tok == DCONE) {
      // draw a cone of given radius and resolution
      cone((float *)cmdptr, ((float *)cmdptr) + 3,
      		(int)(((float *)cmdptr)[7]), ((float *)cmdptr)[6]);

    }  else if(tok == DCONE_I) {
      // draw a cone of given radius and resolution, using indices
      cone(dataBlock +  (int)(((float *)cmdptr)[0]),
      		dataBlock +  (int)(((float *)cmdptr)[1]),
		(int)(((float *)cmdptr)[3]), 
		((float *)cmdptr)[2] );

    } else if(tok == DTEXTPOS) {
      // move the current character position to the point given
      cmov(((float *)cmdptr)[0], ((float *)cmdptr)[1], ((float *)cmdptr)[2]);
      
    } else if(tok == DTEXTPOS_I) {
      // move the current character position to the point given, using indices
      int n1 = ((int *)cmdptr)[0];
      cmov((Coord)dataBlock[n1], (Coord)dataBlock[n1+1],
           (Coord)dataBlock[n1+2]);
      
    } else if(tok == DTEXT) {
      // display text at current position
      charstr(cmdptr);

    } else if(tok == DCOLORINDEX) {
      // set the current color to the given color index ... assumes the
      // color has already been defined
      int n = ((int *)cmdptr)[0];
      
      if(!matDefined[n]) {
        msgErr << "GLRenderer: Error, color " << n << " not defined."
	         << sendmsg;
      } else {
        // set depth cue range if necessary
#ifndef ARCH_AIX3
        if(cueing_enabled()) {
          lRGBrange((short)(255 * backColor[0]),
	 	    (short)(255 * backColor[1]),
      		    (short)(255 * backColor[2]),
		    (short)(255 * matData[n][COLOR_INDEX + 0]),
		    (short)(255 * matData[n][COLOR_INDEX + 1]),
		    (short)(255 * matData[n][COLOR_INDEX + 2]),
		    getgdesc(GD_ZMIN), getgdesc(GD_ZMAX));
	}
#endif
	c3f(&(matData[n][COLOR_INDEX]));

        // bind the material if materials enabled
	materialOn = n;
	if(materialsActive)
	  activate_material();
      }
      
    } else if(tok == DCOLORRGB) {
      // just set the non-material color to the given value.  Turns off
      // materials.
      deactivate_materials();

      // set depth cue range if necessary
#ifndef ARCH_AIX3
      if(cueing_enabled()) {
        lRGBrange((short)(255 * backColor[0]),
	 	  (short)(255 * backColor[1]),
      		  (short)(255 * backColor[2]),
		  (short)(255 * ((float *)cmdptr)[0]),
		  (short)(255 * ((float *)cmdptr)[1]),
		  (short)(255 * ((float *)cmdptr)[2]),
		  getgdesc(GD_ZMIN), getgdesc(GD_ZMAX));
      }
#endif
      c3f((float *)cmdptr);
      
    } else if(tok == DPICKPOINT || tok == DPICKPOINT_I
		|| tok == DPICKLINE || tok == DPICKLINE_I
		|| tok == DPICKBOX || tok == DPICKBOX_I) {
      ; 		// do nothing, these are picking tokens

    } else if(tok == DLIGHTONOFF) {
      // turn on or off a light source
      activate_light(((int *)cmdptr)[0], ((int *)cmdptr)[1]);
      
    } else if(tok == DMATERIALS) {
      // enable/disable material characteristics
      if (my_detail_level > DETAIL_FLAT) {
	if(((int *)cmdptr)[0])
	  activate_material();
	else
	  deactivate_materials();
      }
      
    } else if(tok == DSPHERERES) {
      // set the current sphere resolution
      set_sphere_res(((int *)cmdptr)[0]);
      
    } else if(tok == DSPHERETYPE) {
      // set the current sphere type
      set_sphere_mode(((int *)cmdptr)[0]);
      
    } else if(tok == DLINESTYLE) {
      // set the current line style
      if (my_detail_level > DETAIL_WIREFRAME) 
	set_line_style(((int *)cmdptr)[0]);
      
    } else if(tok == DLINEWIDTH) {
      // set the current line width
      if (my_detail_level > DETAIL_WIREFRAME) 
	set_line_width(((int *)cmdptr)[0]);

    } else if(tok == DPUSH) {
      // push the current trans matrix
      push();

    } else if(tok == DPOP) {
      // pop the current trans matrix
      pop();
      
    } else if(tok == DLOAD) {
      // load the given trans matrix onto the stack
      Matrix *tmpmat = (Matrix *)((float *)cmdptr);
      ::loadmatrix(*tmpmat);

    } else if(tok == DMULT) {
      // multiply the given trans matrix
      Matrix *tmpmat = (Matrix *)((float *)cmdptr);
      ::multmatrix(*tmpmat);

    } else if(tok == DTRANS) {
      // do a translation
      ::translate(((float *)cmdptr)[0], ((float *)cmdptr)[1],
      		((float *)cmdptr)[2]);
      
    } else if(tok == DSCALE) {
      // do a scale
      ::scale(((float *)cmdptr)[0], ((float *)cmdptr)[1],
      		((float *)cmdptr)[2]);

    } else if(tok == DROT) {
      // do a rotation about a given axis
      ::rot(((float *)cmdptr)[0], 'x' + (int)(((float *)cmdptr)[1]));

    } else if(tok == DCOLORDEF) {
      // define a new color
      int n = (int)(((float *)cmdptr)[0]);
      define_material(n, ((float *)cmdptr) + 1);
      
    } else if(tok == DLIGHTDEF) {
      // define a new light source
      int n = (int)(((float *)cmdptr)[0]);
      define_light(n, ((float *)cmdptr) + 1, ((float *)cmdptr) + 4);
      
    } else if(tok == DCLEAR) {
      // clear the display
      clear();

    } else if(tok == DMOREDATA || tok == DNEWDATA) {
      // set the current drawing data block
#ifdef VMDCAVE
      dataBlock = (float *)cmdptr;	// point to current position in list
#else
      dataBlock = ((float **)cmdptr)[0];	// point to given memory loc
#endif

    } else {
      // command not found, help!
      msgErr << "GLRenderer: Unknown drawing token " << tok
             << " encountered ... Skipping this command." << sendmsg;
    }
    
    // update position in array
    cmdptr += cmdsize;
  }  
#if defined(USE_DISKS) || defined(USE_NURBS)
  // if the last with a sphere, need to undo the special matrix
  if (last_tok == DSPHERE || last_tok == DSPHERE_I) {
    ::popmatrix();
  }
#endif
  // restore transformation matrix
  pop();
}

/***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: GLRenderer.C,v $
 * Revision 1.23  1996/05/14  04:50:11  dalke
 * added quadstrip (for a 33% speedup on cylinders)
 *
 * Revision 1.22  1996/02/05  20:51:30  dalke
 * added cylinders with ends
 * testing various fast sphere codes
 *
 * Revision 1.21  1995/09/23  18:40:34  dalke
 * changed linewidth for wireframe
 *
 * Revision 1.20  1995/09/22  15:58:45  dalke
 * added multiple rendering levels for high resolution when fixed and
 * lower one when rotating
 *
 * Revision 1.19  1995/08/15  21:33:20  dalke
 * added support for AIX3
 *
 * Revision 1.18  1995/06/07  02:28:23  billh
 * Moved cursor and button virtual functions to GLDisplayDevice
 * (from GLRenderer).
 *
 * Revision 1.17  1995/05/11  22:49:04  billh
 * Added virtual functions to queue and check events, and to get/modify
 * the cursor position, shape, etc.  All graphics/windowing-specific
 * functions have now been moved to the DisplayDevice.
 *
 * Revision 1.16  95/04/12  20:36:26  billh
 * Added two-sided lighting; made sure polygons drawn counter-clockwise.
 * 
 * Revision 1.15  1995/04/11  09:46:51  dalke
 * new formate for the DTRIANGLE that allows different normals
 * for the triangle verticies.
 *
 * Revision 1.14  1995/04/05  11:27:08  billh
 * Fixed bug with mapping 2D -> 3D coordinates, for special case w1z==w2z
 *
 * Revision 1.13  95/03/24  18:49:36  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.12  1995/03/10  07:49:33  dalke
 * added fast cylinders by precomputing in the DispCmd
 * This makes things about 2.5 times faster for pure cylinder
 * drawing.
 *
 * Revision 1.11  1995/02/22  04:07:05  billh
 * Added virtual routine find_3D_from_2D, which takes a 3D point at pos A,
 * and 2D rel screen position at B, and returns the 3D point corresponding
 * to the 2D point at B in the same plane as point A.
 *
 * Revision 1.10  1995/02/12  07:15:50  dalke
 * now reads the new "DLINKLIST" and continues rendering with
 * the new memory
 *
 * Revision 1.9  1995/02/03  23:18:53  billh
 * Improved cylinder drawing; normals now calculated for each vertex.
 *
 * Revision 1.8  1994/11/03  01:35:56  billh
 * Fixed problem with drawing indexed spheres with NPGL library.
 *
 * Revision 1.7  1994/10/28  18:31:10  billh
 * Added definitions for line styles; disabled near and far clipping for
 * HP's using NPGL library.
 *
 * Revision 1.6  1994/10/05  05:33:01  billh
 * arguments changed to allow to compile on HP's.
 *
 * Revision 1.5  1994/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.4  1994/10/04  22:22:17  billh
 * fixed syntax error with include
 *
 * Revision 1.3  1994/10/04  22:07:55  billh
 * Took out sphere library usage if __NPGL__ is defined, since the npgl lib
 * does not have a version of the sphere drawing routines.  Spheres are
 * just drawn as points for npgl rendering.
 *
 * Revision 1.2  94/09/28  22:06:52  billh
 * changed fsqrt to sqrtf
 * 
 * Revision 1.1  1994/09/23  06:01:39  billh
 * Initial revision
 *
 ***************************************************************************/
