/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: GLDisplayDevice.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.24 $	$Date: 1996/05/14 20:04:19 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that are GL-specific.  Will render drawing
 * commands into a single GL window.
 *
 ***************************************************************************/

#include <stdlib.h>
#include <math.h>
#include "GLDisplayDevice.h"
#include "Inform.h"
#include "utilities.h"

#ifdef VMDFORMS
// GL's "Cursor" conflicts with X's Cursor
#define Cursor XCursor
#include <forms.h>
#undef Cursor
// xforms has a button_down which conflicts with stuff we do
#ifdef button_down
#undef button_down
#endif
#endif

#ifndef YMAXSTEREO
#define YMAXSTEREO      491
#define YOFFSET         532
#endif

#ifdef ARCH_AIX3
// for some #$^%#$^& reason, this wasn't in gl.h, so
// we have to fake it
extern "C" Int32 getgdesc(Int32);
#endif

// static data for this object
static char *glStereoNameStr[GL_STEREO_MODES] = { "Off",
	"CrystalEyes", "InverseCrystal", "CrossEyes", "SideBySide",
	"Left", "Right" };
	
// light model data
static float lightModelData[] = { TWOSIDE, 1.0, LMNULL };

/////////////////////////  constructor and destructor  

// constructor ... open a window and set initial default values
GLDisplayDevice::GLDisplayDevice(int *size, int *loc)
	: GLRenderer("GL Display") {

  MSGDEBUG(1, "Creating GLDisplayDevice ... " << sendmsg);

  // set up data possible before opening window
  stereoNames = glStereoNameStr;
  stereoModes = GL_STEREO_MODES;
  
  // open the window
  windowID = open_window(name, size, loc);

  // set flags for the capabilities of this display
  has2D = TRUE;
#ifndef ARCH_AIX3
  aaAvailable = (getgdesc(GD_PNTSMOOTH_RGB) && getgdesc(GD_LINESMOOTH_RGB));
#else
  // huh????
  aaAvailable = (getgdesc(GD_POINTSMOOTH_RGB) && getgdesc(GD_LINESMOOTH_RGB));
#endif
  cueingAvailable = TRUE;

  // turn on antialiasing, turn off depth-cueing
  aa_on();
  cueing_off();

  // reshape and clear the display, which initializes some other variables
  reshape();
  normal();
  clear();
  update();
}

// destructor ... close the window
GLDisplayDevice::~GLDisplayDevice(void) {
  // go out of stereo mode
  set_stereo_mode(GL_STEREO_OFF);
  winclose(windowID);
}
  
/////////////////////////  protected nonvirtual routines  

// create a new window and set it's characteristics
long GLDisplayDevice::open_window(char *nm, int *size, int *loc) {
  long retval;
  int i;
  
  MSGDEBUG(2,"GLDisplayDevice: opening window ..." << sendmsg);

  // keep this program in the foreground
#ifndef ARCH_AIX3
  foreground();
#endif
  
  // open the graphics window
  if(loc && size)
    prefposition(loc[0], loc[0] + size[0] + 1, loc[1], loc[1] + size[1] + 1);
  else if(size)
    prefsize(size[0], size[1]);
  retval = winopen(nm);
  winconstraints();
  
  // set up graphics parameters
  doublebuffer();
#ifdef GD_STEREO_IN_WINDOW
  if(getgdesc(GD_STEREO_IN_WINDOW) == 1)
      stereobuffer();
#endif
  RGBmode();
  gconfig();
  zbuffer(1);
#ifndef ARCH_AIX3
  glcompat(GLC_ZRANGEMAP, 0);
#endif
  lsetdepth(getgdesc(GD_ZMIN),getgdesc(GD_ZMAX));
  backface(FALSE);
  shademodel(GOURAUD);
  mmode(MVIEWING);
  subpixel(TRUE);

  // define graphics state based on already-set default values
#if !defined(__NPGL__) && !defined(ARCH_AIX3)
  sphmode(SPH_HEMI,TRUE);
  sphmode(SPH_ORIENT,TRUE);
  sphmode(SPH_TESS,SPH_OCT);
#endif
  deflinestyle(1, 0xFFFF);
  deflinestyle(2, 0x3333);
  set_sphere_mode(sphereMode);
  set_sphere_res(sphereRes);
  set_line_width(lineWidth);
  set_line_style(lineStyle);

  // disable clipping, if using NPGL on HP's
#ifdef __NPGL__
#ifdef ARCH_HPUX9
  npglcompat(NPGLC_NEAR_CLIP, NPGLC_CLIP_OFF);
  npglcompat(NPGLC_FAR_CLIP, NPGLC_CLIP_OFF);
#endif
#endif

  MSGDEBUG(2,"GLDisplayDevice: defining lights ..." << sendmsg);

  // turn on lights if necessary
  for(i=0; i < DISP_LIGHTS; i++) {
    if(lightDefined[i]) {
      do_define_light(i, lightColor[i], lightPos[i]);
      do_activate_light(i, lightOn[i]);
    } else
      do_activate_light(i, FALSE);
  }

  // define the light model ... this never changes.
  // material display is turned on/off by binding MATERIALS.
  lmdef(DEFLMODEL,1,sizeof(lightModelData)/sizeof(float) - 1,lightModelData);
  lmbind(LMODEL, 1);

  MSGDEBUG(2,"GLDisplayDevice: defining materials ..." << sendmsg);

  // define materials if necessary
  for(i=0; i < MAXCOLORS; i++) {
    if(matDefined[i])
      do_define_material(i, matData[i]);
  }
  do_activate_material(materialOn, materialsActive);
  
  // load current transformation matrix on stack
  ::loadmatrix((transMat.top()).mat);
  
  return retval;
}


// set the current perspective, based on the eye position and where we
// are looking.  This form is for a non-stereo perspective
void GLDisplayDevice::set_persp(DisplayEye my_eye) {

  if(dim() == 3) {	// use window perspective for 3D view
    // set projection
    mmode(MPROJECTION);
    // get the right call if projection is perspective or orthographic
    if (projection() == PERSPECTIVE) {
      window((Coord)cpLeft, (Coord)cpRight, (Coord)cpDown, (Coord)cpUp,
	     (Coord)nearClip, (Coord)farClip);
    } else {
      // It should be 0.5, but the .25 makes things more like the
      // standard perspective
      ortho( -0.25 * vSize * Aspect, 0.25 * vSize * Aspect,
	     -0.25 * vSize, 0.25 * vSize, 
	     nearClip, farClip);
    }
	   
    // define eye and look at some point.  Assumes up vector = (0,1,0)
    if(my_eye == NOSTEREO) {
      lookat(eyePos[0], eyePos[1], eyePos[2],
      	eyePos[0] + eyeDir[0], eyePos[1] + eyeDir[1], eyePos[2] + eyeDir[2],
	0);
    } else if(my_eye == LEFTEYE) {
      lookat(eyePos[0] - eyeSepDir[0], eyePos[1] - eyeSepDir[1],
      	eyePos[2] - eyeSepDir[2],
	eyePos[0] + eyeDir[0], eyePos[1] + eyeDir[1], eyePos[2] + eyeDir[2],
	0);
    } else {	// right eye
      lookat(eyePos[0] + eyeSepDir[0], eyePos[1] + eyeSepDir[1],
      	eyePos[2] + eyeSepDir[2],
	eyePos[0] + eyeDir[0], eyePos[1] + eyeDir[1], eyePos[2] + eyeDir[2],
	0);
    }
    
    // return matrix mode back to setting the view matrix
    mmode(MVIEWING);
  } else {                     // use (0 .. 1, 0 ..1) window for 2D view
    ortho2(0.0, 1.0, 0.0, 1.0);
  }
}

/////////////////////////  public virtual routines  

//
// get the current state of the device's pointer (i.e. cursor if it has one)
//

// abs pos of cursor from lower-left corner
int GLDisplayDevice::x(void) {
  return (int)getvaluator(MOUSEX);
}


// same, for y direction
int GLDisplayDevice::y(void) {
  return (int)getvaluator(MOUSEY);
}


// whether a button is currently pressed
int GLDisplayDevice::button_down(int b) {
  int retval = FALSE;
  if(b == B_LEFT || b == B2_LEFT)
    retval = (int)getbutton(LEFTMOUSE);
  else if(b == B_MIDDLE || b == B2_MIDDLE)
    retval = (int)getbutton(MIDDLEMOUSE);
  else if(b == B_RIGHT || b == B2_RIGHT)
    retval = (int)getbutton(RIGHTMOUSE);
  else if(b >= B_F1 && b <= B_F12)
    retval = (int)getbutton(F1KEY + (b - B_F1));
  else if(b == B_ESC)
    retval = (int)getbutton(ESCKEY);
  return retval;
}

// which of the "shift" keys are pressed;
//  SHIFT   == left or right shift key
//  CONTROL == left or right control key
//  ALT     == left or right alternate key
int GLDisplayDevice::shift_state(void) {
  int ret_val = 0;
  if (getbutton(LEFTSHIFTKEY) || getbutton(RIGHTSHIFTKEY)) {
    ret_val |= DisplayDevice::SHIFT;
  }
  if (getbutton(LEFTALTKEY) || getbutton(RIGHTALTKEY)) {
    ret_val |= DisplayDevice::ALT;
  }
  if (getbutton(LEFTCTRLKEY) || getbutton(RIGHTCTRLKEY)) {
    ret_val |= DisplayDevice::CONTROL;
  }
  return ret_val;
}

// allow the user to define a special shape for the cursor ... the data
// is assumed to consist of a single-dim array of 32 unsigned shorts; the
// first 16 define a 16x16-bit (row-major) pattern, the last 16 are a
// "second layer" which may be drawn in a different color
// args: which shape (>0, with 0 the "default" case which cannot be changed)
// the bitmap data, and the "hot spot" from the lower-left corner
void GLDisplayDevice::change_cursor_shape(int n, unsigned short *c,
						int x, int y) {
  if(n > 0) {
#if defined(__NPGL__) || defined(ARCH_AIX3)
    curstype(C16X1);
#else
    curstype(C16X2);
#endif
    defcursor(n, c);
    curorigin(n, x, y);
  }
}


// set the Nth cursor shape as the current one.  If no arg given, the
// default shape (n=0) is used.
void GLDisplayDevice::set_cursor(int n) {
   setcursor(n, 0, 0);  
   get_cursor();  // I need to do this or the cursor won't change (!??!)
}

// get the current cursor shape
int GLDisplayDevice::get_cursor(void) {
  short index;
  Colorindex color, wtm;
  Boolean b;
  getcursor(&index, &color, &wtm, &b);
  return index;
}
//
// event handling routines
//

// queue the standard events (need only be called once ... but this is
// not done automatically by the window because it may not be necessary or
// even wanted)
void GLDisplayDevice::queue_events(void) {

  // queueing method is different whether we are using the FORMS library or
  // just GL
#undef VMDFORMS
#ifdef VMDFORMS
//  msgInfo << "Queueing FORMS events" << sendmsg;
  fl_qdevice(REDRAW);
  fl_qdevice(INPUTCHANGE);
  fl_qdevice(KEYBD);
  fl_qdevice(RIGHTMOUSE);
  fl_qdevice(MIDDLEMOUSE);
  fl_qdevice(LEFTMOUSE);
//  fl_qdevice(MOUSEX);
//  fl_qdevice(MOUSEY);
#else
//  msgInfo << "Queueing GL events" << sendmsg;
  qdevice(REDRAW);
  qdevice(INPUTCHANGE);
  qdevice(KEYBD);
  qdevice(RIGHTMOUSE);
  qdevice(MIDDLEMOUSE);
  qdevice(LEFTMOUSE);
  // I have no idea why this is needed
#ifdef ARCH_AIX3
  unqdevice(MODECHANGE);
  qreset();
#endif
//  qdevice(MOUSEX);
//  qdevice(MOUSEY);
#endif
}


// test if there is an event ready
int GLDisplayDevice::test_events(void) {
  long retval;

  // queueing method is different whether we are using the FORMS library or
  // just GL
#ifdef VMDFORMS
  retval = fl_qtest();
#else
  retval = qtest();
//  msgInfo << "qtest just got " << retval << sendmsg;
#endif

  // find what kind of event it was
  switch(retval) {
    case REDRAW:	return WIN_REDRAW;
    case INPUTCHANGE:	return WIN_INPUTCHANGE;
    case KEYBD:		return WIN_KEYBD;
    case RIGHTMOUSE:	return WIN_RIGHT;
    case MIDDLEMOUSE:	return WIN_MIDDLE;
    case LEFTMOUSE:	return WIN_LEFT;
    case MOUSEX:	return WIN_MOUSEX;
    case MOUSEY:	return WIN_MOUSEY;
     // this only needed for the AIX port; and I don't know why.
    case MODECHANGE:    { queue_events(); break;}
  }

  // if here, no proper event found ...
  return WIN_NOEVENT;
}


// read the next event ... returns an event type (one of the above ones),
// and a value.  Returns success, and sets arguments.
int GLDisplayDevice::read_event(long &retdev, long &retval) {

  long dev = test_events();
  short val;

  if(dev != WIN_NOEVENT) {
    // event is pending ... read it
    retdev = dev;
#ifdef VMDFORMS
    dev = fl_qread(&val);
#else
    dev = qread(&val);
#endif
    retval = val;
    return TRUE;
  }

  // if here, no event is ready
  return FALSE;
}


//
// virtual routines for preparing to draw, drawing, and finishing drawing
//

// reshape the display after a shape change
void GLDisplayDevice::reshape(void) {

  MSGDEBUG(2,"GLDisplayDevice: reshaping display." << sendmsg);

#ifdef GD_STEREO_IN_WINDOW
  if(getgdesc(GD_STEREO_IN_WINDOW) < 1 || getgdesc(GD_XPMAX) > 1200) {
    if(inStereo == GL_STEREO_CRYSTAL || inStereo == GL_STEREO_CRYSTAL_REV)
      return;		// now allowed to reshape crystal-eyes mode
  }
#else
  if(inStereo == GL_STEREO_CRYSTAL || inStereo == GL_STEREO_CRYSTAL_REV)
    return;		// now allowed to reshape crystal-eyes mode
#endif

  if(winget() != windowID)  winset(windowID);
  reshapeviewport();
  getsize(&xSize, &ySize);
  getorigin(&xOrig, &yOrig);
  if(inStereo == GL_STEREO_SIDE || inStereo == GL_STEREO_CROSSED) {
    set_screen_pos(0.5 * (float)xSize / (float)ySize);
  } else {
    set_screen_pos((float)xSize / (float)ySize);
  }
}

// clear the display
void GLDisplayDevice::clear(void) {
  unsigned long l;
  
  MSGDEBUG(3, "GLDisplayDevice: clearing display." << sendmsg);

  // set viewport properly
  if(inStereo == GL_STEREO_SIDE || inStereo == GL_STEREO_CROSSED) {
    viewport(0, (short)xSize, 0, (short)ySize);
  } else if(inStereo == GL_STEREO_CRYSTAL ||
	    inStereo == GL_STEREO_CRYSTAL_REV) {
#ifndef GD_STEREO_IN_WINDOW
    viewport(0,(short)getgdesc(GD_XPMAX),0,(short)getgdesc(GD_YPMAX));
#else
    if(getgdesc(GD_STEREO_IN_WINDOW) != 1 || getgdesc(GD_XPMAX) > 1200)
      viewport(0,(short)getgdesc(GD_XPMAX),0,(short)getgdesc(GD_YPMAX));
#endif
  }
  
  l = ( ( (short)(backColor[0] * 255.0) ) |
  	( (short)(backColor[1] * 255.0) << 8) |
  	( (short)(backColor[2] * 255.0) << 16) );
  czclear(l, getgdesc(GD_ZMAX));
}

// prepare to draw a 2D image
int GLDisplayDevice::prepare2D(int do_clear) {

  MSGDEBUG(3, "GLDisplayDevice: preparing to draw 2D." << sendmsg);

  if(winget() != windowID)  winset(windowID);
  Dim = 2;
  if(do_clear)
    clear();
  else
    zclear();
  return 1;
}

// prepare to draw a 3D image
int GLDisplayDevice::prepare3D(int do_clear) {

  MSGDEBUG(3, "GLDisplayDevice: preparing to draw 3D." << sendmsg);

  if(winget() != windowID)  winset(windowID);
  Dim = 3;
  if(do_clear)
    clear();
  else
    zclear();
  return 1;
}

// set up for normal (non-stereo) drawing.  Sets the viewport and perspective.
void GLDisplayDevice::normal(void) {
  viewport(0, (short)xSize, 0, (short)ySize);
  set_persp();
}

// set up for drawing the left eye image.  Assume always the left eye is
// drawn first (so no zclear is needed before it)
void GLDisplayDevice::left(void) {
  if(inStereo == GL_STEREO_SIDE || inStereo == GL_STEREO_CROSSED) {
    viewport(0, (short)(xSize / 2), 0, (short)ySize);
    if (inStereo == GL_STEREO_SIDE) {
      set_persp(RIGHTEYE);	// backwards since we're crossing our eyes
    } else {
      set_persp(LEFTEYE);
    }

  } else if (inStereo == GL_STEREO_LEFT) {
    // set the eye perspective properly
    set_persp(LEFTEYE);

  } else if (inStereo == GL_STEREO_RIGHT) {
    // set the eye perspective properly
    set_persp(RIGHTEYE);

    ///// normal stereo display
  } else if(inStereo == GL_STEREO_CRYSTAL ||
	    inStereo == GL_STEREO_CRYSTAL_REV ) {
#ifndef GD_STEREO_IN_WINDOW
    viewport(0,getgdesc(GD_XPMAX),YOFFSET,YOFFSET + YMAXSTEREO);
#else
    if(getgdesc(GD_STEREO_IN_WINDOW) == 1 && getgdesc(GD_XPMAX) < 1200) {
//      zclear();
      leftbuffer(TRUE);
      rightbuffer(FALSE);
    } else {
      viewport(0,(short)getgdesc(GD_XPMAX),YOFFSET,YOFFSET + YMAXSTEREO);
    }
#endif
    if (inStereo == GL_STEREO_CRYSTAL) {
      set_persp(LEFTEYE);
    } else {
      set_persp(RIGHTEYE);
    }
  } else {			// left called even though we're non-stereo
    normal();
  }

}

// set up for drawing the right eye image.  Assume always the right eye is
// drawn last (so a zclear IS needed before it)
void GLDisplayDevice::right(void) {
  if(inStereo == GL_STEREO_SIDE || inStereo == GL_STEREO_CROSSED) {
    viewport((short)(xSize / 2), (short)xSize, 0, (short)ySize);
    if (inStereo == GL_STEREO_SIDE) {
      set_persp(LEFTEYE);	// backwards since we're crossing our eyes
    } else {
      set_persp(RIGHTEYE);
    }
  } else if (inStereo == GL_STEREO_LEFT || inStereo == GL_STEREO_RIGHT) {
    // no need to to anything, already done in call to left

    ///// normal stereo display
  } else if(inStereo == GL_STEREO_CRYSTAL ||
	    inStereo == GL_STEREO_CRYSTAL_REV ) {
#ifndef GD_STEREO_IN_WINDOW
    viewport(0,getgdesc(GD_XPMAX),0,YMAXSTEREO);
#else
    if(getgdesc(GD_STEREO_IN_WINDOW) == 1 && getgdesc(GD_XPMAX) < 1200) {
      zclear();
      leftbuffer(FALSE);
      rightbuffer(TRUE);
    } else {
      viewport(0,(short)getgdesc(GD_XPMAX),0,YMAXSTEREO);
    }
#endif
    if (inStereo == GL_STEREO_CRYSTAL) {
      set_persp(RIGHTEYE);
    } else {
      set_persp(LEFTEYE);
    }

  } else {			// right called even though we're non-stereo
    normal();
  }
}

// update after drawing
void GLDisplayDevice::update(int do_update) {

  MSGDEBUG(3, "GLDisplayDevice: updating after drawing." << sendmsg);

  if(do_update)
    swapbuffers();

#ifdef GD_STEREO_IN_WINDOW
  if(getgdesc(GD_STEREO_IN_WINDOW) == 1 && getgdesc(GD_XPMAX) < 1200) {
    leftbuffer(TRUE);
    rightbuffer(TRUE);
  }
#endif
}

//
// stereo
//

// change to a different stereo mode (0 means 'off')
void GLDisplayDevice::set_stereo_mode(int newMode) {
  if (newMode == inStereo) return;  // don't constantly change

  if(newMode == GL_STEREO_OFF) {
    //    if(!inStereo)  return;  // if already off, don't re-turn off
    
    if(inStereo == GL_STEREO_CRYSTAL
       || inStereo == GL_STEREO_CRYSTAL_REV) {
#ifdef GD_STEREO_IN_WINDOW
      if(getgdesc(GD_STEREO_IN_WINDOW) < 1 || getgdesc(GD_XPMAX) > 1200) {
#endif
        system("offstereo");

        // restore the window to the size it was before being set to stereo
        if(winget() != windowID)  winset(windowID);
        winposition(xOrig, xOrig + xSize, yOrig, yOrig + ySize);
        winconstraints();
#ifdef GD_STEREO_IN_WINDOW
      }
#endif
      clear();
    }

    inStereo = newMode;
    reshape();
    normal();
    update();

    MSGDEBUG(2,"GLDisplayDevice stereo turned off." << sendmsg);

  } else if(newMode != GL_STEREO_CRYSTAL &&
	    newMode != GL_STEREO_CRYSTAL_REV) {
    set_stereo_mode(GL_STEREO_OFF);
    inStereo = newMode;
    reshape();
    normal();
    MSGDEBUG(2,"GLDisplayDevice set to side-by-side stereo." << sendmsg);

  } else if (newMode == GL_STEREO_CRYSTAL||
	    newMode == GL_STEREO_CRYSTAL_REV) {
    // check if already in a crystal-eyes stereo mode
    if (inStereo == GL_STEREO_CRYSTAL_REV ||
	inStereo == GL_STEREO_CRYSTAL_REV) {
      inStereo = newMode;  // so don't redo all the graphics/windows/etc
    }

    set_stereo_mode(GL_STEREO_OFF);
#ifdef GD_STEREO_IN_WINDOW
    if(getgdesc(GD_STEREO_IN_WINDOW) == 1 && getgdesc(GD_XPMAX) < 1200) {
      if(!getgconfig(GC_DOUBLE) || !getgconfig(GC_STEREO)) {
        msgErr << "Doublebuffer stereo not available." << sendmsg;
        return;
      }
    }
#endif

    inStereo = newMode;

#ifdef GD_STEREO_IN_WINDOW
    if(getgdesc(GD_STEREO_IN_WINDOW) < 1 || getgdesc(GD_XPMAX) > 1200) {
#endif
      // full-screen (not in-a-window) stereo
      if(winget() != windowID)  winset(windowID);

      // execute the SGI program to put the monitor in stereo mode
      system("onstereo");

      // resize the window to be the whole screen
      prefposition(0,getgdesc(GD_XPMAX),0,getgdesc(GD_YPMAX));
      winconstraints();
      reshapeviewport();
      viewport(0,(short)getgdesc(GD_XPMAX),0,(short)getgdesc(GD_YPMAX));
      set_screen_pos(0.5*(float)getgdesc(GD_XPMAX)/(float)YMAXSTEREO);
      clear();
      update();

#ifdef GD_STEREO_IN_WINDOW
    }
#endif

    MSGDEBUG(2,"GLDisplayDevice set to crystal-eyes stereo." << sendmsg);
  }
}

