/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: FormsObj.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.14 $	$Date: 95/10/19 16:05:57 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The FORMS object, which is one single form that can be turned on/off,
 * initialized, etc  It is a base class for specific
 * forms.  It assumes that this is for use with a MoleculeList, and will
 * use the specified DisplayDevice.
 * This class is derived from UIObject; it can generate and receive commands.
 *
 ***************************************************************************/

#include <stdio.h>
#include <strings.h>
#include "FormsObj.h"
#include "Inform.h"
#include "CmdMenu.h"
#include "Mouse.h"
#include "MoleculeList.h"
#include "DisplayDevice.h"
#include "Global.h"


// static member initialization
int FormsObj::formsInitialized = FALSE;
int FormsObj::checkNewEvents = TRUE;


// commands we are interested in
static int numCmds = 6;
static int cmdList[6] = { Command::GUI_EVENT, Command::MOL_NEW,
	Command::MOL_TOP, 		Command::MOL_DEL,
	Command::REMOTE_ATTACH,		Command::REMOTE_RUN
};


// class constructor
FormsObj::FormsObj(char *form_name, UIList *uil, CommandQueue *cq, 
	int show_border, int *new_pos)
  : UIObject(form_name, uil, cq) {
  int i;

  // call create_the_forms, if necessary
  if(!formsInitialized) {
    MSGDEBUG(1,"FormsObj: initializing the forms windows ... " << sendmsg);

    // express interest in certain events
    display->queue_events();

    formsInitialized = TRUE;
  }
  
  MSGDEBUG(1,"Creating form '" << name << "' ..." << sendmsg);

  // register which commands we are interested in
  for(i=0; i < numCmds; command_wanted(cmdList[i++]));
  
  form = NULL;
  formShowing = FALSE;
  border = show_border;
  mol = moleculeList->top();
  lightButton = offButton = NULL;
  
  if(new_pos) {
    placeMethod = FL_PLACE_POSITION;
    formPosition[0] = new_pos[0];
    formPosition[1] = new_pos[1];
  } else
    placeMethod = FL_PLACE_SIZE;
}


// class destructor
FormsObj::~FormsObj(void) {
  Off();
}


// is this a menu object?
int FormsObj::is_menu(void) {
  return TRUE;
}


// return whether the menu is on
int FormsObj::is_on(void) {
  return displayed();
}


// turn the form on
void FormsObj::On(void) {

  if(!formShowing) {
    if(placeMethod == FL_PLACE_POSITION)
      fl_set_form_position(form,formPosition[0],formPosition[1]);
    fl_show_form(form,placeMethod,border,name);
    formShowing = TRUE;
    if(lightButton && fl_get_button(lightButton) != formShowing)
      fl_set_button(lightButton, formShowing);
  }
}


// turn the form off
void FormsObj::Off(void) {

  if(formShowing) {
    placeMethod = FL_PLACE_POSITION;
    formPosition[0] = (int)(form->x);
    formPosition[1] = (int)(form->y);
    fl_hide_form(form);
    formShowing = FALSE;
    if(lightButton && fl_get_button(lightButton) != formShowing)
      fl_set_button(lightButton, formShowing);
  }
}


// move the form
void FormsObj::move(int x, int y) {
  placeMethod = FL_PLACE_POSITION;
  formPosition[0] = x;
  formPosition[1] = y;
  fl_set_form_position(form, x, y);
}

// this version does nothing
void FormsObj::move(float *) { }


// return where the form currently is located
void FormsObj::where(int &x, int &y) {
  x = (int)(form->x);
  y = (int)(form->y);
}

// this version does nothing
void FormsObj::where(float *) { }


// routine to set an input object to an integer value
void FormsObj::set_input_int(FL_OBJECT *obj, int val, int places) {
  char buf[80];

  sprintf(buf,"%-*d",places,val);
  fl_set_input(obj,buf);
}


// update the display
void FormsObj::update(void) {
  checkNewEvents = TRUE;
}


// check for an event, and queue it if found.  Return TRUE if an event
// was generated.
int FormsObj::check_event(void) {
  FL_OBJECT *obj;
  long dev, val;

  // only do this if no other FormsObj's have ...
  if(!checkNewEvents)
    return FALSE;
    
  MSGDEBUG(3,"FormsObj: checking for events ... " << sendmsg);

  // get a forms event if one exists ... keep doing so until break in events
  checkNewEvents = FALSE;
  int foundevent = FALSE;

#ifndef VMDGL
  // when not using GL with XFORMS, one event queue to check
  while((obj = fl_check_forms()) != NULL) {
    if(obj == FL_EVENT) {		         // windowing event
      if(display->read_event(dev, val)) {
	runcommand(new WinEvent(dev, val, id()));
	foundevent = TRUE;
      }
    } else if(obj) {                            // FORMS event
      runcommand(new GUIEvent(obj, id()));
      foundevent = TRUE;
    }
  }
#else
  // GL has a separate event queue from X, must check both
  do {
    if (!(obj = fl_check_forms())) {
      if (display->read_event(dev,val)) {
        runcommand(new WinEvent(dev, val, id()));
        foundevent = TRUE;
      }
    } else {
      runcommand(new GUIEvent(obj, id()));
      foundevent = TRUE;
    }
  } while(obj);
#endif
  
  return foundevent;
}


// update the display due to a command being executed.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int FormsObj::act_on_command(int type, Command *cmd, int suc) {
  int frm_retval;

  MSGDEBUG(2,"FormsObj: acting on command " << type << sendmsg);

  // have derived class check command
  frm_retval = forms_act_on_command(type, cmd, suc);

  // do extra checking
  if(type == Command::GUI_EVENT) {
    FL_OBJECT *obj = ((GUIEvent *)cmd)->obj;

    // check if this form has the associated object
    if(obj == lightButton) {
      MSGDEBUG(2,"On/Off toggle button pressed for menu '" << name << "'.");
      MSGDEBUG(2,sendmsg);
      addcommand(new CmdMenuShow(id(),fl_get_button(obj),id()));
    } else if(obj == offButton) {
      MSGDEBUG(2,"Off button pressed for menu '" << name << "'.");
      MSGDEBUG(2,sendmsg);
      addcommand(new CmdMenuShow(id(),FALSE,id()));
    } else {
      MSGDEBUG(2,"  forms-specific event; checking '" << name << "' ...");
      MSGDEBUG(2,sendmsg);
      return frm_retval || check(obj);
    }

  } else if(type == Command::MOL_TOP	|| type == Command::MOL_DEL ||
	    type == Command::MOL_NEW	|| type == Command::REMOTE_ATTACH ||
	    type == Command::REMOTE_RUN) {
    mol = moleculeList->top();
    reset();
    
  } else
    return frm_retval;

  MSGDEBUG(2,"'" << name << "' ... done acting on command " << type);
  MSGDEBUG(2,sendmsg);

  return TRUE;
}


// do form-specific acting on commands
int FormsObj::forms_act_on_command(int, Command *, int) {
  return FALSE;
}


/////////////////////////////  GUI Event Object

// constructor; must specify what the event type code is
GUIEvent::GUIEvent(FL_OBJECT *o, int newUIid)
	: Command(Command::GUI_EVENT, newUIid) {
  obj = o;
}

