/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DrawForce.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 96/03/24 14:00:26 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Another Child Displayable component for a remote molecule; this displays
 * and stores the information about the interactive forces being applied to
 * the molecule.  If no forces are being used, this draws nothing.
 *
 * The force information is retrieved from the Atom list in the parent
 * molecule.  No forces are stored here.
 *
 * This name is now a misnomer as accelerations are changed, _not_ forces.
 * This eliminates the problem of having hydrogens acceleterating 12 time
 * faster than carbons, etc.
 *
 ***************************************************************************/

#include "DrawForce.h"
#include "DrawMolecule.h"
#include "ColorList.h"
#include "PickModeQuery.h"
#include "DisplayDevice.h"
#include "Scene.h"
#include "Atom.h"
#include "Timestep.h"
#include "Inform.h"
#include "utilities.h"

#define FORCE_SCALE (10.0 / 14.0)

////////////////////////////  constructor  

DrawForce::DrawForce(DrawMolecule *mr)
	: Displayable3D(MULT, mr->name, mr, mr->nAtoms/8) {

  MSGDEBUG(1,"Creating new force representation, for parent molecule:\n");
  MSGDEBUG(1," ==> '" << mr->name << "'" << sendmsg);

  // save data
  mol = mr;

  // initialize variables
  needRegenerate = TRUE;
  forceItemsDrawn = 0;
  colorCat = (-1);

  // register as a pickable item
  register_with_picklist(this->origScene);

  // create new pick modes to allow pointers to add forces
  atomForcePickMode = add_pick_mode("ForceAtom", 0);
  resForcePickMode = add_pick_mode("ForceResidue", 1);
  fragForcePickMode = add_pick_mode("ForceFragment", 2);
}


///////////////////////////  protected virtual routines

// do action when a new color list is provided
// This creates a color category for use for drawing forces.
// These colors can then be edited by the user.
void DrawForce::do_use_colors(void) {

  // add new category (or just get it's index if it exists)
//  colorCat = colorList->add_color_category("Forces");
  
  // add components, and their default colors
//  (colorList->color_category(colorCat))->add_name("Constant", REGORANGE);
}


// do action due to the fact that a color for the given ColorList for the
// specified category has changed.
void DrawForce::do_color_changed(ColorList *, int ccat) {
  // right now this does nothing, since we always redraw the list.  But
  // the general thing it would do is set the flag that a redraw is needed,
  // so looking ahead I'll do this now.
  if(ccat = colorCat) {
    needRegenerate = TRUE;
  }
}


// create a new pick mode object for use as a separate pick mode to add
// forces via the mouse.
PickMode *DrawForce::create_pick_mode(int m) {

  // m is just a code used by this object to tell which one to create.
  if(m >= 0 && m < 3) {
    return new PickModeQuery;
  } else {
    msgErr << "Unknown pick mode " << m << " in DrawForce::create_pick_mode";
    msgErr << sendmsg;
  }

  // if here, error
  return NULL;
}


//////////////////////////////// private routines 

// regenerate the command list
void DrawForce::create_cmdlist(void) {

  // do we need to recreate everything?
  if(needRegenerate) {

    MSGDEBUG(2, "Regenerating display list for forces '" << name << "' ...");
    MSGDEBUG(2, sendmsg);

    // regenerate both data block and display commands
    needRegenerate = FALSE;
    forceItemsDrawn = 0;
    reset_disp_list();

    // only put in commands if there is a current frame
    if(mol->frame() >= 0) {
      Timestep *ts = mol->current();

      // for each atom, if it has a nonzero user force, then display it
      for (int i=0; i < mol->nAtoms; i++) {
	MolAtom *atm = mol->atom(i);

	// check if nonzero force
	if(atm->extra[ATOM_USERFORCEX] > 0.0 ||
	   atm->extra[ATOM_USERFORCEY] > 0.0 ||
	   atm->extra[ATOM_USERFORCEZ] > 0.0) {

	  // get position of atom, and the position of the force vector
	  float *p1 = ts->pos + 3*i;
	  float fval[3], p2[3], p3[3];
	  float mass = atm -> extra[ATOMMASS];
          for(int k = 0; k < 3; k++) {
            fval[k] = atm->extra[ATOM_USERFORCEX + k] / FORCE_SCALE / mass;
	    p2[k] = p1[k] + fval[k];
	    p3[k] = p1[k] + 0.8 * fval[k];
          }

	  // find length of force
	  float p2norm = norm(fval);

	  // put in commands to set line style, etc. if this is thie first
	  // item drawn
	  if(forceItemsDrawn < 1) {
	    // cmdMaterials.putdata(FALSE,this);
	    // cmdLineType.putdata(DASHEDLINE, this);
	    // cmdLineWidth.putdata(3, this);
	    cmdMaterials.putdata(TRUE,this);

	    // int c = REGORANGE;
	    // if(colorCat >= 0)
	    //    c = (colorList->color_category(colorCat))->data("Constant");
	    // cmdColorIndex.putdata(REGCOLOR(c), this);
	  }

	  // set line color
	  int sc = (int)p2norm;
	  if(sc >= MAPCLRS)
	    sc = MAPCLRS - 1;
	  cmdColorIndex.putdata(MAPCOLOR(sc), this);

	  // draw line
//          cmdLine.putdata(p1, p2, this);
	  float rad1 = 0.2 * p2norm;
	  if (rad1 > 0.3)
	    rad1 = 0.3;
	  float rad2 = 1.5 * rad1;
	  cmdCone.putdata(p3, p1, rad1, 9, this);
	  cmdCone.putdata(p3, p2, rad2, 9, this);
	  pickPoint.putdata(p2, i, this);
	  forceItemsDrawn++;
        }
      }
    }
  }
}


//////////////////////////////// public routines 
// prepare for drawing ... do any updates needed right before draw.
void DrawForce::prepare(DisplayDevice *) {

  // see if the frame has changed ... right now, always update
//  if(mol->has_frame_changed())
    needRegenerate = TRUE;
  
  create_cmdlist();
}



// small quick routine to add a force to a given atom in given timestep
void set_drawforce_force(MolAtom *atom, float *force) {

  // change the force
  float mass = atom->extra[ATOMMASS];
  atom->extra[ATOM_USERFORCEX] = force[0] * mass;
  atom->extra[ATOM_USERFORCEY] = force[1] * mass;
  atom->extra[ATOM_USERFORCEZ] = force[2] * mass;
  atom->changed_userforce = TRUE;
}



// return if we are interested in the given pick mode or not ... here, we
// are interested in the force creation picking mode
int DrawForce::want_pick_mode(int) {
  return TRUE;
}


// return whether the pickable object is being displayed
int DrawForce::pickable_on(void) {
  return (forceItemsDrawn > 0 && Displayable3D::pickable_on());
}


// called when a pick is begun:
//    args = display to use, obj picked, button, mode, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
void DrawForce::pick_start(DisplayDevice *, Pickable *p,
                                int btn, int mode, int tag, int, float *) {

  // if we pick an existing force with the middle button, delete it
  if(p == this && btn == MouseEvent::B_MIDDLE && tag >= 0) {

    float newforce[3];
    newforce[0] = newforce[1] = newforce[2] = 0.0;

    // set the force to zero for all necessary atoms
    if(mode == resForcePickMode) {
      // set this force for all the atoms in this atom's residue
      Residue *res = mol->atom_residue(tag);
      if(res) {
	int natm = (res->atoms).num();
	for(int n = 0; n < natm; n++)
	  set_drawforce_force(mol->atom((res->atoms)[n]), newforce);
      }

    } else if(mode == fragForcePickMode) {
      // set this force for all the atoms in this atom's fragment
      Fragment *frag = mol->atom_fragment(tag);
      if(frag) {
	int nres = frag->num();
	for(int r = 0; r < nres; r++) {
	  Residue *res = mol->residue((*frag)[r]);
	  int natm = (res->atoms).num();
	  for(int n = 0; n < natm; n++)
	    set_drawforce_force(mol->atom((res->atoms)[n]), newforce);
	}
      }

    } else {
      set_drawforce_force(mol->atom(tag), newforce);
    }
  }
}


// called when a pick moves:
//    args = display to use, obj picked, button, mode, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
// For the Axes, when they are selected and the pointer moves, we wish
// to move the axes as well.
void DrawForce::pick_move(DisplayDevice *d, Pickable *p,
			  int btn, int mode, int tag, int dim, float *pos) {

  // See if the item selected is one of this molecule's representations.
  // First check if we are selecting a force in this object; then check if
  // the mode is for adding new forces and an atom in our molecule is
  // selected
  int whichrep = (-1);

  if(p == this) {
    whichrep = 1;

  } else if(mode == atomForcePickMode || mode == resForcePickMode ||
	    mode == fragForcePickMode) {
    int repnum = mol->components();
    for(int j=0; j < repnum; j++) {
      if(p == mol->component(j)) {
	whichrep = j;
	break;
      }
    }
  }

  // if an atom in a rep was not selected, we can just quit now
  Timestep *ts = mol->current();
  if(whichrep < 0 || tag < 0 || (ts == NULL) || btn != MouseEvent::B_LEFT)
    return;

  // either add a new force if button 1 is used, or set the force to
  // zero if button 2 is used.
  float newforce[3];
  if(btn == MouseEvent::B_MIDDLE) {
    newforce[0] = newforce[1] = newforce[2] = 0.0;
  } else {
    // an atom in a rep was selected; find the new force and change it
    float *atomPos = ts->pos + tag * 3;
    float atomWorldPos[3], mouseWorldPos[3];
    float *newpos;
    if(dim == 2) {

      // convert atom position to world coordinates
      tm.multpoint3d(atomPos, atomWorldPos);

      // find the 3D position (in world coords) of the mouse
      d->find_3D_from_2D(atomWorldPos, pos, mouseWorldPos);

      // indicate this new position as the mouse position
      newpos = mouseWorldPos;
    } else {
      // for 3D pointer, just use the pointer position as-is, but
      newpos = pos;
    }

    // now convert back from world coords to object coords
    Matrix4 tminv(tm);
    tminv.inverse();
    tminv.multpoint3d(newpos, newforce);

    // find the new force by finding F = mousepos - atompos
    newforce[0] = FORCE_SCALE * (newforce[0] - atomPos[0]);
    newforce[1] = FORCE_SCALE * (newforce[1] - atomPos[1]);
    newforce[2] = FORCE_SCALE * (newforce[2] - atomPos[2]);
  }

  //  msgWarn << "DrawForce: New force is " << newforce[0] <<  ", ";
  //  msgWarn << newforce[1] << ", " << newforce[2] << sendmsg;

  if(mode == resForcePickMode) {
    // set this force for all the atoms in this atom's residue
    Residue *res = mol->atom_residue(tag);
    if(res) {
      int natm = (res->atoms).num();
      for(int n = 0; n < natm; n++)
	set_drawforce_force(mol->atom((res->atoms)[n]), newforce);
      return;
    }

  } else if(mode == fragForcePickMode) {
    // set this force for all the atoms in this atom's fragment
    Fragment *frag = mol->atom_fragment(tag);
    if(frag) {
      int nres = frag->num();
      for(int r = 0; r < nres; r++) {
	Residue *res = mol->residue((*frag)[r]);
	int natm = (res->atoms).num();
	for(int n = 0; n < natm; n++)
	  set_drawforce_force(mol->atom((res->atoms)[n]), newforce);
      }
      return;
    }
  }

  // if here, we have not set the forces, so do so just to the selected
  // atom.
  set_drawforce_force(mol->atom(tag), newforce);
}


