/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Displayable.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.20 $	$Date: 96/02/15 17:03:10 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Base class for all objects which are drawn in the DisplayDevice.
 *
 ***************************************************************************/

#include <string.h>
#include "Displayable.h"
#include "Inform.h"
#include "DispCmds.h"
#include "Scene.h"
#include "PickList.h"
#include "utilities.h"


/*	NOTES:

	1. Each Displayable contains an array of drawing/display commands.
This array actually has some character-sized flags at the beginning, and
then a sequential list of commands.  The format for the cmdList is:
		Integer: On (T) or Off (F)
		Integer: Fixed (T) or Free (F)
		Integer: Dimension
		Push Matrix command
		Multiply Matrix (with this objs trans matrix) command
		... all commands for this object
		Pop Matrix command
	2. Each command has the form (as set up in DispCmds.h):
		command code (integer)
		data for command (0 or more bytes)
	3. A Displayable 'registers' with one or more Scene objects; this
means it gives the display command list pointer to the Scene, which then
uses that to draw the object.  Before drawing, however, the Scene calls
the 'prepare' routine for each Displayable.  When deleted, the Displayable
'unregisters' with all Scene objs it has previously registered with.
If a Scene is deleted, it unregisters the Displayables in it's possession.
	4. A Displayable is either a 'parent' one or a 'child' one.  The
difference is that parent Displayables register with the scene, and have
only one transformation; children do NOT register with the scene, and have
not only their main transformation (which is the same as the parent) but
also a second one which multiplies the first.  Note that children can have
children of their own, but are not 'parents', i.e. child Displayables do not
register with a scene even if they have children of their own.
	5. Children do register their DISPLAY LISTS, just not themselves
(thus, they are not prepared, etc.; just rendered.)
	6. Children are created as normal, but with the parent Displayable
specified instead of the Scene.  As part of the creation, the child will be
added to the parent's list via the 'add child' routine.
*/

// PARENT constructor; maxsz is in units of KB
Displayable::Displayable(TransMethod matMeth, char *nm, int myDim,
	Scene *myscene, int maxsz) : children(16), sceneList(16) {

  // make sure arguments are OK
  if(myDim < 2 || myDim > 3)
    myDim = 3;
  if(maxsz < 1)
    maxsz = 1;

  // make sure there is some extra space allocated
  maxsz++;

  // this is a parent Displayable
  parent = NULL;

  // set which scene we are to register with/get storage from
  origScene = myscene;

  // do common creation action
  do_create(nm, myDim, maxsz, matMeth);

  // parent must register with the scene
  Register(origScene);
}


// CHILD constructor; maxsz is in units of KB
Displayable::Displayable(TransMethod matMeth, char *nm, int myDim,
	Displayable *pops, int maxsz)  : children(16), sceneList(16) {

  // make sure arguments are OK
  if(myDim < 2 || myDim > 3)
    myDim = 3;
  if(maxsz < 1)
    maxsz = 1;

  // make sure there is some extra space allocated
  maxsz++;

  // this is a child Displayable; remember parent
  if(!pops) {
    msgErr << "Error: No parent specified in Displayable '" << name << "'.";
    msgErr << sendmsg;
  }
  parent = pops;
  
  // get copies of all of parents tranformation matrices
  centm = parent->centm;
  rotm = parent->rotm;
  oldrotm = parent->oldrotm;
  globm = parent->globm;
  scalem = parent->scalem;
  tm = parent->tm;

  // set which scene we are to register with/get storage from
  origScene = parent->origScene;

  // do common creation action
  do_create(nm, myDim, maxsz, matMeth);
  
  // finally, add this Displayable as a child to the parent (also registers)
  parent->add_child(this);
}


// does all the creation work after variables have been initialized
void Displayable::do_create(char *nm, int myDim, int maxsz, TransMethod t) {

  // save name
  name = stringdup(nm);

  // method for how the trans matrix is used
  transMethod = t;

  // initialize the display command list ... request space from Scene obj
  initial_block_size = cmdListBlockSize = maxsz * 1024;
  cmdList = origScene->get_disp_storage(cmdListBlockSize);

  MSGDEBUG(1,"Creating new Displayable object '" << name << "'" << sendmsg);
  if(is_child()) {
    MSGDEBUG(2,"   (Parent is '" << parent->name << "')" << sendmsg);
  }
  MSGDEBUG(5,"   dim=" << myDim);
  MSGDEBUG(5,"\n   trans method=" << transMethod);
  MSGDEBUG(5,"\n   message size=" << cmdListBlockSize);
  MSGDEBUG(5,sendmsg);

  // initialize useful DispCmd's
  pushCmd = new DispCmdPush;
  popCmd = new DispCmdPop;
  if(transMethod == MULT) {
    matCmd = new DispCmdMult(&tm);
  } else if(transMethod == LOAD) {
    matCmd = new DispCmdLoad(&tm);
  } else
    matCmd = NULL;

  // initialize flags and scalar settings
  needMatrixRecalc = TRUE;
  Dim = (int *)cmdList;
  displayObj = Dim + 1;
  *Dim = myDim;             // ((int *)cmdList)[0] = dimensionality (=2 or 3)
  
  if(is_child()) {         // ((int *)cmdList)[1] = boolean "is drawn"
    *displayObj = parent->displayed();
    doCent = parent->cent_translating();
    doRot = parent->rotating();
    doGlob = parent->glob_translating();
    doScale = parent->scaling();
  } else {		// get values for this items as default values
    *displayObj = TRUE;
    doCent = doRot = doGlob = doScale = TRUE;
  }
  isFixed = Dim + 2;        // ((int *)cmdList)[2] = boolean "is fixed"
  *isFixed = FALSE;

  // initialize display list
  //
  // This next initialization must be forced so that I
  // don't accidentally try to unlink a non-existant linked list
  cmdListBeg = cmdListPos = cmdList + 3*sizeof(int);
  *((int *)(cmdListPos)) = ::DLASTCOMMAND;  // force end of list
  reset_disp_list();
}

// destructor; free up allocated space
Displayable::~Displayable(void) {

  MSGDEBUG(1,"Deleting Displayable object '" << name << "'" << sendmsg);

  // delete all children still around; also unregistered them
  while(num_children() > 0)
    // delete child object ... the child destructor then removes the
    // child from this parent's list, which is why we always remove the
    // first one, until there are no more.
    delete child(0);

  // unregister from picklist, if necessary.
  // unregister from all scenes
  while(sceneList.num() > 0)
    unRegister(sceneList[0]);

  // free space allocated for disp storage
  origScene->free_disp_storage(cmdList);

  // if this is a child, remove it from it's parent's list of children
  if(is_child())
    parent->remove_child(this);

  // delete objects created in constructor
  delete popCmd;
  delete pushCmd;
  if(matCmd)  delete matCmd;
  
  // delete name string
  delete [] name;
}


///////////////////////////  protected routines 

// recalculate the transformation matrix, and replace matrix in cmdList
// This is composed of these operations (applied as R to L):
//	TM =  GlobalTrans * Scale * Rotation * CenterTrans
void Displayable::recalc_mat(void) {
  if(needMatrixRecalc) {
    if(transMethod != NONE) {
      tm = globm;
      tm *= rotm;
      tm *= scalem;
      tm *= centm;

      // reload this matrix in the display command list
      matCmd->reput(this);

      MSGDEBUG(5, "'" << name<<"' recalculated transformation matrix; now:\n");
      MSGDEBUG(5, tm << sendmsg);
    }
    
    needMatrixRecalc = FALSE;
  }

  // recalc matrix for all children
  for(int i=0; i < num_children(); i++)
    child(i)->recalc_mat();
}

///////////////////////////  public routines 

// turn this object on or off
void Displayable::off(void) { 
  *displayObj = FALSE;
  
  // turn off all children
  for(int i=0; i < num_children(); i++)
    child(i)->off();
}

void Displayable::on(void) { 
  *displayObj = TRUE;
  
  // turn on all children
  for(int i=0; i < num_children(); i++)
    child(i)->on();
}


// routine to set the name of the object
void Displayable::set_name(char *newname) {
  if(name)  delete [] name;
  name = stringdup(newname);
}
  

// register with the given scene object
void Displayable::Register(Scene *sc) {

  MSGDEBUG(2,"Displayable:'" << name<< "' registering with Scene." << sendmsg);

  // only register display list if we are not yet registered
  if(sceneList.find(sc) >= 0) {
    MSGDEBUG(2,"   Displayable: Already registered with that Scene."<<sendmsg);
    return;
  }

  // give command list to scene
  if(is_child()) 
    sc->Register(NULL, cmdList, dim());
  else
    sc->Register(this, cmdList, dim());

  // save scene pointer
  sceneList.append(sc);

  // tell all children to register with this scene as well
  for(int i=0; i < num_children(); i++)
    child(i)->Register(sc);
}


// unregister with the given scene object
void Displayable::unRegister(Scene *sc) {

  MSGDEBUG(2,"Displayable: unregistering with Scene." << sendmsg);

  // tell the scene to forget about our cmdList
  int indx;
  if((indx = sceneList.find(sc)) >= 0) {
    if(is_child())
      sc->unRegister(NULL, cmdList, dim());
    else
      sc->unRegister(this, cmdList, dim());
  
    // remove the scene pointer
    sceneList.remove(indx);

    // attempt to unregister as a pickable object.  If this is not a
    // pickable object, this really does not do anything.
    sc->remove_pickable(this);

  } else {
    msgErr << "Displayable '" << name << "': cannot unregister from Scene.";
    msgErr << sendmsg;
  }

  // tell all children to unregister with this scene as well
  for(int i=0; i < num_children(); i++)
    child(i)->unRegister(sc);
}


// add the given Displayable as a child (assuming it is one)
void Displayable::add_child(Displayable *d) {
  if(d && d->is_child()) {
    // child must register will all of parent's scenes
    for(int i=0; i < sceneList.num(); i++)
      d->Register(sceneList[i]);
    
    // append child to list of children
    int indx = children.append(d);

    MSGDEBUG(1,"Displayable: Parent '" << name << "' added child ");
    MSGDEBUG(1, indx << sendmsg);
  }
}


// remove the given Displayable as a child. return success.
int Displayable::remove_child(Displayable *d) {
  if(d && d->is_child()) {
    // remove first child that matches the pointer, if available.
    int n = child_index(d);
    if(n >= 0) {
      children.remove(n);
      MSGDEBUG(1, "Displayable: Parent '" << name << "' removed child ");
      MSGDEBUG(1, n << "; now " << num_children() << " left." << sendmsg);
      return TRUE;
    } else {
      msgErr << "Displayable: Parent '" << name << "' cannot remove child ";
      msgErr << n << " (out of " << num_children() << ")." << sendmsg;
    }
  }
  
  // if we're here, it didn't work.
  return FALSE;
}


// start a new command with the given integer code and size
// returns TRUE if successful
int Displayable::start_cmd(int cd, int size) {

  // if already processing a command, just exit
  if(entering)
    return FALSE;

  // make sure we have room for this command and
  // both(the "end" command(s) (eg, DPOP and DLASTCOMMAND), and
  //      the "link" command(s) (eg, DLINKLIST and sizeof(char *))
  // (I overestimated)
  if(cmdListSize + size + 2*sizeof(int) + 3*sizeof(int) +
     sizeof(char *) >= cmdListBlockSize) {
     // get more memory and form a linked list
     int n = size;
     if (n < cmdListBlockSize) n = cmdListBlockSize;
     if (n < 10240) n = 10240;  // get at least 11K
     n+=1024;
     char *tmplist = (char *)origScene -> get_disp_storage(n);
     if (!tmplist) {
	msgErr << "Cannot allocate any more space for the display list.";
	msgErr << sendmsg;
	return FALSE;
     }
     *((int *) cmdListPos) = ::DLINKLIST;  // make the link
     cmdListPos = (void *)(((int *) cmdListPos)+1);
     *((char **) cmdListPos) = tmplist;    // to the new data
     
     cmdListPos = tmplist;
     cmdListSize = 0;
     cmdListBlockSize = n;
  }
  
  MSGDEBUG(4,"'" << name << "' starting new command " << cd << sendmsg);

  // put in code, size of command (in bytes), and increment pointers/counters
  *((int *)cmdListPos) = cd;		// code
  *((int *)cmdListPos + 1) = size;	// size of cmd, in bytes (not including
  					// code and this size count)
  cmdListPos = (void *)((char *)cmdListPos + 2*sizeof(int));
  cmdListSize += 2*sizeof(int) + size;
  entering = TRUE;
  return TRUE;
}


// end a new command; arg is how many bytes the command took
void Displayable::end_cmd(void ) {

  // if not processing a command, just exit
  if(!entering)
    return;
  entering = FALSE;
    
  int sz = *((int *) cmdListPos-1);  // get the size back -- yes, this is ugly
                                     // but it is the easiest way to
                                     // change the pre-existing code
  MSGDEBUG(4,"'" << name << "' ending command");
  MSGDEBUG(4,", total size=" << cmdListSize << sendmsg);

  // update pointers, and put pop command at end (if transMethod is not NONE)
  // note that this is _always_ done, even if this isn't the end of the
  // drawing list, 'cause at this level I don't know if this is indeed the
  // end.  If it isn't, then the next step will overwrite.
  cmdListPos = (void *)((char *)cmdListPos + sz);
  if(transMethod != NONE) {
    // put POP command before last command at end of list
    *((int *)cmdListPos) = ::DPOP;
    *(((int *)cmdListPos) + 1) = 0;		// size of DPOP command
    *(((int *)cmdListPos) + 2) = ::DLASTCOMMAND;
  } else {
    // do not put in a POP command
    *((int *)cmdListPos) = ::DLASTCOMMAND;
  }
}


void Displayable::free_linked_list(char *start){
   while (*((int *) start) != ::DLASTCOMMAND &&
	  *((int *) start) != ::DLINKLIST) {
      start += 2 * sizeof(int) +    // bypass the two integers
               *(((int *) start) +1); // and the draw command data
   }
   if (*((int *)start ) == ::DLASTCOMMAND) {  // I'm last on the link
      return;
   }
       // get the pointer to the next block
   start += sizeof(int);                 // by skipping the DLINKLIST
   free_linked_list( *((char **) start) );  // free elements later on
   origScene -> free_disp_storage(*((char **)start));  // and free it
}
      

// reset the display command list; remove all current commands
void Displayable::reset_disp_list(void) {

  MSGDEBUG(3,"'" << name << "' resetting command list." << sendmsg);

  
  // indicate that we have the initial integer data values
  // these three values are:
  // 0) dimensionality (2 or 3)
  // 1) boolean "displayObj"
  // 2) boolean "isFixed"
  cmdListSize = 3 * sizeof(int);
  // this will only be true if the current data block is the first
  // data block, in which case I don't have to search the path to delete
  if (*(((int *) cmdList) + cmdListSize) != ::DLASTCOMMAND) {
     free_linked_list((char *)cmdListBeg );
  }
  cmdListPos = (void *)(cmdList + cmdListSize);
  cmdListBlockSize = initial_block_size;
  entering = FALSE;

  if(transMethod != NONE) {
    // put in push, mult commands (pop always put as last command in list)
    pushCmd->put(this);
    matCmd->put(this);
  } else {
    // just put in last command token; no trans for this object
    *((int *)cmdListPos) = ::DLASTCOMMAND;
  }
}


//
// routines for working with the ColorList
//

// set up to use a specific ColorList.  This stores the list, and calls
// a function to do specific action when given a new list
void Displayable::use_colors(ColorList *collist) {

  // save pointer, and do any special action
  ColorUser::use_colors(collist);
  
  // do this for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->use_colors(collist);
}


// do action due to the fact that a color for the given ColorList for
// the specified category has changed
void Displayable::color_changed(ColorList *changelist, int clr) {
  // do specific preparations for this object
  ColorUser::color_changed(changelist, clr);

  // do so for all the child displayables; this is done after the parent has
  // been processed.
  for(int i=0; i < num_children(); i++)
    child(i)->color_changed(changelist, clr);
}



//
// prepare/update routines
//

// specific prepare for drawing ... do any updates needed right before draw.
// This version called by draw_prepare, and supplied by derived class.
void Displayable::prepare(DisplayDevice *) { }
  

// prepare to draw; possibly recalc the trans matrix, and do particular
// preparations
void Displayable::draw_prepare(DisplayDevice *d) {

  MSGDEBUG(3,"Displayable '" << name << "': preparing." << sendmsg);

  // redo trans matrix, if this is not a child Displayable (otherwise, this
  // is done by the parent already)
  if(!is_child())
    recalc_mat();
    
  // do specific preparations for this object
  prepare(d);

  // now tell all the scenes we are registered with that we're done
  // preparing
  for(int j=0; j < sceneList.num(); j++)
    (sceneList[j])->done_preparing(cmdList, dim());

  // prepare all the child displayables; this is done after the parent has
  // been prepared.
  for(int i=0; i < num_children(); i++)
    child(i)->draw_prepare(d);
}


//
// commands to change transformation
//

// reset to identity the transformation
void Displayable::reset_transformation(void) {
  if(transMethod != NONE) {
    if(scaling())			scalem.identity();
    if(rotating())		rotm.identity();
    if(glob_translating())	globm.identity();
    if(cent_translating())	centm.identity();
    need_matrix_recalc();
  }

  // do reset for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->reset_transformation();
}


// just change the transformation to the one given
void Displayable::load_transformation(Matrix4 &m) {
  scalem.identity();
  globm.identity();
  centm.identity();
  rotm = m;
  need_matrix_recalc();

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->load_transformation(m);
} 

// just get the transformation for this object.  Don't
// worry about the child transmats.
void Displayable::get_transformation(Matrix4 *m) {
  *m = tm;
}

void Displayable::set_scale(float s) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->set_scale(s);

  if(!scaling())  return;
  scalem.identity().scale(s);
  need_matrix_recalc();
}

void Displayable::mult_scale(float s) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->mult_scale(s);

  if(!scaling())  return;
  scalem.scale(s);
  need_matrix_recalc();
}

void Displayable::set_scale(Matrix4 &m) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->set_scale(m);

  if(!scaling())  return;
  scalem = m;
  need_matrix_recalc();
}

void Displayable::mult_scale(Matrix4 &m) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->mult_scale(m);

  if(!scaling())  return;
  scalem *= m;
  need_matrix_recalc();
}

void Displayable::add_rot(float x, char axis) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->add_rot(x, axis);

  if(!rotating())  return;
  // apply rotation to identity, and then multiply this by old rot matrix
  oldrotm.identity().rot(x, axis);
  oldrotm *= rotm;
  rotm = oldrotm;
  need_matrix_recalc();
}

void Displayable::set_rot(float x, char axis) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->set_rot(x, axis);

  if(!rotating())  return;
  // apply rotation to identity, and then multiply this by old rot matrix
  oldrotm.identity().rot(x, axis);
  rotm = oldrotm;
  need_matrix_recalc();
}

void Displayable::add_rot(Matrix4 &m) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->add_rot(m);

  if(!rotating())  return;
  oldrotm = m;
  oldrotm *= rotm;
  rotm = oldrotm;
  need_matrix_recalc();
}

void Displayable::set_rot(Matrix4 &m) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->set_rot(m);

  if(!rotating())  return;
  rotm = m;
  need_matrix_recalc();
}

void Displayable::set_glob_trans(float x, float y, float z) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->set_glob_trans(x, y, z);

  if(!glob_translating())  return;
  globm.identity().translate(x,y,z);
  need_matrix_recalc();
}

void Displayable::set_glob_trans(Matrix4& m) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->set_glob_trans(m);

  if(!glob_translating())  return;
  globm = m;
  need_matrix_recalc();
}

void Displayable::add_glob_trans(float x, float y, float z) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->add_glob_trans(x, y, z);

  if(!glob_translating())  return;
  globm.translate(x,y,z);
  need_matrix_recalc();
}

void Displayable::add_glob_trans(Matrix4& m) {
  if(fixed())  return;		// only transform unfixed objects

  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->add_glob_trans(m);

  if(!glob_translating())  return;
  globm *= m;
  need_matrix_recalc();
}

void Displayable::set_cent_trans(float x, float y, float z) {
  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->set_cent_trans(x, y, z);

  if(!cent_translating())  return;
  centm.identity().translate(x,y,z);
  need_matrix_recalc();
}

void Displayable::set_cent_trans(Matrix4& m) {
  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->set_cent_trans(m);

  if(!cent_translating())  return;
  centm = m;
  need_matrix_recalc();
}

void Displayable::add_cent_trans(float x, float y, float z) {
  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->add_cent_trans(x, y, z);

  if(!cent_translating())  return;
  centm.translate(x,y,z);
  recalc_mat();
}

void Displayable::add_cent_trans(Matrix4& m) {
  // do trans for all children as well
  for(int i=0; i < num_children(); i++)
    child(i)->add_cent_trans(m);

  if(!cent_translating())  return;
  centm *= m;
  need_matrix_recalc();
}

void Displayable::change_center(float x, float y, float z)
{
    // Here's the math:
    //  T = global translation (offset) matrix
    //  M = scaling*rotation matrix
    //  C = centering (offset) matrix
    //  p = picked point
    //  x = any point
    // and let G = T*M*C, the global transformation

    // the current transformation is: T*M*C * p
    // I want a new T', C' such that
    //   C' * p = {0 0 0 1}
    // and
    //   T'*M*C' * x = T M C x

    //  Thus, C' = translate(-p), and
    //   T' = T M C (M C')^-1
    //   T' = G  (M C') ^ -1
    // However, T isn't a pure translation matrix.  
    //   I could fix it -- oh well.

	Matrix4 Tprime;
	Tprime.translate(-x, -y, -z);

	Matrix4 M(rotm);
	M.multmatrix(scalem);
	M.multmatrix(Tprime);
	M.inverse();

	Tprime = tm;
	Tprime.multmatrix(M);

	// and apply the result
	set_cent_trans(-x, -y, -z);
	set_glob_trans(Tprime);

}

//
// routines for working as a Pickable
//
  
// return our list of draw commands with picking draw commands in them
void *Displayable::pick_cmd_list(void) {
  return cmdListBeg;
}

// return name of pickable
char *Displayable::pick_name(void) {
  return name;
}

// return whether the pickable object is being displayed
int Displayable::pickable_on(void) {
  return displayed();
}


