/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CoorDCD.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 95/05/11 22:23:44 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * DCD file class - subclasses of ICoorFile and OCoorfile; used to read and
 * write DCD files.  Since a DCD file contains multiple coordinates, the
 * user must call the read routine multiple times, until no sets are left.
 * For output files, the user must specify in the constructor the type
 * (DCD_BINARY or DCD_ASCII) of the output file.
 *
 * A BaseMolecule must be supplied to both input and output versions,
 * so that the structure data can be accessed.
 *
 * NOTE: currently, this supports only BINARY format.  ASCII uses that
 * strange 12Z6 fortran format, which is a pain to deal with.
 *
 ***************************************************************************/
#ifndef DCDCOORFILE_H
#define DCDCOORFILE_H

#include "CoorFile.h"
class BaseMolecule;

// error flag values; should be > 0
#define DCD_ERR_NOTOPEN		1
#define DCD_ERR_EARLYEOF	2
#define DCD_ERR_NOATOMS		3
#define DCD_ERR_EXTRA		4
#define DCD_ERR_A_NOT_EQ	5
#define DCD_ERR_NOFILE		6
#define DCD_ERR_BADFORMAT	7
#define DCD_ERR_FILEEXISTS	8

// DCD input object

class ICoorDCD : public ICoorFile {

public:
  // file type enum
  enum DCDTYPE { ASCII, BINARY };
  
private:
  // molecule from which to take structure info
  BaseMolecule *mol;  

  // type of file, ascii or binary
  DCDTYPE fileType;
  
  // file number, if file is opened as binary
  int fileNum;
  
  // timestep size (in ps)
  double timeStep;
  
  // starting frame, and frameskip (# timesteps between each frame write)
  int frameStart, frameSkip;
  
  // temp read buffer
  float *X, *Y, *Z;
  
  // free-atom list data
  int namnf, *freeatoms;

public:
  // constructor/destructor: BaseMolecule
  ICoorDCD(BaseMolecule *);
  virtual ~ICoorDCD(void);

  // initialization routine.  Must be called before reading/writing any frames.
  // return TRUE if the file was properly initialized.
  virtual int init(char *);

  // read the next set of coordinates; return list of positions, or NULL if
  // error
  virtual Timestep *read(void);

  //
  // ICoorDCD-specific routines
  //

  // skip over the next set of coordinates; return success
  int skip(void);
 
  // return format of input file
  int file_type(void) { return fileType; }
  
  // return size of timestep (in ps)
  double timestep(void) { return timeStep; }
  
  // frame skip, and starting frame
  int frameskip(void) { return frameSkip; }
  int startframe(void) { return frameStart; }
};


// DCD output object

class OCoorDCD : public OCoorFile {

public:
  // file type enum
  enum DCDTYPE { ASCII, BINARY };
  
private:
  // molecule from which to take structure info
  BaseMolecule *mol;  

  // type of file, ascii or binary
  DCDTYPE fileType;
  
  // file number, if file is opened as binary
  int fileNum;
  
  // timestep size (in ps)
  double timeStep;
  
  // starting frame, and frameskip (# timesteps between each frame write)
  int frameStart, frameSkip;
  
  // temp read buffer
  float *X, *Y, *Z;
  
public:
  // constructor/destructor: BaseMolecule, 
  // type of output file (ascii or binary), and number
  // of frames to write (if > 0).
  OCoorDCD(BaseMolecule *, DCDTYPE, int = 0);
  virtual ~OCoorDCD(void);

  // initialization routine.  Must be called before reading/writing any frames.
  // return TRUE if the file was properly initialized.
  virtual int init(char *);

  // write the next set of coordinates; given list of positions, returns total
  // frames written so far
  virtual int write(Timestep *);

  //
  // OCoorDCD-specific routines
  //
  
  // return format of input file
  int file_type(void) { return fileType; }
  
  // change the output file type, if file has not been opened yet.  Returns
  // type of file.
  int set_type(DCDTYPE typ) { if(!opened()) fileType = typ; return fileType; }
  
  // change the number of frames to write, if file is not yet open.  Returns
  // total number of frames to write.
  int set_frames(int f) { if(!opened()) nFrames = f; return nFrames; }
  
  // change the timestep size, if file is not yet open. Return timestep.
  float set_timestep(double f) { if(!opened()) timeStep = f; return timeStep; }

  // change frame skip, and starting frame.  Return values of parameters.
  int set_frameskip(int f) { if(!opened()) frameSkip = f; return frameSkip; }
  int set_startframe(int f) { if(!opened()) frameStart = f; return frameStart;}
};

#endif

