/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: ConfigList.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 95/05/12 00:39:31 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Read in a configuration file of the form:
 *       keyword = information\n
 *-or-   keyword information\n
 * and produces a database which can return the string (char *)
 * associated with that keyword.  Multiple files can be read.
 * 
 *    A "word" is a seqeunce of characters that are not white space (see
 * isspace(3C) ).  The equals sign ('=') is optional (though if there is more
 * more than one equals sign, then the 2nd is not ignored).  The "information"
 * field may contain more than one word.  White space is ignored except that
 * white space between multiple words in the information field is maintained.
 * Everything on the line at and beyond a pound sign ('#') is ignored.  Hence
 * a data file can be:
 *   fullname = George   Washington # the first president of the U.S.
 *   fullname = Martha Washington   # his second wife
 * Once that is read in, all data associated with "name" can be retreived as
 *  StringList *strList = configFile.find("fullname");
 *  for (StringList *tmp=strList; tmp!=NULL; tmp = tmp -> next)
 *      cout << tmp->data << '\n';
 * Note:
 *   The returned StringList * is NOT new'ed.  Do NOT free it.
 *   Keywords are case INsensitive
 *   In this version of ConfigList, only ONE keyword/data pair is kept,
 * the last one read.  All previous ones are deleted.
 ***************************************************************************/

#include <iostream.h>
#include <strings.h> // for strncpy, strcasecmp
#include <ctype.h>   // for isspace
#include <stdio.h>   // Yes, I do use stdio
#include "Inform.h"
#include "ConfigList.h"

// given a key word, find the element of theList that matches
// This is a private function.
ConfigList::ConfigListNode *ConfigList::find_key_word(const char *key)
{
 ConfigListNode *tmp;
 for (tmp=theList; tmp!= NULL; tmp = tmp -> next) {
   if (!strcasecmp(tmp->name, key))
     return tmp;
 }
 return NULL;
}


//  I have a new name/ data pair.  If the name does not already exist,
// make a new ListData element and put it on the HEAD of theList.  If
// name does already exist, make a new ListNode
//   String 1 starts at s1 and is of length len1
//      "   2    "   "  s2  "   "  "   "    len2
void ConfigList::add_element( char *s1, int len1, char *s2, int len2)
{
    char *temps = new char[len1 + 1];  // what is the name?
    fflush(stdout);
    strncpy(temps, s1, len1);
    temps[len1] = 0;                   //       terminate the string
    ConfigListNode *tmpList;
                                       
    tmpList = find_key_word( temps);  // go through the list
    if (tmpList == NULL )  {          // if not found
       tmpList = new ConfigListNode( theList, temps, NULL);// create a new node
                                    // and stick it on the head of the list
       theList = tmpList;           // note that I can continue to use tmpList
    }

    if (len1 < len2) {                  // if string is smaller, don't realloc
      delete [] temps;
      temps = new char[len2 + 1];
    }
    strncpy(temps, s2, len2);           // get the new string
    temps[len2] = 0;

    StringList *newStrList = new StringList(temps);  // new element
    newStrList -> next = NULL;
    if (tmpList -> data != NULL) {       // oops, there is data
      delete (tmpList -> data);            //       so get rid of it
    }
    tmpList -> data = newStrList;      //    so this is the new data
    
    delete [] temps;
}

// open, read, parse, and close the file
// make a linked list of "AssocList"
// The file is parsed as (I think):
// [w]?([W])*[w]?[=]?[w]?([W][wW])?[w]
// where w == white space; W is non-whitespace
// [x]? says that there can be 0 or more characters of type x.
// [x]* says there must be at least 1 character of type x
// the terms in () are the two that are used.  Everything after a '#'
// sign is excluded.  I'm not sure how to specify that exclusion in
// my regular expression above :)
// Oh, and if there is a problem, a FALSE is returned.  A TRUE
// means it was okay
int ConfigList::read(const char *filename)
{
  FILE *infile;
  
  int linenumber = 0;   // keep track of line numbers for searching out errors

  if (!strcmp(filename,"-")) {  // should the input be from stdin?
    infile = stdin;
  } else {
    if ( (infile = fopen(filename, "r")) == NULL ) {
        return 0;
    }
  }

     // so read and parse it
  char buf[1000]; // give myself lots of space
  char *namestart, *nameend, *datastart, *dataend;
  char *s;
  int spacecount;
  while (fgets(buf, 999, infile)) {
    linenumber ++;        
    namestart = nameend = datastart = dataend = NULL;
    spacecount = 0;
    
    for (s = buf; *s && *s!='\n'; s++) {    // get to the end of the line
       if (*s == '#')                       // found a comment, so break
          break;
       if ( !isspace(*s) )    // dataend will always be the last non-blank char
          dataend = s;
       if ( !isspace(*s) && !namestart)     // found first character of name
          {namestart = s; continue; }
       if ( (isspace(*s)  || *s == '=') &&  // found last character of name
                 namestart && !nameend)
          nameend = s - 1;
       if ( !isspace(*s) && !datastart &&   // found the next char. after name
                 nameend)
          if (*s == '=' && spacecount == 0) // an equals is allowed
             {spacecount++; continue; }     // but only once
            else
             {datastart = s; continue; }    // otherwise, use it
    }
    if (*s == '\n')          // cut out the newline at the end
      *s = 0;
     else
      if (*s != '#') {       // then there was an overflow
        msgWarn << "Line " << linenumber << " of configuration file "
                 << filename << " contains more than 999 characters."
                 << "  Excess characters will be ignored." << sendmsg;
      } else {
        *s = 0;  // delete the '#' character
      }

// I will also ignore line that I can't understand
// If there is any text on the line (as compared to a blank or commented)
//   line, then I will say that there is a problem.
    if (!namestart || !nameend || !datastart || !dataend) {
      if (!namestart && datastart || namestart && !datastart) {// was some data
        msgWarn << "Couldn't parse line " << linenumber << " in "
                 << "configuration file " << filename << ".  The line was: "
                 << buf;
      }
      continue;  // which ever the case, go to the next line
    }

   // now I can add the new values to the linked list
   add_element( namestart, nameend - namestart + 1, datastart,
                  dataend - datastart + 1 );
                  
  } // while I can still get data with fgets
  
  if (strcmp(filename,"-")) {  // close the input file if not stdin
    fclose(infile);
  }
  return 1;  // Okay, all done
}

// class constructor
ConfigList::ConfigList( void) {
  theList = NULL;
}

// destructor for the class - just delete a linked list
ConfigList::~ConfigList( void) {
  ConfigListNode *curr, *next=NULL;
  for (curr = theList; curr!=NULL; curr = next) {
     next = curr -> next;
     delete curr;     // the nasties for deleted the linked list of string
                      // has already been defined in the typedef struct
                      // for ConfigListNode
  }
  theList = NULL;
} // all done


// Given "name", find all the elements of the ConfigList that
//  have have that "name"
//StringList *ConfigList::find(const char *name)
char *ConfigList::find(const char *name)
{
  ConfigListNode *tmpList;
  tmpList = find_key_word(name);
  if (tmpList != NULL)
//    return tmpList -> data;  // to return a StringList *
    return tmpList -> data -> data;
  return NULL;
}

//#define TEST_CONFIGLIST_C
#ifdef TEST_CONFIGLIST_C
#include <iostream.h>

main()
{
  ConfigList dat;
  ; // an example file
  char *strings;
  
  if (!dat.read("vmd.conf"))
    cerr << "Cannot get info from file.";
  cout << "Searching for: 'fullname'\n";
  strings = dat.find("fullname");
  if (!strings) {
    cerr << "Couldn't find fullname.\n";
  } else {
    cout << strings << '\n';
  }
 }
}

#endif

