/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: ColorList.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.12 $	$Date: 96/05/14 04:49:26 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A collection of colored materials; provides ways to define them and edit
 * them.  Each Displayable must turn on or off material characteristics
 * if needed, and say which color to use.
 *
 ***************************************************************************/
#ifndef COLORLIST_H
#define COLORLIST_H

#include "Displayable.h"
#include "NameList.h"
class DispCmdColorDef;

// total number of colors defined here
#define REGCLRS		17
#define EXTRACLRS	1
#define VISCLRS		(REGCLRS - EXTRACLRS)
#define TRANREGCLRS	REGCLRS
#define MAPCLRS		32
#define TRANMAPCLRS	MAPCLRS
#define MAXCOLORS	(REGCLRS + TRANREGCLRS + MAPCLRS + TRANMAPCLRS)

// where different type of colors start in indices
#define BEGREGCLRS	0
#define BEGVISCLRS	VISCLRS
#define BEGTRAN		REGCLRS
#define BEGMAP		(BEGTRAN + TRANREGCLRS)
#define BEGTRANMAP	(BEGMAP + MAPCLRS)

// indices of where each component starts and ends
#define DIFFUSE_INDEX	0
#define ALPHA_INDEX	(DIFFUSE_INDEX + 3)
#define SHININESS_INDEX	(ALPHA_INDEX + 1)
#define AMBIENT_INDEX	(SHININESS_INDEX + 1)
#define SPECULAR_INDEX	(AMBIENT_INDEX + 3)
#define COLOR_ITEMS	(SPECULAR_INDEX + 3)

// which one of these has the non-material color?
#define COLOR_INDEX	DIFFUSE_INDEX

// regular (visible) colors
#define REGBLUE		0
#define REGRED		1
#define REGGREY		2
#define REGORANGE	3
#define REGYELLOW	4
#define REGTAN		5
#define REGSILVER	6
#define REGGREEN	7
#define REGWHITE	8
#define REGPINK		9
#define REGCYAN		10
#define REGPURPLE	11
#define REGLIME		12
#define REGMAUVRE	13
#define REGOCHRE	14
#define REGICEBLUE	15

// extra (possibly non-visible) colors
#define REGBLACK	16

// macro to get regular and translucent color
#define REGCOLOR(a)		(a + BEGREGCLRS)
#define TRANSLUCENT(a)		(a + BEGTRAN)

// macros to get colormap colors, and translucent versions
#define MAPCOLOR(a)		(a + BEGMAP)
#define MAPTRANSLUCENT(a)	(a + BEGTRANMAP)


class ColorList : public Displayable3D {

public:
  // different color scale methods
  enum ColorScaleMethod { RGB, BGR, RWB, BWR, 
			  RWG, GWR, GWB, BWG,
			  BLK_W, W_BLK, SCALE_TOTAL };

  // list of color names, stored as a name list.  Also stores numbers
  // kept as strings. (i.e. "15" for the 15th color)
  NameList<int> colorNames;

private:
  // command to define a color ... used by create_cmdlist
  DispCmdColorDef *dispCmdColorDef;

  // does the color need to be defined?
  int needDefine[MAXCOLORS];
  
  // do any colors need to be defined?
  int needAnyDefined;
  
  // do we need to clean up after a draw where colors were defined?
  int needDefineCleanup;

  // data for each color
  float colorData[MAXCOLORS][COLOR_ITEMS];

  // current color scale method, max/min values, and color scale midpoint
  int scaleMethod;
  float scaleMin, scaleMax, scaleMidpoint;

  // create the color scale based on current settings
  void create_colorscale();

  // regenerate the command list
  void create_cmdlist();

  // list of color category names, and associate namelist with color data
  NameList<NameListIntPtr> categories;
  
public:
  // constructor: the Scene to register with
  ColorList(Scene *);
  virtual ~ColorList(void);
  
  //
  // routines to add new color categories and named colors
  //
  
  // add new color category; return it's index (may already have been added)
  int add_color_category(char *);

  // return Nth color category; assumes a is a valid index.
  NameListIntPtr color_category(int a) {
    return categories.data(a);
  }

  // return the whole set of color categories
  NameList<NameListIntPtr> &color_categories(void) { return categories; }

  //
  // retrieve data about the Nth color
  //
  
  float *color_data(int n) {
    float *retval = NULL;
    if(n >= 0 && n < MAXCOLORS)
      retval = colorData[n];
    return retval;
  }
  
  float *color(int n) {
    float *retval = NULL;
    if(n >= 0 && n < MAXCOLORS)
      retval = colorData[n] + COLOR_INDEX;
    return retval;
  }
  
  //
  // change data for the Nth color
  //
  
  void change_alpha(int, float);
  void change_shininess(int, float);
  void change_ambient(int, float, float, float);
  void change_specular(int, float, float, float);
  void change_diffuse(int, float, float, float);
  void change_color(int, float, float, float);
  void reset_color(int);			// restores Nth color to def

  //
  // get/change settings for the color scale
  //
  
  int scale_method(void) { return scaleMethod; }
  void scale_method(int nmeth) { scaleMethod = nmeth; create_colorscale(); }
  
  float scale_min(void) { return scaleMin; }
  void scale_min(float nv) { scaleMin = nv; create_colorscale(); }
  
  float scale_max(void) { return scaleMax; }
  void scale_max(float nv) { scaleMax = nv; create_colorscale(); }
  
  float scale_midpoint(void) { return scaleMidpoint; }
  void scale_midpoint(float nv) { scaleMidpoint = nv; create_colorscale(); }

  //  find the current closest color index (in the least sq. sense)
  //
  int nearest_index(float r, float g, float b);

  //
  // public virtual routines
  //
  
  // prepare for drawing ... do any updates needed right before draw.
  virtual void prepare(DisplayDevice *);
};

// globally-accessible strings with methods for creating color scale
extern char *colorScaleMethod[ColorList::SCALE_TOTAL];

#endif

