/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdExternal.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.1 $	$Date: 95/05/11 22:06:28 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Commands to allow the user to customize the pop-up menu.
 *
 ***************************************************************************/

#include "CmdUser.h"
#include "Mouse.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

// The following use USER_MENU_ADD, USER_MENU_LIST

//
// static items for this module
//

// strings mapping menu item name --> code
char *menu_item_names[] = {
	"menu", "item", "separator", "subitem", "subseparator", "key" };

int menu_item_codes[] = { CmdUserMenuAdd::MENU_MENU,
	CmdUserMenuAdd::MENU_ITEM,	CmdUserMenuAdd::MENU_SEP,
	CmdUserMenuAdd::MENU_SUBITEM,	CmdUserMenuAdd::MENU_SUBSEP,
	CmdUserMenuAdd::MENU_KEY };

int menu_item_args[] = { 1, 2, 0, 3, 1, 2 };

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'user'; return TRUE if an error occurs.
int text_cmd_user(int argc, char **argv, CommandQueue *cmdQueue, int id) {
  if(argc < 3)
    return TRUE;

  if(!strupncmp(argv[1], "add", CMDLEN)) {
    // add a new item
    if(!strupncmp(argv[2], "menu", CMDLEN) ||
		!strupncmp(argv[2], "subseparator", CMDLEN)) {
      if(argc != 4)
        return TRUE;
      cmdQueue->append(new CmdUserMenuAdd(argv[2], argv[3], id));
    } else if(!strupncmp(argv[2], "item", CMDLEN) ||
		!strupncmp(argv[2], "key", CMDLEN)) {
      if(argc < 5)
        return TRUE;
      char *combstr = combine_arguments(argc, argv, 4);
      cmdQueue->append(new CmdUserMenuAdd(argv[2], argv[3], combstr, id));
      delete [] combstr;
    } else if(!strupncmp(argv[2], "separator", CMDLEN)) {
      if(argc != 3)
        return TRUE;
      cmdQueue->append(new CmdUserMenuAdd(argv[2], id));
    } else if(!strupncmp(argv[2], "subitem", CMDLEN)) {
      if(argc < 6)
        return TRUE;
      char *combstr = combine_arguments(argc, argv, 5);
      cmdQueue->append(new CmdUserMenuAdd(argv[2], argv[3], argv[4],
				combstr, id));
      delete [] combstr;
    }  else
      return TRUE;

  } else if(!strupncmp(argv[1], "print", CMDLEN)) {
    // print out definitions of current items
    if(argc != 3)
      return TRUE;
    cmdQueue->append(new CmdUserMenuPrint(argv[2], id));

  } else
    return TRUE;
    
  // if here, everything worked out ok
  return FALSE;
}


//////////////////////  utility functions  ///////////////////////

// given a menu addition type as a string, return's its code or (-1) if error
int check_menu_item(char *str) {
  for(int i=0; i < CmdUserMenuAdd::MENU_ITEM_TOTAL; i++)
    if(!strupncmp(str, menu_item_names[i], CMDLEN))
      return menu_item_codes[i];
  
  // if here, item was not found
  msgErr << "Unknown menu item type '" << str << "' specified." << sendmsg;
  return (-1);
}


// given a menu addition type code and a number of arguments, makes sure
// the number of arguments is correct.  Returns code for item type if OK, or
// (-1) if error.
int check_menu_argnum(char *typ, int argc) {
  // first check for a proper code
  int typcode = check_menu_item(typ);

  // then make sure the number of args agrees
  if(typcode >= 0) {
    if(menu_item_args[typcode] != argc) {
      msgErr <<"Incorrect number of arguments for item type '" << typ << "'.";
      msgErr << sendmsg;
      typcode = (-1);
    }
  }

  return typcode;
}


////////////////////// add a new user menu item
int CmdUserMenuAdd::do_execute(void) {
  int retval;
  if((retval = (mouse != NULL && itemType >= 0))) {
    if(itemType == MENU_MENU)
      retval = mouse->create_user_menu(menuName);
    else if(itemType == MENU_ITEM)
      retval = mouse->add_user_menu_item(menuName, itemCmd);
    else if(itemType == MENU_SEP)
      retval = mouse->add_user_menu_separator();
    else if(itemType == MENU_SUBITEM)
      retval = mouse->add_user_submenu_item(menuName, subItemName, itemCmd);
    else if(itemType == MENU_SUBSEP)
      retval = mouse->add_user_submenu_separator(menuName);
    else if(itemType == MENU_KEY)
      retval = mouse->add_user_key_command(*menuName, itemCmd);
    else
      retval = FALSE;
  }
  return retval;
}

void CmdUserMenuAdd::create_text(void) {
  if(itemType >= 0) {
    *cmdText << "user add " << menu_item_names[itemType];
    if(itemType != MENU_SEP) {
      *cmdText << " " << menuName;
      if(itemType == MENU_SUBITEM)
        *cmdText << " " << subItemName << " " << itemCmd;
      else if(itemType == MENU_ITEM || itemType == MENU_KEY)
        *cmdText << " " << itemCmd;
    }
  }
  *cmdText << ends;
}

CmdUserMenuAdd::CmdUserMenuAdd(char *typ, int fromUIid)
	: Command(Command::USER_MENU_ADD, fromUIid) {
  itemType = check_menu_argnum(typ, 0);
  menuName = subItemName = itemCmd = NULL;
}

CmdUserMenuAdd::CmdUserMenuAdd(char *typ, char *s1, int fromUIid)
	: Command(Command::USER_MENU_ADD, fromUIid) {
  itemType = check_menu_argnum(typ, 1);
  menuName = subItemName = itemCmd = NULL;
  if(itemType >= 0) {
    menuName = stringdup(s1);
  }
}

CmdUserMenuAdd::CmdUserMenuAdd(char *typ, char *s1, char *s2, int fromUIid)
	: Command(Command::USER_MENU_ADD, fromUIid) {
  itemType = check_menu_argnum(typ, 2);
  menuName = subItemName = itemCmd = NULL;
  if(itemType >= 0) {
    menuName = stringdup(s1);
    itemCmd = stringdup(s2);
  }
}

CmdUserMenuAdd::CmdUserMenuAdd(char *typ, char *s1, char *s2, char *s3,
				int fromUIid)
	: Command(Command::USER_MENU_ADD, fromUIid) {
  itemType = check_menu_argnum(typ, 3);
  menuName = subItemName = itemCmd = NULL;
  if(itemType >= 0) {
    menuName = stringdup(s1);
    subItemName = stringdup(s2);
    itemCmd = stringdup(s3);
  }
}

CmdUserMenuAdd::~CmdUserMenuAdd(void) {
  if(menuName)		delete [] menuName;
  if(subItemName)	delete [] subItemName;
  if(itemCmd)		delete [] itemCmd;
}


////////////////////// add a new user keystroke command
int CmdUserMenuPrint::do_execute(void) {
  int retval;
  if((retval = (mouse != NULL))) {
    if(whichItems == MENU_DATA)
      retval = mouse->print_user_menu(&msgInfo);
    else if(whichItems == KEY_DATA)
      retval = mouse->print_user_keys(&msgInfo);
    else
      retval = FALSE;
  }
  return retval;
}

void CmdUserMenuPrint::create_text(void) {
  if(whichItems >= 0 && whichItems < PRINT_ITEM_TOTAL) {
    *cmdText << "user print " << (whichItems == MENU_DATA ? "menus" : "keys");
    *cmdText << ends;
  }
}

CmdUserMenuPrint::CmdUserMenuPrint(char *wh, int fromUIid)
	: Command(Command::USER_MENU_LIST, fromUIid) {
  if(!strupncmp(wh, "menus", CMDLEN))
    whichItems = MENU_DATA;
  else if(!strupncmp(wh, "keys", CMDLEN))
    whichItems = KEY_DATA;
  else {
    msgErr << "Unknown user-defined command type '" << wh << "'." << sendmsg;
    whichItems = (-1);
  }
}


