/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Axes.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.14 $	$Date: 95/11/15 04:29:29 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A Displayable3D object which consists of a set of axes, which may be
 * drawn anywhere on the screen, at any size.
 *
 ***************************************************************************/

#include <string.h>
#include "Axes.h"
#include "DisplayDevice.h"
#include "Scene.h"
#include "DispCmds.h"
#include "ColorList.h"
#include "PopupMenu.h"
#include "Inform.h"
#include "utilities.h"


// radius for the cylinders that make up the axes
#define AXESRAD		0.12
#define AXESCAPRAD	0.25
#define AXESRODLEN	0.85
#define AXESTXTLEN	1.10
#define AXESRES		12

// default colors to use
#define AXESXCOL	REGRED
#define AXESYCOL	REGGREEN
#define AXESZCOL	REGBLUE
#define AXESOCOL	REGCYAN
#define AXESTCOL	REGWHITE


// string descriptions of axes locations
static char *axesloc[Axes::AXESPOS_TOTAL] = {
	"Off", "Origin", "LowerLeft", "LowerRight",
	"UpperLeft", "UpperRight"};


//////////////////////////  constructor
Axes::Axes(Scene *sc) : Displayable3D(MULT, "Axes", sc, 1),
	 xtxt("x"), ytxt("y"), ztxt("z") {

  // indicate we don't yet have a color object to use
  colorCat = (-1);
  need_create_cmdlist = TRUE;
  movedAxes = FALSE;

  // initialize coordinates for axes lines
  origin[0] = yLine[0] = zLine[0] = 0.0;  
  xLine[0] = AXESRODLEN; xLineCap[0] = 1.0;

  origin[1] = xLine[1] = zLine[1] = 0.0;  
  yLine[1] = AXESRODLEN; yLineCap[1] = 1.0;

  origin[2] = xLine[2] = yLine[2] = 0.0;
  zLine[2] = AXESRODLEN; zLineCap[2] = 1.0;

  xText[0] = AXESTXTLEN * xLine[0]; xText[1] = xLine[1]; xText[2] = xLine[2];
  yText[1] = AXESTXTLEN * yLine[1]; yText[0] = yLine[0]; yText[2] = yLine[2];
  zText[2] = AXESTXTLEN * zLine[2]; zText[0] = zLine[0]; zText[1] = zLine[1];
  
  // Displayable characteristics
  rot_on();
  glob_trans_off();
  cent_trans_off();
  
  // set scaling factor to a small amount
  scale_on();
  set_scale(0.25);
  scale_off();
  
  // put axes in lower left corner by default
  axesPos = AXES_LOWERLEFT;
  Aspect = (-1.0);
  
  // register as a pickable object
  register_with_picklist(sc);

}


//////////////////////////  destructor
Axes::~Axes(void) {
  // do nothing
}


//////////////////////////  protected virtual routines  ////////////////////

// do action when a new color list is provided
// This creates a color category for use by the Axes, with the colors for
// the components.  These colors can then be edited by the user.
void Axes::do_use_colors(void) {
  // add new category (or just get it's index if it exists)
  colorCat = colorList->add_color_category("Axes");
  
  // add components, and their default colors
  (colorList->color_category(colorCat))->add_name("X", AXESXCOL);
  (colorList->color_category(colorCat))->add_name("Y", AXESYCOL);
  (colorList->color_category(colorCat))->add_name("Z", AXESZCOL);
  (colorList->color_category(colorCat))->add_name("Origin", AXESOCOL);
  (colorList->color_category(colorCat))->add_name("Labels", AXESTCOL);

  // indicate we need to recreate the command list
  need_create_cmdlist = TRUE;
}

// do action due to the fact that a color for the given ColorList for
// the specified category has changed
void Axes::do_color_changed(ColorList *changelist, int clr) {
  if(changelist == colorList && clr == colorCat) {

    // color changed for us, recreate command list
    need_create_cmdlist = TRUE;
  }
}

//////////////////////////  public virtual routines  ////////////////////

// create the drawing command list
void Axes::create_cmdlist(void) {
  int usecolors[5];

  MSGDEBUG(2,"Axes: creating command list ..." << sendmsg);

  reset_disp_list();

  // find colors to use
  if(colorCat >= 0) {
    usecolors[0] = (colorList->color_category(colorCat))->data("X");
    usecolors[1] = (colorList->color_category(colorCat))->data("Y");
    usecolors[2] = (colorList->color_category(colorCat))->data("Z");
    usecolors[3] = (colorList->color_category(colorCat))->data("Origin");
    usecolors[4] = (colorList->color_category(colorCat))->data("Labels");
  } else {
    usecolors[0] = AXESXCOL;
    usecolors[1] = AXESYCOL;
    usecolors[2] = AXESZCOL;
    usecolors[3] = AXESOCOL;
    usecolors[4] = AXESTCOL;
  }

  // turn on material characteristics
  cmdMaterials.putdata(TRUE, this);

  // set sphere type and resolution
  sphres.putdata(AXESRES,  this);
  sphtype.putdata(SOLIDSPHERE, this);

  // put in commands to draw lines
  // x-axis
  xcol.putdata(REGCOLOR(usecolors[0]), this);
  xcyl.putdata(origin, xLine, AXESRAD, AXESRES, this);
  xcap.putdata(xLine, xLineCap, AXESCAPRAD, AXESRES, this);
  pickPoint.putdata(xLine, 1, this);
  pickPoint.putdata(xLineCap, 2, this);

  // y-axis
  xcol.putdata(REGCOLOR(usecolors[1]), this);
  xcyl.putdata(origin, yLine, AXESRAD, AXESRES, this);
  xcap.putdata(yLine, yLineCap, AXESCAPRAD, AXESRES, this);
  pickPoint.putdata(yLine, 3, this);
  pickPoint.putdata(yLineCap, 4, this);

  // z-axis
  xcol.putdata(REGCOLOR(usecolors[2]), this);
  xcyl.putdata(origin, zLine, AXESRAD, AXESRES, this);
  xcap.putdata(zLine, zLineCap, AXESCAPRAD, AXESRES, this);
  pickPoint.putdata(zLine, 5, this);
  pickPoint.putdata(zLineCap, 6, this);

  // put in commands to draw point at origin
  xcol.putdata(REGCOLOR(usecolors[3]), this);
  sph.putdata(origin, AXESRAD, this);
  pickPoint.putdata(origin, 0, this);

  // turn off material characteristics
  cmdMaterials.putdata(FALSE, this);
  xcol.putdata(REGCOLOR(usecolors[4]), this);

  // put in commands to label the axes
  txtpos.putdata(xText, this);
  xtxt.put(this);
  txtpos.putdata(yText, this);
  ytxt.put(this);
  txtpos.putdata(zText, this);
  ztxt.put(this);

  need_create_cmdlist = FALSE;
}


// set axes display mode; return success
int Axes::location(int ap) {

  axesPos = ap;
  movedAxes = FALSE;
  if(ap == NO_AXES) {
    off();
  } else {
    on();
    Aspect = (-1.0);
  }

  return TRUE;
}


// return descripton of location
char *Axes::loc_description(int ap) {
  return axesloc[ap];
}


//////////////////  public virtual Displayable routines  ////////////////////

// routine to prepare the displayable object; must set the origin properly
void Axes::prepare(DisplayDevice *display) {
  float asp, xpos, ypos;
  float poscale = 0.95;

  // recreate command list if needed
  if(need_create_cmdlist)
    create_cmdlist();

  if (axesPos == NO_AXES || movedAxes) {  // don't do nuthin'
    return;
  }
  
  if((asp = display->aspect()) != Aspect) {
    // move the axes to their proper position
    if(axesPos == AXES_LOWERLEFT) {
      xpos = -poscale * asp;
      ypos = -poscale;
    } else if(axesPos == AXES_LOWERRIGHT) {
      xpos = poscale * asp;
      ypos = -poscale;
    } else if(axesPos == AXES_UPPERLEFT) {
      xpos = -poscale * asp;
      ypos = poscale;
    } else if(axesPos == AXES_UPPERRIGHT) {
      xpos = poscale * asp;
      ypos = poscale;
    } else {
      xpos = ypos = 0.0;
    }

    // update the current transformation
    MSGDEBUG(2,"Axes: changing axes position to " << xpos << ", " << ypos);
    MSGDEBUG(2," for aspect ratio " << asp << sendmsg);

    Aspect = asp;
    glob_trans_on();
    set_glob_trans(xpos, ypos, 0.0);
    glob_trans_off();
  }
}


//////////////////  public virtual Pickable routines  ////////////////////


// Create a popup menu, with the given name; return NULL if none available
// subclasses should have the parent first create a menu, and then add items
// to the end of it.  The argument is the tag of the selected item, it is
// ignored here.
PopupMenu *Axes::create_popup_menu(int tag) {

  // get (maybe create) a specialized popup menu
  PopupMenu *pm = Displayable::create_popup_menu(tag);
  if(!pm) {
    pm = new PopupMenu(name);
  }

  // add items to the menu
  for(int i=0; i < locations(); i++) {
    char cmdbuf[64];
    strcpy(cmdbuf, "axes location ");
    strcat(cmdbuf, loc_description(i));
    pm->add_item(loc_description(i), cmdbuf, TRUE, TRUE, location() == i);
  }

  // return the new menu
  return pm;
}


// return if we are interested in the given pick mode or not ... here, we
// are interested in ALL modes
int Axes::want_pick_mode(int) {
  return TRUE;
}


//
// When the Axes are picked and moved with a pointer, this is used to move
// the Axes to a new position.  The initial pointer position is remembered,
// and subsequent motions add a global translation to the Axes.
// This is done for any pick mode, and any button.  This is only done if the
// item selected is actually the specific Axes object.
//

// called when a pick moves:
//	args = display to use, obj picked, button, mode, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
// For the Axes, when they are selected and the pointer moves, we wish
// to move the axes as well.
void Axes::pick_move(DisplayDevice *d, Pickable *p,
  				int, int, int tag, int dim, float *pos) {
  float moveAxesOrigPos[3], newAxesOrigPos[3], *newpos;

  // if p is the same as this object, move the axes in the X-Y plane by the
  // required amount
  if(p == this && tag == 0) {
    
    // calculate amount to translate axes
    if(dim == 2) {
      tm.multpoint3d(origin, moveAxesOrigPos);
      d->find_3D_from_2D(moveAxesOrigPos, pos, newAxesOrigPos);
      newpos = newAxesOrigPos;
    } else {
      newpos = pos;
    }
    
    // apply transformation
    glob_trans_on();
    set_glob_trans(newpos[0], newpos[1], newpos[2]);
    glob_trans_off();
    
    movedAxes = TRUE;
  }
}


