/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Animation.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.8 $	$Date: 96/03/24 07:06:06 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Animation class, which stores a list of pointers to Timestep objects
 * that contain 3D coordinates and other data that varies with time.
 *
 ***************************************************************************/
#ifndef ANIMATION_H
#define ANIMATION_H

#include "ResizeArray.h"
#include "Timestep.h"

// animation insert position defines
#define APPEND_AT_BEG	(-1)
#define APPEND_AT_END	(-2)

// speed factor: maximum fraction of a second between redraws
#define SPEED_FACTOR	0.5

// number of frames to initially allocate space for
#define INITIAL_FRAME_COUNT	1024


class Animation {

public:
  // enums for Animation options
  enum AnimDir 	{ FORWARD, FORWARD1, REVERSE, REVERSE1, PAUSE };
  enum AnimPos 	{ END, START, BEFORE, AFTER, DELETE, WRITE };
  enum AnimFrames	{ ALL, SELECTION };	// frames to append/del/write
  enum AnimStyle	{ ONCE, LOOP, ROCK, TOTAL_STYLES };

private:
  // last time the image was drawn, for use with determining speed
  double lastTime;

  // has the current frame changed in any way recently?
  int frameChanged;

  // flag saying whether we should set the frameChanged setting
  int needSetFrameChanged;

  // storage for the frames ... a resizeable array of Timestep pointers
  ResizeArray<Timestep *> tsList;

  // set the current frame position to a new value
  void change_current_frame(int n, int setChangeFlag = TRUE) {
    if (n < -1 || n >= tsList.num()) {
      msgErr << "animation request for frame " << n
	     << "outside of range 0(-1) to " << tsList.num() - 1 << sendmsg;
    } else {
      currFrame = n;
      if(setChangeFlag)
	curr_frame_changed();
    }
  }

protected:
  // current frame number (0 ... # frames - 1)
  int currFrame;
  
  // frames to skip to next position when animating
  int frameSkip;

  // animation speed, from 0.0 (slowest) to 1.0 (fastest)
  float Speed;

  // current method for appending frames
  AnimPos appendPos;

  // frame to append before/after, if method is an insert method
  int appendFrame;

  // current animation direction
  AnimDir animDir;
  
  // what to do when you get to the end of the loop
  AnimStyle animStyle;

  // frame to which the animation should be moved to, if a jump is requested
  // (no jump if moveTo < 0)
  int moveTo;

  // current 'time' elapsed since beginning of animation
  float currTime;

public:
  // class constructor/destructor
  Animation(void);
  virtual ~Animation(void);
  
  // total number of frames currently stored
  int num(void) { return tsList.num(); }

  // return current time
  float time(void) { return currTime; }

  // return the current frame number (frames 0...(frames -1); -1 => no frames)
  int frame(void) { return currFrame; }

  // return whether there is a 'current' frame (i.e. are any frames present)
  int is_current(void) { return (num() > 0 && currFrame >= 0); }

  // return 'current' frame, if available; NULL otherwise
  Timestep *current(void) {
    Timestep *retval = NULL;
    if(frame() >= 0)
      retval = tsList[frame()];
    return retval;
  }

  // return Nth frame, referred to here as an 'item'
  Timestep *item(int n) {
    Timestep *retval = NULL;
    if(n >= 0 && n < num())
      retval = tsList[n];
    return retval;
  }
  
  // explicitely move the current frame to the given one.  This does NOT
  // update anything else, so it should only be used when temporarily
  // changing the value to be returned by current().
  // There are more error message here than change_current_frame since
  // this is the one available to the Tcl 'molinfo' command
  void override_current_frame(int n) {
    if (tsList.num() == 0) {
      msgErr << "no animation data; cannot go to frame " << n
	     << sendmsg;
    } else {
      if (n < 0) {
	msgErr << "animation request for frame " << n
	       << " is less than 0; setting to 0" << sendmsg;
	currFrame = 0;
      } else if (n >= tsList.num()) {
	msgErr << "animation request for frame " << n
	       << " is larger than the number of\navailable frames; "
	       << "setting to " << tsList.num() - 1 << sendmsg;
	currFrame = tsList.num()-1;
      } else {
	currFrame = n;
      }
    }
  }

  // signal that the current frame has changed somehow
  void curr_frame_changed(void) { needSetFrameChanged = TRUE; }

  // check if the curr frame has changed any
  int has_frame_changed(void) { return frameChanged; }

  // move to the specified frame, if possible
  void goto_frame(int fr) { moveTo = fr; }

  // delete all currently stored frames, or just a specific one
  void delete_animation(void);
  void delete_frame(int);

  // append a new frame and return its number
  int append_frame(Timestep *);

  // update the animation list based on current mode; return current frame.
  int anim_update(void);

  // set append method to append at the end of the list
  void append_end(void) { appendPos = END; }

  // set the append method to appending at beginning
  void append_start(void) { appendPos = START; }

  // set the append method to append after the specified frame
  void append_after(int fr) { appendPos = AFTER; appendFrame = fr; }

  // set the append method to append before the specified frame
  void append_before(int fr) { appendPos = BEFORE; appendFrame = fr; }
  
  // set the # of frames to skip each step; returns new value
  void skip(int newsk) { frameSkip = ( newsk >= 1 ? newsk : 1); }

  // return the # frames to skip each step
  int skip(void) { return frameSkip; }
  
  // get/set the animation dir
  void anim_dir(AnimDir ad) { animDir = ad; }
  AnimDir anim_dir(void) { return animDir; }
  
  // get/set the animation style
  void anim_style(AnimStyle as) { animStyle = as; }
  AnimStyle anim_style(void) { return animStyle; }

  // animation speed methods
  //	newsp should be from 0 (min speed) to 1 (max speed)
  float speed(float newsp) {
    if(newsp < 0.0)
      Speed = 0.0;
    else if(newsp > 1.0)
      Speed = SPEED_FACTOR;
    else
      Speed = newsp*SPEED_FACTOR;
    return Speed;
  }

  float speed(void) { return Speed/SPEED_FACTOR; }

};


// static storage for strings describing animation styles
extern char *animationStyleName[Animation::TOTAL_STYLES];

#endif

