# Yo Emacs this is  -*-Tcl-*-
# Last changed Time-stamp: <96/03/25 00:44:21 ivo>
# A script to save current viewpoint, representations etc to a file.
# By Ivo Hofacker <ivo@ks.uiuc.edu>
# Usage:
# In the vmd console type 
#       save_state foo
# to save your work in file foo. Restarting vmd as
#       vmd -e foo
# should restore most of your previous session.
#
# Essentially everything that can be changed from the graphics and mol and
# color forms will be saved, but nothing else.
#  
# In particular the script will restore
#     molecules read in from files
#     graphics objects (spheres, cylenders...)
#     all representations
#     the transformation matrices for each molecule
#     which color to use for which representation, definitions of colors 0..33
# It will not restore:
#     trajectories loaded via the edit menu
#     labels
#     interactive MD sessions
#     anything in the display form, such as lighting, axes, stage, ...


proc save_viewpoint {} {
   global viewpoints
   if [info exists viewpoints] {unset viewpoints}
   # get the current matricies
   foreach mol [molinfo list] {
      set viewpoints($mol) [molinfo $mol get {
        center_matrix rotate_matrix scale_matrix global_matrix}]
   }
}
proc restore_viewpoint {} {
   global viewpoints
   foreach mol [molinfo list] {
      if [info exists viewpoints($mol)] {
         molinfo $mol set {center_matrix rotate_matrix scale_matrix
           global_matrix} $viewpoints($mol)
      }
   }
}

proc save_reps {} {
    global representations
    foreach mol [molinfo list] {
	set representations($mol) ""
	for {set i 0} {$i < [molinfo $mol get numreps]} {incr i} {
	    lappend representations($mol) \
		[molinfo $mol get "{rep $i} {selection $i} {color $i}"]
	}
    }
}

proc restore_reps {} {
    global representations
    foreach mol [molinfo list] {
	if [info exists representations($mol)] {
	    #delete current representations
	    for {set i 0} {$i < [molinfo $mol get numreps]} {incr i} {
		mol delrep $i $mol
	    }
	    #restore saved representations
	    foreach rep $representations($mol) {
		puts $rep
		lassign $rep r s c
		mol representation $r
		mol color $c
		mol selection $s
		mol addrep $mol
	    }
	}
    }
}
	    
proc save_state {file} {
    global representations
    global viewpoints
    save_viewpoint
    save_reps
    
    set fildes [open $file w]
    puts $fildes "\#!/usr/local/bin/vmd"
    puts $fildes "\# vmd script saved by save_state"
    puts $fildes "set viewplist {}"
    puts $fildes "set fixedlist {}"
    foreach mol [molinfo list] {
	# find source file for coordinates if any
	if {[molinfo $mol get source] == "File"} {
	    set sfile1 [molinfo $mol get {filetype  filename}]
	    set sfile2 [molinfo $mol get {filetype2 filename2}]
	    if [lsearch -exact $sfile2 {}] {
		puts $fildes "mol load $sfile1 $sfile2"
	    } else {
		puts $fildes "mol load $sfile1"
	    }
	    puts $fildes "mol delrep 0 top"
	    if [info exists representations($mol)] {
		foreach rep $representations($mol) {
		    lassign $rep r s c
		    puts $fildes "mol representation $r"
		    puts $fildes "mol color $c"
		    puts $fildes "mol selection {$s}"
		    puts $fildes "mol addrep top"
		}
	    }
	} elseif { [molinfo $mol get source] == "Graphics"} {
	    puts $fildes "mol load graphics [molinfo $mol get name]"
	    puts $fildes "mol top \[lindex \[molinfo list\] end\]"
	    foreach g [graphics $mol list] {
		puts $fildes "graphics top [graphics $mol info $g]"
	    }
	} else {
	    puts $fildes "\# can't save molecule $mol, source=[molinfo $mol get source]"
	    continue
	}
	if {[molinfo $mol get drawn] == 0} {
	    puts $fildes "molinfo top set drawn 0"
	}
	if {[molinfo $mol get active] == 0} {
	    puts $fildes "molinfo top set active 0"
	}
	if {[molinfo $mol get fixed] == 1} {
	    puts $fildes "lappend fixedlist \[molinfo top\]"
	}
	
	puts $fildes "set viewpoints(\[molinfo top\]) [list $viewpoints($mol)]"
	puts $fildes "lappend viewplist \[molinfo top\]"
	if {$mol == [molinfo top]} {
	    puts $fildes "set topmol \[molinfo top\]"
	}
	puts $fildes "\# done with molecule $mol"
    }
    puts $fildes "foreach v \$viewplist \{"
    puts $fildes "  molinfo \$v set {center_matrix rotate_matrix scale_matrix
global_matrix} \$viewpoints(\$v)"
    puts $fildes "\}"
    puts $fildes "foreach v \$fixedlist \{"
    puts $fildes "  molinfo \$v set fixed 1"
    puts $fildes "\}"
    puts $fildes "unset viewplist"
    puts $fildes "unset fixedlist"
    puts $fildes "mol top \$topmol"
    puts $fildes "unset topmol"
    save_colors $fildes
    close $fildes
}

# an up-to-date list of color categories can be produced using
#    foreach c [colorinfo categories] {
#	foreach cc [colorinfo category $c] {
#	    lappend colcatlist "$c,$cc" [colorinfo category $c $cc]
#	}
#    }
# the list of default rgb values is generated by
# for {set c 0} {$c < 2*[colorinfo num]} {incr c} {
#    lappend def_rgb [colorinfo rgb $c]
# }

proc save_colors {fildes} {
    set colcatlist { Display,Background black
	Axes,X red Axes,Y green Axes,Z blue Axes,Origin cyan Axes,Labels white
	Stage,Even blue Stage,Odd green
	Labels,Dihedrals cyan Labels,Angles yellow Labels,Bonds white
	Labels,Atoms green
	Name,H white Name,O red Name,N blue Name,C cyan Name,S yellow Name,P
	tan Name,Z silver
	Type,H white Type,O red Type,N blue Type,C cyan Type,S yellow Type,P
	tan Type,Z silver
	Resname,ALA blue Resname,ARG white Resname,ASN tan Resname,ASP red 
	Resname,CYS yellow Resname,GLY white Resname,GLU pink Resname,GLN
	orange Resname,HIS cyan Resname,ILE green Resname,LEU pink
	Resname,LYS cyan Resname,MET yellow Resname,PHE purple Resname,PRO
	ochre Resname,SER yellow Resname,THR mauve Resname,TRP silver
	Resname,TYR green Resname,VAL tan Resname,ADE blue Resname,CYT orange
	Resname,GUA yellow Resname,THY purple Resname,URA green Resname,TIP
	cyan Resname,TIP3 cyan Resname,WAT cyan Resname,SOL cyan Resname,H2O
	cyan Resname,LYR purple Resname,ZN silver Resname,NA yellow
	Resname,CL green 
	Restype,Unassigned cyan Restype,Solvent yellow Restype,Nucleic_Acid
	purple Restype,Basic blue Restype,Acidic red Restype,Polar green
	Restype,Nonpolar white Restype,Ion tan
	Highlight,Proback green Highlight,Nucback yellow Highlight,Nonback 
	blue
	{Structure,Alpha Helix} purple Structure,3_10_Helix mauve
	Structure,Pi_Helix red Structure,Extended_Beta yellow
	Structure,Bridge_Beta tan Structure,Turn cyan Structure,Coil white
    }

    set def_rgb {
	{0.25 0.25 1.0} {1.0 0.0 0.0} {0.35 0.35 0.35} {0.8 0.5 0.2}
	{0.8 0.8 0.0} {0.5 0.5 0.2} {0.6 0.6 0.6} {0.2 0.7 0.2} {1.0 1.0 1.0}
	{1.0 0.6 0.6} {0.25 0.75 0.75} {0.65 0.3 0.65} {0.5 0.9 0.4}
	{0.9 0.4 0.7} {0.5 0.3 0.0} {0.5 0.75 0.75} {0.0 0.0 0.0}
    } 
    
    set def_alpha 1.0
    set def_alphat 0.3
    set def_shiny 40.0
    set def_ambient {0.0 0.0 0.0}
    set def_specular {1.0 1.0 1.0}
    
    array set def_colcat $colcatlist

    foreach c [colorinfo categories] {
	foreach cc [colorinfo category $c] {
	    set col [colorinfo category $c $cc]
	    if {![info exists def_colcat($c,$cc)] ||
		[string compare $col $def_colcat($c,$cc)]} {
		puts $fildes "color $c \{$cc\} $col"
	    }
	}
    }
    set cnum [colorinfo num]
    for {set c 0} {$c < 2*$cnum} {incr c} {
	if {$c >= $cnum} {
	    set c2 [expr $c - $cnum]
	    set def_a $def_alphat
	} else {
	    set c2 $c
	    set def_a $def_alpha
	}
	
	set rgb [colorinfo rgb $c]
	if  {[string compare $rgb [lindex $def_rgb $c2]]}  {
	    puts $fildes "color change rgb $c $rgb"
	}
	set alpha [colorinfo alpha $c]
	if {$alpha != $def_a} {
	    puts $fildes "color change alpha $c $alpha"
	}
	set shiny [colorinfo shininess $c]
	if {$shiny != $def_shiny} {
	    puts $fildes "color change shininess $c $shiny"
	}
	set ambient [colorinfo ambient $c]
	if {[string compare $ambient $def_ambient]} {
	    puts $fildes "color change ambient $c $ambient"
	}
	set specular [colorinfo specular $c]
	    if {[string compare $specular $def_specular]} {
	    puts $fildes "color change specular $c $specular"
	}
    }
}
