# A utility for safely running untrusted VMD scripts.

package provide sandbox 0.1

namespace eval ::Sandbox:: {
    # create a safe interpreter
    variable safe [interp create -safe]

    # allow the safe interp to print messages
    interp share {} stdout $safe

    # atomselections are fine except for "writepdb"
    variable safe_atomselection_cmds {
        num list text molid frame delete global uplevel get set
        getbonds move moveby lmoveby moveto lmoveto update
    }

    # Animate is unsafe because of "write" and "read"
    variable safe_animate_commands {
        dup forward for reverse rev pause prev next skip delete speed
        style styles goto
    }

    # label is OK except for graph, which takes a command string
    variable safe_label_commands {
        list add show hide delete addspring
    }

    # mol needs to be prevented from loading files
    # ("new","load","importgraphics")
    # Hence "cancel" is of no use...
    # pdbload and urlload are okay though
    variable safe_mol_commands {
        urlload pdbload list color representation selection modcolor
        modmaterial modstyle modselect addrep delrep modrep delete
        active inactive on off fix free top ssrecalc rename repname
        repindex selupdate material
    }

    # The VMD commands that should be totally safe
    variable safe_cmds {axes color colorinfo display echo quit exit
        help material measure menu light molinfo mouse rock rotate
        scale stage translate wait sleep graphics draw}
    foreach cmd $safe_cmds {
        $safe alias $cmd $cmd
    }

    # play, render are unsafe
    # no tool; we don't people destroying our haptic devices
    # user is unsafe because the script could set up a malicious event on "r"
    # vmdinfo is probably okay...but let's be safe and leave it out
    # imd is unneeded and possibly unsafe
    # logfile is unneeded, unsafe

    ################################## command aliases
    proc safe_animate { args } {
        variable safe_animate_commands
        test_args animate $safe_animate_commands $args
        eval [concat animate $args]
    }
    $safe alias animate ::Sandbox::safe_animate

    # label is OK except for graph, which takes a command string
    proc safe_label { args } {
        variable safe_label_commands
        test_args label $safe_label_commands $args
        eval [concat animate $args]
    }
    $safe alias label ::Sandbox::safe_label

    # mol needs to be prevented from loading files
    # ("new","load","importgraphics")
    # pdbload and urlload are okay though
    proc safe_mol { args } {
        variable safe_mol_commands
        test_args mol $safe_mol_commands $args
        eval [concat mol $args]
    }
    $safe alias mol ::Sandbox::safe_mol
    $safe alias molecule ::Sandbox::safe_mol

    # Atomselect is fine except for writepdb - this is a bit complicated
    proc safe_atomselect { args } {

        # get a sel and move it up one level to get out of this proc safely
        set sel [eval [concat atomselect $args]]
        $sel uplevel 1

        # make a new proc that will encapsulate this one
        proc ::Sandbox::safe_$sel { args } "
            global ::Sandbox::safe_atomselection_cmds
            test_args $sel \$safe_atomselection_cmds \$args
            eval \[concat $sel \$args\]
        "

        # return the safely encapsulated selection command
        variable safe
        $safe alias $sel ::Sandbox::safe_$sel
        return $sel
    }
    $safe alias atomselect ::Sandbox::safe_atomselect

    # make a command for starting a webserver
    $safe alias startServer ::Sandbox::startServer

    # make a command for loading pdbs on the web
    $safe alias newurlload ::Sandbox::newurlload

    ########################################
    # these should be the only commands necessary for package users
    namespace export safeSource
    namespace export startServer
}

# test_args(funcname,safe_args,myargs) returns only if the first
# member of myarg is found in safe_args or myargs has length 0.
# Otherwise, it causes an unsafe error.
proc ::Sandbox::test_args { funcname safe_args myargs } {
    if { [lsearch -exact $safe_args [lindex $myargs 0]] == -1 ||
         [llength $myargs] == 0} {
        unsafe $funcname $myargs
    }
}

# safely source a script
proc ::Sandbox::safeSource { script } {
    variable safe
    $safe invoke source $script
}
    

# a procedure for reporting errors
proc ::Sandbox::unsafe { args } {
    error "Unsafe command requested: $args"
}

# the accept command for startServer below
proc ::Sandbox::accept { channel address port } {
    # check for an allowed address
    if { $address != "127.0.0.1" } {
        puts -nonewline $channel "HTTP/1.0 403 Forbidden\n"
        puts -nonewline $channel "Content-type: text/html\n\n"
        puts $channel "Sorry, you must connect from the local machine."
        close $channel
        puts "Connection attempt from $address denied."
        return
    }

    # get the http request
    set request [gets $channel]

    # parse it
    set range 0
    set encoded_command ""
    set command ""
    regexp {GET\s+/?(\S+)} $request range encoded_command
    regsub -all {\%20} $encoded_command " " command
    puts $command

    # return a nice response
    puts -nonewline $channel "HTTP/1.0 200 OK\n"
    puts -nonewline  $channel "Content-type: text/html\n\n"
    puts $channel "Ran command: <code>$command</code>\n"
    flush $channel
    close $channel

    # run the command!
    variable safe
    $safe eval $command
}

# Sets up VMD as a webserver 
proc ::Sandbox::startServer { {port 2000} } {
    socket -server ::Sandbox::accept 2000
}

# This is a version of urlload that works under Windows
proc ::Sandbox::newurlload { type url } {
  global env
  global tcl_platform

  switch $tcl_platform(platform) {
    windows {
      set tmpfile [file join / vmd[pid]url.tcl]
    }
    default {
      set tmpfile [file join $env(TMPDIR) vmd[pid]url.tcl]
    }
  }     

  if {[string length $url] > 0} {
    puts "Initiating automatic download..."
    puts "URL: $url"
    ::Sandbox::vmdhttpcopy $url $tmpfile
    if {[file exists $tmpfile] > 0} {
      mol load $type $tmpfile
      file delete -force $tmpfile
    } else {
      puts "Failed to create temporary file."
    }
  }
}

# Copy a URL to a file
proc ::Sandbox::vmdhttpcopy { url file {chunk 4096} } {
  set out [open $file w]
  set token [::http::geturl $url -channel $out \
                 -progress ::Sandbox::vmdhttpProgress \
                 -blocksize $chunk]
  close $out
  # This ends the line started by http::Progress
  puts stderr ""
  upvar #0 $token state
  set max 0
  foreach {name value} $state(meta) {
    if {[string length $name] > $max} {
      set max [string length $name]
    }
    if {[regexp -nocase ^location$ $name]} {
      # Handle URL redirects
      puts stderr "Location:$value"
      return [copy [string trim $value] $file $chunk]
    }
  }
  incr max

  return $token
}

proc ::Sandbox::vmdhttpProgress {args} {
  puts -nonewline stderr . ; flush stderr
}
