#include "dock.h"

/************************************************/
//constructor
Bump_Filter::Bump_Filter()
{

    bump = NULL;

}
/************************************************/
//destructor
Bump_Filter::~Bump_Filter()
{

    delete[]bump;
    bump = NULL;

}
/************************************************/
//collect user defined parameters
void
Bump_Filter::input_parameters(Parameter_Reader & parm)
{
    string          tmp;

    cout << "\nBump Filter Parameters" << endl;
    cout <<
        "------------------------------------------------------------------------------------------"
        << endl;

    bump_filter = (parm.query_param("bump_filter", "yes", "yes no") ==
                 "yes") ? true : false;

    if (bump_filter) {
        file_prefix = parm.query_param("bump_grid_prefix", "grid");
    
        anchor_bump_max = atoi(parm.query_param("max_bumps_anchor", "2").c_str());
        if (anchor_bump_max <= 0) {
            cout <<
                "ERROR:  Parameter must be integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
	growth_bump_max = atoi(parm.query_param("max_bumps_growth", "2").c_str());
        if (growth_bump_max <= 0) {
            cout <<
                "ERROR:  Parameter must be integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
    }

}

/************************************************/
//call to read in data from pregenerated Bump Grid
//and calculate location of grid in space
void
Bump_Filter::initialize()
{

    if (bump_filter) {
        cout << "Initializing Bump Filter Routines..." << endl;
        read_bump_grid();
        calc_corner_coords();
        use_score = true;
    } else
        use_score = false;

}

/************************************************/
//read in dimensions of Bump Grid and bumps calculated
//for receptor by GRID accessory program
void
Bump_Filter::read_bump_grid()
{
    string          fname;

    fname = file_prefix + ".bmp";
    grid_in = fopen(fname.c_str(), "rb");

    if (grid_in == NULL) {
        cout << "\n\nCould not open " << fname <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    cout << " Reading the bump grid from " << fname << endl;

    fread(&size, sizeof(int), 1, grid_in);
    fread(&spacing, sizeof(float), 1, grid_in);
    fread(origin, sizeof(float), 3, grid_in);
    fread(span, sizeof(int), 3, grid_in);

    bump = new unsigned char[size];
    fread(bump, sizeof(unsigned char), size, grid_in);

    fclose(grid_in);

}

/************************************************/
//call to calculate number of bumps for anchor orientation
//if number of bumps exceeds user defined cutoff, do
//not allow anchor to continue
bool
Bump_Filter::check_anchor_bumps(DOCKMol & mol, bool more_orients)
{
   int             num_bumps;

   if(!more_orients)
	return true;

   if (bump_filter) {

       num_bumps = get_bump_score(mol);

       if((num_bumps <= anchor_bump_max)&&(num_bumps != -1))
	    return true;
       else
	    return false;
   } else
	    return true;
 
}
/************************************************/
//call to calculate number of bumps for grown conformation
//if number of bumps exceeds user defined cutoff, do
//not allow conformation to continue
bool
Bump_Filter::check_growth_bumps(DOCKMol & mol)
{
    int num_bumps = 0;

    if(bump_filter) {
        
       num_bumps = get_bump_score(mol);

       if((num_bumps <= growth_bump_max)&&(num_bumps != -1))
	    return true;
       else
	    return false;
   } else
	    return true;
}
/************************************************/
//calculate number of bumps
int
Bump_Filter::get_bump_score(DOCKMol & mol)
{
    int             bump_count = 0;
    int             vdw_threshold = 0;

    for (int atom = 0; atom < mol.num_atoms; atom++) {
        if (mol.atom_active_flags[atom] && mol.amber_at_heavy_flag[atom]) {
            if (check_box_boundaries(mol.x[atom], mol.y[atom], mol.z[atom])) {
                find_grid_neighbors(mol.x[atom], mol.y[atom], mol.z[atom]);
                vdw_threshold = NINT(10 * mol.amber_at_radius[atom]);

                if (vdw_threshold >= (int) bump[nearest_neighbor])
                    bump_count++;
            } else
                return -1;  //atom is outside the grid box
        }
    }

    return (bump_count);
}


/************************************************/
/************************************************/
/************************************************/
Energy_Score::Energy_Score()
{

    avdw = NULL;
    bvdw = NULL;
    es = NULL;

}
/************************************************/
Energy_Score::~Energy_Score()
{

    delete[]avdw;
    delete[]bvdw;
    delete[]es;

}
/************************************************/
void
Energy_Score::input_parameters(Parameter_Reader & parm, bool & primary_score,
                               bool & secondary_score)
{
    string          tmp;

    use_primary_score = false;
    use_secondary_score = false;

    cout << "\nGrid Score Parameters" << endl;
    cout <<
        "------------------------------------------------------------------------------------------"
        << endl;

    if (!primary_score) {
        tmp = parm.query_param("grid_score_primary", "yes", "yes no");
        if (tmp == "yes")
            use_primary_score = true;
        else
            use_primary_score = false;

        primary_score = use_primary_score;
    }

    if (!secondary_score) {
        tmp = parm.query_param("grid_score_secondary", "no", "yes no");
        if (tmp == "yes")
            use_secondary_score = true;
        else
            use_secondary_score = false;

        secondary_score = use_secondary_score;
    }

    if (use_primary_score || use_secondary_score)
        energy_score = 1;
    else
        energy_score = 0;

    if (energy_score == 1) {
        rep_radius_scale = atof(parm.query_param("grid_score_rep_rad_scale", "1").c_str());
        if (rep_radius_scale <= 0.0) {
                cout <<
                    "ERROR:  Parameter must be a float greater than zero. Program will terminate."
                    << endl;
                exit(0);
        }
        vdw_scale = atof(parm.query_param("grid_score_vdw_scale", "1").c_str());
        if (vdw_scale <= 0.0) {
            bool            off;
            off =
                (parm.query_param("grid_score_turn_off_vdw", "yes", "yes no") ==
                 "yes") ? true : false;
            if (!off) {
                cout <<
                    "ERROR:  Parameter must be a float greater than zero. Program will terminate."
                    << endl;
                exit(0);
            }
        }
        es_scale = atof(parm.query_param("grid_score_es_scale", "1").c_str());
        if (es_scale <= 0.0) {
            bool            off;
            off =
                (parm.query_param("grid_score_turn_off_es", "yes", "yes no") ==
                 "yes") ? true : false;
            if (!off) {
                cout <<
                    "ERROR:  Parameter must be a float greater than zero. Program will terminate."
                    << endl;
                exit(0);
            }
        }
	

        file_prefix = parm.query_param("grid_score_grid_prefix", "grid");
        use_score = true;
    }
}

/************************************************/
void
Energy_Score::initialize(AMBER_TYPER & typer)
{

    if (energy_score == 1) {

        cout << "Initializing Grid Score Routines..." << endl;

        read_nrg_grid();
	att_exp = 6;
	rep_exp = 12;
        init_vdw_energy(typer);
        calc_corner_coords();
        use_score = true;
    } else
        use_score = false;


}

/************************************************/
void
Energy_Score::read_nrg_grid()
{
    string          fname;

    fname = file_prefix + ".nrg";
    grid_in = fopen(fname.c_str(), "rb");

    if (grid_in == NULL) {
        cout << "\n\nCould not open " << fname <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    cout << " Reading the energy grid from " << fname << endl;

    fread(&size, sizeof(int), 1, grid_in);
    fread(&atom_model, sizeof(int), 1, grid_in);
    fread(&att_exp, sizeof(int), 1, grid_in);
    fread(&rep_exp, sizeof(int), 1, grid_in);

    avdw = new float[size];
    bvdw = new float[size];
    es = new float[size];

    fread((void *) bvdw, sizeof(float), size, grid_in);
    fread((void *) avdw, sizeof(float), size, grid_in);
    fread((void *) es, sizeof(float), size, grid_in);

    fclose(grid_in);

    fname = file_prefix + ".bmp";
    grid_in = fopen(fname.c_str(), "rb");

    if (grid_in == NULL) {
        cout << "\n\nCould not open " << fname <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    fread(&size, sizeof(int), 1, grid_in);
    fread(&spacing, sizeof(float), 1, grid_in);
    fread(origin, sizeof(float), 3, grid_in);
    fread(span, sizeof(int), 3, grid_in);

    fclose(grid_in);
}

/************************************************/
bool
Energy_Score::compute_score(DOCKMol & mol)
{
    float           vdw_val,
                    es_val,
                    total;
    int             atom;
    string          text;
    char            line[1000];


    total = vdw_val = es_val = 0.0;
    if (energy_score == 1) {

        // check to see if molecule is inside grid box
        for (atom = 0; atom < mol.num_atoms; atom++) {
            if (!check_box_boundaries(mol.x[atom], mol.y[atom], mol.z[atom])) {
                mol.current_score = -MIN_FLOAT;  // arbitrarily large score
                sprintf(line, "ERROR:  Conformation could not be scored by"
                    " DOCK.\nConformation not completely within grid box.\n");
                mol.current_data = line;
                return false;
            }
        }

        for (atom = 0; atom < mol.num_atoms; atom++) {

            if (mol.atom_active_flags[atom]) {

                find_grid_neighbors(mol.x[atom], mol.y[atom], mol.z[atom]);

                vdw_val +=
                    ((vdwA[mol.amber_at_id[atom]] * interpolate(avdw)) -
                     (vdwB[mol.amber_at_id[atom]] * interpolate(bvdw))) *
                    vdw_scale;
                es_val += mol.charges[atom] * interpolate(es) * es_scale;
            }


        }

        total = vdw_val + es_val;

        vdw_component = vdw_val;
        es_component = es_val;

        mol.current_score = total;
        mol.current_data = output_score_summary(total);

    }

    return true;
}

/************************************************/
string
Energy_Score::output_score_summary(float score)
{
    ostringstream text;

    if (use_score) {
        text << DELIMITER << setw(STRING_WIDTH) << "Grid Score:"
             << setw(FLOAT_WIDTH) << fixed << score << endl
             << DELIMITER << setw(STRING_WIDTH) << "vdw:"
             << setw(FLOAT_WIDTH) << fixed << vdw_component << endl
             << DELIMITER << setw(STRING_WIDTH) << "es:"
             << setw(FLOAT_WIDTH) << fixed << es_component << endl
        ;
    }
    return text.str();
}

/************************************************/
/************************************************/
/************************************************/
Continuous_Energy_Score::Continuous_Energy_Score()
{
}
/************************************************/
Continuous_Energy_Score::~Continuous_Energy_Score()
{
}
/************************************************/
void
Continuous_Energy_Score::input_parameters(Parameter_Reader & parm,
                                          bool & primary_score,
                                          bool & secondary_score)
{
    string          tmp;

    use_primary_score = false;
    use_secondary_score = false;

    cout << "\nContinuous Energy Score Parameters" << endl;
    cout <<
        "------------------------------------------------------------------------------------------"
        << endl;

    if (!primary_score) {
        tmp = parm.query_param("continuous_score_primary", "no", "yes no");
        if (tmp == "yes")
            use_primary_score = true;
        else
            use_primary_score = false;

        primary_score = use_primary_score;
    }

    if (!secondary_score) {
        tmp = parm.query_param("continuous_score_secondary", "no", "yes no");
        if (tmp == "yes")
            use_secondary_score = true;
        else
            use_secondary_score = false;

        secondary_score = use_secondary_score;
    }

    if (use_primary_score || use_secondary_score)
        use_score = true;
    else
        use_score = false;

    if (use_score) {
        receptor_filename =
            parm.query_param("cont_score_rec_filename", "receptor.mol2");

        att_exp = atoi(parm.query_param("cont_score_att_exp", "6").c_str());
        if (att_exp <= 0) {
            cout <<
                "ERROR: Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
        rep_exp = atoi(parm.query_param("cont_score_rep_exp", "12").c_str());
        if (rep_exp <= 0) {
            cout <<
                "ERROR: Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
	rep_radius_scale = atof(parm.query_param("cont_score_rep_rad_scale", "1").c_str());
        if (rep_radius_scale <= 0.0) {
                cout <<
                    "ERROR:  Parameter must be a float greater than zero. Program will terminate."
                    << endl;
                exit(0);
        }
        diel_screen =
            atof(parm.query_param("cont_score_dielectric", "4.0").c_str());
        if (diel_screen <= 0.0) {
            cout <<
                "ERROR: Parameter must be a float greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }

        vdw_scale = atoi(parm.query_param("cont_score_vdw_scale", "1").c_str());
        if (vdw_scale <= 0) {
            bool            off;
            off =
                (parm.query_param("cont_score_turn_off_vdw", "yes", "yes no") ==
                 "yes") ? true : false;
            if (!off) {
                cout <<
                    "ERROR: Parameter must be an integer greater than zero.  Program will terminate."
                    << endl;
                exit(0);
            }
        }
        es_scale = atoi(parm.query_param("cont_score_es_scale", "1").c_str());
        if (es_scale <= 0) {
            bool            off;
            off =
                (parm.query_param("cont_score_turn_off_es", "yes", "yes no") ==
                 "yes") ? true : false;
            if (!off) {
                cout <<
                    "ERROR: Parameter must be an integer greater than zero.  Program will terminate."
                    << endl;
                exit(0);
            }
        }

    }
}

/************************************************/
void
Continuous_Energy_Score::initialize(AMBER_TYPER & typer)
{
    ifstream        rec_file;
    bool            read_vdw,
                    use_chem;

    if (use_score) {

        init_vdw_energy(typer);

        rec_file.open(receptor_filename.c_str());

        if (rec_file.fail()) {
            cout << "Error Opening Receptor File!" << endl;
            exit(0);
        }

        if (!Read_Mol2(receptor, rec_file, false, false, false)) {
            cout << "Error Reading Receptor Molecule!" << endl;
            exit(0);
        }

        rec_file.close();

        read_vdw = true;
        use_chem = false;
        typer.prepare_molecule(receptor, read_vdw, use_chem);

    }
}

/************************************************/
bool
Continuous_Energy_Score::compute_score(DOCKMol & mol)
{
    float           vdw_val,
                    es_val,
                    total;
    int             i,
                    j;
    float           dist;

    total = vdw_val = es_val = 0.0;

    if (use_score == 1) {

        for (i = 0; i < mol.num_atoms; i++) {

            for (j = 0; j < receptor.num_atoms; j++) {

                if (mol.atom_active_flags[i]) {

                    dist =
                        sqrt(pow((mol.x[i] - receptor.x[j]), 2) +
                             pow((mol.y[i] - receptor.y[j]),
                                 2) + pow((mol.z[i] - receptor.z[j]), 2));
                    vdw_val +=
                        (((vdwA[mol.amber_at_id[i]] *
                           vdwA[receptor.amber_at_id[j]]) / pow(dist,
                                                                rep_exp)) -
                         ((vdwB[mol.amber_at_id[i]] *
                           vdwB[receptor.amber_at_id[j]]) / pow(dist,
                                                                att_exp))) *
                        vdw_scale;
                    es_val +=
                        ((332 * mol.charges[i] * receptor.charges[j]) /
                         (pow(dist, 2) * diel_screen)) * es_scale;

                }

            }

        }

        total = vdw_val + es_val;

        vdw_component = vdw_val;
        es_component = es_val;

        mol.current_score = total;
        mol.current_data = output_score_summary(total);

        // add_score_to_mol(mol, "Energy Score", total);

    }

    return true;
}

/************************************************/
string
Continuous_Energy_Score::output_score_summary(float score)
{
    ostringstream text;

    if (use_score) {
        text << DELIMITER << setw(STRING_WIDTH) << "Energy Score:"
             << setw(FLOAT_WIDTH) << fixed << score << endl
             << DELIMITER << setw(STRING_WIDTH) << "vdw:"
             << setw(FLOAT_WIDTH) << fixed << vdw_component << endl
             << DELIMITER << setw(STRING_WIDTH) << "es:"
             << setw(FLOAT_WIDTH) << fixed << es_component << endl
        ;
    }
    return text.str();
}

/************************************************/
/************************************************/
/************************************************/
Contact_Score::Contact_Score()
{

    cnt = NULL;
    bump = NULL;

}
/************************************************/
Contact_Score::~Contact_Score()
{

    delete[]cnt;
    delete[]bump;

}
/************************************************/
void
Contact_Score::input_parameters(Parameter_Reader & parm, bool & primary_score,
                                bool & secondary_score)
{
    string          tmp;

    use_primary_score = false;
    use_secondary_score = false;

    cout << "\nContact Score Paramters" << endl;
    cout <<
        "------------------------------------------------------------------------------------------"
        << endl;

    if (!primary_score) {
        tmp = parm.query_param("contact_score_primary", "no", "yes no");
        if (tmp == "yes")
            use_primary_score = true;
        else
            use_primary_score = false;

        primary_score = use_primary_score;
    }

    if (!secondary_score) {
        tmp = parm.query_param("contact_score_secondary", "no", "yes no");
        if (tmp == "yes")
            use_secondary_score = true;
        else
            use_secondary_score = false;

        secondary_score = use_secondary_score;
    }

    if (use_primary_score || use_secondary_score)
        contact_score = 1;
    else
        contact_score = 0;

    if (contact_score == 1) {
        use_score = true;
        cutoff_distance =
            atof(parm.query_param("contact_score_cutoff_distance", "4.5").
                 c_str());
        if (cutoff_distance <= 0.0) {
            cout <<
                "ERROR:  Parameter must be a float greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
        clash_overlap =
            atof(parm.query_param("contact_score_clash_overlap", "0.75").
                 c_str());
        if (clash_overlap <= 0.0) {
            cout <<
                "ERROR:  Parameter must be a float greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
        clash_penalty =
            atof(parm.query_param("contact_score_clash_penalty", "50").c_str());
        if (clash_penalty <= 0.0) {
            cout <<
                "ERROR:  Parameter must be a float greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
        file_prefix = parm.query_param("contact_score_grid_prefix", "grid");
    }

}

/************************************************/
void
Contact_Score::initialize()
{

    if (contact_score == 1) {
        cout << "Initializing Contact Score Routines..." << endl;
        read_cnt_grid();
        calc_corner_coords();
        use_score = true;
    } else
        use_score = false;

}

/************************************************/
void
Contact_Score::read_cnt_grid()
{
    string          fname;

    fname = file_prefix + ".cnt";
    grid_in = fopen(fname.c_str(), "rb");

    if (grid_in == NULL) {
        cout << "\n\nCould not open " << fname <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    cout << " Reading the contact grid from " << fname << endl;

    fread(&size, sizeof(int), 1, grid_in);
    cnt = new short int[size];
    fread(cnt, sizeof(short int), size, grid_in);

    fclose(grid_in);

    fname = file_prefix + ".bmp";
    grid_in = fopen(fname.c_str(), "rb");

    if (grid_in == NULL) {
        cout << "\n\nCould not open " << fname <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    fread(&size, sizeof(int), 1, grid_in);
    fread(&spacing, sizeof(float), 1, grid_in);
    fread(origin, sizeof(float), 3, grid_in);
    fread(span, sizeof(int), 3, grid_in);

    bump = new unsigned char[size];
    fread(bump, sizeof(unsigned char), size, grid_in);

    fclose(grid_in);

}

/************************************************/
bool
Contact_Score::compute_score(DOCKMol & mol)
{
    float           cnt_score;
    int             vdw_threshold;
    int             atom;
    char            line[1000];

    cnt_score = 0;
    vdw_threshold = 0;

    if (contact_score == 1) {

        for (atom = 0; atom < mol.num_atoms; atom++) {
            if (!check_box_boundaries(mol.x[atom], mol.y[atom], mol.z[atom])) {
                mol.current_score = -MIN_FLOAT;  // arbitrarily large score
                sprintf(line, "ERROR:  Conformation could not be scored by"
                    " DOCK.\nConformation not completely within grid box.\n");
                mol.current_data = line;
                return false;
            }
        }

        for (atom = 0; atom < mol.num_atoms; atom++) {

            if (mol.amber_at_heavy_flag[atom]) {

                if (mol.atom_active_flags[atom]) {

                    find_grid_neighbors(mol.x[atom], mol.y[atom], mol.z[atom]);

                    vdw_threshold = NINT(10 * mol.amber_at_radius[atom]);

                    if (vdw_threshold >= (int) bump[nearest_neighbor]) {
                        cnt_score += clash_penalty;
                    } else {
                        cnt_score += (float) cnt[nearest_neighbor];
                    }

                }

            }

        }

        mol.current_score = cnt_score;
        mol.current_data = output_score_summary(cnt_score);

    }

    return true;
}

/************************************************/
string
Contact_Score::output_score_summary(float score)
{
    ostringstream text;

    if (use_score) {
        text << DELIMITER << setw(STRING_WIDTH) << "Contact Score:"
             << setw(FLOAT_WIDTH) << fixed << score << endl
        ;
    }
    return text.str();
}
