#include <tcl/tcl.h>
#include <cstdlib>
#include <cstring>
#include <cmath>
#include <vector>

/*
 * qwrap is a Tcl routine for VMD, with a C++(ish) implementation
 * it is equivalent to "pbc wrap -all -center origin -compound res"
 * there are two differences:
 * 1) the center of each "wrapping block" is the reference point, rather than one given atom
 * 2) it's faster (up to 30 times in my tests)
 * 3) some options are hard-coded right now, they're the most likely ones for a 
 *    trajectory from a NAMD biomolecular simulation.
 * 4) it only deals with orthorhombic boxes
 * 5) I can't count!
 *
 * Jerome Henin <jerome.henin@ibpc.fr> March 2013
 */

// TODO: implement "centersel" and "sel" options

extern "C" {
int Qwrap_Init(Tcl_Interp *interp);
}
static int obj_qwrap(ClientData, Tcl_Interp *interp, int argc, Tcl_Obj * const objv[]);

// this is "truncating" a to integer number of times b
// such as a is between -b/2 and +b/2

void truncate(float *a, const std::vector<float> &b)
{
  for (int c=0; c<3; c++)
    a[c] = floor (a[c] / b[c] + 0.5) * b[c];
  return;
}

int parse_vector (Tcl_Obj * const obj, std::vector<float> &vec, Tcl_Interp *interp)
{
  Tcl_Obj **data;
  int num;
  double d;

  if (Tcl_ListObjGetElements(interp, obj, &num, &data) != TCL_OK) {
    Tcl_SetResult(interp, (char *) "qwrap: error parsing arguments", TCL_STATIC);
    return -1;
  }

  vec.resize(num);

  for (int i = 0; i < num; i++) {
    if (Tcl_GetDoubleFromObj(interp, data[i], &d) != TCL_OK) {
      Tcl_SetResult(interp, (char *) "qwrap: error parsing vector element as floating-point", TCL_STATIC);
      return -1;
    }
    // Tcl gives us doubles, make them float
    vec[i] = float (d);
  }
  return num;
}

int parse_ivector (Tcl_Obj * const obj, std::vector<int> &vec, Tcl_Interp *interp, bool fromDouble)
{
  Tcl_Obj **data;
  int num;
  double d;

  if (Tcl_ListObjGetElements(interp, obj, &num, &data) != TCL_OK) {
    Tcl_SetResult(interp, (char *) "qwrap: error parsing arguments", TCL_STATIC);
    return -1;
  }

  vec.resize(num);

  if (fromDouble == false) {
    for (int i = 0; i < num; i++) {
      if (Tcl_GetIntFromObj(interp, data[i], &vec[i]) != TCL_OK) {
        Tcl_SetResult(interp, (char *) "qwrap: error parsing vector element as integer", TCL_STATIC);
        return -1;
      }
    }
  } else {
    // do a double-to-int conversion first
    for (int i = 0; i < num; i++) {
      if (Tcl_GetDoubleFromObj(interp, data[i], &d) != TCL_OK) {
        Tcl_SetResult(interp, (char *) "qwrap: error parsing vector element as integer", TCL_STATIC);
        return -1;
      }
      vec[i] = int (d);
    }
  }
  return num;
}

// ***************************************************************************************
// ***************************************************************************************

static int obj_qwrap(ClientData data, Tcl_Interp *interp, int argc, Tcl_Obj * const objv[])
{
  Tcl_Obj *atomselect, *object, *bytes;
  int num, ncoords, result, length;
  int num_frames, first_frame, last_frame;
  enum { NONE, RES, BETA } compound;
  char *sel_text = NULL;
  char *center_sel_text = NULL;

  std::vector<int> blockID;
  float *coords;
  std::vector<float> PBC;
  unsigned char *array;

  if (argc % 2 != 1) {
    Tcl_WrongNumArgs(interp, 1, objv, (char *)"[first n] [last n] [compound none|res|beta] [sel seltext] [centersel seltext]");
    return TCL_ERROR;
  }

  // Default Values
  compound = RES;
  first_frame = 0;
  last_frame = -1;

  for (int i = 1; i + 1 < argc; i += 2) { 
    const char *cmd = Tcl_GetString(objv[i]);
    if (!strncmp(cmd, "first", 4)) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &first_frame) != TCL_OK) { return TCL_ERROR; }

    } else if (!strncmp(cmd, "last", 4)) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &last_frame) != TCL_OK) { return TCL_ERROR; }
    } else if (!strncmp(cmd, "compound", 4)) {

      const char *comp = Tcl_GetString(objv[i+1]);
      if (!strncmp(comp, "res", 4)) compound = RES;
      else if (!strncmp(comp, "none", 4)) compound = NONE;
      else if (!strncmp(comp, "beta", 4)) compound = BETA;
      else {
        Tcl_SetResult(interp, (char *) "qwrap: unknown compound type", TCL_STATIC);
        return TCL_ERROR;
      }

    } else if (!strncmp(cmd, "sel", 4)) {
      sel_text = Tcl_GetString(objv[i+1]);

    } else if (!strncmp(cmd, "center", 4)) {
      center_sel_text = Tcl_GetString(objv[i+1]);
      // TODO: create centering selection etc.

    } else {
      Tcl_SetResult(interp, (char *) "Usage: qwrap [first n] [last n] [compound res|none] [sel seltext] [centersel seltext]", TCL_STATIC);
      return TCL_ERROR;
    }
  }

  result = Tcl_EvalEx(interp, "atomselect top all", -1, 0);
  if (result != TCL_OK) {
    Tcl_SetResult(interp, (char *) "qwrap: error calling atomselect", TCL_STATIC);
    return TCL_ERROR;
  }
  atomselect = Tcl_GetObjResult(interp);
  Tcl_IncrRefCount(atomselect);   // needed to retain the atomselect object beyond this point!


  // ********* block IDs *******

  Tcl_Obj *script = Tcl_DuplicateObj(atomselect);
  if ( compound == RES ) 
    Tcl_AppendToObj (script, " get residue", -1);
  else if ( compound == BETA )
    Tcl_AppendToObj (script, " get beta", -1);
  else // this case is just to find out how many atoms we have
    Tcl_AppendToObj (script, " get occupancy", -1);
  result = Tcl_EvalObjEx(interp, script, TCL_EVAL_DIRECT);
  if (result != TCL_OK) {
    Tcl_SetResult(interp, (char *) "qwrap: error calling atomselect", TCL_STATIC);
    return TCL_ERROR;
  }
  ncoords = parse_ivector(Tcl_GetObjResult(interp), blockID, interp, (compound != RES) );
  if (ncoords == -1) {
    Tcl_SetResult(interp, (char *) "qwrap: error parsing atomselect result", TCL_STATIC);
    return TCL_ERROR;
  }

  result = Tcl_EvalEx(interp, "molinfo top get numframes", -1, 0);
  if (result != TCL_OK) {
    Tcl_SetResult(interp, (char *) "qwrap: error calling molinfo", TCL_STATIC);
    return TCL_ERROR;
  }
  object = Tcl_GetObjResult(interp);
  if (Tcl_GetIntFromObj(interp, object, &num_frames) != TCL_OK) {
    Tcl_SetResult(interp, (char *) "qwrap: error parsing number of frames", TCL_STATIC);
    return TCL_ERROR;
  }

  if ( first_frame < 0 || first_frame >= num_frames ) {
    Tcl_SetResult(interp, (char *) "qwrap: illegal value of first_frame", TCL_STATIC);
    return TCL_ERROR;
  }
  if ( last_frame == -1 || last_frame >= num_frames ) last_frame = num_frames - 1;  
  int print = ((last_frame - first_frame) / 10);
  if (print < 10) print = 10;
  if (print > 100) print = 100;

  for (int frame = first_frame; frame <= last_frame; frame++) {

    if (frame % print == 0) {
      Tcl_Obj *msg = Tcl_ObjPrintf ("puts \"Wrapping frame %i\"", frame);
      result = Tcl_EvalObjEx(interp, msg, TCL_EVAL_DIRECT);
      if (result != TCL_OK) { return TCL_ERROR; }
    }

    Tcl_Obj *chgframe = Tcl_DuplicateObj(atomselect);
    Tcl_AppendPrintfToObj (chgframe, " frame %i", frame);
    result = Tcl_EvalObjEx(interp, chgframe, TCL_EVAL_DIRECT);
    if (result != TCL_OK) { return TCL_ERROR; }
     
    Tcl_Obj *mol_chgframe = Tcl_ObjPrintf ("molinfo top set frame %i", frame);
    result = Tcl_EvalObjEx(interp, mol_chgframe, TCL_EVAL_DIRECT);
    if (result != TCL_OK) { return TCL_ERROR; }

    // ********* PBC *******

    Tcl_Obj *get_abc = Tcl_ObjPrintf ("molinfo top get {a b c}");
    result = Tcl_EvalObjEx(interp, get_abc, TCL_EVAL_DIRECT);
    if (result != TCL_OK) { return TCL_ERROR; }

    object = Tcl_GetObjResult(interp);
    num = parse_vector(object, PBC, interp); 
    if (num != 3 || PBC[0]*PBC[1]*PBC[2] == 0.0) {
      Tcl_SetResult(interp, (char *) "qwrap: error parsing PBC", TCL_STATIC);
      return TCL_ERROR;
    }

    Tcl_Obj *get_ts = Tcl_ObjPrintf ("gettimestep %s %i", "top", frame);
    result = Tcl_EvalObjEx(interp, get_ts,  TCL_EVAL_DIRECT);
    if (result != TCL_OK) {
      Tcl_SetResult(interp, (char *) "qwrap: error getting coordinates", TCL_STATIC);
      return TCL_ERROR;
    }
    
    bytes = Tcl_GetObjResult(interp);
    Tcl_IncrRefCount(bytes);
    Tcl_InvalidateStringRep (bytes);
    coords = reinterpret_cast<float *> (Tcl_GetByteArrayFromObj(bytes, &length));

    if ( length != 3 * ncoords * sizeof(float) ) {
      Tcl_SetResult(interp, (char *) "qwrap: error getting coordinates (wrong data size)", TCL_STATIC);
      return TCL_ERROR;
    }

    // ******** actual wrapping *******
    float avg[3];
    int current, size, end;

    for (int start = 0; start < ncoords; ) {

      if ( compound != NONE ) {
        current = blockID[start];
        size = 0;
        for (int c = 0; c < 3; c++) avg[c] = 0.0;

        for (end = start; end < ncoords && blockID[end] == current; end++) {
          for (int c = 0; c < 3; c++) avg[c] += coords[3*end + c];
          size++;
        }
        for (int c = 0; c < 3; c++) avg[c] /= size;

      } else {
        end = start;
        for (int c = 0; c < 3; c++) avg[c] = coords[3*end + c];
        end++;
      }

      truncate (avg, PBC);
      
      for (int i = start; i < end; i++) {
        for (int c = 0; c < 3; c++) coords[3*i + c] -= avg[c];
      } 

      start = end;
    }
    // ******** wrapping done *******

    // call rawtimestep to set byte array
    Tcl_Obj *set_ts[5];

    set_ts[0] = Tcl_NewStringObj("rawtimestep", -1);
    set_ts[1] = Tcl_NewStringObj("top", -1);
    set_ts[2] = bytes;
    set_ts[3] = Tcl_NewStringObj("-frame", -1);
    set_ts[4] = Tcl_NewIntObj(frame);

    result = Tcl_EvalObjv (interp, 5, set_ts, 0);
    if (result != TCL_OK) { return TCL_ERROR; }
    Tcl_DecrRefCount(bytes);
  } // end loop on frames

  Tcl_DecrRefCount(atomselect);
  Tcl_SetResult(interp, (char *) "", TCL_STATIC);
  return TCL_OK;
}

extern "C" {
  int Qwrap_Init(Tcl_Interp *interp) {
    Tcl_CreateObjCommand(interp, "qwrap", obj_qwrap,
                    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    return TCL_OK;
  }
}
