
namespace eval ::molywood {
    variable version 1.0
    # General variables
    variable scenes {}
    variable ui_elements [dict create]
    # global options
    variable movie_name "my_movie"
    variable frames_per_second 15
    variable keep_frames false
    variable gif ""
    variable breakpoints ""
    variable render "snapshot"
    variable restart true
    variable ambient_occlusion 0
    variable scene_rows 1
    variable scene_columns 1
    variable master_overlays {}
    variable current_canvas
    # scene-specific options
    variable scene_source "PDB 4-letter code" ;# the value that's displayed
    variable scene_source__ pdb_code ;# the value that's passed internally
    variable source_file
    variable scene_row 0
    variable scene_column 0
    variable scene_name_var scene1
    variable resolution_x 750
    variable resolution_y 750
    variable after_scene ""
    variable max_transparent_surfaces 3
    # other variables
    variable python python
    variable conda_prefix ""
    variable selected_template ""
    variable currently_selected_scene ""
    variable current_total_scene_time "Total scene time: 0 s"
    variable show_advanced_global
    variable show_advanced_scene
    variable actions {}
}

package provide molywood $::molywood::version
package require TclOO

set script_dir [file dirname [info script]]

source [file join $script_dir molywood_actions.tcl]
source [file join $script_dir molywood_scene.tcl]

# widgets with displays:
# list of scenes  $w.scns.scnl           (ttk::treeview)
# list of actions $w.nb.sceneProp.container.actl  (ttk::treeview)
# list of MOs     $w.nb.general.topFrame.advancedFrame.movr

# Main entry point for the plugin
proc molywood-gui {} {

    # Create and show the main window if it doesn't exist
    if {[winfo exists .molywood_window]} {
        wm deiconify .molywood_window
        return .molywood_window
    }
    set w [eval ::molywood::create_main_window]

    # ::molywood::check_install

    # initial setup
    if {[llength $::molywood::scenes] == 0} {
        ::molywood::add_scene
        set items [$w.scns.scnl children {}]
        $w.scns.scnl selection set [lindex $items end]
    }

    update
    # Initialize the main window
    return $w
}

proc ::molywood::create_main_window {} {
    # Create the main window
    set w [toplevel .molywood_window]
    wm title $w "Molywood Movie Maker v$::molywood::version"
    wm geometry $w 800x740

    frame $w.topFrame

    # Top buttons
    ttk::button $w.topFrame.docs -text "Documentation" -command {::molywood::invokeBrowser "https://mmb.irbbarcelona.org/molywood/docs"}
    ttk::button $w.topFrame.samples -text "Sample library" -command {::molywood::invokeBrowser "https://mmb.irbbarcelona.org/molywood/samples"}
    ttk::button $w.topFrame.tutorial -text "Tutorial" -command {::molywood::invokeBrowser "https://www.youtube.com/watch?v=7rHwlxm3d00&list=PLl6-JGevY8O1bExfakK8yr-sczsuBjooV"}
    ttk::button $w.topFrame.load -text "Load input" -command ::molywood::read_from_input
    ttk::button $w.topFrame.write -text "Save input" -command ::molywood::write_input
    ttk::button $w.topFrame.templates -text "Load template" -command ::molywood::choose_movie_template
    ::molywood::create_tooltip $w.topFrame.docs "Open online documentation in your browser"
    ::molywood::create_tooltip $w.topFrame.samples "Open an online library of minimal movie examples"
    ::molywood::create_tooltip $w.topFrame.tutorial "Open a YouTube playlist with step-by-step tutorials"
    ::molywood::create_tooltip $w.topFrame.load "Load a movie setup/script from a text file \[Shift+O\]"
    ::molywood::create_tooltip $w.topFrame.templates "Choose from pre-existing movie templates and samples \[Shift+T\]"
    ::molywood::create_tooltip $w.topFrame.write "Write a movie setup/script to a text file \[Shift+F\]"

    bind $w <Shift-O> {::molywood::read_from_input}
    bind $w <Shift-F> {::molywood::write_input}
    bind $w <Shift-T> {::molywood::choose_movie_template}

    grid $w.topFrame.docs      -row 1 -column 1 -sticky ew
    grid $w.topFrame.samples   -row 1 -column 2 -sticky ew
    grid $w.topFrame.tutorial  -row 1 -column 3 -sticky ew
    grid $w.topFrame.load      -row 2 -column 1 -sticky ew
    grid $w.topFrame.templates -row 2 -column 2 -sticky ew
    grid $w.topFrame.write     -row 2 -column 3 -sticky ew

    grid columnconfigure $w.topFrame 1 -weight 1
    grid columnconfigure $w.topFrame 2 -weight 1
    grid columnconfigure $w.topFrame 3 -weight 1

    if {[lsearch -exact [font names] tree_font] == -1} {
        font create tree_font -family Helvetica -size 11 -weight bold
    }
    ttk::style configure Treeview.Heading -font tree_font -background gray65 -foreground white
    ttk::style configure Treeview -font tree_font

    pack $w.topFrame -side top -fill x -expand false
    frame $w.scns
    ttk::style configure Custom.Treeview
    ttk::style map Custom.Treeview -background [list selected "#2D5FF5"] -foreground [list selected "white"]
    #ttk::style configure Treeview -indent 5
    ttk::treeview $w.scns.scnl -style Custom.Treeview -selectmode browse -height 4
    $w.scns.scnl configure -column {#1 #2}
    $w.scns.scnl column #0 -width 120 -stretch 1 -anchor w
    $w.scns.scnl column #1 -width 100 -stretch 1 -anchor w
    $w.scns.scnl column #2 -width 480 -stretch 1 -anchor w
    $w.scns.scnl heading #0 -text "Scene name"
    $w.scns.scnl heading #1 -text "Time"
    $w.scns.scnl heading #2 -text "Scene attributes"

    ttk::scrollbar $w.scns.scnl_scroll -orient vertical -command "$w.scns.scnl yview"
    $w.scns.scnl configure -yscrollcommand "$w.scns.scnl_scroll set"

    grid $w.scns.scnl -row 0 -column 0 -sticky nsew
    grid $w.scns.scnl_scroll -row 0 -column 1 -sticky ns
    grid columnconfigure $w.scns 0 -weight 1
    pack $w.scns -side top -fill x -expand false -pady 25 -padx 15

    # Set up tabs or frames
    set nb [ttk::notebook $w.nb]
    pack $nb -fill both -expand true

    # General Tab
    frame $nb.general
    $nb add $nb.general -text "General"

    # Default scene1 Tab
    frame $nb.sceneProp
    $nb add $nb.sceneProp -text "Scene contents"

     # Default Add scene Tab
    frame $nb.addScene -padx 5 -pady 5 
    $nb add $nb.addScene -text "Scenes manager"

    ########################################
    #        Scene contents go here        #
    ########################################

    # Icon definitions
    set down_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "arrow-big-down.png"]]
    set ::molywood::idown [image create photo icdown]
    $::molywood::idown copy $down_osize -subsample 12 12
    set up_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "arrow-big-up.png"]]
    set ::molywood::iup [image create photo icup]
    $::molywood::iup copy $up_osize -subsample 12 12
    set add_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "square-plus.png"]]
    set ::molywood::iadd [image create photo icadd]
    $::molywood::iadd copy $add_osize -subsample 12 12
    set copy_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "copy-plus.png"]]
    set ::molywood::icopy [image create photo iccopy]
    $::molywood::icopy copy $copy_osize -subsample 12 12
    set edit_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "edit.png"]]
    set ::molywood::iedit [image create photo icedit]
    $::molywood::iedit copy $edit_osize -subsample 12 12
    set link_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "link.png"]]
    set ::molywood::ilink [image create photo iclink]
    $::molywood::ilink copy $link_osize -subsample 12 12
    set ulink_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "link-off.png"]]
    set ::molywood::iulink [image create photo iculink]
    $::molywood::iulink copy $ulink_osize -subsample 12 12 
    set remove_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "trash-x.png"]]
    set ::molywood::iremove [image create photo icremove]
    $::molywood::iremove copy $remove_osize -subsample 12 12 
    set preview_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "player-play.png"]]
    set ::molywood::ipreview [image create photo icpreview]
    $::molywood::ipreview copy $preview_osize -subsample 12 12 
    set previewsel_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "player-track-next.png"]]
    set ::molywood::ipreviewsel [image create photo icpreviewsel]
    $::molywood::ipreviewsel copy $previewsel_osize -subsample 12 12 
    set render_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "video.png"]]
    set ::molywood::irender [image create photo icrender]
    $::molywood::irender copy $render_osize -subsample 12 12 
    set diagram_osize [image create photo -format png -file [file join $::env(MOLYWOOD_DIR) icons "layout-dashboard.png"]]
    set ::molywood::idiagram [image create photo icdiagram]
    $::molywood::idiagram copy $diagram_osize -subsample 12 12 

    # Top container for the actionlist
    frame $nb.sceneProp.container -width 700
    ttk::treeview $nb.sceneProp.container.actl -style Custom.Treeview -selectmode extended -height 8 -columns {#1 #2}
    $nb.sceneProp.container.actl column #0 -width 150 -stretch 0 -anchor w
    $nb.sceneProp.container.actl column #1 -width 100 -stretch 0 -anchor w
    $nb.sceneProp.container.actl column #2 -width 450 -stretch 0 -anchor w

    $nb.sceneProp.container.actl heading #0 -text "Action"
    $nb.sceneProp.container.actl heading #1 -text "Duration"
    $nb.sceneProp.container.actl heading #2 -text "   Non-default action parameters" -anchor w
    $nb.sceneProp.container.actl tag configure "light_gray_tag" -background gray95
    $nb.sceneProp.container.actl tag configure "dark_gray_tag" -background gray85
    bind $nb.sceneProp.container.actl <Double-1> {::molywood::edit_action}
    # A vertical scrollbar linked to the treeview
    ttk::scrollbar $nb.sceneProp.container.actl_scroll -orient vertical -command "$nb.sceneProp.container.actl yview"
    $nb.sceneProp.container.actl configure -yscrollcommand "$nb.sceneProp.container.actl_scroll set"
    # A horizontal scrollbar too (useful if we merge actions)
    ttk::scrollbar $nb.sceneProp.container.actl_hscroll -orient horizontal -command "$nb.sceneProp.container.actl xview"
    $nb.sceneProp.container.actl configure -xscrollcommand "$nb.sceneProp.container.actl_hscroll set"
    
    grid $nb.sceneProp.container.actl -row 0 -column 0 -sticky nsew
    grid $nb.sceneProp.container.actl_scroll -row 0 -column 1 -sticky ns
    grid $nb.sceneProp.container.actl_hscroll -row 1 -column 0 -sticky ew

    grid rowconfigure $nb.sceneProp.container 0 -weight 1
    grid columnconfigure $nb.sceneProp.container 0 -weight 1
    grid $nb.sceneProp.container -row 0 -column 0 -sticky nw -pady 25 -padx 15

    # This fn will update list of scenes when needed
    bind $w.scns.scnl <<TreeviewSelect>> [list ::molywood::update_on_scene_selection $w.scns.scnl $nb.sceneProp]
    # Double-click will preview scene
    bind $w.scns.scnl <Double-1> {::molywood::preview_scene}

    # Everything below the action list goes here
    frame $nb.sceneProp.midFrame
    set smfr $nb.sceneProp.midFrame
    grid $nb.sceneProp.midFrame -row 1 -column 0 -sticky nw -columnspan 2

    # Add the "Move up/down" and "Merge/unmerge" buttons, row 1
    frame $smfr.buttons
    # grid columnconfigure $smfr.buttons 0 -weight 1
    # grid columnconfigure $smfr.buttons 1 -weight 1
    # grid columnconfigure $smfr.buttons 2 -weight 1
    # grid columnconfigure $smfr.buttons 3 -weight 1
    # grid columnconfigure $smfr.buttons 4 -weight 1

    frame $smfr.buttons.frameUp -width 80 -height 40
    grid $smfr.buttons.frameUp -row 0 -column 0 -sticky ew -padx 5 -pady 5
    button $smfr.buttons.btnUp -text "Move up" -command {::molywood::move_action "up"} -image $::molywood::iup -compound left -width 80
    grid $smfr.buttons.btnUp -in $smfr.buttons.frameUp -row 0 -column 0 -sticky ew
    ::molywood::create_tooltip $smfr.buttons.btnUp "Move action up in the actions list \[Shift+W\]"
    bind $w <Shift-W> {::molywood::move_action "up"}

    frame $smfr.buttons.frameDown -width 92 -height 80
    grid $smfr.buttons.frameDown -row 0 -column 1 -sticky ew -padx 5 -pady 5
    button $smfr.buttons.btnDown -text "Move down" -command {::molywood::move_action "down"} -image $::molywood::idown -compound left -width 92
    grid $smfr.buttons.btnDown -in $smfr.buttons.frameDown -row 0 -column 0 -sticky ew
    ::molywood::create_tooltip $smfr.buttons.btnDown "Move action down in the actions list \[Shift+S\]"
    bind $w <Shift-S> {::molywood::move_action "down"}

    frame $smfr.buttons.frameMerge -width 225 -height 80
    grid $smfr.buttons.frameMerge -row 0 -column 2 -sticky ew -padx 5 -pady 5 -columnspan 2
    button $smfr.buttons.btnMerge -text "Make simultaneous" -command ::molywood::merge_actions -image $::molywood::ilink -compound left -width 225
    grid $smfr.buttons.btnMerge -in $smfr.buttons.frameMerge -row 0 -column 0 -sticky ew -columnspan 2
    ::molywood::create_tooltip $smfr.buttons.btnMerge "Combine two actions so that they happen simultaneously \[Shift+M\]"
    bind $w <Shift-M> {::molywood::merge_actions}
    # button $smfr.buttons.btnMerge -text "Make actions simultaneous" -command ::molywood::merge_actions -width 21
    # grid $smfr.buttons.btnMerge -row 0 -column 2 -sticky ew -padx 5 -pady 5  -columnspan 2

    frame $smfr.buttons.frameUnmerge -width 180 -height 80
    grid $smfr.buttons.frameUnmerge -row 0 -column 4 -sticky ew -padx 5 -pady 5
    button $smfr.buttons.btnUnmerge -text "Make separate" -command ::molywood::unmerge_action -image $::molywood::iulink -compound left -width 180
    grid $smfr.buttons.btnUnmerge -in $smfr.buttons.frameUnmerge -row 0 -column 0 -sticky ew
    ::molywood::create_tooltip $smfr.buttons.btnUnmerge "Separate simultaneous actions, making them consecutive again \[Shift+U\]"
    bind $w <Shift-U> {::molywood::unmerge_actions}
    # button $smfr.buttons.btnUnmerge -text "Make actions separate" -command ::molywood::unmerge_action -width 21
    # grid $smfr.buttons.btnUnmerge -row 0 -column 4 -sticky ew -padx 5 -pady 5 

    # Add the "Add/edit/remove action" buttons, row 2
    frame $smfr.buttons.frameAdd -width 213 -height 40
    grid $smfr.buttons.frameAdd -row 1 -column 0 -sticky nsew -padx 5 -pady 5  -columnspan 2
    button $smfr.buttons.btnAdd -text "Add action" -command ::molywood::open_action_popup -image $::molywood::iadd -compound left -background lavender -width 213
    grid $smfr.buttons.btnAdd -in $smfr.buttons.frameAdd -row 0 -column 0 -sticky nsew -columnspan 2
    ::molywood::create_tooltip $smfr.buttons.btnAdd "Add new action to the actions list (opens a popup) \[Shift+A\]"
    bind $w <Shift-A> {::molywood::open_action_popup}
    # button $smfr.buttons.btnAdd -text "Add action" -command ::molywood::open_action_popup -width 21 -background lavender
    # grid $smfr.buttons.btnAdd -row 1 -column 0 -sticky ew -padx 5 -pady 5 -columnspan 2
    
    frame $smfr.buttons.frameEdit -width 92 -height 80
    grid $smfr.buttons.frameEdit -row 1 -column 2 -sticky ew -padx 5 -pady 5
    button $smfr.buttons.btnEdit -text "Edit" -command ::molywood::edit_action -image $::molywood::iedit -compound left -width 92
    grid $smfr.buttons.btnEdit -in $smfr.buttons.frameEdit -row 0 -column 0 -sticky ew
    ::molywood::create_tooltip $smfr.buttons.btnEdit "Edit action properties (double-click on action or \[Shift+E\])"
    bind $w <Shift-E> {::molywood::edit_action}
    # button $smfr.buttons.btnEdit -text "Edit action" -command ::molywood::edit_action
    # grid $smfr.buttons.btnEdit -row 1 -column 2 -sticky ew -padx 5 -pady 5 

    frame $smfr.buttons.frameCopy -width 92 -height 80
    grid $smfr.buttons.frameCopy -row 1 -column 3 -sticky ew -padx 5 -pady 5
    button $smfr.buttons.btnCopy -text "Copy" -command ::molywood::copy_actions -image $::molywood::icopy -compound left -width 92
    grid $smfr.buttons.btnCopy -in $smfr.buttons.frameCopy -row 0 -column 0 -sticky ew
    ::molywood::create_tooltip $smfr.buttons.btnCopy "Copy one or more actions (use Ctrl or Shift to select many) \[Shift+C\]"
    bind $w <Shift-C> {::molywood::copy_actions}
    # button $smfr.buttons.btnCopy -text "Copy action(s)" -command ::molywood::copy_actions
    # grid $smfr.buttons.btnCopy -row 1 -column 3 -sticky ew -padx 5 -pady 5 

    frame $smfr.buttons.frameRemove -width 180 -height 80
    grid $smfr.buttons.frameRemove -row 1 -column 4 -sticky ew -padx 5 -pady 5
    button $smfr.buttons.btnRemove -text "Remove" -command ::molywood::remove_actions -image $::molywood::iremove -compound left -width 180
    grid $smfr.buttons.btnRemove -in $smfr.buttons.frameRemove -row 0 -column 0 -sticky ew
    ::molywood::create_tooltip $smfr.buttons.btnRemove "Remove one or more actions (use Ctrl or Shift to select many) \[Delete\]"
    bind $w <Delete> {::molywood::remove_actions}
    # button $smfr.buttons.btnRemove -text "Remove action(s)" -command ::molywood::remove_actions -width 21
    # grid $smfr.buttons.btnRemove -row 1 -column 4 -sticky ew -padx 5 -pady 5 

    # Final "Preview" and total time counter
    frame $smfr.buttons.framePreview  -height 80 -width 345
    grid $smfr.buttons.framePreview -row 2 -column 0 -sticky ew -padx 5 -pady 5 -columnspan 3
    button $smfr.buttons.btnPreview -text "Preview in VMD" -command ::molywood::preview_scene -image $::molywood::ipreview -compound left -width 345 -background gray75 
    grid $smfr.buttons.btnPreview -in $smfr.buttons.framePreview -row 0 -column 0 -sticky ew -columnspan 3
    ::molywood::create_tooltip $smfr.buttons.btnPreview "Preview this scene only \[Shift+P\]"
    bind $w <Shift-P> {::molywood::preview_scene}
    # button $smfr.buttons.btnPreview -text "Preview in VMD" -width 21 -command ::molywood::preview_scene -background gray75 
    # grid $smfr.buttons.btnPreview -row 2 -column 0 -sticky ew -padx 5 -pady 5 -columnspan 3

    frame $smfr.buttons.framePreviewSel -height 80 -width 315
    grid $smfr.buttons.framePreviewSel -row 2 -column 3 -sticky ew -padx 5 -pady 5 -columnspan 3
    button $smfr.buttons.btnPreviewSel -text "Preview selected action(s) only" -command [list ::molywood::preview_scene true] -image $::molywood::ipreviewsel -compound left -width 315
    grid $smfr.buttons.btnPreviewSel -in $smfr.buttons.framePreviewSel -row 0 -column 0 -sticky ew -columnspan 3
    ::molywood::create_tooltip $smfr.buttons.btnPreviewSel "Preview only the actions that are selected (use Ctrl or Shift to select many)"
    # button $smfr.buttons.btnPrevSel -text "Preview selected scenes only" -width 25 -command [list ::molywood::preview_scene true]
    # grid $smfr.buttons.btnPrevSel -row 2 -column 3 -sticky ew -padx 5 -pady 5 -columnspan 3

    pack $smfr.buttons -side top -fill y -padx 5 -pady 5

    ########################################
    #         Scene adder goes here        #
    ########################################

    frame $nb.addScene.topFrame
    set atfr $nb.addScene.topFrame
    pack $atfr -side top -fill both -expand false

    frame $atfr.sceneName
    grid columnconfigure $atfr 0 -minsize 300
    grid columnconfigure $atfr 1 -minsize 400

    # The main scene name entry widget
    label $atfr.sceneName.lblSceneName -text "Scene name:"
    ::molywood::create_tooltip $atfr.sceneName.lblSceneName "Choose a name for the Scene (to create a new one, set a unique name!)"
    entry $atfr.sceneName.entrySceneName -textvariable ::molywood::scene_name_var -width 36
    pack $atfr.sceneName.lblSceneName -side left
    pack $atfr.sceneName.entrySceneName -side right -padx 5
    grid $atfr.sceneName -row 0 -column 0 -sticky ew -padx 5 -pady 5 -columnspan 2

    # 'Scene source' label and combobox that will dictate what to display as "Source file"
    frame $atfr.sceneSourceFrame
    label $atfr.sceneSourceFrame.lblSceneSource -text "Scene source:"
    ::molywood::create_tooltip $atfr.sceneSourceFrame.lblSceneSource "Set the starting point for the movie: a pre-made .vmd visualization state,\na local structure file, a structure defined by a 4-letter PDB code,\na wildcard path matching multiple .cube files, or the VMD state that's currently displayed"
    set sources_dict {
        "Existing VMD state" visualization 
        "Structure file" structure 
        "PDB 4-letter code" pdb_code 
        "Set of .cube files" cubes 
        "Current VMD state" "current VMD state"
    }
    ttk::combobox $atfr.sceneSourceFrame.comboSceneSource -values [dict keys $sources_dict] -textvariable ::molywood::scene_source -width 15
    pack $atfr.sceneSourceFrame.lblSceneSource -side left
    pack $atfr.sceneSourceFrame.comboSceneSource -side right -padx 5
    grid $atfr.sceneSourceFrame -row 1 -column 0 -sticky ew -padx 5
    bind $atfr.sceneSourceFrame.comboSceneSource <<ComboboxSelected>> [list ::molywood::switch_source_file_widget $w 0 0]
    #trace add variable ::molywood::scene_source write ::molywood::switch_source_file_widget

    # This will be interactively edited, default is pdb_code
    frame $atfr.sourceFileFrame
    label $atfr.sourceFileFrame.lblSourceFile -text "Source file:"
    ::molywood::create_tooltip $atfr.sourceFileFrame.lblSourceFile "Choose the file that will be loaded at the beginning of your movie"
    entry $atfr.sourceFileFrame.entrySourceFile -textvariable ::molywood::source_file -width 25
    pack $atfr.sourceFileFrame.lblSourceFile -side left -padx 5
    pack $atfr.sourceFileFrame.entrySourceFile -side right -padx 5
    grid $atfr.sourceFileFrame -row 1 -column 1 -sticky ew -padx 5
    ::molywood::switch_source_file_widget $w 0 0

    # The X/Y Resolution label and entries
    frame $atfr.resolutionFrame
    label $atfr.resolutionFrame.lblResolution -text "Resolution:"
    ::molywood::create_tooltip $atfr.resolutionFrame.lblResolution "Set the Scene's resolution in px (make sure they match if you're using multiple panels!)"
    label $atfr.resolutionFrame.lblX -text "X"
    entry $atfr.resolutionFrame.entryResolutionX -textvariable ::molywood::resolution_x -width 5
    label $atfr.resolutionFrame.lblY -text "Y"
    entry $atfr.resolutionFrame.entryResolutionY -textvariable ::molywood::resolution_y -width 5
    pack $atfr.resolutionFrame.lblResolution -side left
    pack $atfr.resolutionFrame.entryResolutionY -side right -padx 5
    pack $atfr.resolutionFrame.lblY -side right
    pack $atfr.resolutionFrame.entryResolutionX -side right -padx 5
    pack $atfr.resolutionFrame.lblX -side right
    grid $atfr.resolutionFrame -row 3 -column 0 -sticky ew -padx 5 -pady 5

    # The Scene position row/column comboboxes
    frame $atfr.scenePositionFrame
    label $atfr.scenePositionFrame.lblSceneRC -text "Multiview position:"
    ::molywood::create_tooltip $atfr.scenePositionFrame.lblSceneRC "Set the position of this Scene in the multi-panel video"
    label $atfr.scenePositionFrame.lblRow -text "row"
    ttk::combobox $atfr.scenePositionFrame.sceneRow -values [::molywood::get_consec $::molywood::scene_rows] -textvariable ::molywood::scene_row -width 5
    label $atfr.scenePositionFrame.lblCol -text "column"
    ttk::combobox $atfr.scenePositionFrame.sceneCol -values [::molywood::get_consec $::molywood::scene_columns] -textvariable ::molywood::scene_column -width 5
    pack $atfr.scenePositionFrame.lblSceneRC -side left -padx 5
    pack $atfr.scenePositionFrame.sceneCol -side right -padx 5
    pack $atfr.scenePositionFrame.lblCol -side right
    pack $atfr.scenePositionFrame.sceneRow -side right -padx 5
    pack $atfr.scenePositionFrame.lblRow -side right
    grid $atfr.scenePositionFrame -row 3 -column 1 -sticky ew -padx 5 -pady 5

    # The 'After scene' label and combobox
    frame $atfr.afterSceneFrame
    label $atfr.afterSceneFrame.lblAfterScene -text "After scene:"
    ::molywood::create_tooltip $atfr.afterSceneFrame.lblAfterScene "Put the current Scene after the end of another Scene"
    ttk::combobox $atfr.afterSceneFrame.comboAfterScene -values [list "" {*}[::molywood::get_all_other_scene_names $::molywood::scene_name_var]] -textvariable ::molywood::after_scene -width 10
    pack $atfr.afterSceneFrame.lblAfterScene -side left
    pack $atfr.afterSceneFrame.comboAfterScene -side right -padx 5
    grid $atfr.afterSceneFrame -row 4 -column 0 -sticky ew -padx 5 -pady 5
    bind $atfr.afterSceneFrame.comboAfterScene <<ComboboxSelected>> ::molywood::update_afters

    frame $atfr.maxTransparentSurfacesFrame
    label $atfr.maxTransparentSurfacesFrame.lblMaxTransparentSurfaces -text "Max transparent surfaces:"
    ::molywood::create_tooltip $atfr.maxTransparentSurfacesFrame.lblMaxTransparentSurfaces "Choose how many transparent surfaces light can pass through"
    spinbox $atfr.maxTransparentSurfacesFrame.spinMaxTransparentSurfaces -from 0 -to 10 -textvariable ::molywood::max_transparent_surfaces -width 5
    pack $atfr.maxTransparentSurfacesFrame.lblMaxTransparentSurfaces -side left -padx 5
    pack $atfr.maxTransparentSurfacesFrame.spinMaxTransparentSurfaces -side right -padx 5
    grid $atfr.maxTransparentSurfacesFrame -row 4 -column 1 -sticky ew -padx 5 -pady 5

    trace add variable ::molywood::scene_name_var write ::molywood::highlight_update_name
    trace add variable ::molywood::resolution_x write ::molywood::highlight_update
    trace add variable ::molywood::resolution_y write ::molywood::highlight_update
    trace add variable ::molywood::after_scene write ::molywood::highlight_update
    trace add variable ::molywood::max_transparent_surfaces write ::molywood::highlight_update
    trace add variable ::molywood::scene_row write ::molywood::highlight_update
    trace add variable ::molywood::scene_column write ::molywood::highlight_update
    trace add variable ::molywood::source_file write ::molywood::highlight_update
    trace add variable ::molywood::source_source__ write ::molywood::highlight_update

    # The 'Add/edit/remove/clone scene' buttons
    button $atfr.btnAddScene -text "Add new scene" -command ::molywood::add_scene  -width 80
    ::molywood::create_tooltip $atfr.btnAddScene "Adds a Scene with properties specified in this tab\n(make sure it has a unique name!)"
    grid $atfr.btnAddScene -row 6 -column 0 -sticky ew -padx 5 -pady 5 -columnspan 2

    button $atfr.btnEditScene -text "Update currently selected scene" -command ::molywood::edit_scene -width 80
    ::molywood::create_tooltip $atfr.btnEditScene "Set the properties specified in this tab for the currently selected Scene \[Shift+U\]"
    grid $atfr.btnEditScene -row 7 -column 0 -sticky ew -padx 5 -pady 5 -columnspan 2
    bind $w <Shift-U> {::molywood::edit_scene}

    button $atfr.btnRemoveScene -text "Remove currently selected scene" -command ::molywood::remove_scene -width 38
    ::molywood::create_tooltip $atfr.btnRemoveScene "Permanently remove the Scene that is currently selected"
    grid $atfr.btnRemoveScene -row 8 -column 0 -sticky ew -padx 5 -pady 5

    button $atfr.btnCloneScene -text "Clone selected scene with contents" -command ::molywood::clone_scene -width 38
    ::molywood::create_tooltip $atfr.btnCloneScene "Create a new Scene with the same properties and actions as currently selected \[Shift+L\]"
    grid $atfr.btnCloneScene -row 8 -column 1 -sticky ew -padx 5 -pady 5
    bind $w <Shift-L> {::molywood::clone_scene}

    ########################################
    #       General settings go here       #
    ########################################

    # Top frame for the General tab
    frame $nb.general.topFrame
    set gtfr $nb.general.topFrame 
    pack $gtfr -side top -fill x

    # The main 'Movie name' label and entry
    frame $gtfr.mvNameDir
    if {[string length [pwd]] > 20} {
        set pathdisp "...[string range [pwd] end-20 end]"
    } else {
        set pathdisp [pwd]
    }
    label $gtfr.mvNameDir.lblMovieName -text "Movie output file: $pathdisp/"
    ::molywood::create_tooltip $gtfr.mvNameDir.lblMovieName "Your movie will be located at [pwd]/$::molywood::movie_name.mp4"
    entry $gtfr.mvNameDir.entryMovieName -textvariable ::molywood::movie_name -width 12
    label $gtfr.mvNameDir.lblExtension -text ".mp4"
    pack $gtfr.mvNameDir.lblMovieName -side left
    pack $gtfr.mvNameDir.entryMovieName -side left
    pack $gtfr.mvNameDir.lblExtension -side left -padx 1
    grid $gtfr.mvNameDir -row 0 -column 0 -sticky e -padx 5 -pady 5

    # The 'Frames per second' and 'Render mode' widgets
    frame $gtfr.cwd
    button $gtfr.cwd.cwd -text "Change working directory" -command ::molywood::cwd -compound left
    ::molywood::create_tooltip $gtfr.cwd.cwd "Change the directory in which the movie will be produced; if you specify a non-existent directory, it will be created"
    pack $gtfr.cwd.cwd -side left
    grid $gtfr.cwd -row 0 -column 1 -sticky ew -padx 5 -pady 5

    # The 'Frames per second' and 'Render mode' widgets
    frame $gtfr.fps
    label $gtfr.fps.lbl -text "Frames per second:"
    ::molywood::create_tooltip $gtfr.fps.lbl "Adjust the FPS rate. Large values will slow down production"
    spinbox $gtfr.fps.spin -from 1 -to 60 -textvariable ::molywood::frames_per_second -width 5
    pack $gtfr.fps.lbl -side left
    pack $gtfr.fps.spin -side right -padx 30
    grid $gtfr.fps -row 1 -column 0 -sticky ew -padx 5 -pady 5

    frame $gtfr.drf
    ttk::combobox $gtfr.drf.combo -values {preview snapshot "high quality"} -textvariable ::molywood::render 
    label $gtfr.drf.lbl -text "Render mode:"
    ::molywood::create_tooltip $gtfr.drf.lbl "Choose between preview (only shows in VMD), snapshot (composes a movie from screenshots; fast)\nand high quality (uses the internal renderer to produce individual frames; slow)"
    pack $gtfr.drf.lbl -side left
    pack $gtfr.drf.combo -side right -padx 30
    grid $gtfr.drf -row 2 -column 0 -sticky ew -padx 5 -pady 5

    # Scene layout can be chosen here:
    frame $gtfr.layout
    label $gtfr.layout.label -text "Layout:"
    ::molywood::create_tooltip $gtfr.layout.label "Enables multi-panel videos with multiple Scenes;\nadjust 'Multiview position' in Scenes manager"
    pack $gtfr.layout.label -side left

    frame $gtfr.layout.frame
    # Parameters for grid size
    set gtfr_rows 1
    set gtfr_cols 5
    set canvas_width 37
    set canvas_height 37

    # Variable to keep track of the currently highlighted canvas
    set currentCanvas ""

    # Dictionary for pastel colors
    array set pastelColors {
        red    "#FF6961"
        blue   "#AEC6CF"
        green  "#77DD77"
        purple "#CFCFFF"
    }

    # Create a grid of canvases
    for {set r 0} {$r < $gtfr_rows} {incr r} {
        for {set c 0} {$c < $gtfr_cols} {incr c} {
            set canvasName [format "%s.canvas%d_%d" $gtfr.layout.frame $r $c]
            canvas $canvasName -width $canvas_width -height $canvas_height -bg white -highlightbackground white -highlightthickness 2
            grid $canvasName -row $r -column $c  -sticky news -padx 3 -pady 3

            # Bind mouse click event
            bind $canvasName <Button-1> [list ::molywood::on_canvas_click $canvasName]
            
        }
    }

    # Calculate center coordinates
    set centerX [expr {$canvas_width / 2}]
    set centerY [expr {$canvas_height / 2}]

    ########################################################################
    # Layout 1 - one panel
    ########################################################################
    set r 0 ; set c 0
    set i 1
    set canvasName [format "%s.canvas%d_%d" $gtfr.layout.frame $r $c]
    $canvasName create rectangle 2 2 $canvas_width $canvas_height -fill $pastelColors(red)

    # Calculate center coordinates
    $canvasName create text 19.5 19.5 -text $i -font "Helvetica 12" -fill black

    ########################################################################
    # Layout 2 - side by side
    ########################################################################
    set r 0 ; set c 1
    set i 1
    set canvasName [format "%s.canvas%d_%d" $gtfr.layout.frame $r $c]
    $canvasName create rectangle 2 2 18 37 -fill $pastelColors(red)
    $canvasName create text 9 19.5 -text $i -font "Helvetica 12" -fill black
    incr i
    $canvasName create rectangle 21 2 37 37 -fill $pastelColors(blue)
    $canvasName create text 29 19.5 -text $i -font "Helvetica 12" -fill black

    ########################################################################
    # Layout 3 - bottom/top
    ########################################################################
    set r 0 ; set c 2
    set i 1
    set canvasName [format "%s.canvas%d_%d" $gtfr.layout.frame $r $c]
    $canvasName create rectangle 2 2 37 18 -fill $pastelColors(red)
    $canvasName create text 19.5 11 -text $i -font "Helvetica 12" -fill black
    incr i
    $canvasName create rectangle 2 21 37 37 -fill $pastelColors(blue)
    $canvasName create text 19.5 30 -text $i -font "Helvetica 12" -fill black

    ########################################################################
    # Layout 4 - 2x2 grid
    ########################################################################
    set r 0 ; set c 3
    set i 1
    set canvasName [format "%s.canvas%d_%d" $gtfr.layout.frame $r $c]
    $canvasName create rectangle 2 2 18 18 -fill $pastelColors(red)
    $canvasName create text 10 11 -text $i -font "Helvetica 12" -fill black
    incr i
    $canvasName create rectangle 21 2 37 18 -fill $pastelColors(blue)
    $canvasName create text 29 11 -text $i -font "Helvetica 12" -fill black
    incr i
    $canvasName create rectangle 2 21 18 37 -fill $pastelColors(green)
    $canvasName create text 10 30 -text $i -font "Helvetica 12" -fill black
    incr i
    $canvasName create rectangle 21 21 37 37 -fill $pastelColors(purple)
    $canvasName create text 29 30 -text $i -font "Helvetica 12" -fill black

    ########################################################################
    # Layout 5 - custom
    ########################################################################
    set r 0 ; set c 4
    set canvasName [format "%s.canvas%d_%d" $gtfr.layout.frame $r $c]
    $canvasName create text 19.5 19.5 -text "Custom" -font "Helvetica 8" -fill black

    $gtfr.layout.frame.canvas0_0 configure -bg gray40 -highlightbackground gray40 -highlightthickness 2
    set ::molywood::current_canvas $gtfr.layout.frame.canvas0_0

    pack $gtfr.layout.frame -side right

    grid $gtfr.layout -row 1 -column 1 -sticky ew -padx 5 -pady 5 -rowspan 2

    # Checkbutton to show/hide advanced settings
    checkbutton $gtfr.chkAdvanced -text "Show Advanced Settings" -variable ::molywood::show_advanced_global -command ::molywood::toggle_advanced_global_settings
    ::molywood::create_tooltip $gtfr.chkAdvanced "Manage restarting, shading, GIF conversion, breakpoints, and global (master) overlays"
    grid $gtfr.chkAdvanced -row 3 -column 0 -sticky ew -padx 5 -pady 5 -columnspan 3

    # Advanced frame (initially hidden)
    frame $gtfr.advancedFrame

    # The 'Ambient occlusion' checkbox (we're making it apply to all scenes without exceptions, 
    # in the rare cases where it's relevant can be modified directly in the text file for final rendering)
    frame $gtfr.advancedFrame.ao
    checkbutton $gtfr.advancedFrame.ao.chk -text "" -variable ::molywood::ambient_occlusion -command ::molywood::toggle_ao
    label $gtfr.advancedFrame.ao.lbl -text "Ambient occlusion:" 
    pack $gtfr.advancedFrame.ao.lbl -side left -padx 5
    pack $gtfr.advancedFrame.ao.chk -side left -padx 5
    grid $gtfr.advancedFrame.ao -row 4 -column 0 -sticky ew

    # Checkboxes for 'Restart' and 'Keep frames'
    frame $gtfr.advancedFrame.rst
    checkbutton $gtfr.advancedFrame.rst.chk -text "" -variable ::molywood::restart
    label $gtfr.advancedFrame.rst.lbl -text "Resume rendering:"
    pack $gtfr.advancedFrame.rst.lbl -side left -padx 15
    pack $gtfr.advancedFrame.rst.chk -side left -padx 5
    grid $gtfr.advancedFrame.rst -row 4 -column 1 -sticky ew

    frame $gtfr.advancedFrame.kpf
    checkbutton $gtfr.advancedFrame.kpf.chk -text "" -variable ::molywood::keep_frames
    label $gtfr.advancedFrame.kpf.lbl -text "Keep frames:"
    pack $gtfr.advancedFrame.kpf.lbl -side left -padx 15
    pack $gtfr.advancedFrame.kpf.chk -side left -padx 5
    grid $gtfr.advancedFrame.kpf -row 4 -column 2 -sticky ew

    # Checkbox for 'Gif' and optional breakpoints
    frame $gtfr.advancedFrame.gif
    entry $gtfr.advancedFrame.gif.chk -textvariable ::molywood::gif -width 25
    label $gtfr.advancedFrame.gif.lbl -text "Convert to GIF (fps, width_px):"
    pack $gtfr.advancedFrame.gif.lbl -side left -padx 5 -pady 2
    pack $gtfr.advancedFrame.gif.chk -side right -padx 5 -pady 2
    grid $gtfr.advancedFrame.gif -row 5 -column 0 -sticky ew -columnspan 3

    frame $gtfr.advancedFrame.brkp
    entry $gtfr.advancedFrame.brkp.chk -textvariable ::molywood::breakpoints -width 25
    label $gtfr.advancedFrame.brkp.lbl -text "Movie breakpoints (comma-separated times in \[s\]):"
    pack $gtfr.advancedFrame.brkp.lbl -side left -padx 5 -pady 2
    pack $gtfr.advancedFrame.brkp.chk -side right -padx 5 -pady 2
    grid $gtfr.advancedFrame.brkp -row 6 -column 0 -sticky ew -columnspan 3

    # Add treeview for master overlays
    ttk::treeview $gtfr.advancedFrame.movr -selectmode browse -height 2 -columns {#1 #2}
    $gtfr.advancedFrame.movr column #0 -width 70 -stretch 1 -anchor w
    $gtfr.advancedFrame.movr column #1 -width 35 -stretch 1 -anchor w
    $gtfr.advancedFrame.movr column #2 -width 350 -stretch 1 -anchor w
    $gtfr.advancedFrame.movr heading #0 -text "Master overlays"
    $gtfr.advancedFrame.movr heading #1 -text "From:To"
    $gtfr.advancedFrame.movr heading #2 -text "Overlay parameters"
    grid $gtfr.advancedFrame.movr -row 7 -column 0 -sticky ew -padx 5 -pady 5 -columnspan 3
    button $gtfr.advancedFrame.btnAddMOvr -text "Add master overlay" -width 23 -command ::molywood::add_mo
    grid $gtfr.advancedFrame.btnAddMOvr -row 8 -column 0 -sticky ew -padx 5 -pady 5
    button $gtfr.advancedFrame.btnEditMOvr -text "Edit master overlay" -width 23 -command ::molywood::edit_mo
    grid $gtfr.advancedFrame.btnEditMOvr -row 8 -column 1 -sticky ew -padx 5 -pady 5
    button $gtfr.advancedFrame.btnRmMOvr -text "Remove master overlay" -width 23 -command ::molywood::remove_mo
    grid $gtfr.advancedFrame.btnRmMOvr -row 8 -column 2 -sticky ew -padx 5 -pady 5
    bind $gtfr.advancedFrame.movr <Double-1> {::molywood::edit_mo}

    # Initially hide the advanced frame content
    grid remove $gtfr.advancedFrame

    # The "Render" and "Display script diagram" buttons in a separate bottom frame
    frame $nb.general.bottomFrame
    set gbfr $nb.general.bottomFrame
    pack $gbfr -side top -fill x -padx 5 -pady 5

    frame $gbfr.buttonR -width 200 -height 80
    button $gbfr.buttonR.btnRender -text "Render" -background gray75 -command ::molywood::do_render -image $::molywood::irender -compound left
    ::molywood::create_tooltip $gbfr.buttonR.btnRender "Produce the movie (if render mode is 'snapshot' or 'high quality') or preview (otherwise) \[Shift+R\]"
    pack $gbfr.buttonR.btnRender -side left -padx 5 -pady 5 -expand yes -fill both -anchor center
    frame $gbfr.buttonD -width 200 -height 80
    button $gbfr.buttonD.btnDisplay -text "Display script diagram" -command ::molywood::draw_diagram -background lavender -image $::molywood::idiagram -compound left
    ::molywood::create_tooltip $gbfr.buttonD.btnDisplay "Show a block diagram visualizing the temporal organization of actions and scenes \[Shift+D\]"
    pack $gbfr.buttonD.btnDisplay -side left -padx 5 -pady 5 -expand yes -fill both -anchor center
    pack $gbfr.buttonR $gbfr.buttonD -side left -padx 5 -pady 5 -expand yes -fill both
    bind $w <Shift-R> {::molywood::do_render}
    bind $w <Shift-D> {::molywood::draw_diagram}


    frame $nb.general.depsFrame
    set gbdp $nb.general.depsFrame
    pack $gbdp -side top -fill x -padx 5 -pady 5

    button $gbdp.btnDeps -text "Verify dependencies" -width 21 -background gray85 -command ::molywood::verify_deps
    ::molywood::create_tooltip $gbdp.btnDeps "Check all imports to make sure you can run Molywood"
    button $gbdp.moly -text "Molywood" -relief flat -background "gray80"  -padx 5 -pady 5
    button $gbdp.imgm -text "Imagemagick" -relief flat -background "gray80"  -padx 5 -pady 5
    button $gbdp.ffmp -text "ffmpeg" -relief flat -background "gray80"  -padx 5 -pady 5
    button $gbdp.pyth -text "Python" -relief flat -background "gray80"  -padx 5 -pady 5
    lassign [::molywood::get_conda_envs] env_list path_list
    set displayList {}
    for {set i 0} {$i < [llength $env_list]} {incr i} {
        set n [lindex $env_list  $i]
        set p [lindex $path_list $i]
        lappend displayList "$n ($p)"
    }
    ttk::combobox $gbdp.cond -values $displayList -textvariable molywood::selectedEnv -state readonly -width 30

    pack $gbdp.btnDeps $gbdp.moly $gbdp.imgm $gbdp.ffmp $gbdp.pyth -side left -padx 5 -pady 5

    bind $gbdp.cond <<ComboboxSelected>> {
        ::molywood::set_conda_prefix
    }

    ##############################
    #     initial setup done     #
    ##############################

    # Registering the main window with the namespace
    dict set ::molywood::ui_elements window $w
    ::molywood::verify_deps
    ::molywood::update_scenes_table
    return $w
}

proc ::molywood::verify_deps {} {
    # backend that tries running all the commands and detects
    # which ones are not available, then colors buttons accordingly
    set w [dict get $::molywood::ui_elements window]
    set gbdp $w.nb.general.depsFrame
    set rcfile [::molywood::which_molywoodrc]
    if {$rcfile != ""} {
        source $rcfile
    }
    if {[info exists molywood_conda]} {
        puts "MOLY-Logger\) Will use conda environment from .molywoodrc, $molywood_conda"
        set ::molywood::conda_prefix $molywood_conda
    }
    set missing 0
    if {[::molywood::command_exists "molywood"]} {
        $gbdp.moly configure -background "chartreuse" -activebackground "chartreuse" -highlightbackground "chartreuse" -activeforeground "black" -text "Molywood"
    } else {
        $gbdp.moly configure -background "tomato" -relief raised -command [list ::molywood::install_dep "molywood"] -text "install Molywood"
        incr missing
    }
    if {[::molywood::command_exists "ffmpeg"]} {
        $gbdp.ffmp configure -background "chartreuse" -activebackground "chartreuse" -highlightbackground "chartreuse" -activeforeground "black" -text "ffmpeg"
    } else {
        $gbdp.ffmp configure -background "tomato" -relief raised -command [list ::molywood::install_dep "ffmpeg"] -text "install ffmpeg"
        incr missing
    }
    if {[::molywood::command_exists "convert"] || [::molywood::command_exists "magick"]} {
        $gbdp.imgm configure -background "chartreuse" -activebackground "chartreuse" -highlightbackground "chartreuse" -activeforeground "black" -text "imagemagick"
    } else {
        $gbdp.imgm configure -background "tomato" -relief raised -command [list ::molywood::install_dep "magick"] -text "install imagemagick"
        incr missing
    }
    if {[catch {exec $::molywood::python -c "import numpy matplotlib"}]} {
        $gbdp.pyth configure -background "chartreuse" -activebackground "chartreuse" -highlightbackground "chartreuse" -activeforeground "black" -text "Python"
    } else {
        $gbdp.pyth configure -background "tomato" -relief raised -command [list ::molywood::install_dep "python"] -text "install Python deps"
        incr missing
    }
    if {$missing > 0} {
        pack $gbdp.cond -side left -padx 5 -pady 5
    }
    return $missing
}

proc ::molywood::get_conda_envs {} {
    # runs 'conda env list' in the command line and returns two arrays,
    # one with names of envs and the other with paths to them
    if {[catch {exec conda env list 2>@1} raw]} {
        return {} {}
    }
    set ::molywood::selectedEnv ""
    set env_list {}
    set path_list {}

    foreach line [split $raw "\n"] {
        set t [string trim $line]
        if {$t eq "" || [string index $t 0] eq "#"} continue
        # Try active‐env pattern:  name  *  /full/path
        if {[regexp {^(\S+)\s+\*\s+(\S+)$} $t -> name path]} {
            set isActive 1
        # Try named‐env pattern: name  /full/path
        } elseif {[regexp {^(\S+)\s+(\S+)$} $t -> name path]} {
            set isActive 0
        # Try path‐only pattern: /full/path
        } elseif {[regexp {^(\S+)$} $t -> path]} {
            set name [file tail $path]
            set isActive 0
        } else {
            continue
        }
        lappend env_list $name
        lappend path_list $path
        if {$isActive} {
            set ::molywood::selectedEnv "$name \($path\)"
        }
    }
    return [list $env_list $path_list]
}

proc ::molywood::which_molywoodrc {} {
    set osname [string trimleft [array get ::tcl_platform os] "os "]
    if {$osname eq "Windows NT"} {
        set name molywood.rc
    } else {
        set name .molywoodrc
    }
    foreach f [list [file join [pwd]        $name] \
                  [file join $::env(HOME)   $name] \
                  [file join $::env(VMDDIR) $name]] {
        if {[file exists $f]} { return $f }
    }
    return ""
}

proc ::molywood::create_molywoodrc {} {
    set osname [string trimleft [array get ::tcl_platform os] "os "]
    if {$osname eq "Windows NT"} {
        set name molywood.rc
    } else {
        set name .molywoodrc
    }
    set filename [file join $::env(HOME)   $name]
    set fh [open $filename w]
    close $fh
    return $filename
}

proc ::molywood::cwd {} {
    set w [dict get $::molywood::ui_elements window]
    set gtfr $w.nb.general.topFrame 
    set dir [tk_chooseDirectory -initialdir [pwd] -mustexist false -title "Select working directory"]
    if {$dir != "" && ![file exists $dir]} {file mkdir $dir}
    if {[string length $dir]} {
        if {[file isdirectory $dir]} {
            cd $dir
        } else {
            tk_messageBox -icon error -title "Error" -message "“$dir” is not a directory."
            return ""
        }
    }
    if {[string length [pwd]] > 20} {
        set pathdisp "...[string range [pwd] end-20 end]"
    } else {
        set pathdisp [pwd]
    }
    $gtfr.mvNameDir.lblMovieName configure -text "Movie output file: $pathdisp/"
    ::molywood::create_tooltip $gtfr.mvNameDir.lblMovieName "Your movie will be located at [pwd]/$::molywood::movie_name.mp4"
    return $dir
}

proc ::molywood::show_dataset_window {} {
    # shows (or re-shows) the dataset creator window,
    # initializes an array of relevant properties
    if {[winfo exists .molywood_dataset_window]} {
        wm deiconify .molywood_dataset_window
        return
    }
    array set ::molywood::dataset_headers    {}
    array set ::molywood::dataset_contents   {}
    array set ::molywood::dataset_text_widgets {}
    array set ::molywood::dataset_color      {}
    array set ::molywood::dataset_width      {}
    array set ::molywood::dataset_style      {}
    array set ::molywood::dataset_label      {}
    array set ::molywood::point_dynamic      {}
    array set ::molywood::selected_label     {}
    array set ::molywood::selected_measure   {}
    array set ::molywood::measure_sel1       {}
    array set ::molywood::measure_sel2       {}
    array set ::molywood::measure_param1     {}
    array set ::molywood::measure_param2     {}
    array set ::molywood::label_spec         {}
    set ::molywood::dataset_count 0
    set ::molywood::dataset_xlimit "min,max"
    set ::molywood::dataset_ylimit "min,max"

    toplevel .molywood_dataset_window
    wm title .molywood_dataset_window "Dataset Creator"
    wm geometry .molywood_dataset_window 720x750

    frame .molywood_dataset_window.datasetsFrame
    pack .molywood_dataset_window.datasetsFrame -fill both -expand true -padx 10 -pady 10

    button .molywood_dataset_window.addBtn -text "Add Dataset" -command ::molywood::add_dataset_window
    ::molywood::create_tooltip .molywood_dataset_window.addBtn "Adds a new dataset to be displayed on the same plot"
    pack .molywood_dataset_window.addBtn -pady 5

    labelframe .molywood_dataset_window.generalFrame -text "General Plot Options"
    pack .molywood_dataset_window.generalFrame -fill x -padx 10 -pady 5

    frame .molywood_dataset_window.generalFrame.xrow
    label .molywood_dataset_window.generalFrame.xrow.xlabel -text "X-axis label:"
    ::molywood::create_tooltip .molywood_dataset_window.generalFrame.xrow.xlabel "Label for the X-axis; can contain special characters and TeX inserts"
    entry .molywood_dataset_window.generalFrame.xrow.xentry -textvariable ::molywood::dataset_x_label -width 30
    pack .molywood_dataset_window.generalFrame.xrow.xlabel .molywood_dataset_window.generalFrame.xrow.xentry -side left -padx 5 -pady 2
    label .molywood_dataset_window.generalFrame.xrow.xlabel2 -text "X-axis limit:"
    ::molywood::create_tooltip .molywood_dataset_window.generalFrame.xrow.xlabel2 "Specify lower and upper limits for the X-axis, separated by a comma; 'min,max' leaves the default"
    entry .molywood_dataset_window.generalFrame.xrow.xentry2 -textvariable ::molywood::dataset_xlimit -width 30
    pack .molywood_dataset_window.generalFrame.xrow.xlabel2 .molywood_dataset_window.generalFrame.xrow.xentry2 -side left -padx 5 -pady 2
    pack .molywood_dataset_window.generalFrame.xrow -fill x

    frame .molywood_dataset_window.generalFrame.yrow
    label .molywood_dataset_window.generalFrame.yrow.ylabel -text "Y-axis label:"
    ::molywood::create_tooltip .molywood_dataset_window.generalFrame.yrow.ylabel "Label for the Y-axis; can contain special characters and TeX inserts"
    entry .molywood_dataset_window.generalFrame.yrow.yentry -textvariable ::molywood::dataset_y_label -width 30
    pack .molywood_dataset_window.generalFrame.yrow.ylabel .molywood_dataset_window.generalFrame.yrow.yentry -side left -padx 5 -pady 2
    label .molywood_dataset_window.generalFrame.yrow.ylabel2 -text "Y-axis limit:"
    ::molywood::create_tooltip .molywood_dataset_window.generalFrame.yrow.ylabel2 "Specify lower and upper limits for the Y-axis, separated by a comma; 'min,max' leaves the default"
    entry .molywood_dataset_window.generalFrame.yrow.yentry2 -textvariable ::molywood::dataset_ylimit -width 30
    pack .molywood_dataset_window.generalFrame.yrow.ylabel2 .molywood_dataset_window.generalFrame.yrow.yentry2 -side left -padx 5 -pady 2    
    pack .molywood_dataset_window.generalFrame.yrow -fill x

    frame .molywood_dataset_window.saveBtns
    pack .molywood_dataset_window.saveBtns -fill x 
    button .molywood_dataset_window.saveBtns.save -text "Save Datasets" -command ::molywood::save_datasets_to_file -width 25
    ::molywood::create_tooltip .molywood_dataset_window.saveBtns.save "Write the dataset to a file but keep the window open to allow further edits"
    button .molywood_dataset_window.saveBtns.preview -text "Preview Plot" -command ::molywood::preview_dataset_plot -width 25
    ::molywood::create_tooltip .molywood_dataset_window.saveBtns.preview "See the plot as will be inserted into the video (without the dynamic point)"
    button .molywood_dataset_window.saveBtns.saveClose -text "Save Datasets & Close" -command [list ::molywood::save_datasets_to_file .molywood_dataset_window] -width 25
    ::molywood::create_tooltip .molywood_dataset_window.saveBtns.saveClose "Write the dataset to a file, and set the file as input for the overlay"
    pack .molywood_dataset_window.saveBtns.save -in .molywood_dataset_window.saveBtns -side left -padx 25 -pady 10
    pack .molywood_dataset_window.saveBtns.preview -in .molywood_dataset_window.saveBtns -side left -padx 25 -pady 10
    pack .molywood_dataset_window.saveBtns.saveClose -in .molywood_dataset_window.saveBtns  -side right -padx 25 -pady 10

    ::molywood::add_dataset_window
}

proc ::molywood::preview_dataset_plot {} {
    # runs Molywood to preview a single frame before the plot is added to the movie
    ::molywood::do_dataset_saving moly_plot_preview.dat
    set fh [open "moly_plot_preview.txt" "w"]
    puts $fh {$ global name=preview render=f}
    puts $fh {$ scene1}
    puts $fh {# scene1}
    puts $fh "{add_overlay origin=0,0 relative_size=1 datafile=moly_plot_preview.dat; do_nothing t=1}"
    close $fh
    ::molywood::run_tool molywood preview_plot moly_plot_preview.txt
}

proc ::molywood::add_dataset_window {} {
    # adds a new dataset panel and initializes its defaults
    incr ::molywood::dataset_count
    set idx $::molywood::dataset_count
    set parent .molywood_dataset_window.datasetsFrame

    # Header frame
    set hdr $parent.header$idx
    frame $hdr -relief groove -borderwidth 1
    set ::molywood::dataset_headers($idx) $hdr
    pack $hdr -fill x -pady 2
    button $hdr.toggle$idx -text "-" -width 2  -command [list ::molywood::toggle_dataset_frame $idx]
    ::molywood::create_tooltip $hdr.toggle$idx "Toggles the visibility of the dataset for convenience"
    pack $hdr.toggle$idx -side left -padx 4 -pady 4
    label $hdr.label$idx -text "Dataset $idx"
    pack $hdr.label$idx -side left -padx 4
    button $hdr.remove$idx -text "Remove" -command [list ::molywood::remove_dataset_window $idx]
    ::molywood::create_tooltip $hdr.remove$idx "Removes the current dataset"
    pack $hdr.remove$idx -side right -padx 4 -pady 4

    # Content frame
    set content $parent.content$idx
    frame $content
    set ::molywood::dataset_contents($idx) $content
    pack $content -after $hdr -fill x -padx 10 -pady 2
    # Sub-frames for layout
    # Row 1: left half text area, right column with import/load
    frame $content.row1$idx
    pack $content.row1$idx -fill x -padx 5 -pady 2
    # Create left and right panes inside row1
    frame $content.row1_left$idx
    frame $content.row1Right$idx
    pack $content.row1_left$idx -in $content.row1$idx -side left -fill y -padx 5 -pady 5
    pack $content.row1Right$idx -in $content.row1$idx -side right -padx 5 -pady 5
    # Left pane: Data text area
    text $content.text$idx -height 6 -width 40 -wrap none
    set ::molywood::dataset_text_widgets($idx) $content.text$idx
    pack $content.text$idx -in $content.row1_left$idx -fill both -expand true
    # Dropdown and Load button in a sub-frame
    frame $content.row1Right_ctrl$idx
    pack $content.row1Right_ctrl$idx -in $content.row1Right$idx -side top -pady 2
    set ::molywood::selected_label($idx) ""
    set ::molywood::dataset_dropdowns($idx) $content.labelcb$idx
    ttk::combobox $content.labelcb$idx -textvariable ::molywood::selected_label($idx) -state readonly -postcommand [list ::molywood::refresh_label_dropdown $idx]
    pack $content.labelcb$idx -in $content.row1Right_ctrl$idx -side left -padx 1
    button $content.loadlabel$idx -text "Load from Label" -command [list ::molywood::load_dataset_from_label $idx] -width 20
    ::molywood::create_tooltip $content.loadlabel$idx "Create a new dataset by calculating a geometric property over the trajectory from an existing label (choose label from the dropdown)"
    pack $content.loadlabel$idx -in $content.row1Right_ctrl$idx -side left -padx 1
    # Another line for measurements
    frame $content.row1RightMeasure$idx
    pack $content.row1RightMeasure$idx -in $content.row1Right$idx -side top -pady 2
    ttk::combobox $content.measurecb$idx -textvariable ::molywood::selected_measure($idx) -state readonly \
        -values {"Number of Contacts" "Number of H-Bonds" "Radius of Gyration" "Solvent-accessible Surface" "RMSD (no extra alignment)"}
    pack $content.measurecb$idx -in $content.row1RightMeasure$idx -side left -padx 1
    button $content.loadmeasure$idx -text "Measure (choose selections)" -command [list ::molywood::load_dataset_from_measure $idx] -width 20
    ::molywood::create_tooltip $content.loadmeasure$idx "Create a new dataset by calculating a property from the loaded trajectory (opens a dialog box)"
    pack $content.loadmeasure$idx -in $content.row1RightMeasure$idx -side left -padx 1
    set ::molywood::selected_measure($idx) "Number of Contacts"
    # Right pane: Import File on top, then dropdown and Load side by side
    frame $content.row0$idx
    pack $content.row0$idx -in $content.row1Right$idx -side top -pady 2
    button $content.import$idx -text "Import Data from File" -command [list ::molywood::import_dataset_file $idx] -width 20
    ::molywood::create_tooltip $content.import$idx "Read a new dataset from a selected 2-column data file"
    pack $content.import$idx -in $content.row0$idx -side left -padx 1
    button $content.scale$idx -text "Rescale X Axis" -command [list ::molywood::scale_x $idx] -width 20
    ::molywood::create_tooltip $content.scale$idx "Rescale the X coordinate in-place by a chosen factor/offset, or to a specified range (opens a dialog box)"
    pack $content.scale$idx -in $content.row0$idx -side left -padx 1
    # Yet another line for marks and averages
    frame $content.row1RightMark$idx
    pack $content.row1RightMark$idx -in $content.row1Right$idx -side top -pady 2
    button $content.loadmark$idx -text "Horizontal/vertical mark" -command [list ::molywood::load_dataset_from_mark $idx] -width 20
    ::molywood::create_tooltip $content.loadmark$idx "Adds a fake dataset that creates a horizontal or vertical line to mark a selected value on the plot, possibly with a text label"
    pack $content.loadmark$idx -in $content.row1RightMark$idx -side left -padx 1
    button $content.loadravg$idx -text "Running Average" -command [list ::molywood::load_dataset_from_ravg $idx] -width 20
    ::molywood::create_tooltip $content.loadravg$idx "Create a new dataset as a running average of the current one (opens a dialog box)"
    pack $content.loadravg$idx -in $content.row1RightMark$idx -side left -padx 1
    # Now to row2: fundamental options; first all the mess of color choices in matplotlib
    frame $content.row2$idx
    pack $content.row2$idx -fill x -padx 1 -pady 2
    label $content.colorLabel$idx -text "Line color:"
    ::molywood::create_tooltip $content.colorLabel$idx "Choose one of the matplotlib-supported colors"
    menubutton $content.colorMenu$idx -textvariable ::molywood::dataset_color($idx) -indicatoron 1 -borderwidth 1 -relief raised -width 8
    menu $content.colorMenu$idx.menu -tearoff 0
    $content.colorMenu$idx configure -menu $content.colorMenu$idx.menu
    # Neutral & Grayscale
    set neutral_colors [list \
        black dimgray dimgrey gray grey darkgray darkgrey silver lightgray lightgrey gainsboro whitesmoke \
        white snow linen ivory beige oldlace floralwhite antiquewhite blanchedalmond seashell ghostwhite]

    # Reds & Pinks
    set red_pink_colors [list \
        red darkred firebrick maroon crimson indianred lightcoral salmon lightsalmon darksalmon tomato \
        orangered mistyrose lavenderblush pink lightpink hotpink deeppink palevioletred mediumvioletred]

    # Oranges, Yellows & Browns
    set orange_yellow_brown_colors [list \
        bisque navajowhite blanchedalmond papayawhip moccasin peachpuff wheat cornsilk lemonchiffon \
        lightgoldenrodyellow lightyellow palegoldenrod khaki darkkhaki gold goldenrod darkgoldenrod \
        orange darkorange burlywood tan sandybrown peru chocolate saddlebrown sienna brown rosybrown]

    # Greens, Cyans, Blues & Purples
    set cool_colors [list \
        green darkgreen forestgreen limegreen lime chartreuse lawngreen yellowgreen seagreen mediumseagreen \
        lightseagreen darkseagreen springgreen mediumspringgreen aquamarine mediumaquamarine turquoise mediumturquoise \
        lightcyan paleturquoise azure cyan aqua darkcyan darkturquoise cadetblue blue darkblue mediumblue \
        royalblue dodgerblue cornflowerblue skyblue deepskyblue lightskyblue steelblue lightblue powderblue \
        lightsteelblue slateblue darkslateblue mediumslateblue purple indigo violet plum thistle lavender \
        mediumorchid darkorchid orchid blueviolet rebeccapurple magenta fuchsia darkmagenta]

    menu $content.colorMenu$idx.css41 -tearoff 0
    $content.colorMenu$idx.menu add cascade -label "CSS4 Neutral Colors" -menu $content.colorMenu$idx.css41
    foreach col $neutral_colors {
        $content.colorMenu$idx.css41 add radiobutton -label $col -variable ::molywood::dataset_color($idx) -value $col
    }
    menu $content.colorMenu$idx.css42 -tearoff 0
    $content.colorMenu$idx.menu add cascade -label "CSS4 Red/Pink Colors" -menu $content.colorMenu$idx.css42
    foreach col $red_pink_colors {
        $content.colorMenu$idx.css42 add radiobutton -label $col -variable ::molywood::dataset_color($idx) -value $col
    }
    menu $content.colorMenu$idx.css43 -tearoff 0
    $content.colorMenu$idx.menu add cascade -label "CSS4 Orange/Yellow Colors" -menu $content.colorMenu$idx.css43
    foreach col $orange_yellow_brown_colors {
        $content.colorMenu$idx.css43 add radiobutton -label $col -variable ::molywood::dataset_color($idx) -value $col
    }
    menu $content.colorMenu$idx.css44 -tearoff 0
    $content.colorMenu$idx.menu add cascade -label "CSS4 Green/Blue Colors" -menu $content.colorMenu$idx.css44
    foreach col $cool_colors {
        $content.colorMenu$idx.css44 add radiobutton -label $col -variable ::molywood::dataset_color($idx) -value $col
    }
    menu $content.colorMenu$idx.tableau -tearoff 0
    $content.colorMenu$idx.menu add cascade -label "Tableau Colors" -menu $content.colorMenu$idx.tableau
    set tt_list {tab:blue tab:orange tab:green tab:red tab:purple tab:brown tab:pink tab:gray tab:olive tab:cyan}
    foreach col $tt_list {
        $content.colorMenu$idx.tableau add radiobutton -label $col -variable ::molywood::dataset_color($idx) -value $col
    }
    menu $content.colorMenu$idx.base -tearoff 0
    $content.colorMenu$idx.menu add cascade -label "Base Colors" -menu $content.colorMenu$idx.base
    set base_list {red green blue cyan magenta yellow black}
    foreach col $base_list {
        $content.colorMenu$idx.base add radiobutton -label $col -variable ::molywood::dataset_color($idx) -value $col
    }
    # remaining matplotlib options
    label $content.widthLabel$idx -text "Line width:"
    ::molywood::create_tooltip $content.widthLabel$idx "Specify line width; 1.5 is matplotlib's default"
    entry $content.widthEntry$idx -width 3 -textvariable ::molywood::dataset_width($idx)
    label $content.styleLabel$idx -text "Line style:"
    ::molywood::create_tooltip $content.styleLabel$idx "Choose line style"
    ttk::combobox $content.styleCombo$idx -textvariable ::molywood::dataset_style($idx) -values [list solid dotted dashed dashdot] -state readonly -width 5
    label $content.labLabel$idx -text "Label:"
    ::molywood::create_tooltip $content.labLabel$idx "If set, will add a legend to the plot (unless a horizontal/vertical mark)"
    entry $content.labEntry$idx -width 7 -textvariable ::molywood::dataset_label($idx)
    label $content.dynLabel$idx -text "Dynamic point:"
    ::molywood::create_tooltip $content.dynLabel$idx "If selected, this dataset will feature a dynamically moving point indicating position in a trajectory"
    checkbutton $content.dynCheck$idx -text "" -variable ::molywood::point_dynamic($idx)
    pack \
        $content.colorLabel$idx $content.colorMenu$idx \
        $content.widthLabel$idx $content.widthEntry$idx \
        $content.styleLabel$idx $content.styleCombo$idx \
        $content.labLabel$idx $content.labEntry$idx \
        $content.dynLabel$idx $content.dynCheck$idx \
        -in $content.row2$idx -side left -padx 5 -pady 5

    # Defaults:
    set ::molywood::dataset_color($idx) "tab:blue"
    set ::molywood::dataset_width($idx) "1.5"
    set ::molywood::dataset_style($idx) "solid"
    set ::molywood::dataset_label($idx) ""
    set ::molywood::point_dynamic($idx) 1
}

proc ::molywood::refresh_label_dropdown {idx} {
    # refreshes dropdown for labels-based measurement with existing VMD labels
    set dd $::molywood::dataset_dropdowns($idx)
    array unset ::molywood::label_spec
    set items {}
    foreach cat [label list] {
        if {$cat eq "Atoms"} continue
        foreach raw_spec [label list $cat] {
            set ids {}
            foreach elem $raw_spec {
                if {[regexp {^(\d+)\s+(\d+)$} $elem -> mol atom]} {
                    lappend ids $atom
                } else {
                    break
                }
            }
            if {[llength $ids] == 0} continue
            set disp "$cat: [join $ids -]"
            set ::molywood::label_spec($idx,$disp) $raw_spec
            lappend items $disp
        }
    }
    $dd configure -values $items
    if {$::molywood::selected_label($idx) eq "" && [llength $items] > 0} {
        set ::molywood::selected_label($idx) [lindex $items 0]
    }
}

proc ::molywood::load_dataset_from_measure {idx} {
    # calculates data from a user-selected measure (5 options),
    # asks for the relevant selections and parameters
    set measure $::molywood::selected_measure($idx)
    if {$measure eq ""} return
    set top [toplevel .molywood_measure_popup$idx]
    wm title $top "Configure $measure"

    set ::molywood::measure_sel1($idx) "all"
    set ::molywood::measure_sel2($idx) ""
    set ::molywood::measure_param1($idx) ""
    set ::molywood::measure_param2($idx) ""
    set row 0
    switch $measure {
        "Number of Contacts" {
            label $top.label1 -text "Selection group 1:"
            grid $top.label1 -row $row -column 0 -sticky w
            entry $top.entry1 -width 30 -textvariable ::molywood::measure_sel1($idx)
            grid $top.entry1 -row $row -column 1
            incr row
            label $top.label2 -text "Selection group 2:"
            grid $top.label2 -row $row -column 0 -sticky w
            entry $top.entry2 -width 30 -textvariable ::molywood::measure_sel2($idx)
            grid $top.entry2 -row $row -column 1
            incr row
            label $top.label3 -text "Cutoff (Å):"
            grid $top.label3 -row $row -column 0 -sticky w
            entry $top.entry3 -width 10 -textvariable ::molywood::measure_param1($idx)
            grid $top.entry3 -row $row -column 1
            set ::molywood::measure_param1($idx) 4.0
            incr row
        }
        "Number of H-Bonds" {
            label $top.label1 -text "Selection group 1:"
            grid $top.label1 -row $row -column 0 -sticky w
            entry $top.entry1 -width 30 -textvariable ::molywood::measure_sel1($idx)
            grid $top.entry1 -row $row -column 1
            incr row
            label $top.label2 -text "Selection group 2:"
            grid $top.label2 -row $row -column 0 -sticky w
            entry $top.entry2 -width 30 -textvariable ::molywood::measure_sel2($idx)
            grid $top.entry2 -row $row -column 1
            incr row
            label $top.label3 -text "Distance cutoff (Å):"
            grid $top.label3 -row $row -column 0 -sticky w
            entry $top.entry3 -width 10 -textvariable ::molywood::measure_param1($idx)
            grid $top.entry3 -row $row -column 1
            set ::molywood::measure_param1($idx) 3.5
            incr row
            label $top.label4 -text "Angle cutoff (°):"
            grid $top.label4 -row $row -column 0 -sticky w
            entry $top.entry4 -width 10 -textvariable ::molywood::measure_param2($idx)
            grid $top.entry4 -row $row -column 1
            set ::molywood::measure_param2($idx) 30
            incr row
        }
        "Radius of Gyration" {
            label $top.label1 -text "Selection:"
            grid $top.label1 -row $row -column 0 -sticky w
            entry $top.entry1 -width 30 -textvariable ::molywood::measure_sel1($idx)
            grid $top.entry1 -row $row -column 1
            incr row
        }
        "Solvent-accessible Surface" {
            label $top.label1 -text "Selection (solute):"
            grid $top.label1 -row $row -column 0 -sticky w
            entry $top.entry1 -width 30 -textvariable ::molywood::measure_sel1($idx)
            grid $top.entry1 -row $row -column 1
            incr row
            label $top.label2 -text "Selection (surface):"
            grid $top.label2 -row $row -column 0 -sticky w
            entry $top.entry2 -width 30 -textvariable ::molywood::measure_sel2($idx)
            grid $top.entry2 -row $row -column 1
            incr row
            label $top.label3 -text "Probe radius (Å):"
            grid $top.label3 -row $row -column 0 -sticky w
            entry $top.entry3 -width 10 -textvariable ::molywood::measure_param1($idx)
            grid $top.entry3 -row $row -column 1
            set ::molywood::measure_param1($idx) 1.4
            incr row
        }
        "RMSD (no extra alignment)" {
            label $top.label1 -text "Selection:"
            grid $top.label1 -row $row -column 0 -sticky w
            entry $top.entry1 -width 30 -textvariable ::molywood::measure_sel1($idx)
            grid $top.entry1 -row $row -column 1
            incr row
            label $top.label3 -text "Reference frame ID:"
            grid $top.label3 -row $row -column 0 -sticky w
            entry $top.entry3 -width 10 -textvariable ::molywood::measure_param1($idx)
            grid $top.entry3 -row $row -column 1
            set ::molywood::measure_param1($idx) 0
            incr row
        }
    }
    button $top.calcBtn -text "Calculate" -command [list ::molywood::_run_measure $idx $top]
    grid $top.calcBtn -row $row -column 0 -columnspan 2 -pady 10
}

proc ::molywood::_run_measure {idx top} {
    # backend: runs measure and populates widget, then destroys popup
    set m       $::molywood::selected_measure($idx)
    set mol     [molinfo top]
    set currframe [molinfo $mol get frame]
    set nFrames [molinfo $mol get numframes]
    set vals    {}
    # Loops over each frame and compute
    for {set f 0} {$f < $nFrames} {incr f} {
        animate goto $f
        switch $m {
            "Number of Contacts" {
                set raw [measure contacts $::molywood::measure_param1($idx) [atomselect top $::molywood::measure_sel1($idx)] [atomselect top $::molywood::measure_sel2($idx)]]
                lappend vals [llength [lindex $raw 0]]
            }
            "Number of H-Bonds" {
                set raw [measure hbonds $::molywood::measure_param1($idx) $::molywood::measure_param2($idx) [atomselect top $::molywood::measure_sel1($idx)] [atomselect top $::molywood::measure_sel2($idx)]]
                lappend vals [llength [lindex $raw 0]]
            }
            "Radius of Gyration" {
                set v [measure rgyr [atomselect top $::molywood::measure_sel1($idx)]]
                lappend vals $v
            }
            "Solvent-accessible Surface" {
                set sasa [measure sasa $::molywood::measure_param1($idx) [atomselect top $::molywood::measure_sel1($idx)]]
                lappend vals $sasa
            }
            "RMSD (no extra alignment)" {
                set selCur [atomselect top $::molywood::measure_sel1($idx)]
                set sel0 [atomselect top $::molywood::measure_sel1($idx)]
                $selCur frame $f
                $sel0 frame $::molywood::measure_param1($idx)
                set v [measure rmsd $sel0 $selCur]
                $selCur delete
                #$sel0 delete
                lappend vals $v
            }
        }
    }
    if {[info exists sel0]} {
        $sel0 delete
    }
    animate goto $currframe
    set txtw $::molywood::dataset_text_widgets($idx)
    $txtw delete 1.0 end
    for {set i 0} {$i < $nFrames} {incr i} {
        set v [lindex $vals $i]
        $txtw insert end "$i\t[format %.3f $v]\n"
    }
    destroy $top
}

proc ::molywood::load_dataset_from_mark {idx} {
    # creates a new horizontal/vertical line dataset following user-specified
    # selection of settings
    set top [toplevel .molywood_mark_popup$idx]
    wm title $top "Create a line"
    set ::molywood::line_orientation  "horizontal"
    set ::molywood::line_mode         "chosen"
    set ::molywood::line_value        ""
    set ::molywood::line_label        ""
    label    $top.lblOrient -text "Orientation:" 
    radiobutton $top.rbtnH   -text Horizontal -variable ::molywood::line_orientation -value horizontal -command  [list ::molywood::_update_mark_options $top $idx]
    radiobutton $top.rbtnV   -text Vertical -variable ::molywood::line_orientation -value vertical -command  [list ::molywood::_update_mark_options $top $idx]
    grid $top.lblOrient -row 0 -column 0 -sticky w -padx 5 -pady 5
    grid $top.rbtnH    -row 0 -column 1 -sticky w
    grid $top.rbtnV    -row 0 -column 2 -sticky w
    # Horizontal‐mode radios (only shown when horizontal)
    label $top.lblHMode -text "If horizontal, use:"
    foreach {mode col} {chosen 1 min 2 max 3 mean 4} {
        radiobutton $top.rbH$mode -text $mode -variable ::molywood::line_mode -value $mode -command [list ::molywood::_update_mark_options $top $idx]
        grid $top.rbH$mode -row 1 -column $col -sticky w
    }   
    grid $top.lblHMode -row 1 -column 0 -sticky w -padx 5 -pady 5
    label $top.lblValue -text "Value:"
    entry $top.entValue -textvariable ::molywood::line_value -width 10
    grid $top.lblValue -row 2 -column 0 -sticky w -padx 5 -pady 5
    grid $top.entValue -row 2 -column 1 -columnspan 2 -sticky we -padx 5 -pady 5
    label $top.lblLabel -text "Label:"
    entry $top.entLabel -textvariable ::molywood::line_label -width 20
    grid $top.lblLabel -row 3 -column 0 -sticky w -padx 5 -pady 5
    grid $top.entLabel -row 3 -column 1 -columnspan 2 -sticky we -padx 5 -pady 5
    button $top.btnAdd -text "Add as new dataset" -command [list ::molywood::make_new_mark_dataset $idx $top]
    grid $top.btnAdd -row 5 -column 0 -columnspan 3 -pady 10
    ::molywood::_update_mark_options $top $idx
}

proc ::molywood::_compute_stat {idx stat {col 1}} {
    # computes the minimum/maximum/mean value in the dataset,
    # optionally specifying the column (second as default)
    set txt_widget $::molywood::dataset_text_widgets($idx)
    set content [$txt_widget get 1.0 end]
    set vals {}
    foreach line [split $content "\n"] {
        if {[string trim $line] ne ""} {
            # regexp -all -inline {\S+} keeps runs of non‐space chars
            set fields [regexp -all -inline {\S+} $line]
            if {[llength $fields] >= 2} {
                lappend vals [lindex $fields $col]
            }
        }
    }
    if {[llength $vals] == 0} { return "" }
    switch -- $stat {
        min {
            set sorted [lsort -real $vals]
            return [lindex $sorted 0]
        }
        max {
            set sorted [lsort -real $vals]
            return [lindex $sorted end]
        }
        mean {
            set sum 0
            foreach v $vals { set sum [expr $sum + $v] }
            return [expr {double($sum) / [llength $vals]}]
        }
        default {
            return ""
        }
    }
}

proc ::molywood::_update_mark_options {top idx} {
    # updates the selections available to the user when swithing from
    # horizontal to vertical mark and vice versa
    set hwidgets [list $top.lblHMode $top.rbHchosen $top.rbHmin $top.rbHmax $top.rbHmean]
    if {$::molywood::line_orientation eq "horizontal"} {
        set val [::molywood::_compute_stat $idx $::molywood::line_mode]
        set ::molywood::line_value $val
        foreach w $hwidgets { grid $w }
    } else {
        foreach w $hwidgets { grid remove $w }
        set ::molywood::line_value ""
    }
}

proc ::molywood::make_new_mark_dataset {idx top} {
    # creates a dataset containing 2 points that specify either a horizontal
    # or a vertical line
    ::molywood::add_dataset_window
    set dst_txt $::molywood::dataset_text_widgets($::molywood::dataset_count)
    $dst_txt delete 1.0 end
    if {$::molywood::line_orientation eq "horizontal"} {
        set minx [::molywood::_compute_stat $idx "min" 0]
        set maxx [::molywood::_compute_stat $idx "max" 0]
        $dst_txt insert end "$minx\t$::molywood::line_value\n"
        $dst_txt insert end "$maxx\t$::molywood::line_value\n"
    } else {
        set miny [::molywood::_compute_stat $idx "min" 1]
        set maxy [::molywood::_compute_stat $idx "max" 1]
        $dst_txt insert end "$::molywood::line_value\t$miny\n"
        $dst_txt insert end "$::molywood::line_value\t$maxy\n"
    }
    set ::molywood::dataset_label($::molywood::dataset_count) $::molywood::line_label
    set ::molywood::point_dynamic($::molywood::dataset_count) 0
    set ::molywood::dataset_color($::molywood::dataset_count) "darkgray"
    set ::molywood::dataset_width($::molywood::dataset_count) "2.0"
    set ::molywood::dataset_style($::molywood::dataset_count) "dashed"
    destroy $top
}

proc ::molywood::load_dataset_from_ravg {idx} {
    # creates a dataset in the dataset manager as running average of an existing one (frontend)
    set top [toplevel .molywood_ravg_popup$idx]
    wm title $top "Running average from dataset"
    set row 0
    set ::molywood::smoothing_window 1
    label $top.lblWindow -text "Smoothing window:" 
    grid  $top.lblWindow  -row $row -column 0 -sticky w -padx 5 -pady 5
    entry $top.entWindow -textvariable ::molywood::smoothing_window -width 10
    grid  $top.entWindow -row $row -column 1 -sticky we -padx 5 -pady 5
    incr row
    button $top.btnAdd -text "Add as new dataset" -command [list ::molywood::make_new_ravg_dataset $idx $top]
    grid  $top.btnAdd -row $row -column 0 -columnspan 2 -pady 10
}

proc ::molywood::scale_x {idx} {
    # compute original range
    set ::molywood::orig_min [::molywood::_compute_stat $idx min 0]
    set ::molywood::orig_max [::molywood::_compute_stat $idx max 0]
    # initialize vars
    set ::molywood::min_x   $::molywood::orig_min
    set ::molywood::max_x   $::molywood::orig_max
    set ::molywood::offset  0
    set ::molywood::scale   1

    # add traces
    trace add variable ::molywood::min_x   write ::molywood::onMinMax
    trace add variable ::molywood::max_x   write ::molywood::onMinMax
    trace add variable ::molywood::offset  write ::molywood::onOffScale
    trace add variable ::molywood::scale   write ::molywood::onOffScale

    # build UI
    set top [toplevel .molywood_scalex_popup$idx]
    wm title $top "Rescaling X coordinate"
    set row 0

    label   $top.lblMin   -text "Minimum X value:"
    entry   $top.entMin   -textvariable ::molywood::min_x   -width 10
    grid    $top.lblMin   -row $row -column 0 -sticky w -padx 5 -pady 5
    grid    $top.entMin   -row $row -column 1 -sticky we -padx 5 -pady 5
    incr row

    label   $top.lblMax   -text "Maximum X value:"
    entry   $top.entMax   -textvariable ::molywood::max_x   -width 10
    grid    $top.lblMax   -row $row -column 0 -sticky w -padx 5 -pady 5
    grid    $top.entMax   -row $row -column 1 -sticky we -padx 5 -pady 5
    incr row

    label   $top.lblOff   -text "Offset:"
    entry   $top.entOff   -textvariable ::molywood::offset  -width 10
    grid    $top.lblOff   -row $row -column 0 -sticky w -padx 5 -pady 5
    grid    $top.entOff   -row $row -column 1 -sticky we -padx 5 -pady 5
    incr row

    label   $top.lblScale -text "Scaling factor:"
    entry   $top.entScale -textvariable ::molywood::scale   -width 10
    grid    $top.lblScale -row $row -column 0 -sticky w -padx 5 -pady 5
    grid    $top.entScale -row $row -column 1 -sticky we -padx 5 -pady 5
    incr row

    button  $top.btnApply -text "Edit dataset in-place" -command [list ::molywood::edit_x_dataset $idx $top]
    grid    $top.btnApply -row $row -column 0 -columnspan 2 -pady 10
}

proc ::molywood::onMinMax {idx var op} {
    # grab into locals
    set minx   $::molywood::min_x
    set maxx   $::molywood::max_x
    # skip if user has cleared either field or it’s not a valid number yet
    if {![string is double -strict $minx] || ![string is double -strict $maxx]} {
        return
    }
    # compute new scale and offset
    set new_scale  [expr {1.0 * ($maxx - $minx) / ($::molywood::orig_max - $::molywood::orig_min)}]
    set new_offset [expr {1.0 * $minx - $new_scale * $::molywood::orig_min}]

    # avoid recursion
    trace remove variable ::molywood::scale  write ::molywood::onOffScale
    trace remove variable ::molywood::offset write ::molywood::onOffScale
        set ::molywood::scale  $new_scale
        set ::molywood::offset $new_offset
    trace add    variable ::molywood::scale  write ::molywood::onOffScale
    trace add    variable ::molywood::offset write ::molywood::onOffScale
}

# Callback when offset or scale changes: update min_x & max_x
proc ::molywood::onOffScale {idx var op} {
    # grab into locals
    set off   $::molywood::offset
    set scl   $::molywood::scale
    # skip if either field is empty or not numeric
    if {![string is double -strict $off] || ![string is double -strict $scl]} {
        return
    }
    # compute new min and max
    set new_min [expr {1.0 * $off + $scl * $::molywood::orig_min}]
    set new_max [expr {1.0 * $off + $scl * $::molywood::orig_max}]

    # avoid recursion
    trace remove variable ::molywood::min_x write ::molywood::onMinMax
    trace remove variable ::molywood::max_x write ::molywood::onMinMax
        set ::molywood::min_x $new_min
        set ::molywood::max_x $new_max
    trace add    variable ::molywood::min_x write ::molywood::onMinMax
    trace add    variable ::molywood::max_x write ::molywood::onMinMax
}

proc ::molywood::edit_x_dataset {idx top {col 0}} {
    set txt_widget $::molywood::dataset_text_widgets($idx)
    set content    [$txt_widget get 1.0 end]
    set vals {}
    foreach line [split $content "\n"] {
        set line [string trim $line]
        if {$line eq ""} continue
        set fields [regexp -all -inline {\S+} $line]
        if {[llength $fields] < 2} continue

        # scale column 0
        set raw [lindex $fields 0]
        if {[string is double -strict $raw]} {
            set raw [expr {$raw * $::molywood::scale + $::molywood::offset}]
        }
        # grab column 1 unchanged
        set y [lindex $fields 1]

        # store the pair
        lappend vals [list $raw $y]
    }

    # rewrite the widget with X<tab>Y
    $txt_widget delete 1.0 end
    foreach pair $vals {
        lassign $pair x y
        $txt_widget insert end \
            "[format %.3f $x]\t[format %.3f $y]\n"
    }
    destroy $top
}

proc ::molywood::make_new_ravg_dataset {idx top} {
    # creates a dataset in the dataset manager as running average of an existing one (backend)
    set src_txt  $::molywood::dataset_text_widgets($idx)
    set raw     [$src_txt get 1.0 end]
    set lines   [lmap L [split [string trimright $raw "\n"] "\n"] {
        string trim $L
    }]
    set lines   [lsearch -all -inline $lines *]  ;# remove any empty
    set xs {}
    set ys {}
    foreach line $lines {
        # if tab-separated:
        if {[regexp {^([^\t]+)\t(.+)$} $line -> x y]} {
            lappend xs $x
            lappend ys [expr {double($y)}]
        } else {
            # fallback: split on whitespace
            set parts [split $line]
            lassign $parts x y
            lappend xs $x
            lappend ys [expr {double($y)}]
        }
    }
    # running-average on ys
    set W     $::molywood::smoothing_window
    if {$W < 1} { set W 1 }
    set n     [llength $ys]
    set half  [expr {$W/2.0}]
    set ySm   {}
    for {set i 0} {$i < $n} {incr i} {
        set sum   0.0
        set count 0
        set start [expr {int($i - $half)}]
        set end   [expr {int($i + $half)}]
        for {set j $start} {$j <= $end} {incr j} {
            if {$j>=0 && $j<$n} {
                set sum   [expr {$sum + [lindex $ys $j]}]
                incr count
            }
        }
        if {$count>0} {
            lappend ySm [expr {$sum/$count}]
        } else {
            lappend ySm 0.0
        }
    }

    # register new dataset
    ::molywood::add_dataset_window
    set dst_txt $::molywood::dataset_text_widgets($::molywood::dataset_count)
    $dst_txt delete 1.0 end
    for {set i 0} {$i < $n} {incr i} {
        set x   [lindex $xs $i]
        set yv  [format %.3f [lindex $ySm $i]]
        $dst_txt insert end "$x\t$yv\n"
    }
    destroy $top
}

proc ::molywood::load_dataset_from_label {idx} {
    # reads data into a dataset in the dataset manager based on a user-created label
    set disp $::molywood::selected_label($idx)
    if {$disp eq ""} return
    set spec $::molywood::label_spec($idx,$disp)
    # Extract atom indices (stop as soon as you hit the numeric/show tokens)
    set atoms {}
    foreach elem $spec {
        if {[regexp {^(\d+)\s+(\d+)$} $elem -> mol atom]} {
            lappend atoms $atom
        } else {
            break
        }
    }
    set n [llength $atoms]
    if {$n < 2 || $n > 4} {
        tk_messageBox -message "Unsupported label (must have 2–4 atoms)" -type ok
        return
    }
    # Use measure ... frame all directly on your atom index list
    switch $n {
        2 { set vals [measure bond  $atoms frame all] }
        3 { set vals [measure angle $atoms frame all] }
        4 { set vals [measure dihed $atoms frame all] }
    }
    set txtw $::molywood::dataset_text_widgets($idx)
    $txtw delete 1.0 end
    for {set i 0} {$i < [llength $vals]} {incr i} {
        set v [lindex $vals $i]
        $txtw insert end "$i\t[format %.3f $v]\n"
    }
}

proc ::molywood::toggle_dataset_frame {idx} {
    # toggles collapse/expand in the dataset manager
    set hdr $::molywood::dataset_headers($idx)
    set cnt $::molywood::dataset_contents($idx)
    if {[winfo viewable $cnt]} {
        pack forget $cnt
        $hdr.toggle$idx config -text "+"
    } else {
        pack $cnt -after $hdr -fill x -padx 20 -pady 2
        $hdr.toggle$idx config -text "-"
    }
}

proc ::molywood::remove_dataset_window {idx} {
    # removes a panel and cleans up vars in the dataset manager
    foreach var {dataset_headers dataset_contents dataset_text_widgets selected_label dataset_color dataset_width dataset_style\
                 point_dynamic dataset_label label_spec measure_sel1 measure_sel2 measure_param1 measure_param2} {
        set fullName "::molywood::${var}($idx)"
        if {[info exists $fullName]} {
            destroy [set $fullName]
            unset $fullName
        }
    }
}

proc ::molywood::import_dataset_file {idx} {
    # imports dataset from file
    set path [tk_getOpenFile -filetypes {{{Text Files} {.txt .dat}} {{All Files} *}}]
    if {$path eq ""} return
    set f [open $path r]
    set lines {}
    while {[gets $f line] >= 0} {
        if {[regexp {^[#@!]} $line]} continue
        lappend lines $line
    }
    close $f
    set txtw $::molywood::dataset_text_widgets($idx)
    $txtw delete 1.0 end
    foreach L $lines { $txtw insert end "$L\n" }
}

proc ::molywood::save_datasets_to_file {{widget_to_close ""}} {
    # saves all datasets to a file
    set save [tk_getSaveFile -filetypes {{{Data Files} {.dat .txt}} {{All Files} *}}]
    if {$save eq ""} return
    ::molywood::do_dataset_saving $save
    set ::molywood::datafileMode "file"
    ::molywood::toggle_datafile_mode $::molywood::pending_datafile_entry
    set ent $::molywood::pending_datafile_entry.fileChooser.entry 
    $ent delete 0 end
    $ent insert 0 $save
    if {$widget_to_close != ""} {
       destroy $widget_to_close
    }
}

proc ::molywood::do_dataset_saving {filename} {
    # internal function that saves the dataset(s) with proper headers
    set out [open $filename w]
    if {$::molywood::dataset_x_label == ""} {
        set ::molywood::dataset_x_label "Frame ID"
    }
    if {$::molywood::dataset_y_label == ""} {
        set ::molywood::dataset_y_label "Value"
    }
    puts $out "# $::molywood::dataset_x_label ; $::molywood::dataset_y_label"
    for {set i 1} {$i <= $::molywood::dataset_count} { incr i } {
        if {[info exists ::molywood::dataset_text_widgets($i)]} {
            set parts {}
            set data [$::molywood::dataset_text_widgets($i) get 1.0 end]
            if {$::molywood::dataset_color($i) ne ""} {lappend parts "color=$::molywood::dataset_color($i)"}
            if {$::molywood::dataset_width($i) ne ""} {lappend parts "lw=$::molywood::dataset_width($i)"}
            if {$::molywood::dataset_style($i) ne ""} {lappend parts "ls=$::molywood::dataset_style($i)"}
            if {$::molywood::dataset_xlimit ne "" && $::molywood::dataset_xlimit ne "min,max" && [llength [split $data "\n"]] > 0} {
                lappend parts "xlim=\[$::molywood::dataset_xlimit\]"
                }
            if {$::molywood::dataset_ylimit ne "" && $::molywood::dataset_ylimit ne "min,max" && [llength [split $data "\n"]] > 0} {
                lappend parts "ylim=\[$::molywood::dataset_ylimit\]"
                }
            if {$::molywood::dataset_label($i) ne ""} {lappend parts "label=$::molywood::dataset_label($i)"}
            if {!$::molywood::point_dynamic($i)} {lappend parts "dynamic=f"}
            puts $out "! [join $parts " "]"
            foreach line [split $data "\n"] {
                if {$line ne ""} { puts $out $line }
            }
            puts $out ""
        }
    }
    close $out
}

proc ::molywood::_install_choice {w dep choiceVarName} {
    # helper fn to figure out the real deps string at install time
    set choice [set ::$choiceVarName]
    # for pip+python, install numpy+matplotlib
    if {$choice eq "pip" && $dep eq "python"} {
        set actualDeps "numpy matplotlib"
    } else {
        set actualDeps $dep
    }
    ::molywood::installer $w $actualDeps $choice
}

proc ::molywood::install_dep {dep} {
    # dependency installer manager through which the user can choose
    # the preferred installation method
    set w [toplevel .installq]
    wm title $w "Install Dependency"
    wm resizable $w 0 0
    ttk::style configure TLabel   -font {Arial 11}
    ttk::style configure TButton  -font {Arial 10} -padding {5 10}
    ttk::style configure TLabelframe        -padding {10 10 10 10}
    ttk::style configure TLabelframe.Label  -font {Arial 12 bold}

    ttk::frame $w.main -padding {20 20 20 20}
    pack $w.main -fill both -expand 1
    ttk::label $w.main.header -text "Install $dep" -font {Arial 14 bold}
    grid $w.main.header -row 0 -column 0 -columnspan 2 -pady 10
    ttk::labelframe $w.main.methods -text "Choose installation method"
    grid $w.main.methods -row 1 -column 0 -columnspan 2 -sticky nws -pady 5

    set selVar installChoice
    set ::$selVar conda

    # conda option (always)
    ttk::radiobutton $w.main.methods.conda -text   "Create new conda environment (might take a while)" -variable $selVar -value conda
    grid $w.main.methods.conda -row 0 -column 0 -sticky w -pady 2
    # pip option (when appropriate)
    if {$dep eq "molywood" || $dep eq "python"} {
        ttk::radiobutton $w.main.methods.pip -text "Install via pip" -variable $selVar -value pip
        grid $w.main.methods.pip -row 1 -column 0 -sticky w -pady 2
    }
    # pkg option (for ffmpeg & magick)
    if {$dep eq "ffmpeg" || $dep eq "magick"} {
        ttk::radiobutton $w.main.methods.pkg -text "Use system package installer (run VMD with sudo)" -variable $selVar -value pkg
        grid $w.main.methods.pkg -row 2 -column 0 -sticky w -pady 2
    }

    ttk::separator $w.main.sep -orient horizontal
    grid $w.main.sep -row 2 -column 0 -columnspan 2 -sticky ew -pady 10
    ttk::button $w.main.install -text "Install" -width 12 -command [list ::molywood::_install_choice $w $dep $selVar]
    ttk::button $w.main.cancel -text "Cancel" -width 12 -command [list destroy $w]
    grid $w.main.install -row 3 -column 0 -sticky e -padx 5
    grid $w.main.cancel  -row 3 -column 1 -sticky w -padx 5
    update idletasks
    set x [expr {([winfo screenwidth $w]  - [winfo width $w])  / 2}]
    set y [expr {([winfo screenheight $w] - [winfo height $w]) / 2}]
    wm geometry $w +$x+$y
    tkwait window $w
    ::molywood::verify_deps
}

proc ::molywood::installer {widget dependency mode} {
    # performs the installation depending on the chosen method
    set osname [string trimleft [array get ::tcl_platform os] "os "]
    if {$mode == "conda"} {
        puts "MOLY-Logger\) Creating a new conda environment..."
        ::molywood::make_conda_env
    } elseif {$mode == "pip"} {
        puts "MOLY-Logger\) Installing $dependency using pip..."
        exec pip install $dependency
    } elseif {$mode == "pkg"} {
        if {$osname == "darwin"} {
            puts "MOLY-Logger\) Installing $dependency using Homebrew..."
            exec brew install $dependency
        } elseif {$osname == "linux"} {
            puts "MOLY-Logger\) Installing $dependency using apt-get..."
            exec sudo apt-get update
            exec sudo apt-get install -y $dependency
        } else {
            puts "MOLY-Logger\) Unsupported OS: $osname for package-based installation."
        }
    } else {
        puts "MOLY-Logger\) Unknown installation mode: $mode"
    }
    destroy $widget
}

proc ::molywood::make_conda_env {} {
    # writes a temporary .yml file and uses it to create a new environment,
    # then sets it as the current env for all future tasks to be run with it
    set filename "temp_cnd_environment.yml"
    if {[catch {set fh [open $filename w]} err]} {
        tk_messageBox -type ok -icon error -title "Error" -message "Cannot open $filename for writing:\n$err"
        return
    }
    puts $fh "name: molywood"
    puts $fh "channels:"
    puts $fh "  - conda-forge"
    puts $fh "dependencies:"
    foreach pkg {imagemagick ffmpeg matplotlib numpy pip "pip:"} {
        puts $fh "  - $pkg"
    }
    puts $fh "    - molywood"
    close $fh
    #    Note: exec will block until conda finishes
    if {[catch {exec conda env create -q -f $filename 2>@1} result]} {
        tk_messageBox -type ok -icon error -title "Conda Error" -message "Failed to create environment:\n$result"
    } else {
        tk_messageBox -type ok -icon info -title "Done" -message "Conda environment 'molywood' created successfully!"
        set w [dict get $::molywood::ui_elements window]
        set gbdp $w.nb.general.depsFrame
        lassign [::molywood::get_conda_envs] env_list path_list
        set displayList {}
        set selEnv ""
        for {set i 0} {$i < [llength $env_list]} {incr i} {
            set n [lindex $env_list  $i]
            set p [lindex $path_list $i]
            lappend displayList "$n ($p)"
            if {$n == "molywood"} {
                puts "MOLY-Logger\) Setting 'molywood' as the default environment"
                set selEnv "$n ($p)"
            }
        }
        $gbdp.cond configure -values $displayList
        set ::molywood::selectedEnv $selEnv
        ::molywood::set_conda_prefix
    }
    file delete $filename
}

proc ::molywood::set_conda_prefix {} {
    # sets conda_prefix to the conda env path, and rechecks
    # dependencies within the new environment
    if {[regexp {^[^(]+\(([^)]+)\)$} $molywood::selectedEnv -> path]} {
        set ::molywood::conda_prefix $path
    }
    set missing_deps [::molywood::verify_deps]
    if {$missing_deps == 0} {
        ::molywood::save_conda_setting
    }
}

proc ::molywood::save_conda_setting {} {
    set path $::molywood::conda_prefix
    if {$path eq ""} {
        return
    }
    set answer [tk_messageBox -icon question -type yesno -title "Save Conda Settings?" \
        -message "Save this conda prefix to your .molywoodrc configuration file?\n\n$path"
    ]
    if {$answer ne "yes"} {
        return
    }
    set rcfile [::molywood::which_molywoodrc]
    if {$rcfile eq ""} {
        # tk_messageBox -icon error -title "No .molywoodrc Found" -message "Could not locate any .molywoodrc (or molywood.rc) file.\nNo changes made."
        # return
        ::molywood::create_molywoodrc
        set rcfile [::molywood::which_molywoodrc]
    }
    if {[catch {
        set fh [open $rcfile a]
        puts $fh "set molywood_conda \"$path\""
        close $fh
    } err]} {
        tk_messageBox -icon error -title "Write Error" -message "Failed to write to $rcfile:\n$err"
        return
    }
    tk_messageBox -icon info -title "Saved" -message "Appended conda setting to:\n$rcfile"
}

proc ::molywood::command_exists {cmd} {
    # uses 'which' to identify whether a command exists or not
    if {$::molywood::conda_prefix == ""} {
        if {[catch {exec which $cmd} result]} {
            return 0
        }
    } else {
        if {[catch {exec conda run --no-capture-output -p $::molywood::conda_prefix which $cmd} result]} {
            return 0
        }
    }
    return 1
}

 proc ::molywood::run_tool {args} {
    # wrapper to run molywood commands in a smarter way, accounting for 'conda run'
    # if an environment is chosen, and preventing warnings to be turned into errors
    set cmdList {}
    if {[string length $::molywood::conda_prefix]} {
        lappend cmdList conda run --no-capture-output -p $::molywood::conda_prefix
    }
    foreach a $args { lappend cmdList $a }
    puts "MOLY-Logger\) Molywood-GUI: running command $cmdList"
    if {[catch {exec {*}$cmdList 2>@1} result]} {
        # tk_messageBox -type ok -icon error -title "Command Failed" \
        #     -message "Failed to run:\n    [join $cmdList { }]\n\nOutput was:\n$result"
        # return -code error $result
        # error -level 0 $options(-errorInfo)
        return -code error "$result"
    }
    return $result
}

proc ::molywood::add_scene {} {
    # produces a new scene based on the current window settings
    # and appends to the global list of scenes, then updates widgets
    set scene_setup $::molywood::scene_name_var
    variable scenes
    if {$scene_setup eq ""} {
        tk_messageBox -message "Scene name cannot be empty!"
        return
    }
    set scene_names [::molywood::get_all_scene_names]
    set count 1
    while {[lsearch -exact $scene_names $scene_setup] != -1} {
        set scene_setup "scene$count"
        incr count
    }
    # foreach scene_obj $scenes {
    #     set existing_name [$scene_obj get_scene_name]
    #     if {$existing_name eq $scene_setup} {
    #         tk_messageBox -message "A scene with the name '$scene_setup' already exists!"
    #         return
    #     }
    # }
    set new_scene [scene new $scene_setup [::molywood::new_scene_params]]
    lappend scenes $new_scene
    ::molywood::update_scenes_table
    puts "MOLY-Logger\) Added Molywood scene: $scene_setup"
}

proc ::molywood::edit_scene {} {
    # takes the current states of the widgets and applies them
    # to the currently selected scene, then updates widgets
    variable scenes
    set scene_setup $::molywood::scene_name_var
    if {$scene_setup eq ""} {
        tk_messageBox -message "Scene name cannot be empty!"
        return
    }

    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    $sc set_scene_parameters [::molywood::new_scene_params]
    $sc set_scene_name $scene_setup 

    ::molywood::update_scenes_table
}


proc ::molywood::remove_scene {} {
    # removes the currently selected scene (only 1 at a time)
    # by copying all but selected scenes to new_scene, then updates widgets
    set new_scenes {}
    foreach scene $::molywood::scenes {
        if {[$scene get_scene_name] != $::molywood::currently_selected_scene} {
            lappend new_scenes $scene
        }
    }
    set ::molywood::scenes $new_scenes
    if {[llength $::molywood::scenes] > 0} {
        set ::molywood::currently_selected_scene [lindex ::molywood::scenes 0]
    } else {
        set ::molywood::currently_selected_scene ""
    }
    ::molywood::populate_actions
    ::molywood::update_scenes_table
}

proc ::molywood::clone_scene {} {
    # creates a new scene with a name taken from the widget,
    # copies all actions and properties from the selected scene,
    # adds it to the list of scenes, then updates widgets
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    set depdict [dict merge {*}[::molywood::resolve_deps]]
    foreach {key value} [dict get $depdict] {
        if {$::molywood::currently_selected_scene == $value || $::molywood::currently_selected_scene == $key} {
            error "Cannot clone scenes involved in dependencies (after/overlay)"
        }
    }
    set new_scene_name $::molywood::scene_name_var
    set sc_clone [scene new $::molywood::scene_name_var [::molywood::existing_scene_params $sc]]

    set scene_names [::molywood::get_all_scene_names]
    set new_scene_name $::molywood::scene_name_var
    set count 1
    while {[lsearch -exact $scene_names $new_scene_name] != -1} {
        set new_scene_name "scene$count"
        incr count
    }


    $sc_clone set_scene_name $new_scene_name
    for {set i 0} {$i < [llength [$sc get_all_action_times]]} {incr i} {
        $sc_clone add_action_wrapper [$sc get_action_wrapper $i] $i
    }
    lappend ::molywood::scenes $sc_clone

    ::molywood::populate_actions
    ::molywood::update_scenes_table
}

proc ::molywood::update_afters {} {
    # Sets correct values of possible choices for the dropdown "after"
    # (all scenes except for the currently selected ones)
    set w [dict get $::molywood::ui_elements window]
    $w.nb.addScene.topFrame.afterSceneFrame.comboAfterScene configure -values [list "" {*}[::molywood::get_all_other_scene_names $::molywood::currently_selected_scene]]
}
  
proc ::molywood::update_on_scene_selection {treeview_widget tab_widget} {
    # Updates all things that need updating (e.g. list of actions)
    # when a different scene is selected in the list
    set selected_item [$treeview_widget selection]
    set w [dict get $::molywood::ui_elements window]

    if {[llength $selected_item] == 0 && $::molywood::currently_selected_scene != ""} {
        foreach treeitem [$treeview_widget children {}] {
            set tmp_scene_name [$treeview_widget item $treeitem -text]
            if {$::molywood::currently_selected_scene == $tmp_scene_name} {
                $treeview_widget selection set $treeitem
                set selected_item [$treeview_widget selection]
            }
        }
    }

    if {[llength $selected_item] > 0} {
        set scene_name [$treeview_widget item $selected_item -text]
        set scene_attributes [lindex [$treeview_widget item $selected_item -values] 0]
        set sc [::molywood::get_scene_by_name $scene_name]
        set scene_params [$sc get_scene_parameters]

        set ::molywood::currently_selected_scene $scene_name

        set ::molywood::scene_name_var $scene_name
        set ::molywood::scene_row [dict get $scene_params rnum]
        set ::molywood::scene_column [dict get $scene_params cnum]
        set ::molywood::resolution_x [dict get $scene_params xres]
        set ::molywood::resolution_y [dict get $scene_params yres]
        set ::molywood::after_scene [dict get $scene_params aftr]
        set ::molywood::ambient_occlusion [dict get $scene_params aocc]
        set ::molywood::max_transparent_surfaces [dict get $scene_params mxts]
        set ::molywood::source_file [dict get $scene_params srcf]
        set ::molywood::scene_source__ [dict get $scene_params srcs]
        set srcdict {
            visualization "Existing VMD state" 
            structure "Structure file" 
            pdb_code "PDB 4-letter code" 
            cubes "Set of .cube files" 
            "current VMD state" "Current VMD state" 
        }
        set ::molywood::scene_source [dict get $srcdict [dict get $scene_params srcs]]
        if {[$sc is_dep]} {
            $w.nb.addScene.topFrame.scenePositionFrame.sceneRow configure -state disabled
            $w.nb.addScene.topFrame.scenePositionFrame.sceneCol configure -state disabled
            set master_scene_param [[::molywood::get_scene_by_name [$sc depends_on]] get_scene_parameters]
            set ::molywood::scene_row [dict get $master_scene_param rnum]
            set ::molywood::scene_column [dict get $master_scene_param cnum]
        } else {
            $w.nb.addScene.topFrame.scenePositionFrame.sceneRow configure -state readonly
            $w.nb.addScene.topFrame.scenePositionFrame.sceneCol configure -state readonly
        }

        ::molywood::populate_actions
        ::molywood::update_afters 

    } else {
        puts "MOLY-Logger\) No item selected"
    }
}

proc ::molywood::toggle_advanced_global_settings {} {
    # Switches between the full and hidden view of 
    # the general properties tab
    set w [dict get $::molywood::ui_elements window]
    if {$::molywood::show_advanced_global} {
        grid $w.nb.general.topFrame.advancedFrame -row 6 -column 0 -sticky ew -padx 5 -pady 5 -columnspan 2
    } else {
        grid remove $w.nb.general.topFrame.advancedFrame
    }
}

proc ::molywood::set_rows_cols {} {
    # Sets correct values of possible choices for the dropdown column/row
    # (depending on the chosen layout via scene_rows/scene_columns variables)
    set w [dict get $::molywood::ui_elements window]
    $w.nb.addScene.topFrame.scenePositionFrame.sceneRow configure -values [::molywood::get_consec $::molywood::scene_rows]
    $w.nb.addScene.topFrame.scenePositionFrame.sceneCol configure -values [::molywood::get_consec $::molywood::scene_columns]
}

proc ::molywood::on_canvas_click {canvasName} {
    # Handle mouse clicks and highlight the canvas

    # Reset the previous highlighted canvas
    if {$::molywood::current_canvas ne ""} {
        $::molywood::current_canvas configure -bg white -highlightbackground white -highlightthickness 2
    }

    # # Highlight the clicked canvas
    $canvasName configure -bg gray40 -highlightbackground gray40 -highlightthickness 2
     set ::molywood::current_canvas $canvasName
    
    if {[string index $canvasName end] == "0"} {
        set ::molywood::scene_rows 1
        set ::molywood::scene_columns 1
    } elseif {[string index $canvasName end] == "1"} {
        set ::molywood::scene_rows 1
        set ::molywood::scene_columns 2
    } elseif {[string index $canvasName end] == "2"} {
        set ::molywood::scene_rows 2
        set ::molywood::scene_columns 1
    } elseif {[string index $canvasName end] == "3"} {
        set ::molywood::scene_rows 2
        set ::molywood::scene_columns 2
    } elseif {[string index $canvasName end] == "4"} {
        ::molywood::custom_layout
    }
    ::molywood::set_rows_cols
    if {[llength $::molywood::scenes] > 1} {
            ::molywood::set_default_locs $::molywood::scene_rows $::molywood::scene_columns
        }
}

proc ::molywood::set_default_locs {nrows ncols} {
    set maxscenes [expr $nrows * $ncols]
    set scenecount 0
    foreach scene $::molywood::scenes {
        $scene set_scene_parameter rnum [expr $scenecount / $ncols]
        $scene set_scene_parameter cnum [expr $scenecount % $ncols]
        incr scenecount
    }

}

proc ::molywood::custom_layout {} {
    # Creates a widget to set a layout different than the defaults
    set w [toplevel .customlayout]
    wm geometry $w 270x200
    wm title $w "Set custom layout"
    set ::molywood::custom_layout_nrows $::molywood::scene_rows
    set ::molywood::custom_layout_ncols $::molywood::scene_columns
    # Create a dropdown to choose an action
    label $w.lblAction -text "Choose layout:"
    grid $w.lblAction -row 0 -column 0 -columnspan 2 -pady 25 -padx 15 -sticky ew

    label $w.rows_lbl -text "Number of rows:"
    entry $w.rows_entry -text $::molywood::scene_rows -textvariable ::molywood::custom_layout_nrows -width 10
    grid $w.rows_lbl -row 1 -column 0 -sticky e -padx 15 -pady 5
    grid $w.rows_entry -row 1 -column 1 -sticky w -padx 15 -pady 5

    label $w.cols_lbl -text "Number of columns:"
    entry $w.cols_entry -text "$::molywood::scene_columns" -textvariable ::molywood::custom_layout_ncols -width 10
    grid $w.cols_lbl -row 2 -column 0 -sticky e -padx 15 -pady 5
    grid $w.cols_entry -row 2 -column 1 -sticky w -padx 15 -pady 5

    # Button to confirm action
    button $w.select -text "Set layout" -command [list ::molywood::set_layout $w]
    grid $w.select -row 3 -column 0 -columnspan 2 -pady 25 -padx 15 -sticky ew

    tkwait window $w
}

proc ::molywood::set_layout {widget} {
    # Enacts all the changes to variables and dropdowns when layout is chosen
    set ::molywood::scene_rows $::molywood::custom_layout_nrows
    set ::molywood::scene_columns $::molywood::custom_layout_ncols
    ::molywood::set_rows_cols
    destroy $widget
}

proc ::molywood::switch_source_file_widget {var_name ext op} {
    # Based on the value of scene source, a different scene file widget
    # should be shown and read; in the case of "current VMD state",
    # it allows to save the state to a default filename, then uses that filename
    if {[dict exists $::molywood::ui_elements window]} {
        set w [dict get $::molywood::ui_elements window]
    } else {
        set w $var_name
    }
    set frame $w.nb.addScene.topFrame.sourceFileFrame 
    foreach ch [winfo children $frame] {
        destroy $ch
    }

    set sources_dict {
        "Existing VMD state" visualization 
        "Structure file" structure 
        "PDB 4-letter code" pdb_code 
        "Set of .cube files" cubes 
        "Current VMD state" "current VMD state"
    }

    set ::molywood::scene_source__ [dict get $sources_dict $::molywood::scene_source]

    # Depending on the value of scene_source, either show an entry or a file dialog button:
    switch $::molywood::scene_source__ {
        pdb_code {
            label $frame.lblSourceFile -text "4-letter PDB code:" -height 2
            pack $frame.lblSourceFile -side left -padx 5
            entry $frame.entrySourceFile -textvariable ::molywood::source_file -width 15
            pack $frame.entrySourceFile -side right -padx 5
            if {$ext == 0} {
                set ::molywood::source_file "1ubq"
            }
            
        }
        cubes {
            label $frame.lblSourceFile -text "Pattern for .cube files:" -height 2
            pack $frame.lblSourceFile -side left -padx 5
            entry $frame.entrySourceFile -textvariable ::molywood::source_file -width 22
            pack $frame.entrySourceFile -side right -padx 5
            if {$ext == 0} {
                set ::molywood::source_file "myorbital*cube"
            }
        }
        visualization {
            label $frame.lblSourceFile -text "Source .vmd file:" -height 2
            pack $frame.lblSourceFile -side left -padx 5
            button $frame.btnOpenFile -text "Choose" -command {
                set ::molywood::source_file [tk_getOpenFile]
            }
            pack $frame.btnOpenFile -side right -padx 5
            entry $frame.entrySourceFile -textvariable ::molywood::source_file -width 15
            pack $frame.entrySourceFile -side right -padx 5
            if {$ext == 0} {
                set ::molywood::source_file ""
            }
        }
        structure {
            label $frame.lblSourceFile -text "Source structure file:" -height 2
            pack $frame.lblSourceFile -side left -padx 5
            button $frame.btnOpenFile -text "Choose" -command {
                set ::molywood::source_file [tk_getOpenFile]
            }
            pack $frame.btnOpenFile -side right -padx 5
            entry $frame.entrySourceFile -textvariable ::molywood::source_file -width 15
            pack $frame.entrySourceFile -side right -padx 5
            if {$ext == 0} {
                set ::molywood::source_file ""
            }
        }
        "current VMD state" {
            label $frame.lblSourceFile -text "Save current state" -height 2
            pack $frame.lblSourceFile -side left -padx 5
            button $frame.btnOpenFile -text "Save" -command ::molywood::save_current_vis
            pack $frame.btnOpenFile -side right -padx 5
        }
        
    }
    set original_font [$w.nb.addScene.topFrame.sceneSourceFrame.lblSceneSource cget -font]
    $frame.lblSourceFile configure -font $original_font
}

proc ::molywood::save_current_vis {{update_all 0} {widget 0}} {
    # Saves vis state to a standard-named .vmd file
    set timestr [clock format [clock seconds] -format "%y-%m-%d_%H:%M:%S"]
    save_state state_${::molywood::currently_selected_scene}_${timestr}.vmd
    set ::molywood::scene_source "Existing VMD state"
    set ::molywood::source_file "state_${::molywood::currently_selected_scene}_${timestr}.vmd"
    set w [dict get $::molywood::ui_elements window]
    ::molywood::switch_source_file_widget $w 1 0
    if {$update_all} {
        set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
        $sc set_scene_parameters [::molywood::new_scene_params] 
        ::molywood::update_scenes_table
        if {$widget != 0} {
            $widget.ifsource configure -text "Your current scene preview is based on: $::molywood::scene_source $::molywood::source_file.\n\nIf you want to use the current VMD state instead, click here to change the source:"
            $widget.button configure -bg gray85 -relief flat

        }
    }
}

proc ::molywood::populate_actions {} {
    # Builds the list of actions in the treeview widget
    # based on the actions stored in the currently selected scene
    set w [dict get $::molywood::ui_elements window]
    set active_action_indices [::molywood::get_active_actions_indices]
    set items [$w.nb.sceneProp.container.actl children {}]
    if {[llength $items] > 0} {
        $w.nb.sceneProp.container.actl delete $items
    }

    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    if {$sc == ""} {
        return
    }
    set action_names [$sc get_all_action_names]
    set params_names [$sc get_all_action_params]
    set times [$sc get_all_action_times]

    # Loop through the list of scenes and add each to the treeview
    for {set i 0} {$i < [llength $action_names]} {incr i} {
        set allparams [lindex $params_names $i] 
        set allactions [lindex $action_names $i]
        set action_counter 0
        set allnondefparams {}
        foreach action_name $allactions {
            set nondefparams {}
            set defaults [dict get [dict get $::molywood::actions $action_name] defaults]
            set alwayson [dict get [dict get $::molywood::actions $action_name] alwayson]
            set parameters [lindex $allparams $action_counter]
            # Loop through the keys in parameters and compare with defaults
            # TODO this defo should have better logic
            foreach {key value} [dict get $parameters] { 
                if {[dict exists $alwayson $key]} {
                    dict set nondefparams $key $value
                } elseif {![dict exists $defaults $key] || $value ne [dict get $defaults $key]} {
                    # If key is not in defaults or values don't match, add to nondefparams
                    if {$value != {} && $value != "" } {
                        if {![dict exists $defaults $key]} {
                            dict set nondefparams $key $value
                        } else {
                            if { !($value == 0 && [dict get $defaults $key] == "f") && !($value == 1 && [dict get $defaults $key] == "t") } {
                                dict set nondefparams $key $value
                            }
                        }
                    }
                }
            }
            lappend allnondefparams $nondefparams
            incr action_counter
        }
        set actions [join $allactions " || "]
        set params [join $allnondefparams " || "]

        set time [concat [lindex $times $i]]
        # Alternating between the two styles
        if {[expr $i % 2] == 0} {
            $w.nb.sceneProp.container.actl insert "" end -text $actions -values [list "$time s" $params] -tags "light_gray_tag"
        }  else {
            $w.nb.sceneProp.container.actl insert "" end -text $actions -values [list "$time s" $params] -tags "dark_gray_tag"
        }
    }
    
    # Call after populating the Treeview
    ::molywood::adjust_column_width $w.nb.sceneProp.container.actl 2
    $w.nb.sceneProp.container configure -width 700
    # resetting the same active entries
    foreach active_wrapper_index $active_action_indices {
        $w.nb.sceneProp.container.actl selection set [lindex [$w.nb.sceneProp.container.actl children {}] $active_wrapper_index]
    }
}

proc ::molywood::adjust_column_width {{tree 0} {column_id 2}} { 
    if {$tree == 0} {
        set w [dict get $::molywood::ui_elements window]
        set tree $w.nb.sceneProp.container.actl
    }
    set max_width 0
    foreach iid [$tree children {}] {
        set text [$tree item $iid -values]
        set text_width [string length [lindex $text [expr $column_id - 1]]]
        set max_width [expr {max($max_width, $text_width * 7)}] 
    }
    $tree column #$column_id -width [expr {max($max_width, 475)}]
}

proc ::molywood::new_scene_params {} {
    # collects all currently defined variables into a struct
    # that's understood by the scene's constructor
    array set params {}
    set params(xres) $::molywood::resolution_x
    set params(yres) $::molywood::resolution_y
    set params(rnum) $::molywood::scene_row
    set params(cnum) $::molywood::scene_column
    set params(mxts) $::molywood::max_transparent_surfaces
    set params(aocc) $::molywood::ambient_occlusion
    set params(aftr) $::molywood::after_scene
    set params(srcs) $::molywood::scene_source__
    set params(srcf) $::molywood::source_file
    return [array get params]
}

proc ::molywood::existing_scene_params {scene} {
    set ex_params [$scene get_scene_parameters]
    foreach {key value} [dict get $ex_params] {
        set params($key) $value
    }
    return [array get params]
}

proc ::molywood::update_mo_table {} {
    # Updates the table of master overlays
    set w [dict get $::molywood::ui_elements window]
    set tree $w.nb.general.topFrame.advancedFrame.movr
    set items [$tree children {}]
    if {[llength $items] > 0} {
        $tree delete $items
    }
    set counter 1
    set defaults [dict get [dict get $::molywood::actions_mo add_master_overlay] defaults]

    foreach mo $::molywood::master_overlays {
        set nondefaults {}
        foreach {key val} [dict get $mo] {
            if {[dict exists $defaults $key]} {
                if {$val != [dict get $defaults $key] && $val != {}} {
                    dict set nondefaults $key $val
                }
            }
        }
        $tree insert "" end -text "MO_$counter" -values [list "[dict get $mo start_time] - [expr [dict get $mo start_time] + [dict get $mo t]] s" $nondefaults] -open 1
        incr counter
    }
}

proc ::molywood::update_scenes_table {} {
    # rebuilds the treeview with scenes taken from
    # the global scenes list
    set w [dict get $::molywood::ui_elements window]

    set items [$w.scns.scnl children {}]
    if {[llength $items] > 0} {
        $w.scns.scnl delete $items
    }
    # Loop through the list of scenes and add each to the treeview
    foreach scene_obj $::molywood::scenes {
        if {![$scene_obj is_dep]} {
            $scene_obj set_time_offset 0
        }
    }
    lassign [::molywood::resolve_deps] afters withins
    set active 0
    set count 0
    foreach scene_obj $::molywood::scenes {
        set scene_name [$scene_obj get_scene_name]
        set scene_attributes [$scene_obj get_scene_attributes]
        if {$scene_name == $::molywood::currently_selected_scene} {
            set active $count
        }
        incr count
        set scene_duration [$scene_obj calc_total_time]
        set time_offset [$scene_obj get_time_offset]
        set final_time [expr $time_offset + $scene_duration]
        # Insert scene name in column #0 and scene attributes in column #1
        if {[dict exists $afters $scene_name]} {
            $w.scns.scnl insert [dict get $afters $scene_name] end -id $scene_name -text "$scene_name" -values [list "$time_offset - $final_time s (cont.)" $scene_attributes] -open 1 -tags "after"
            $w.scns.scnl tag configure after -background lavender
        } elseif {[dict exists $withins $scene_name]} {
            $w.scns.scnl insert [dict get $withins $scene_name] end -id $scene_name -text "$scene_name" -values [list "$time_offset - $final_time s (inset)" $scene_attributes] -open 1 -tags "within"
            $w.scns.scnl tag configure within -background "pale turquoise"
        } else {
            $w.scns.scnl insert "" end -id $scene_name -text $scene_name -values [list "$time_offset - $final_time s" $scene_attributes] -open 1
        }
    }
    set items [$w.scns.scnl children {}]
    $w.scns.scnl selection set [lindex $items $active]
    ::molywood::update_afters
}

proc ::molywood::get_scene_by_name { scene_name } {
    # accessory, returns the scene that has the requested name
    foreach sc $::molywood::scenes {
        if {[$sc get_scene_name] == $scene_name} {
            return $sc
        }
    }
    return 
}

proc ::molywood::open_action_popup {{mode add} {default_action "do_nothing"} {wrapper_index 0} {action_index 0} {current_action ""} {current_params ""}} {
    # opens a popup to choose an action, has two modes (add/edit)
    # so accepts a bunch of default parameters for adding, for editing
    # they will be passed with custom values
    if {[::molywood::get_scene_by_name $::molywood::currently_selected_scene] == ""} {
        error "Please select or create a scene to which actions can be added!"
    }
    set w [toplevel .popup]
    if {$mode == "add"} {
        wm title $w "Add Action"
    } elseif {$mode == "edit"} {
        wm title $w "Edit Action"
    } elseif {$mode == "mo"} {
        wm title $w "Add Master Overlay"
    } elseif {$mode == "edit_mo"} {
        wm title $w "Edit Master Overlay"
    }
    wm geometry $w 350x750
    ### ::molywood::open_action_popup edit_mo $item $mo_counter "" $item [lindex $::molywood::master_overlays $mo_counter]
    # Create a dropdown to choose an action
    label $w.lblAction -text "Choose Action:"
    grid $w.lblAction -row 0 -column 0 -pady 25 -sticky ew

    # Dropdown (combobox) for selecting action
    if {$mode != "mo" && $mode != "edit_mo"} {
        ttk::combobox $w.actionCombo -values [dict keys $::molywood::actions] -state readonly -height [llength [array names ::molywood::actions]]
    } else {
        ttk::combobox $w.actionCombo -values [dict keys $::molywood::actions_mo] -state readonly -height [llength [array names ::molywood::actions_mo]]
    }
    grid $w.actionCombo -row 0 -column 1 -sticky ew -padx 5 -pady 5 

    # Frame to hold dynamically created parameter entry widgets
    frame $w.paramFrame
    grid $w.paramFrame -row 1 -column 0 -columnspan 2

    # Button to confirm action
    if {$mode == "add"} {
        button $w.btnAdd -text "Add Action" -command [list ::molywood::add_action_from_popup $w ]
    } elseif {$mode == "edit"} {
        button $w.btnAdd -text "Save Properties" -command [list ::molywood::save_action_properties_from_popup $w $wrapper_index $action_index]
    } elseif {$mode == "mo"} {
        button $w.btnAdd -text "Add Master Overlay" -command [list ::molywood::add_action_from_popup $w]
    } elseif {$mode == "edit_mo"} {
        button $w.btnAdd -text "Edit Master Overlay" -command [list ::molywood::save_action_properties_from_popup $w $wrapper_index $action_index]
    }
    grid $w.btnAdd -row 2 -column 0 -columnspan 2 -pady 10

    bind $w <Return> {.popup.btnAdd invoke}
    # When an action is selected from the dropdown, update the parameter entry fields
    bind $w.actionCombo <<ComboboxSelected>> [list ::molywood::update_params $w.paramFrame $w.actionCombo]

    # Set the initial action to the default
    if {$mode == "add"} {
        $w.actionCombo set $default_action
        ::molywood::update_params $w.paramFrame $w.actionCombo
    } elseif {$mode == "edit"} {
        $w.actionCombo set $current_action
        ::molywood::update_params $w.paramFrame $w.actionCombo $current_params
    } elseif {$mode == "mo"} {
        $w.actionCombo set "add_master_overlay"
        ::molywood::update_params $w.paramFrame $w.actionCombo
    } elseif {$mode == "edit_mo"} {
        $w.actionCombo set "add_master_overlay"
        ::molywood::update_params $w.paramFrame $w.actionCombo $current_params
    }

    tkwait window $w
}

proc ::molywood::add_action_from_popup {w } {
    # Executed once "Add Action" is clicked in the popup,
    # creates a new action and appends it to the scene,
    # then updates widgets
    set chosen_action [$w.actionCombo get]
    if {$chosen_action == "add_master_overlay"} {
        set params [dict get [dict get $::molywood::actions_mo $chosen_action] params]
    } else {
        set params [dict get [dict get $::molywood::actions $chosen_action] params]
    }
    set action_name [$w.actionCombo get]
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    set action_params [dict create]
    foreach param $params {
        set value "::molywood::add_action_widget_$param"
        if {$param == "mode" || $param == "sigmoid" || $param == "style"} {
            dict set action_params $param [dict get $::molywood::substitute_values [set $value]]
        } else {
            dict set action_params $param [set $value]
        }
    }
    # Extra check here, otherwise the on-the-fly time calculation
    # will keep crashing and sending errors
    if {[dict exists $action_params t]} {
        if {[dict get $action_params t] != "" && ![string is double [dict get $action_params t]]} {
            error "Time has to be a number"
        }
    }

    if {$chosen_action == "add_master_overlay"} {
        lappend ::molywood::master_overlays $action_params
        ::molywood::update_mo_table
    } else {
        $sc add_action [llength [$sc get_all_action_times]] $action_name $action_params
        ::molywood::populate_actions
        ::molywood::update_scenes_table
        set wmain [dict get $::molywood::ui_elements window]
        update
        set actlist [$wmain.nb.sceneProp.container.actl children {}]
        $wmain.nb.sceneProp.container.actl selection set [lindex $actlist [expr [llength $actlist] - 1]]
    }
    # After adding the action, close the popup
    destroy $w
}

proc ::molywood::save_action_properties_from_popup {w wrapper_index action_index} {
    # Same logic as add_actions_from_popup but serves for editing,
    # so needs some extra parameters to pass down
    set chosen_action [$w.actionCombo get]
    if {$chosen_action == "add_master_overlay"} {
        set params [dict get [dict get $::molywood::actions_mo $chosen_action] params]
    } else {
        set params [dict get [dict get $::molywood::actions $chosen_action] params]
    }
    set action_params [dict create]
    foreach param $params {
        set value "::molywood::add_action_widget_$param"
        if {$param == "mode" || $param == "sigmoid" || $param == "style"} {
            dict set action_params $param [dict get $::molywood::substitute_values [set $value]]
        } else {
            dict set action_params $param [set $value]
        }
    }

    if {$chosen_action == "add_master_overlay"} {
        lset ::molywood::master_overlays $wrapper_index $action_params
    } else {
        set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
        $sc edit_action $wrapper_index $action_index $chosen_action $action_params
        ::molywood::populate_actions
        ::molywood::update_scenes_table
    }
    destroy $w
}

proc ::molywood::update_params {frame_to_update frame_combo {defaults ""}} {
    # Shows the correct list of parameters/entry widgets
    # when a different action is chosen from the dropdown
    foreach w [winfo children $frame_to_update] {
        destroy $w
    }
    set action_name [$frame_combo get]
    if {$action_name == "add_master_overlay"} {
        set params [dict get [dict get $::molywood::actions_mo $action_name] params]
        if {$defaults == ""} {
            set defaults [dict get [dict get $::molywood::actions_mo $action_name] defaults]
        }
        set types [dict get [dict get $::molywood::actions_mo $action_name] types]
    } else {
    # Get the list of parameters, types and defaults (if not provided externally) for the selected action
        set params [dict get [dict get $::molywood::actions $action_name] params]
        if {$defaults == ""} {
            set defaults [dict get [dict get $::molywood::actions $action_name] defaults]
        }
        set types [dict get [dict get $::molywood::actions $action_name] types]
    }
    ::molywood::destroy_all_tooltips $frame_to_update
    # Dynamically create widgets for each parameter
    # TODO run below twice, once first if param name not in secondary, then if is in secondary
    set secs [dict get [dict get $::molywood::actions $action_name] secondary]
    foreach paramkind {"primary" "secondary"} {
        if {$paramkind == "secondary" && [llength $secs] > 0} {
            frame $frame_to_update.sep1 -height 2 -bd 0 -relief flat -background "dim gray"
            grid $frame_to_update.sep1 -row [llength [winfo children $frame_to_update]] -column 0 -sticky ew -padx 25 -pady 5 -columnspan 2
            label $frame_to_update.lbl -text "Advanced/less common parameters:"
            grid $frame_to_update.lbl -row [llength [winfo children $frame_to_update]] -column 0 -sticky ew -padx 25 -pady 2 -columnspan 2
            frame $frame_to_update.sep2 -height 2 -bd 0 -relief flat -background "dim gray"
            grid $frame_to_update.sep2 -row [llength [winfo children $frame_to_update]] -column 0 -sticky ew -padx 25 -pady 5 -columnspan 2
            set row_offset 2
        }
        foreach param $params {
            if {$paramkind == "primary"} {if {[lsearch -exact $secs $param] >= 0} {continue}}
            if {$paramkind == "secondary"} {if {[lsearch -exact $secs $param] < 0} {continue}}
            # Label for each parameter
            label $frame_to_update.lbl_$param -text "$param:"
            grid $frame_to_update.lbl_$param -row [llength [winfo children $frame_to_update]] -column 0 -sticky e -padx 5 -pady 2

            # Create an entry field for each parameter and set its default value
            set default_value ""
            set param_type [dict get $types $param]
            if {[dict exists $defaults $param]} {
                set default_value [dict get $defaults $param]
            }
            set paramvar [namespace current]::add_action_widget_$param
            set $paramvar $default_value
            if {$param_type == "boolean"} {
                set $paramvar [expr {$default_value eq "t" || $default_value eq "y" || $default_value == 1}]
                checkbutton $frame_to_update.widg_$param -text "" -variable $paramvar
            } elseif {$param_type == "sel"} {
                set avail_values {}
                switch $param {
                    material {
                        set avail_values {Opaque Transparent BrushedMetal Diffuse Ghost Glass1 Glass2 Glass3 Glossy HardPlastic MetallicPastel Steel Translucent Edgy EdgyShiny EdgyGlass Goodsell AOShiny AOChalky AOEdgy BlownGlass GlassBubble RTChrome}
                        set revdict {Opaque Opaque Transparent Transparent BrushedMetal BrushedMetal Diffuse Diffuse Ghost Ghost Glass1 Glass1 Glass2 Glass2 Glass3 Glass3 Glossy Glossy HardPlastic HardPlastic MetallicPastel MetallicPastel Steel Steel Translucent Translucent Edgy Edgy EdgyShiny EdgyShiny EdgyGlass EdgyGlass Goodsell Goodsell AOShiny AOShiny AOChalky AOChalky AOEdgy AOEdgy BlownGlass BlownGlass GlassBubble GlassBubble RTChrome RTChrome}
                        ttk::combobox $frame_to_update.widg_$param -values $avail_values -state normal -width 12 -textvariable $paramvar
                        if {[dict exists $revdict $default_value]} {
                            $frame_to_update.widg_$param set [dict get $revdict $default_value]
                        } else {
                            $frame_to_update.widg_$param set ""
                        }
                    }
                    mode {
                        #set avail_values { u d ud n}
                        set avail_values {"create (u)" "remove (d)" "create&remove (ud)" "modify (n)"}
                        set revdict {u "create (u)" d "remove (d)" ud "create&remove (ud)" n "modify (n)"}
                        ttk::combobox $frame_to_update.widg_$param -values $avail_values -state normal -width 12 -textvariable $paramvar
                        if {[dict exists $revdict $default_value]} {
                            $frame_to_update.widg_$param set [dict get $revdict $default_value]
                        } else {
                            $frame_to_update.widg_$param set ""
                        }
                    }
                    style {
                        set avail_values {NewCartoon Surf QuickSurf Licorice VDW CPK DynamicBonds Tube IsoSurface}
                        set revdict {newcartoon NewCartoon surf Surf quicksurf QuickSurf licorice Licorice vdw VDW cpk CPK dynamicbonds DynamicBonds tube Tube isosurface IsoSurface}
                        ttk::combobox $frame_to_update.widg_$param -values $avail_values -state normal -width 12 -textvariable $paramvar
                        if {[dict exists $revdict $default_value]} {
                            $frame_to_update.widg_$param set [dict get $revdict $default_value]
                        } else {
                            $frame_to_update.widg_$param set ""
                        }
                    }
                    sigmoid {
                        set avail_values {"smooth (t)" "sharp (n)" "smooth&steady (sls)"}
                        set revdict {t "smooth (t)" n "sharp (n)" sls "smooth&steady (sls)"}
                        ttk::combobox $frame_to_update.widg_$param -values $avail_values -state normal -width 12 -textvariable $paramvar
                        if {[dict exists $revdict $default_value]} {
                            $frame_to_update.widg_$param set [dict get $revdict $default_value]
                        } else {
                            $frame_to_update.widg_$param set ""
                        }
                    }
                }
            } elseif {$param_type == "dfile"} {
                # 1) per-param mode variable (“file” vs “create”)
                set datafileModeVar datafileMode
                variable ::molywood::$datafileModeVar
                set ::molywood::$datafileModeVar file

                # 2) make the outer container (it will be grid()’d by update_params)
                frame $frame_to_update.widg_$param

                # 3) subframe for the two radios
                # frame $frame_to_update.widg_${param}.radioFrame
                # pack  $frame_to_update.widg_${param}.radioFrame -fill x -pady 1

                # radiobutton $frame_to_update.widg_${param}.radioFrame.fileRb -text File -variable ::molywood::$datafileModeVar -value file \
                #     -command  [list ::molywood::toggle_datafile_mode $frame_to_update.widg_$param]
                # radiobutton $frame_to_update.widg_${param}.radioFrame.createRb -text "New dataset" -variable ::molywood::$datafileModeVar -value    create \
                #     -command  [list ::molywood::toggle_datafile_mode $frame_to_update.widg_$param]
                # pack $frame_to_update.widg_${param}.radioFrame.fileRb $frame_to_update.widg_${param}.radioFrame.createRb -side left -padx 1 -pady 1

                # 4a) subframe #1: existing-file chooser (below the radios)
                frame $frame_to_update.widg_${param}.fileChooser
                entry  $frame_to_update.widg_${param}.fileChooser.entry -textvariable $paramvar -width 12

                # build the command string once, so tk_getOpenFile is *not* run yet
                button $frame_to_update.widg_${param}.fileChooser.browse -text Change -command [list ::molywood::on_datafile_create $frame_to_update.widg_${param}]

                pack $frame_to_update.widg_${param}.fileChooser.entry $frame_to_update.widg_${param}.fileChooser.browse -side left -padx 1 -pady 1
                pack $frame_to_update.widg_${param}.fileChooser -fill x -padx 2 -pady 1

                # 4b) subframe #2: “Create DS…” button (also below the radios)
                frame $frame_to_update.widg_${param}.createFrame
                button $frame_to_update.widg_${param}.createFrame.create -text    "Create/Load Dataset" \
                    -command [list ::molywood::on_datafile_create $frame_to_update.widg_${param}]
                pack $frame_to_update.widg_${param}.createFrame.create -padx 2 -pady 1

                # 5) initial state: show file-chooser, hide create
                pack $frame_to_update.widg_${param}.createFrame
                pack forget $frame_to_update.widg_${param}.fileChooser
            } else {
                entry $frame_to_update.widg_$param -textvariable $paramvar -width 12
            }
            grid $frame_to_update.widg_$param -row [expr [llength [winfo children $frame_to_update]] - 1] -column 1 -sticky w -padx 5 -pady 2
            if {$param == "ignore"} {
                grid remove $frame_to_update.widg_$param
                grid remove $frame_to_update.lbl_$param
            }
            ::molywood::create_tooltip $frame_to_update.lbl_$param [dict get $::molywood::tooltips $param]
            if {[dict exists $::molywood::tooltips_values $param]} {
                set text "[dict get $::molywood::tooltips_values $param]"
                ::molywood::create_tooltip $frame_to_update.widg_$param $text
            }
        }
    }
}

# Toggle between the two sub-frames
proc ::molywood::toggle_datafile_mode {parent} {
    set mode $::molywood::datafileMode
    if {$mode eq "file"} {
        # show file-chooser
        pack    $parent.fileChooser     -padx 20 -pady 2
        pack forget $parent.createFrame
    } else {
        # show create-dataset button
        pack    $parent.createFrame     -padx 20 -pady 2
        pack forget $parent.fileChooser
    }
}

# Called when “Create dataset…” is clicked
proc ::molywood::on_datafile_create {entryWidget} {
    # remember to fill this entry once the user saves
    set ::molywood::pending_datafile_entry $entryWidget
    ::molywood::show_dataset_window
}

proc ::molywood::get_active_actions_indices {} {
    # Returns 0-based indices of the actions 
    # that are selected in the treeview widget
    set w [dict get $::molywood::ui_elements window]
    set active_action_indices {}
    set action_counter 0
    foreach item [$w.nb.sceneProp.container.actl children {}] {
        foreach selected [$w.nb.sceneProp.container.actl selection] {
            if {$item == $selected} {
                lappend active_action_indices $action_counter
            }
        }
        incr action_counter
    }
    return $active_action_indices
}

proc ::molywood::remove_actions {} {
    # Removes all selected actions
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    set active_action_indices [lsort -integer -decreasing [::molywood::get_active_actions_indices]]
    set active_action_indices_incrsorted [lsort -integer [::molywood::get_active_actions_indices]]
    if {[llength $active_action_indices] == 0} {
        error "Please select an action to remove"
    }
    foreach index_to_remove $active_action_indices {
        $sc remove_action $index_to_remove
    }
    ::molywood::populate_actions
    ::molywood::update_scenes_table
    set w [dict get $::molywood::ui_elements window]
    update
    set actionlist [$w.nb.sceneProp.container.actl children {}]
    if {[llength $actionlist] == 0} {
        return
    }
    if {[lindex $active_action_indices_incrsorted 0] == 0} {
        set new_ind 0
    } else {
        set new_ind [expr [lindex $active_action_indices_incrsorted 0] - 1]
    }
    $w.nb.sceneProp.container.actl selection set [lindex $actionlist $new_ind]
}

proc ::molywood::copy_actions {} {
    # Copies all selected actions (adding copies at the bottom of the list)
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    set active_action_indices [lsort -integer [::molywood::get_active_actions_indices]]
    if {[llength $active_action_indices] == 0} {
        error "Please select an action to copy"
    }
    foreach index_to_copy $active_action_indices {
        set wrap [$sc get_action_wrapper $index_to_copy]
        $sc add_action_wrapper $wrap [llength [$sc get_all_action_times]]
    }
    ::molywood::populate_actions
    ::molywood::update_scenes_table
    set w [dict get $::molywood::ui_elements window]
    update
    $w.nb.sceneProp.container.actl selection set [lindex [$w.nb.sceneProp.container.actl children {}] [lindex $active_action_indices 0]]
}


proc ::molywood::edit_action {} {
    # Edits selected actions; if there are merged actions,
    # it should call the editor widget multiple times
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    set active_action_indices [::molywood::get_active_actions_indices]
    if {[llength $active_action_indices] == 0} {
        error "Please select an action to edit"
    }
    foreach active_action_index $active_action_indices {
        set active_wrapper_index [lindex $active_action_index 0]
        set active_wrapper [$sc get_action_wrapper $active_wrapper_index]
        set action_counter 0
        foreach action $active_wrapper {
            lassign $action name params
            ::molywood::open_action_popup edit $name $active_wrapper_index $action_counter $name $params
            incr action_counter
        }
    }
 }

proc ::molywood::add_mo {} {
    # Adds a master overlay by recycling the existing widget
    ::molywood::open_action_popup mo
} 

proc ::molywood::edit_mo {} {
    # Edits selected master overlays
    set w [dict get $::molywood::ui_elements window]

    set selected_items [$w.nb.general.topFrame.advancedFrame.movr selection]
    set all_items [$w.nb.general.topFrame.advancedFrame.movr children {}]
    set selected_item_index [lsearch $all_items $selected_items]
    set item [lindex $all_items $selected_item_index]
    ::molywood::open_action_popup edit_mo $item $selected_item_index $selected_item_index $item [lindex $::molywood::master_overlays $selected_item_index]
    ::molywood::update_mo_table
 }


proc ::molywood::remove_mo {} {
    # Removes selected master overlays
    set w [dict get $::molywood::ui_elements window]
    set new_mos {}
    set mo_counter 0
    foreach item [$w.nb.general.topFrame.advancedFrame.movr children {}] {
        if {$item != [$w.nb.general.topFrame.advancedFrame.movr selection]} {
            lappend new_mos [lindex $::molywood::master_overlays $mo_counter]
        }
        incr mo_counter
    }
    set ::molywood::master_overlays $new_mos
    ::molywood::update_mo_table
}

proc ::molywood::move_action {direction} {
    # Can reorder actions up or down, delegating to actions' methods
    set w [dict get $::molywood::ui_elements window]
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    set active_action_indices [::molywood::get_active_actions_indices]
    if {[llength $active_action_indices] > 1} {
        error "Can only move one action at a time"
    }
    if {[llength $active_action_indices] < 1} {
        error "Please select an action to move"
    }
    set active_wrapper_index [lindex $active_action_indices 0]

    if {$direction == "up"} {
        $sc move_action_up $active_wrapper_index
    }
    if {$direction == "down"} {
        $sc move_action_down $active_wrapper_index
    }

    ::molywood::populate_actions
    ::molywood::update_scenes_table

    if {$direction == "up"} {
        $w.nb.sceneProp.container.actl selection set [lindex [$w.nb.sceneProp.container.actl children {}] [expr $active_wrapper_index - 1]]
    }
    if {$direction == "down"} {
        $w.nb.sceneProp.container.actl selection set [lindex [$w.nb.sceneProp.container.actl children {}] [expr $active_wrapper_index + 1]]
    }
}

proc ::molywood::merge_actions {} {
    # Merges two or more actions into a single action wrapper,
    # indicating these should happen simultaneously
    set active_action_indices [::molywood::get_active_actions_indices]
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    if {[llength $active_action_indices] < 2} {
        error "Need at least two actions to merge"
    }
    if {[expr [lindex $active_action_indices end] - [lindex $active_action_indices 0]] != [expr [llength $active_action_indices] - 1]} {
        error "Can only merge consecutive actions"
    }
    set inverse_indices [lsort -integer -decreasing $active_action_indices]
    for {set i 0} {$i <= [expr [llength $inverse_indices] - 2]} {incr i} {
        $sc merge_actions [lindex $inverse_indices $i+1] [lindex $inverse_indices [expr $i]]
    }
    ::molywood::populate_actions
    ::molywood::update_scenes_table
    set w [dict get $::molywood::ui_elements window]
    update
    $w.nb.sceneProp.container.actl selection set [lindex [$w.nb.sceneProp.container.actl children {}] [lindex $active_action_indices 0]]
}

proc ::molywood::unmerge_action {} {
    # Unmerges a wrapper into individual actions
    set w [dict get $::molywood::ui_elements window]
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    set active_action_indices [::molywood::get_active_actions_indices]
    if {[llength $active_action_indices] != 1} {
        error "Can only unmerge one action at a time"
    }
    $sc unmerge_action [lindex $active_action_indices 0]
    ::molywood::populate_actions
    ::molywood::update_scenes_table
    set w [dict get $::molywood::ui_elements window]
    update
    $w.nb.sceneProp.container.actl selection set [lindex [$w.nb.sceneProp.container.actl children {}] [lindex $active_action_indices 0]]
 }

proc ::molywood::invokeBrowser {url} {
    # Stolen from the colvars plugin, there stolen from http://wiki.tcl.tk/557
    # open is the OS X equivalent to xdg-open on Linux, start is used on Windows
    set commands {xdg-open open start}
    foreach browser $commands {
        if {$browser eq "start"} {
          set command [list {*}[auto_execok start] {}]
        } else {
          set command [auto_execok $browser]
        }
        if {[string length $command]} {
          break
        }
    }
    if {[string length $command] == 0} {
        return -code error "couldn't find browser"
    }
    if {[catch {exec {*}$command $url &} error]} {
        return -code error "couldn't execute '$command': $error"
    }
}

proc ::molywood::write_input {{outfilename ""}} {
    # Writes a Molywood input to the specified file
    set all_input [::molywood::write_global_options]
    
    foreach scene $::molywood::scenes {
        $scene sanitize_values
        append all_input "[$scene write_actions_to_string]\n"
    }

    if {$outfilename == ""} {
        set outfilename [::molywood::choose_save_file]
    }
    if {$outfilename == ""} {
        return
    }

    set outfile [open $outfilename w+]
    puts $outfile $all_input
    close $outfile
}

proc ::molywood::choose_save_file {} {
    # Opens the save-file dialog and gets the file path
    set file_path [tk_getSaveFile -title "Save File" -filetypes {{"Text Files" {.txt}} {"All Files" {*}}}]
    
    # Check if a file was selected (i.e., not canceled)
    if {$file_path ne ""} {
        puts "MOLY-Logger\) File will be saved to: $file_path"
        return $file_path
    } else {
        puts "MOLY-Logger\) File save was canceled."
    }
}

proc ::molywood::write_global_options {} {
    # Writes currently defined global options as a formated string
    set output "\$ global "
    append output "fps=$::molywood::frames_per_second "
    append output "name=$::molywood::movie_name "
    if {$::molywood::render == "preview"} {
        append output "draft=t render=f "
    } elseif {$::molywood::render == "snapshot"} {
        append output "draft=t render=t "
    } else {
        append output "draft=f render=t "
    }
    append output "keepframes=$::molywood::keep_frames "
    if {$::molywood::breakpoints != "" && $::molywood::breakpoints != "{}"} {
        append output "breakpoints=[string map {" " ""} $::molywood::breakpoints] "
    }
    if {$::molywood::gif != "" && $::molywood::gif != "{}"} {
        append output "gif=$::molywood::gif "
    }
    append output "restart=$::molywood::restart \n"
    foreach mo $::molywood::master_overlays {
        append output "\$ master_overlay"
        foreach {key val} [dict get $mo] {
            if {$val != ""} {
                append output " $key=$val"
            }
        }
        append output "\n"
    }
    if {$::molywood::scene_rows > 1 || $::molywood::scene_columns > 1} { 
        append output "\$ layout rows=$::molywood::scene_rows columns=$::molywood::scene_columns\n"
    }
    foreach scene $::molywood::scenes {
        set params [$scene get_formatted_scene_parameters]
        append output "\$ [$scene get_scene_name] "
        foreach {key value} [dict get $params] {
            append output "$key=$value "
        }
        append output "\n\n"
    }
    return $output
}

proc ::molywood::preview_scene {{only_selected false} {selected ""}} {
    # Launches a preview by asking what to do, and optionally
    # saving the current state and clearing everything, running
    # Molywood externally to produce .tcl, and running the latter in VMD
    if {[[::molywood::get_scene_by_name $::molywood::currently_selected_scene] calc_total_time] == 0} {
        error "Nothing to preview, add some actions first"
    }
    set active_actions [join [::molywood::get_active_actions_indices] ","] ;# TODO
    if {$selected == ""} {
        set whattodo [::molywood::save_reload_popup]
    } else {
        set whattodo $selected
    }
    set tempren $::molywood::render
    set ::molywood::render preview
    switch $whattodo {
        continue {
            set dispsize [display get size]
            save_state current_state.vmd
            try {
                mol delete all
                ::molywood::write_input tmp_input.txt
                if {$only_selected} {
                    ::molywood::run_tool molywood tmp_input.txt -only_scene=$::molywood::currently_selected_scene -only_actions=$active_actions
                } else {
                    ::molywood::run_tool molywood tmp_input.txt -only_scene=$::molywood::currently_selected_scene
                }
                source script_$::molywood::currently_selected_scene.tcl
            } finally {
                mol delete all
                play current_state.vmd
                display resize {*}$dispsize
            }
        }
        keep {
            ::molywood::write_input tmp_input.txt
            if {[molinfo num] > 0} {
                if {$only_selected} {
                    ::molywood::run_tool molywood tmp_input.txt -only_scene=$::molywood::currently_selected_scene -only_actions=$active_actions -skipsource -vmdexec="[info nameofexecutable]"
                } else {
                    ::molywood::run_tool molywood tmp_input.txt -only_scene=$::molywood::currently_selected_scene -skipsource -vmdexec="[info nameofexecutable]"
                }
            } else {
                if {$only_selected} {
                    ::molywood::run_tool molywood tmp_input.txt -only_scene=$::molywood::currently_selected_scene -only_actions=$active_actions -vmdexec="[info nameofexecutable]"
                } else {
                    ::molywood::run_tool molywood tmp_input.txt -only_scene=$::molywood::currently_selected_scene -vmdexec="[info nameofexecutable]"
                }
            }
            source script_$::molywood::currently_selected_scene.tcl
        }
        cancel {
            puts "MOLY-Logger\) Skipped preview of scene $::molywood::currently_selected_scene"
        }
    }
    display update on
    set ::molywood::render $tempren
    return $whattodo
}

proc ::molywood::warning_default_pdb {} {
    # When doing a preview, makes a popup window asking the user
    # whether they're fine with saving and restoring the current VMD state
    # or should they chicken out/proceed bravely
    set w [toplevel .warnpopup]
    wm title $w "Warning: confirm choice of source"
    wm geometry $w 450x150
    label $w.ifsource -text "Your current scene preview is based on: $::molywood::scene_source $::molywood::source_file\nbut your main molecule seems to be different.\n\nIf you want to use e.g. the current VMD state instead,\ncancel and change the sources in the 'Scenes manager' tab."
    pack $w.ifsource -padx 8 -pady 8
    label $w.spacer -text ""
    pack $w.spacer -pady 5
    set ::molywood::whattodo ""
    button $w.btnContinue -text "Continue rendering" -command [list ::molywood::sr_response $w "continue"]
    button $w.btnCancel -text "Cancel preview" -command [list ::molywood::sr_response $w "cancel"]
    pack $w.btnContinue -side left -padx 7 -pady 7 -expand true -fill both
    pack $w.btnCancel -side left -padx 7 -pady 7 -expand true -fill both
    tkwait window $w
    return $::molywood::whattodo
}


proc ::molywood::do_render {} {
    # Launches the rendering in two modes, "preview" simply runs preview of
    # each scene one by one, the other two run the Python script and open
    # the resulting .mp4; whatever the errors, at the end the active scene
    # is reset to the first one on the list
    if {[regexp {[/\\~]} $::molywood::movie_name]} {
        error "Your movie name is: $::molywood::movie_name \nFor security reasons, movie name cannot contain slashes, backslashes, or tildes!"
    }
    if {[::molywood::calc_total_movie_time] == 0} {
        error "Nothing to render, add some actions first"
    } elseif {$::molywood::render != "preview" && $::molywood::scene_source == "PDB 4-letter code" && [molinfo top] > -1} {
        if { [string toupper [molinfo top get name]] != [string toupper $::molywood::source_file]} {
            set dowe_continue [::molywood::warning_default_pdb]
            if {$dowe_continue == "cancel"} {
                return
                }
            }
    }
    try {
        if {$::molywood::render == "preview"} {
            set whattodo ""
            set tmp_css $::molywood::currently_selected_scene
            foreach sc $molywood::scenes {
                set ::molywood::currently_selected_scene [$sc get_scene_name]
                set whattodo [::molywood::preview_scene "false" $whattodo]
            }
            set ::molywood::currently_selected_scene $tmp_css
        } else {
            ::molywood::write_input $::molywood::movie_name.txt
            toplevel .busy
            wm title .busy "Rendering in progress"
            wm iconbitmap .busy hourglass
            label .busy.lbl -text "Running molywood, please stand by while \n rendering, converting and composing frames…" -padx 20 -pady 15
            pack .busy.lbl
            update idletasks
            ::molywood::run_tool molywood $::molywood::movie_name.txt -vmdexec="[info nameofexecutable]"
            destroy .busy
            set osname [string trimleft [array get ::tcl_platform os] "os "]
            switch -- $osname {
                "Linux" {
                    exec xdg-open $::molywood::movie_name.mp4 &
                }
                "Darwin" {
                    exec open $::molywood::movie_name.mp4 &
                }
                "Windows" {
                    exec {*}[auto_execok start] "" $::molywood::movie_name.mp4 &
                }
                default {
                    error "Can't auto-open the .mp4 file on detected platform: $osname, please check the output manually"
                }
            }
        }
    } finally {
        set ::molywood::currently_selected_scene [[lindex $::molywood::scenes 0] get_scene_name]
    }
}

proc ::molywood::save_reload_popup {} {
    # When doing a preview, makes a popup window asking the user
    # whether they're fine with saving and restoring the current VMD state
    # or should they chicken out/proceed bravely
    set w [toplevel .srpopup]
    wm title $w "Preview - choose how to proceed"
    wm geometry $w 650x350
    label $w.ifsource -text "Your current preview is based on: $::molywood::scene_source $::molywood::source_file.\n\nTo use the current VMD state instead, click here to change the source:"
    pack $w.ifsource -padx 8 -pady 8
    button $w.button -text "Save current state & use as source" -command [list ::molywood::save_current_vis 1 $w]
    if {$::molywood::scene_source != "Existing VMD state"} {
        $w.button configure -bg goldenrod1
    }
    pack $w.button -padx 2 -pady 2
    label $w.spacer -text ""
    pack $w.spacer -pady 5
    label $w.label -text "Choose from the two preview modes:\n\n1. The current state is saved, the source is loaded to play animation, and saved state is restored.\nNOTE: unsaved progress MIGHT BE LOST if can't be saved to a .vmd state.\n\n2. If a molecule is loaded, only the actions will be performed, and left 'as is' once animation is over.\nNOTE: you'll need to manually undo the actions to return to previous state.\n\nWhat do you want to do?"
    pack $w.label -padx 10 -pady 10
    set ::molywood::whattodo ""
    button $w.btnSaveRestore -text "1. Continue with save/restore\n(always loads from source)" -command [list ::molywood::sr_response $w "continue"]
    button $w.btnKeepState -text "2. Continue on top of current state\n(only load source if no mols present)" -command [list ::molywood::sr_response $w "keep"]
    button $w.btnCancel -text "Cancel preview" -command [list ::molywood::sr_response $w "cancel"]
    pack $w.btnSaveRestore -side left -padx 7
    pack $w.btnKeepState -side left -padx 7
    pack $w.btnCancel -side left -padx 7
    tkwait window $w
    return $::molywood::whattodo
}

proc ::molywood::sr_response {w option} {
    # Records the response from the "whattodo" widget and closes it
    set ::molywood::whattodo $option
    destroy $w
}

proc ::molywood::toggle_ao {} {
    # Sets all AO settings in agreement with the global checkbox
    foreach scene $molywood::scenes {
        $scene set_scene_parameter aocc $::molywood::ambient_occlusion
    }
}

proc ::molywood::choose_movie_template {} {
    # Creates a window with available templates
    set w [toplevel .templates]
    wm title $w "Choose a template"
    wm geometry $w 850x600
    set ::molywood::selected_template ""
    set numgentemp 6

    proc hseparator path {
        frame $path -relief flat -borderwidth 3 -height 0 -pady 0
        pack [frame $path.inner -relief groove -borderwidth 2 -height 2 -pady 0] -expand yes -fill x
        set path
    }

    label $w.lab1 -text "Generic templates for customization:" -font "Helvetica 12 bold"
    grid $w.lab1 -row 0 -column 0 -sticky ew -padx 8 -pady 3  -columnspan 6
    radiobutton $w.template1 -text "Rotate and zoom on selection" -variable ::molywood::selected_template -value "rotzoom" -justify left 
    grid $w.template1 -row 1 -column 0 -sticky ew -padx 8 -pady 3  -columnspan 6
    radiobutton $w.template2 -text "Fade background, animate and highlight" -variable ::molywood::selected_template -value "fade" -justify left 
    grid $w.template2 -row 2 -column 0 -sticky ew -padx 8 -pady 3  -columnspan 6
    radiobutton $w.template3 -text "Two systems side by side, animated in turns" -variable ::molywood::selected_template -value "sidebyside" -justify left 
    grid $w.template3 -row 3 -column 0 -sticky ew -padx 8 -pady 3  -columnspan 6
    radiobutton $w.template4 -text "Still frames from an animation moved to the side" -variable ::molywood::selected_template -value "frameshooter" -justify left 
    grid $w.template4 -row 4 -column 0 -sticky ew -padx 8 -pady 3  -columnspan 6

    grid [hseparator $w.sep2] -row [expr $numgentemp - 1] -column 0 -sticky ew -padx 8 -pady 8  -columnspan 6

    label $w.lab2 -text "Samples from documentation:" -font "Helvetica 12 bold"
    grid $w.lab2 -row $numgentemp -column 0 -sticky ew -padx 8 -pady 8  -columnspan 6
    

    label $w.labsam0 -text "Basic options" -font "Helvetica 10 bold"
    grid $w.labsam0 -row [expr $numgentemp + 1] -column 0 -sticky w -padx 2 -pady 2
    radiobutton $w.samp00 -text "End credits *" -variable ::molywood::selected_template -value "basic1" -justify left 
    grid $w.samp00 -row [expr $numgentemp + 2] -column 0 -sticky w -padx 2 -pady 2
    radiobutton $w.samp01 -text "Custom TCL inserts" -variable ::molywood::selected_template -value "basic2" -justify left 
    grid $w.samp01 -row [expr $numgentemp + 3] -column 0 -sticky w -padx 2 -pady 2
    radiobutton $w.samp02 -text "Ambient occlusion" -variable ::molywood::selected_template -value "render1" -justify left 
    grid $w.samp02 -row [expr $numgentemp + 4] -column 0 -sticky w -padx 2 -pady 2
    radiobutton $w.samp03 -text "Draft vs high-quality" -variable ::molywood::selected_template -value "render2" -justify left 
    grid $w.samp03 -row [expr $numgentemp + 5] -column 0 -sticky w -padx 2 -pady 2
    radiobutton $w.samp04 -text "Background music *" -variable ::molywood::selected_template -value "audio1" -justify left 
    grid $w.samp04 -row [expr $numgentemp + 6] -column 0 -sticky w -padx 2 -pady 2
    radiobutton $w.samp05 -text "Audio effects *" -variable ::molywood::selected_template -value "audio2" -justify left 
    grid $w.samp05 -row [expr $numgentemp + 7] -column 0 -sticky w -padx 2 -pady 2


    label $w.labsam1 -text "Camera options" -font "Helvetica 10 bold"
    grid $w.labsam1 -row [expr $numgentemp + 1] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp10 -text "Zoom on selection" -variable ::molywood::selected_template -value "centering1" -justify left 
    grid $w.samp10 -row [expr $numgentemp + 2] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp11 -text "Rotate around selection" -variable ::molywood::selected_template -value "centering2" -justify left 
    grid $w.samp11 -row [expr $numgentemp + 3] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp12 -text "Zoom while rotating" -variable ::molywood::selected_template -value "concurrency1" -justify left 
    grid $w.samp12 -row [expr $numgentemp + 4] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp13 -text "Highlight while rotating" -variable ::molywood::selected_template -value "concurrency2" -justify left 
    grid $w.samp13 -row [expr $numgentemp + 5] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp14 -text "Rotation modes" -variable ::molywood::selected_template -value "rotation1" -justify left 
    grid $w.samp14 -row [expr $numgentemp + 6] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp15 -text "Off-axis rotation" -variable ::molywood::selected_template -value "rotation2" -justify left 
    grid $w.samp15 -row [expr $numgentemp + 7] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp16 -text "Action smoothness" -variable ::molywood::selected_template -value "rotation3" -justify left 
    grid $w.samp16 -row [expr $numgentemp + 8] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp161 -text "Saving/retrieving viewpoints" -variable ::molywood::selected_template -value "rotation4" -justify left 
    grid $w.samp161 -row [expr $numgentemp + 9] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp17 -text "2D camera translations" -variable ::molywood::selected_template -value "translation1" -justify left 
    grid $w.samp17 -row [expr $numgentemp + 10] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp18 -text "One scene after another" -variable ::molywood::selected_template -value "translation2" -justify left 
    grid $w.samp18 -row [expr $numgentemp + 11] -column 1 -sticky w -padx 2 -pady 2
    radiobutton $w.samp19 -text "Zoom modes" -variable ::molywood::selected_template -value "zoom1" -justify left 
    grid $w.samp19 -row [expr $numgentemp + 12] -column 1 -sticky w -padx 2 -pady 2


    label $w.labsam2 -text "Highlights" -font "Helvetica 10 bold"
    grid $w.labsam2 -row [expr $numgentemp + 1] -column 2 -sticky w -padx 2 -pady 2
    radiobutton $w.samp20 -text "Adding representations" -variable ::molywood::selected_template -value "highlight1" -justify left 
    grid $w.samp20 -row [expr $numgentemp + 2] -column 2 -sticky w -padx 2 -pady 2
    radiobutton $w.samp21 -text "Overlaying representations" -variable ::molywood::selected_template -value "highlight2" -justify left 
    grid $w.samp21 -row [expr $numgentemp + 3] -column 2 -sticky w -padx 2 -pady 2
    radiobutton $w.samp22 -text "Working with densities *" -variable ::molywood::selected_template -value "highlight3" -justify left 
    grid $w.samp22 -row [expr $numgentemp + 4] -column 2 -sticky w -padx 2 -pady 2
    radiobutton $w.samp23 -text "Densities and structures *" -variable ::molywood::selected_template -value "highlight4" -justify left 
    grid $w.samp23 -row [expr $numgentemp + 5] -column 2 -sticky w -padx 2 -pady 2
    radiobutton $w.samp24 -text "Visualizing ensembles" -variable ::molywood::selected_template -value "highlight5" -justify left 
    grid $w.samp24 -row [expr $numgentemp + 6] -column 2 -sticky w -padx 2 -pady 2
    radiobutton $w.samp25 -text "Full/partial transparency" -variable ::molywood::selected_template -value "transparency1" -justify left 
    grid $w.samp25 -row [expr $numgentemp + 7] -column 2 -sticky w -padx 2 -pady 2
    radiobutton $w.samp26 -text "Modifying base VMD reps" -variable ::molywood::selected_template -value "highlight6" -justify left 
    grid $w.samp26 -row [expr $numgentemp + 8] -column 2 -sticky w -padx 2 -pady 2
    radiobutton $w.samp27 -text "Dynamic selections" -variable ::molywood::selected_template -value "highlight7" -justify left 
    grid $w.samp27 -row [expr $numgentemp + 9] -column 2 -sticky w -padx 2 -pady 2


    label $w.labsam3 -text "Overlays" -font "Helvetica 10 bold"
    grid $w.labsam3 -row [expr $numgentemp + 1] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp30 -text "Simple plot *" -variable ::molywood::selected_template -value "overlay1" -justify left 
    grid $w.samp30 -row [expr $numgentemp + 2] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp31 -text "Skipping frames *" -variable ::molywood::selected_template -value "overlay2" -justify left 
    grid $w.samp31 -row [expr $numgentemp + 3] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp32 -text "Editing plot styles *" -variable ::molywood::selected_template -value "overlay3" -justify left 
    grid $w.samp32 -row [expr $numgentemp + 4] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp33 -text "2D plots *" -variable ::molywood::selected_template -value "overlay4" -justify left 
    grid $w.samp33 -row [expr $numgentemp + 5] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp34 -text "Text overlays" -variable ::molywood::selected_template -value "overlay5" -justify left 
    grid $w.samp34 -row [expr $numgentemp + 6] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp35 -text "Dynamic text" -variable ::molywood::selected_template -value "overlay6" -justify left 
    grid $w.samp35 -row [expr $numgentemp + 7] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp36 -text "Using TeX commands" -variable ::molywood::selected_template -value "overlay7" -justify left 
    grid $w.samp36 -row [expr $numgentemp + 8] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp37 -text "Pictures & videos *" -variable ::molywood::selected_template -value "overlay8" -justify left 
    grid $w.samp37 -row [expr $numgentemp + 9] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp38 -text "Overlaying scenes" -variable ::molywood::selected_template -value "overlay9" -justify left 
    grid $w.samp38 -row [expr $numgentemp + 10] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp39 -text "Master overlays" -variable ::molywood::selected_template -value "overlay11" -justify left 
    grid $w.samp39 -row [expr $numgentemp + 11] -column 3 -sticky w -padx 2 -pady 2
    radiobutton $w.samp310 -text "Fancy text boxes" -variable ::molywood::selected_template -value "overlay12" -justify left 
    grid $w.samp310 -row [expr $numgentemp + 12] -column 3 -sticky w -padx 2 -pady 2
    

    label $w.labsam4 -text "Trajectories & Multi-scenes" -font "Helvetica 10 bold"
    grid $w.labsam4 -row [expr $numgentemp + 1] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp40 -text "Grid of scenes" -variable ::molywood::selected_template -value "scenes1" -justify left 
    grid $w.samp40 -row [expr $numgentemp + 2] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp41 -text "Overriding defaults" -variable ::molywood::selected_template -value "scenes2" -justify left 
    grid $w.samp41 -row [expr $numgentemp + 3] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp42 -text "Multiple molecules" -variable ::molywood::selected_template -value "scenes3" -justify left 
    grid $w.samp42 -row [expr $numgentemp + 4] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp43 -text "Moving sep molecules" -variable ::molywood::selected_template -value "scenes4" -justify left 
    grid $w.samp43 -row [expr $numgentemp + 5] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp44 -text "Animating sep molecules" -variable ::molywood::selected_template -value "scenes5" -justify left 
    grid $w.samp44 -row [expr $numgentemp + 6] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp45 -text "Aligning molecules" -variable ::molywood::selected_template -value "scenes6" -justify left 
    grid $w.samp45 -row [expr $numgentemp + 7] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp46 -text "Smooth trajectories" -variable ::molywood::selected_template -value "animate1" -justify left 
    grid $w.samp46 -row [expr $numgentemp + 8] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp47 -text "Playback speed" -variable ::molywood::selected_template -value "animate2" -justify left 
    grid $w.samp47 -row [expr $numgentemp + 9] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp48 -text "Align trajectory" -variable ::molywood::selected_template -value "fitting1" -justify left 
    grid $w.samp48 -row [expr $numgentemp + 10] -column 4 -sticky w -padx 2 -pady 2
    radiobutton $w.samp49 -text "Fit to axis" -variable ::molywood::selected_template -value "fitting2" -justify left 
    grid $w.samp49 -row [expr $numgentemp + 11] -column 4 -sticky w -padx 2 -pady 2

    label $w.lab3 -text "* requires external files accessible on gitlab.com/KomBioMol/molywood" -font "Helvetica 9"
    grid $w.lab3 -row [expr $numgentemp + 13] -column 0 -sticky ew -padx 8 -pady 8  -columnspan 6

    frame $w.butts
    button $w.butts.submit -text "Load template" -command [list ::molywood::load_template $w]
    pack $w.butts.submit -pady 10 -padx 10 -side left
    button $w.butts.exit -text "Exit" -command [list destroy $w]
    pack $w.butts.exit -pady 10 -padx 10 -side right
    grid $w.butts -row [expr $numgentemp + 14] -column 2 -sticky ew -padx 2 -pady 2 
}

proc ::molywood::load_template {window} {
    # Given the chosen template, loads it and closes the window
    if {$::molywood::selected_template == "rotzoom" || $::molywood::selected_template == "fade" || $::molywood::selected_template == "sidebyside" || $::molywood::selected_template == "frameshooter"} {
        ::molywood::read_from_input [dict get $::molywood::templates $::molywood::selected_template]
        destroy $window
    } elseif {$::molywood::selected_template != ""} {
        set full_input [dict get $::molywood::samples $::molywood::selected_template]
        set replacement [file join $::env(MOLYWOOD_DIR) examples TransAtlas.vmd]
        set full_input [string map [list INPUTPATH $replacement] $full_input]
        set full_input "$ global name=$::molywood::selected_template $full_input"
        puts "MOLY-Logger\) Loading movie template from input:"
        puts $full_input
        ::molywood::read_from_input $full_input
        destroy $window
    } else {
        error "Please choose an option"
    }
}

proc ::molywood::do_cleanup {{scene_params_only 0}} {
    if {!$scene_params_only} {
        set ::molywood::scenes {}
        set ::molywood::movie_name "my_movie"
        set ::molywood::frames_per_second 15
        set ::molywood::keep_frames 0
        set ::molywood::gif ""
        set ::molywood::breakpoints ""
        set ::molywood::render "snapshot"
        set ::molywood::restart true
        set ::molywood::ambient_occlusion 0
        set ::molywood::scene_rows 1
        set ::molywood::scene_columns 1
        set ::molywood::master_overlays {}
    }
    set ::molywood::scene_source "PDB 4-letter code" ;# the value that's displayed
    set ::molywood::scene_source__ pdb_code ;# the value that's passed internally
    set ::molywood::source_file
    set ::molywood::scene_row 0
    set ::molywood::scene_column 0
    set ::molywood::scene_name_var scene1
    set ::molywood::resolution_x 750
    set ::molywood::resolution_y 750
    set ::molywood::after_scene ""
    set ::molywood::max_transparent_surfaces 3
}


proc ::molywood::read_from_input {{input_text ""}} {
    # A bit spaghettified but reads an input line by line,
    # recognizes global/scene parameters, and expedites
    # the processing of actions to other fns
    if {$input_text == ""} {
        set all_lines {}
        set inputfile [tk_getOpenFile]
        if {$inputfile == ""} {
            return
        }
        set handle [open $inputfile r]
        while {[gets $handle line] >= 0} {
            lappend all_lines $line
        }
    } else {
        set all_lines [split $input_text "\n"]
    }
    ::molywood::do_cleanup
    set global_params {}
    set current_scene_name ""
    set scenedefs {}
    set sceneacts {}
    set movers {}
    foreach line $all_lines {
        if {[string trim $line] eq "" || [regexp {^\s*\!(.*)} $line]} {
            continue
        }
        set line [::molywood::lineprocessor $line]
        if {[string match "\$*" $line]} {
            if {[regexp {\$\s+global(.*)} $line -> kv_pairs]} {
                set regex {(\S+)=([^\s]+)}
                foreach {_ key value} [regexp -all -inline $regex $kv_pairs] {
                    dict set global_params $key $value
                }
            } elseif {[regexp {\$\s+layout(.*)} $line -> kv_pairs]} {
                set regex {(\S+)=([^\s]+)}
                foreach {_ key value} [regexp -all -inline $regex $kv_pairs] {
                    dict set global_params $key $value
                }
            } elseif {[regexp {\$\s+master_overlay(.*)} $line -> kv_pairs]} {
                set regex {(\S+)=(".*?"|'.*?'|\S+)}
                set new_mo {}
                foreach {_ key value} [regexp -all -inline $regex $kv_pairs] {
                    if {$key == t} {
                        set val [string trim $value s]
                    } else {
                        set val $value
                    }
                    dict set new_mo $key $val
                }
                lappend movers $new_mo
            } else {
                if {[regexp {\$\s+(\S+)\s+(.*)} $line -> scene_name kv_pairs]} {
                    set kv_dict {}
                    # Extract key-value pairs using a regular expression
                    set regex {(\S+)=([^\s]+)}
                    foreach {_ key value} [regexp -all -inline $regex $kv_pairs] {
                        dict set kv_dict $key $value
                    }
                    dict set scenedefs $scene_name $kv_dict
                }
            }
        } elseif {[regexp {\#\s+(.+)} $line -> current_scene_name]} {
            foreach cscn [split $current_scene_name ,] {
                if {! [dict exists $sceneacts [string trim $cscn]]} {
                    dict set sceneacts [string trim $cscn] ""
                }
            }
        } elseif {$current_scene_name != ""} {
            foreach cscn [split $current_scene_name ,] {
                set current_content [dict get $sceneacts [string trim $cscn]]
                append current_content "$line\n"
                dict set sceneacts [string trim $cscn] $current_content
            }
        }
    }
    dict set global_params "master_overlays" $movers
    foreach {key value} [dict get $global_params] {
        switch $key {
            name {
                set ::molywood::movie_name $value
            }
            fps {
                set ::molywood::frames_per_second $value
            }
            keepframes {
                set ::molywood::keep_frames [::molywood::read_bool $value]
            }
            restart {
                set ::molywood::restart [::molywood::read_bool $value]
            }
            gif {
                set ::molywood::gif $value
            }
            breakpoints {
                set ::molywood::breakpoints $value
            }
            render {
                if {! [::molywood::read_bool $value]} {
                    set ::molywood::render "preview"
                } elseif {[dict exists $global_params draft] && ![::molywood::read_bool [dict get $global_params draft]]} {
                    set ::molywood::render "high quality"
                } else {
                    set ::molywood::render "snapshot"
                }
            }
            rows {
                set ::molywood::scene_rows $value
            }
            columns {
                set ::molywood::scene_columns $value
            }
            master_overlays {
                set ::molywood::master_overlays $value
                ::molywood::update_mo_table
            }
        }
    }

    foreach {scene_name} [dict keys $scenedefs] {
        ::molywood::do_cleanup 1
        set ::molywood::scene_name_var $scene_name
        foreach {key value} [dict get [dict get $scenedefs] $scene_name] {
            switch $key {
                visualization -
                structure -
                cubes -
                pdb_code {
                    set sources_dict {
                        visualization "Existing VMD state" 
                        structure "Structure file" 
                        pdb_code "PDB 4-letter code" 
                        cubes "Set of .cube files" 
                    }   
                    set ::molywood::scene_source__ $key
                    set ::molywood::scene_source [dict get $sources_dict $key]
                    set ::molywood::source_file $value
                    set w [dict get $::molywood::ui_elements window]
                    ::molywood::switch_source_file_widget $w 1 0
                }
                position {
                    lassign [split $value ","] rp cp 
                    set ::molywood::scene_row $rp
                    set ::molywood::scene_column $cp
                }
                resolution {
                    lassign [split $value ","] xr yr 
                    set ::molywood::resolution_x $xr 
                    set ::molywood::resolution_y $yr
                }
                after {
                    set ::molywood::after_scene $value
                }
                ambient_occlusion {
                    set ::molywood::ambient_occlusion [::molywood::read_bool $value]
                }
                max_transparent_surfaces {
                    set ::molywood::max_transparent_surfaces $value
                }
            }
        }
        set new_scene [scene new $scene_name [::molywood::new_scene_params]]
        set action_wrappers [::molywood::read_actions [dict get $sceneacts $scene_name]]
        set i 0
        foreach action_wrapper $action_wrappers {
            $new_scene add_action_wrapper $action_wrapper $i
            incr i
        }
        lappend ::molywood::scenes $new_scene
    }
    ::molywood::populate_actions
    ::molywood::update_scenes_table
    set w [dict get $::molywood::ui_elements window]
    set items [$w.scns.scnl children {}]
    $w.scns.scnl selection set [lindex $items 0]
    set ::molywood::currently_selected_scene [[lindex $::molywood::scenes 0] get_scene_name]
}

proc ::molywood::read_bool {bool_str} {
    # Deals with different legal ways of setting boolean values
    if {$bool_str == "t" || $bool_str == "y" || $bool_str == "true" || $bool_str == "yes" || $bool_str == "1"} {
        return true
    } elseif {$bool_str == "f" || $bool_str == "n" || $bool_str == "false" || $bool_str == "no" || $bool_str == "0"} {
        return false
    } else {
        error "Value $bool_str cannot be converted to a boolean type"
    }
}

proc ::molywood::highlight_update { var_name _ op } {
    # Makes the "Update currently selected scene" highlighted in goldenrod1
    # if any changes were made to the scene attributes
    set varmatch {
        "::molywood::after_scene" aftr
        "::molywood::resolution_x" xres
        "::molywood::resolution_y" yres
        "::molywood::scene_row" rnum
        "::molywood::scene_column" cnum
        "::molywood::source_file" srcf
        "::molywood::source_source__" srcs
        "::molywood::max_transparent_surfaces" mxts
    }
    set w [dict get $::molywood::ui_elements window]
    if {![winfo exists $w.nb.addScene.topFrame.btnEditScene]} {
        return
    }
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    if {$sc != ""} {
        set sc_params [$sc get_scene_parameters]
        if { [set $var_name] != [dict get $sc_params [dict get $varmatch $var_name]] } {
            $w.nb.addScene.topFrame.btnEditScene configure -bg goldenrod1
            $w.nb.addScene.topFrame.btnEditScene configure -text "CLICK to update"
        } else {
            $w.nb.addScene.topFrame.btnEditScene configure -bg gray85
            $w.nb.addScene.topFrame.btnEditScene configure -text "Update currently selected scene"
        }
    } else {
        return
    }
}

proc ::molywood::highlight_update_name { ins_name _ op } {
    # Makes the "Update currently selected scene" highlighted in goldenrod1
    # if any changes were made to the scene attributes
    set w [dict get $::molywood::ui_elements window]
    if {![winfo exists $w.nb.addScene.topFrame.btnEditScene]} {
        return
    }
    set sc [::molywood::get_scene_by_name $::molywood::currently_selected_scene]
    if {$sc != ""} {
        set sc_name [$sc get_scene_name]
        if { $ins_name != $sc_name } {
            $w.nb.addScene.topFrame.btnEditScene configure -bg goldenrod1
            $w.nb.addScene.topFrame.btnEditScene configure -text "CLICK to update"
        } else {
            $w.nb.addScene.topFrame.btnEditScene configure -bg gray85
            $w.nb.addScene.topFrame.btnEditScene configure -text "Update currently selected scene"
        }
    } else {
        return
    }
}

proc ::molywood::read_actions { input_text } {
    # Reads actions either delimited by newlines
    # or enclosed in curly brackets; in both cases,
    # runs them through process_single_action to 
    # obtain a single wrapper to pass on further
    set result {}
    set in_braces 0 
    set brace_block ""
    set lines [split $input_text "\n"]
    foreach line $lines {
        if {$in_braces} {
            append brace_block " $line"
            if {[regexp {\}$} $line]} {
                set inner_content [string trim $brace_block \{\}]
                set wrapper {}
                foreach item [split $inner_content ";"] {
                    lappend wrapper [::molywood::process_single_action [string trim $item]]
                }
                lappend result $wrapper
                set in_braces 0 
                set brace_block ""
            }
        } elseif {[regexp {^\{} $line]} {
            set in_braces 1
            set brace_block $line
        } else {
            if {$line ne ""} {
                lappend result [list [::molywood::process_single_action $line]]
            }
        }
    }

    return $result
}


proc ::molywood::lineprocessor {textline} {
    # remove comments if ! not encircled in quotation marks
    set line [string trim $textline]
    if {[regexp {^([^"']*|"[^"]*"|'[^']*')*?!} $line match]} {
        set line [string range $line 0 [expr {[string first "!" $line] - 1}]]
    }
    return $line
}

proc ::molywood::resolve_deps {} {
    # Finds scenes that are either consecutive in time (through the "after=" scene option),
    # or are embedded one within another (as an overlay). It calculates their time offsets,
    # and reorders scenes so that dependencies always come immediately after the base ones.
    set afters {}
    set withins {}
    set afters_immediate_offsets {}
    set withins_immediate_offsets {}
    foreach scene $::molywood::scenes {
        set scname [$scene get_scene_name]
        dict set afters_immediate_offsets $scname 0
        dict set withins_immediate_offsets $scname 0
    }
    foreach scene $::molywood::scenes {
        set scparams [$scene get_scene_parameters]
        set sc_actparams_wrappers [$scene get_all_action_params]
        set sc_acttimes [$scene get_all_action_times]
        set scname [$scene get_scene_name]
        set after [dict get $scparams aftr]
        if {[string trim $after] != ""} {
            if {[lsearch -exact [::molywood::get_all_other_scene_names $scname] $after] >= 0} {
                dict set afters $scname $after 
                dict set afters_immediate_offsets $scname [[::molywood::get_scene_by_name $after] calc_total_time]
            }
        }
        set wrapcount 0
        set elapsed 0
        foreach sc_actparams_wrapper $sc_actparams_wrappers {
            foreach sc_actparams $sc_actparams_wrapper {
                if {[dict exists $sc_actparams scene]} {
                    set dependent_scene [dict get $sc_actparams scene]
                    if {[lsearch -exact [::molywood::get_all_other_scene_names $scname] $dependent_scene] >= 0} {
                        dict set withins $dependent_scene $scname 
                        dict set withins_immediate_offsets $dependent_scene $elapsed
                    }
                }
            }
            set elapsed [expr $elapsed + [lindex $sc_acttimes $wrapcount]]
            incr wrapcount
        }
    }

    ::molywood::reorder_scenes [dict merge $afters $withins]
    foreach {dep_scene base_scene} [dict get $afters] {
        set base_offset [[::molywood::get_scene_by_name $base_scene] get_time_offset]
        set total_offset [expr $base_offset + [dict get $afters_immediate_offsets $dep_scene]]
        [::molywood::get_scene_by_name $dep_scene] set_time_offset $total_offset
    }
    foreach {dep_scene base_scene} [dict get $withins] {
        set base_offset [[::molywood::get_scene_by_name $base_scene] get_time_offset]
        set total_offset [expr $base_offset + [dict get $withins_immediate_offsets $dep_scene]]
        [::molywood::get_scene_by_name $dep_scene] set_time_offset $total_offset
    }
    return [list $afters $withins]
}

proc ::molywood::reorder_scenes {dep_dict} {
    # Makes sure the ordering respects the hierarchical order of scenes
    set ordered_scenes {}
    set seen_scenes {}
    
    proc handle_scene {scene_name dep_dict ordered_scenes_name seen_scenes_name} {
        upvar $ordered_scenes_name ordered_scenes
        upvar $seen_scenes_name seen_scenes

        if {[dict exists $seen_scenes $scene_name]} {
            return
        }
        # Check if the scene has dependencies
        if {[dict exists $dep_dict $scene_name]} {
            set dependency [dict get $dep_dict $scene_name]
            # Ensure the dependency is handled first
            handle_scene $dependency $dep_dict $ordered_scenes_name $seen_scenes_name
        }
        # Add the current scene to the order and mark it as seen
        lappend ordered_scenes [::molywood::get_scene_by_name $scene_name]
        dict set seen_scenes $scene_name 1
    }
    
    foreach scene $::molywood::scenes {
        handle_scene [$scene get_scene_name] $dep_dict ordered_scenes seen_scenes
    }
    set ::molywood::scenes $ordered_scenes
}

proc ::molywood::get_all_scene_names {} {
    # Returns all scene names
    set all_names {}
    foreach scene $::molywood::scenes {
        lappend all_names [$scene get_scene_name]
    }
    return $all_names
}

proc ::molywood::get_all_other_scene_names {curr_scene_name} {
    # Returns all scene names except for the one that's specified
    set scnames [::molywood::get_all_scene_names]
    set ind [lsearch -exact $scnames $curr_scene_name]
    if {$ind >= 0} {
        set scnames [lreplace $scnames $ind $ind]
    }
    return $scnames
}

proc ::molywood::calc_total_movie_time {} {
    # calculates the total time of the movie (largest offset+duration)
    set maxt 0
    foreach scene $::molywood::scenes {
        set sumt [expr [$scene calc_total_time] + [$scene get_time_offset]]
        if {$sumt > $maxt} {
            set maxt $sumt
        }
    }
    return $maxt
}


proc ::molywood::draw_diagram {} {
    # Draws the diagram representing the movie's conceptual layout
    set total_time [::molywood::calc_total_movie_time]
    if {$total_time == 0} {
        error "Cannot display an empty diagram, add some actions first"
    }
    set w [toplevel .diagram]
    wm title $w "Block Diagram"
    wm geometry $w 900x600
    set frame [frame $w.frame]
    pack $frame -fill both -expand true
    set canvas_width [expr max(900,[expr $total_time * 20])]
    set usable_width [expr $canvas_width - 100]
    set c [canvas $frame.canvas -width 900 -height 600 -scrollregion "0 0 $canvas_width 600"]
    # pack $c -fill both -expand true
    set hscroll [scrollbar $frame.hscroll -orient horizontal -command "$c xview"]
    pack $hscroll -side bottom -fill x
    pack $c -side left -fill both -expand true
    $c configure -xscrollcommand "$hscroll set"

    # Function to add a block (bar) to the canvas
    proc add_block {canvas x y width height label color nacts} {
        $canvas create rectangle $x $y [expr {$x + $width}] [expr {$y + $height}] -fill $color
        set textwidth 10
        set textmid [expr $x + $width/2]
        if {$height > $width} {
            set ang 90
        } else {
            set ang 0
        }
        if {$height > 35 || $width > 35} {
            $canvas create text $textmid [expr {$y + $height/2}] -text $label -fill black -angle $ang
        } else {
            $canvas create text $textmid [expr {$y + $height/2}] -text [string index $label 0] -fill black -angle $ang
        }
    }

    proc draw_axes {canvas width height total_time usable_width} {
        # Function to draw axes (optional)
        # Horizontal axis
        $canvas create line 35 25 [expr {$width - 25}] 25 -arrow last
        $canvas create text [expr {$width - 55}] 11 -text "Time axis \[s\]"
        for {set tick 1} {$tick < $total_time} {incr tick} {
            set rel_time [expr double($tick) / $total_time]
            set absx [expr $usable_width * $rel_time]
            $canvas create line [expr 35 + $absx] 25 [expr 35 + $absx] 22 -arrow none
            if {$tick % 5 == 0} {
                $canvas create text [expr 35 + $absx] 11 -text $tick
            }
        }
        
        # Vertical axis
        $canvas create line 35 [expr {$height - 25}] 35 25 -arrow none
    }


    # Draw the blocks
    set tothei [expr 550.0 - 400 / [llength $::molywood::scenes]]
    set indivhei [expr $tothei / [llength $::molywood::scenes]]
    set y_offset 50
    if {[llength $::molywood::scenes] > 7} {
        set scangle 60
    } else {
        set scangle 90
    }
    foreach scene $::molywood::scenes {
        set x_offset [expr 35 + $usable_width * [$scene get_time_offset] / $total_time]
        $c create text 18 [expr $y_offset + $indivhei / 2.0] -text [$scene get_scene_name] -angle $scangle
        set names_wrapper [$scene get_all_action_names]
        set times [$scene get_all_action_times]
        set counter 0
        foreach time $times {
            if {$time == 0} {
            incr counter 
            continue
            }
            set rel_time [expr double($time) / $total_time]
            set width [expr $usable_width * $rel_time]
            set names [lindex $names_wrapper $counter]
            set yoffs_int 0
            set height [expr [expr $indivhei - 10] / [llength $names]]
            foreach name $names {
                set label [dict get [dict get $::molywood::actions $name] abbr]
                set color [dict get [dict get $::molywood::actions $name] color]
                add_block $c $x_offset [expr $y_offset + $yoffs_int] $width $height $label $color [llength $names]
                set yoffs_int [expr $yoffs_int + $height]
            }
            set x_offset [expr {$x_offset + $width}]
            incr counter 
        }
        set y_offset [expr $y_offset + $indivhei]
    }

    # Draw axes
    draw_axes $c $canvas_width 600 $total_time $usable_width

    # Run the main loop
    tkwait window $w
}

proc ::molywood::create_tooltip {widget text} {
    # Creates a tooltip for a specific widget; generates a unique name 
    # for the tooltip based on the widget's path
    set parent [winfo toplevel $widget]
    set tooltip_name [format "%s.tooltip_%s" $parent [regsub -all {[^a-zA-Z0-9]} $widget {_}]]
    
    # If the tooltip already exists, destroy it
    if {[winfo exists $tooltip_name]} {
        destroy $tooltip_name
    }
    
    # Create a new top-level window as the tooltip
    toplevel $tooltip_name
    wm overrideredirect $tooltip_name 1  ;# Remove window decorations
    wm withdraw $tooltip_name            ;# Hide initially

    # Add a label with the tooltip text
    label $tooltip_name.label -text $text -background cornsilk -relief solid -borderwidth 1
    pack $tooltip_name.label

    # Bind the widget to show the tooltip on hover
    bind $widget <Enter> [list ::molywood::show_tooltip $tooltip_name]
    bind $widget <Leave> [list ::molywood::hide_tooltip $tooltip_name]
}

proc ::molywood::show_tooltip {tooltip_name} {
    # Ensure tooltip still exists
    if {![winfo exists $tooltip_name]} {
        return
    }
    # Shows the tooltip with a small offset from the mouse pointer
    set x [winfo pointerx [winfo toplevel $tooltip_name]]
    set y [winfo pointery [winfo toplevel $tooltip_name]]
    if {$x < 0 || $y < 0} { return }
    wm geometry $tooltip_name +[expr {$x + 10}]+[expr {$y + 10}]
    wm deiconify $tooltip_name
}

proc ::molywood::hide_tooltip {tooltip_name} {
    # Hides the tooltip
    if {[winfo exists $tooltip_name]} {
        wm withdraw $tooltip_name
    }
}

proc ::molywood::destroy_all_tooltips {tlevel} {
    # Destroys all tooltips before rebuilding the main window
    if {![winfo exists $tlevel]} {
        return
    }
    foreach win [winfo children $tlevel] {
        if {[string match "$toplevel.tooltip_*" $win]} {
            destroy $win
        }
    }
}

proc ::molywood::process_single_action {input_str} {
    # Processes a single action, relegated to the end because of regexps
    # that break brace matching in the IDE :/
    set parameters {}  ;# Dictionary to store the key-value pairs
    set action_name "" ;# Variable to hold the first word (the action name)
    # Regex to capture the unquoted initial word, key="value" pairs, or key=value pairs
    #set regex {(\S+|(\S+=(['"]).*?\3)|\S+=\S+)}
    set regex {(\S+="[^"]*"|\S+='[^']*'|\S+=\S+|\S+)}


    
    set first_word 1  ;# To track the initial unquoted word
    
    # Loop through the matched segments
    foreach match [regexp -all -inline $regex $input_str] {
        if {$first_word} {
            # The first match is the unquoted initial word (e.g., "rotate")
            set action_name $match
            set first_word 0  ;# After first word, process key-value pairs
        } else {
            set match [string trim $match "\}"]
            # If it's a key="value" pair
            if {[regexp {^(\S+)=((['"]).*?\3)$} $match -> key value]} {
                dict set parameters $key $value
            # If it's a simple key=value pair without quotes
            } elseif {[regexp {^(\S+)=(\S+)$} $match -> key value]} {
                if {$key == "t"} {
                    set value [string trim $value "s"]
                }
                dict set parameters $key $value
            }
        }
    }
    
    return [list $action_name $parameters]
}

proc ::molywood::get_consec {fint} {
    # helper fn to get consecutive numbers up to fint
    set outp {}
    for {set i 0 } {$i < $fint} {incr i} {
        lappend outp $i
    }
    return $outp
}