/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: webpdbplugin.c,v $
 *      $Author: johns $       $Locker:  $             $State: Exp $
 *      $Revision: 1.19 $       $Date: 2003/06/12 19:39:25 $
 *
 ***************************************************************************/

#include <tcl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "molfile_plugin.h"
#include "readpdb.h"

/*
 * Load pdb from the RCSB
 * Uses Tcl
 */

/*
 * Need my own read_pdb_record because the one in readpdb takes a FILE*.
 * This one will be better anyway since I don't recopy the string ;-)
 * Read the given pdb string.  On returning, pos will point to the start of
 * the next read. 
 */ 
static int my_read_pdb_record(const char *pdb, char **pos) {
   
  int recType = PDB_UNKNOWN;
  char *nlpos;  /* newline position */
  nlpos = strchr(pdb, '\n');
  if (!nlpos) {
    return PDB_EOF;
  } else {
    *pos = nlpos + 1;
  }

  /* what was it? */
  if (!strncmp(pdb, "REMARK", 6)) {
    recType = PDB_REMARK;

  } else if (!strncmp(pdb, "CRYST1", 6)) {
    recType = PDB_CRYST1;

  } else if (!strncmp(pdb, "ATOM  ", 6) ||
         !strncmp(pdb, "HETATM", 6)) {
    recType = PDB_ATOM;

    /* the only two END records are "END   " and "ENDMDL" */
  } else if (!strcmp(pdb, "END") ||       /* If not space " " filled */
         !strncmp(pdb, "END ", 4) ||      /* Allows other stuff      */
         !strncmp(pdb, "ENDMDL", 6)) {    /* NMR records             */
    recType = PDB_END;
  }
  return recType;
}
   
typedef struct {
  char *pdbstr; 
  char *pos;
  int natoms;
} pdbdata;


static void *pdb_read(char *pdbstr, int *natoms) {
  pdbdata *pdb;
  int indx;
  char *pos = pdbstr;
  char *next;

  if (!pdbstr) return NULL;
  do {
    if((indx = my_read_pdb_record(pos, &next)) == PDB_ATOM)
      *natoms += 1;
    pos = next;
  } while (indx != PDB_END && indx != PDB_EOF);
  pdb = (pdbdata *)malloc(sizeof(pdbdata));
  pdb->pdbstr = pdbstr;
  pdb->pos = pdbstr;
  pdb->natoms = *natoms;

  return pdb;
}

static const char *rcsbmsg[] = {
  "  The PDB is supported by RCSB, the NSF, US PHS, NIH, NCRP, NIGMS, NLM,",
  "and US DoE, who are not liable for the data.  PDB files shall not be",
  "sold.  See ftp://ftp.rcsb.org/advisory.doc for full details."
};

static int show_msg = 1;

static void *open_file_read(const char *filename, const char *filetype,
    int *natoms) {

  Tcl_Interp *interp;
  char url[300];
  char cmd[300]; 
  char *pdbfile;
  const char *result;
  void *v;

  /*
   * Create and initialize the interpreter
   */
  interp = Tcl_CreateInterp();
  if (!interp) {
    fprintf(stderr, "Could not create new Tcl Interp\n");
    return NULL; 
  }
  if (Tcl_Init(interp) != TCL_OK) {
    fprintf(stderr, "Warning, could not create initialize Tcl Interp\n");
  }
  if (!Tcl_PkgRequire(interp, (char *)"http", (char *)"2.0", 0)) {
    fprintf(stderr, "Could not load http package\n");
    Tcl_DeleteInterp(interp);
    return NULL;
  }

  if (strlen(filename) != 4) {
    fprintf(stderr, "PDB code %s is invalid; PDB accession codes have four letters.\n", filename);
    Tcl_DeleteInterp(interp);
    return NULL;
  }

  if (show_msg) {
    int i;
    show_msg = 0;
    for (i=0; i<3; i++) printf("%s\n", rcsbmsg[i]);
  }
 
  sprintf(url, "http://www.rcsb.org/pdb/cgi/export.cgi/%s.pdb?pdbId=%s;format=PDB", filename, filename);

  sprintf(cmd, "set token [::http::geturl \"%s\"]", url);
  if (Tcl_Eval(interp, cmd) != TCL_OK) {
    fprintf(stderr, "Error loading PDB: %s\n",interp->result);
    Tcl_DeleteInterp(interp);
    return NULL;
  } 
  sprintf(cmd, "upvar #0 $token state");
  Tcl_Eval(interp, cmd); 
  
  result = Tcl_GetVar2(interp, (char *)"state", "body", TCL_GLOBAL_ONLY); 
  if (!result) {
    fprintf(stderr, "Error loading PDB: %s\n", interp->result);
    Tcl_DeleteInterp(interp);
    return NULL;
  } 
  pdbfile = strdup(result);
  Tcl_DeleteInterp(interp);

  /* XXX this code needs updating still */
  /* pdbfile will be free'd by close_pdb() */
  v = pdb_read(pdbfile, natoms); 
  return v;
}
   
static int read_pdb_structure(void *mydata, int *optflags, 
    molfile_atom_t *atoms) {

  pdbdata *pdb = (pdbdata *)mydata;
  char *pos = pdb->pdbstr;
  char *next;
  int i, rectype;
  char ridstr[8];
  float newpos;
  molfile_atom_t *atom;

  *optflags = MOLFILE_INSERTION | MOLFILE_OCCUPANCY | MOLFILE_BFACTOR;
  i=0; /* Count atoms */
  do {
    rectype = my_read_pdb_record(pos, &next);
    switch (rectype) {
    case PDB_ATOM:
      atom = atoms+i;
      get_pdb_fields(pos, next-pos, atom->name, atom->resname, atom->chain, 
          atom->segid, ridstr, atom->insertion, &newpos, &newpos, &newpos,
                     &atom->occupancy, &atom->bfactor);
      atom->resid = atoi(ridstr);
      strcpy(atom->type, atom->name);
      i++;
      break;

#if 0
    case PDB_CRYST1:
      get_pdb_cryst1(pdbrec, &ts->alpha, &ts->beta, &ts->gamma,
                     &ts->a_length, &ts->b_length, &ts->c_length);
      break;
#endif

    default:
      break;
    }
    pos = next;
  } while (rectype != PDB_END && rectype != PDB_EOF);

  return MOLFILE_SUCCESS;
}

static int read_next_timestep(void *v, int natoms, molfile_timestep_t *ts) {
  pdbdata *pdb = (pdbdata *)v;
  char *pos = pdb->pos;
  char *next;
  float *x, *y, *z;
  float occup, bfac;
  int indx, i = 0;

  if (ts) {
    x = ts->coords;
    y = x+1;
    z = x+2;
  } else {
    x = y = z = 0;
  }
  do {
    indx = my_read_pdb_record(pos, &next);
    if((indx == PDB_END || indx == PDB_EOF) && (i < pdb->natoms)) {
      return MOLFILE_ERROR;
    } else if(indx == PDB_ATOM) {
      if(i >= pdb->natoms) {
        break;
      }
      /* just get the coordinates, and store them */
      if (ts) {
        get_pdb_coordinates(pos, x, y, z, &occup, &bfac);
        x += 3;
        y += 3;
        z += 3;
        i++;
      }
    } else if (indx == PDB_CRYST1) {
      if (ts) {
        get_pdb_cryst1(pos, &ts->alpha, &ts->beta, &ts->gamma,
                               &ts->A, &ts->B, &ts->C);
      }
    }
    pos = next;
  } while(!(indx == PDB_END || indx == PDB_EOF));
  pdb->pos = pos;

  return MOLFILE_SUCCESS;
}

static void close_pdb_read(void *v) {
  pdbdata *pdb = (pdbdata *)v;
  if (!pdb) return;
  free(pdb->pdbstr);
  free(pdb);
}

/* 
 * Registration stuff
 */

static molfile_plugin_t plugin = {
  vmdplugin_ABIVERSION,         /* ABI version */
  MOLFILE_PLUGIN_TYPE,          /* type */
  "webpdb",                     /* name */
  "Justin Gullingsrud",         /* author */
  0,                            /* major version */
  2,                            /* minor version */
  VMDPLUGIN_THREADSAFE,         /* is reentrant */
  "",                           /* filename extension */
  open_file_read,
  read_pdb_structure,
  0,
  read_next_timestep,
  close_pdb_read
};

int VMDPLUGIN_init() {
  return VMDPLUGIN_SUCCESS;
}

int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  (*cb)(v, (vmdplugin_t *)&plugin);
  return VMDPLUGIN_SUCCESS;
}

int VMDPLUGIN_fini() {
  return VMDPLUGIN_SUCCESS;
}


#ifdef TEST_WEBPDB_PLUGIN

int main(int argc, char *argv[]) {
  char *file;
  if (argc < 2) {
    fprintf(stderr, "Usage: %s <pdbcode>\n", argv[0]);
    return -1;
  }
  file = (char *)open_file_read(argv[1], "webpdb",  NULL);
  printf("%s\n", file);
  free(file);
  return 0;
}

#endif
