/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: readpdb.h,v $
 *      $Author: johns $       $Locker:  $             $State: Exp $
 *      $Revision: 1.9 $       $Date: 2003/09/14 19:25:47 $
 *
 ***************************************************************************/

#ifndef READ_PDB_H
#define READ_PDB_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#define PDB_RECORD_LENGTH   80

/*  record type defines */
enum {PDB_REMARK, PDB_ATOM, PDB_UNKNOWN, PDB_END, PDB_EOF, PDB_CRYST1};

/* read the next record from the specified pdb file, and put the string found
   in the given string pointer (the caller must provide adequate (81 chars)
   buffer space); return the type of record found
*/
static int read_pdb_record(FILE *f, char *retStr) {

  char inbuf[PDB_RECORD_LENGTH+2];
  int recType = PDB_UNKNOWN;
 
  /* XXX This PDB record reading code breaks with files that use
   * Mac or DOS style line breaks with ctrl-M characters.  We need
   * to replace the use of fgets() and comparisons against \n with
   * code that properly handles the other cases.
   */
 
  /*	read the next line	*/
  if(inbuf != fgets(inbuf, PDB_RECORD_LENGTH+1, f)) {
    strcpy(retStr,"");
    recType = PDB_EOF;
  } else {
    /*	remove the newline character, if there is one */
    if(inbuf[strlen(inbuf)-1] == '\n')
      inbuf[strlen(inbuf)-1] = '\0';

    /* what was it? */
    if (!strncmp(inbuf, "REMARK", 6)) {
      recType = PDB_REMARK;

    } else if (!strncmp(inbuf, "CRYST1", 6)) {
      recType = PDB_CRYST1;
      
    } else if (!strncmp(inbuf, "ATOM  ", 6) ||
	       !strncmp(inbuf, "HETATM", 6)) {
      recType = PDB_ATOM;

      /* the only two END records are "END   " and "ENDMDL" */
    } else if (!strcmp(inbuf, "END") ||       /* If not space " " filled */
	       !strncmp(inbuf, "END ", 4) ||      /* Allows other stuff      */
	       !strncmp(inbuf, "ENDMDL", 6)) {    /* NMR records             */
      recType = PDB_END;

    } else {
      recType = PDB_UNKNOWN;
    }

    if(recType == PDB_REMARK || recType == PDB_ATOM || 
       recType == PDB_CRYST1) {
      strcpy(retStr,inbuf);
    } else {
      strcpy(retStr,"");
    }
  }

  /* read the '\r', if there was one */
  {
    int ch = fgetc(f);
    if (ch != '\r') {
      ungetc(ch, f);
    }
  }
  
  return recType;
}

static void get_pdb_cryst1(char *record, float *alpha, float *beta,
		float *gamma,float *a, float *b, float *c) {

  int i;
  char tmp[81];
  char ch, *s = tmp;
  for (i=0; i<81; i++) tmp[i] = 0;
  strncpy(tmp, record, 80);
  tmp[80] = 0;

  s = tmp+6 ;          ch = tmp[15]; tmp[15] = 0;
  *a = (float) atof(s);
  s = tmp+15; *s = ch; ch = tmp[24]; tmp[24] = 0;
  *b = (float) atof(s);
  s = tmp+24; *s = ch; ch = tmp[33]; tmp[33] = 0;
  *c = (float) atof(s);
  s = tmp+33; *s = ch; ch = tmp[40]; tmp[40] = 0;
  *alpha = (float) atof(s);
  s = tmp+40; *s = ch; ch = tmp[47]; tmp[47] = 0;
  *beta = (float) atof(s);
  s = tmp+47; *s = ch; ch = tmp[54]; tmp[54] = 0;
  *gamma = (float) atof(s);
}

/* Extract the x,y, and z coordinates from the given ATOM record.	*/
static void get_pdb_coordinates(char *record, float *x, float *y, float *z,
	float *occup, float *beta) {
  char numstr[9];

  /* get X, Y, and Z */
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 30, 8);
  *x = (float) atof(numstr);
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 38, 8);
  *y = (float) atof(numstr);
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 46, 8);
  *z = (float) atof(numstr);
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 54, 6);
  *occup = (float) atof(numstr);
  memset(numstr, 0, 9 * sizeof(char));
  strncpy(numstr, record + 60, 6);
  *beta = (float) atof(numstr);
}


/* Break a pdb ATOM record into its fields.  The user must provide the
   necessary space to store the atom name, residue name, and segment name.
   Character strings will be null-terminated.
*/
static void get_pdb_fields(char *record, int reclength, char *name, char *resname,
                   char *chain,
		   char *segname, char *resid, char *insertion, float *x,
		   float *y, float *z, float *occup, float *beta) {
  int i,len;
  
  /* get atom name */
  strncpy(name,record + 12, 4);
  name[4] = '\0';
  while((len = strlen(name)) > 0 && name[len-1] == ' ')
    name[len-1] = '\0';
  while(len > 0 && name[0] == ' ') {
    for(i=0; i < len; i++)  name[i] = name[i+1];
    len--;
  }

  /* get residue name */
  strncpy(resname,record + 17, 4);
  resname[4] = '\0';
  while((len = strlen(resname)) > 0 && resname[len-1] == ' ')
    resname[len-1] = '\0';
  while(len > 0 && resname[0] == ' ') {
    for(i=0; i < len; i++)  resname[i] = resname[i+1];
    len--;
  }

  chain[0] = record[21];
  chain[1] = 0;

  /* get residue id number */
  strncpy(resid,record + 22, 4);
  resid[4] = '\0';
  while((len = strlen(resid)) > 0 && resid[len-1] == ' ')
    resid[len-1] = '\0';
  while(len > 0 && resid[0] == ' ') {
    for(i=0; i < len; i++)  resid[i] = resid[i+1];
    len--;
  }

  insertion[0] = record[26];
  insertion[1] = 0;

  /* get x, y, and z coordinates */
  get_pdb_coordinates(record, x, y, z, occup, beta);

  /* get segment name	*/
  if(reclength >= 73) {
    strncpy(segname, record + 72, 4);
    segname[4] = '\0';
    while((len = strlen(segname)) > 0 && segname[len-1] == ' ')
      segname[len-1] = '\0';
    while(len > 0 && segname[0] == ' ') {
      for(i=0; i < len; i++)  segname[i] = segname[i+1];
      len--;
    }
  } else {
    strcpy(segname,"");
  }
}  

/* Write PDB data to given file descriptor; return success. */
static int write_raw_pdb_record(FILE *fd, const char *recordname,
    int index,const char *atomname, const char *resname,int resid, 
    const char *insertion, float x, float y, float z, float occ, float beta, 
    const char *chain, const char *segname) {

  int rc;
  if (index < 100000) {
    rc = fprintf(fd,
      "%s%5d %4s%c%-4s%c%4d%c   %8.3f%8.3f%8.3f%6.2f%6.2f      %-4s\n",
          recordname, index, atomname, ' ', resname, chain[0], resid,
      insertion[0], x, y, z, occ, beta, segname);
  } else {
    rc = fprintf(fd,
      "%s***** %4s%c%-4s%c%4d%c   %8.3f%8.3f%8.3f%6.2f%6.2f      %-4s\n",
          recordname, atomname, ' ', resname, chain[0], resid,
      insertion[0], x, y, z, occ, beta, segname);
  }
  return (rc > 0);
}

#endif
