/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: readdcd.h,v $
 *      $Author: johns $       $Locker:  $             $State: Exp $
 *      $Revision: 1.15 $       $Date: 2003/06/12 19:39:24 $
 *
 ***************************************************************************/

#ifndef READ_DCD_H
#define READ_DCD_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*  DEFINE ERROR CODES THAT MAY BE RETURNED BY DCD ROUTINES		*/
#define DCD_EOF         -1  /*  Normal EOF                      */
#define DCD_DNE		    -2  /*  DCD file does not exist		    */
#define DCD_OPENFAILED	-3	/*  Open of DCD file failed		    */
#define DCD_BADREAD 	-4	/*  read call on DCD file failed	*/
#define DCD_BADEOF	    -5  /*  premature EOF found in DCD file	*/
#define DCD_BADFORMAT	-6	/*  format of DCD file is wrong		*/
#define DCD_FILEEXISTS  -7	/*  output file already exists		*/
#define DCD_BADMALLOC   -8	/*  malloc failed			        */


/*
 * Read the header information from a dcd file.
 * Input: fd - a file struct opened for binary reading.
 * Output: 0 on success, negative error code on failure.
 * Side effects: *natoms set to number of atoms per frame
 *               *nsets set to number of frames in dcd file
 *               *istart set to starting timestep of dcd file
 *               *nsavc set to timesteps between dcd saves
 *               *delta set to value of trajectory timestep
 *               *nfixed set to number of fixed atoms 
 *               *freeind may be set to heap-allocated space
 *               *reverse set to one if reverse-endian, zero if not.
 *               *charmm set to internal code for handling charmm data.
 */
static int read_dcdheader(FILE *fd, int *natoms, int *nsets, int *istart, int *nsavc, 
                   double *delta, int *nfixed, int **freeind, 
                   float **fixedcoords, int *reverse, int *charmm);	

/* 
 * Read a dcd timestep from a dcd file
 * Input: fd - a file struct opened for binary reading, from which the 
 *             header information has already been read.
 *        natoms, nfixed, first, *freeind, reverse, charmm - the corresponding 
 *             items as set by read_dcdheader
 *        first - true if this is the first frame we are reading.
 *        x, y, z: space for natoms each of floats.
 *        unitcell - space for six floats to hold the unit cell data.  
 *                   Not set if no unit cell data is present.
 * Output: 0 on success, negative error code on failure.
 * Side effects: x, y, z contain the coordinates for the timestep read.
 *               unitcell holds unit cell data if present.
 */
static int read_dcdstep(FILE *fd, int natoms, float *x, float *y, float *z, 
                        float *unitcell, int nfixed, int first, int *freeind, 
                        float *fixedcoords, int reverse, int charmm);	

/* 
 * Skip past a timestep.  If there are fixed atoms, this cannot be used with
 * the first timestep.  
 * Input: fd - a file struct from which the header has already been read
 *        natoms - number of atoms per timestep
 *        nfixed - number of fixed atoms
 *        charmm - charmm flags as returned by read_dcdheader
 * Output: 0 on success, negative error code on failure.
 * Side effects: One timestep will be skipped; fd will be positioned at the
 *               next timestep.
 */
static int skip_dcdstep(FILE *fd, int natoms, int nfixed, int charmm);

/*
 * clean up dcd data
 * Input: nfixed, freeind - elements as returned by read_dcdheader
 * Output: None
 * Side effects: Space pointed to by freeind is freed if necessary.
 */
static void close_dcd_read(int *freeind, float *fixedcoords);

/*
 * Write a header for a new dcd file
 * Input: fd - file struct opened for binary writing
 *        remarks - string to be put in the remarks section of the header.  
 *                  The string will be truncated to 70 characters.
 *        natoms, istart, nsavc, delta - see comments in read_dcdheader
 * Output: 0 on success, negative error code on failure.
 * Side effects: Header information is written to the dcd file.
 */
static int write_dcdheader(FILE *fd, const char *remarks, int natoms, 
                    int istart, int nsavc, double delta);	

/* 
 * Write a timestep to a dcd file
 * Input: fd - a file struct for which a dcd header has already been written
 *       curframe: Count of frames written to this file, starting with 1.
 *       curstep: Count of timesteps elapsed = istart + curframe * nsavc.
 *        natoms - number of elements in x, y, z arrays
 *        x, y, z: pointers to atom coordinates
 * Output: 0 on success, negative error code on failure.
 * Side effects: coordinates are written to the dcd file.
 */
static int write_dcdstep(FILE *fd, int curstep, int curframe, 
                 int natoms, const float *x, const float *y, const float *z);



#define DCD_IS_CHARMM       0x01
#define DCD_HAS_4DIMS       0x02
#define DCD_HAS_EXTRA_BLOCK 0x04

/* READ Macro to make porting easier */
#define READ(fd, buf, size) \
        fread((buf), (size), 1, (fd))


/* WRITE Macro to make porting easier */
#define WRITE(fd, buf, size) \
        fwrite((buf), (size), 1, (fd))


static void reverseFourByteWord(int* N)
{
  char byteArray[4] = {'0','0','0','0'};

  char *bytePointer;


  bytePointer = (char*)N;

  byteArray[0]  =  *bytePointer;
  byteArray[1]  =  *(bytePointer+1);
  byteArray[2]  =  *(bytePointer+2);
  byteArray[3]  =  *(bytePointer+3);

  *bytePointer     = byteArray[3];
  *(bytePointer+1) = byteArray[2];
  *(bytePointer+2) = byteArray[1];
  *(bytePointer+3) = byteArray[0];
}

static void reverseEightByteWord(double* N)
{
  char byteArray[8] = {'0','0','0','0','0','0','0','0'};
  char *bytePointer;

  bytePointer = (char*)N;

  byteArray[0]  =  *bytePointer;
  byteArray[1]  =  *(bytePointer+1);
  byteArray[2]  =  *(bytePointer+2);
  byteArray[3]  =  *(bytePointer+3);
  byteArray[4]  =  *(bytePointer+4);
  byteArray[5]  =  *(bytePointer+5);
  byteArray[6]  =  *(bytePointer+6);
  byteArray[7]  =  *(bytePointer+7);

  *bytePointer     = byteArray[7];
  *(bytePointer+1) = byteArray[6];
  *(bytePointer+2) = byteArray[5];
  *(bytePointer+3) = byteArray[4];
  *(bytePointer+4) = byteArray[3];
  *(bytePointer+5) = byteArray[2];
  *(bytePointer+6) = byteArray[1];
  *(bytePointer+7) = byteArray[0];
}

/* XXX This is broken - fread never returns -1 */
#define CHECK_FREAD(X, msg)  if (X==-1) \
			     { \
				return(DCD_BADREAD); \
			     }

#define CHECK_FEOF(X, msg)  if (X==0) \
			     { \
				return(DCD_BADEOF); \
			     }


static int read_dcdheader(FILE * fd, int *N, int *NSET, int *ISTART, 
		   int *NSAVC, double *DELTA, int *NAMNF, 
		   int **FREEINDEXES, float **fixedcoords, int *reverseEndian, 
                   int *charmm)
{
  int input_integer;	/*  Integer buffer space	*/
  int ret_val;		/*  Return value from read	*/
  int i;		/*  Loop counter		*/
  char hdrbuf[84];	/*  Char buffer used to store header	*/
  int NTITLE;

  /*  First thing in the file should be an 84		*/
  ret_val = READ(fd, &input_integer, sizeof(int));
  CHECK_FREAD(ret_val, "reading first int from dcd file");
  CHECK_FEOF(ret_val, "reading first int from dcd file");

  /* Check magic number in file header and determine byte order*/
  if (input_integer != 84) {
    /* check to see if its merely reversed endianism     */
    /* rather than a totally incorrect file magic number */
    reverseFourByteWord(&input_integer);

    if (input_integer == 84) {
      *reverseEndian=1;
    }
    else {
      return(DCD_BADFORMAT);
    }
  }
  else {
    *reverseEndian=0;    
  }

  /*  Buffer the entire header for random access	*/
  ret_val = READ(fd, hdrbuf, 84);
  CHECK_FREAD(ret_val, "buffering header");
  CHECK_FEOF(ret_val, "buffering header");

  /*  Check for the ID string "COORD"		*/
  if (hdrbuf[0] != 'C' || hdrbuf[1] != 'O' ||
      hdrbuf[2] != 'R' || hdrbuf[3] != 'D') {
    return DCD_BADFORMAT;
  }

  /* CHARMm-genereate DCD files set the last integer in the     */
  /* header, which is unused by X-PLOR, to its version number.  */
  /* Checking if this is nonzero tells us this is a CHARMm file */
  /* and to look for other CHARMm flags.                        */
  if (*((int *) (hdrbuf + 80)) != 0) {
	(*charmm) = DCD_IS_CHARMM;
	if (*((int *) (hdrbuf + 44)) != 0)
		(*charmm) |= DCD_HAS_EXTRA_BLOCK;
	if (*((int *) (hdrbuf + 48)) == 1)
		(*charmm) |= DCD_HAS_4DIMS;
  }
  else (*charmm) = 0;

  /*  Store the number of sets of coordinates (NSET) */
  (*NSET) = *((int *) (hdrbuf + 4));
  if (*reverseEndian) reverseFourByteWord(NSET);

  /*  Store ISTART, the starting timestep	     */
  (*ISTART) = *((int *) (hdrbuf + 8));
  if (*reverseEndian) reverseFourByteWord(ISTART);

  /*  Store NSAVC, the number of timesteps between   */
  /*  dcd saves				               */
  (*NSAVC) = *((int *) (hdrbuf + 12));
  if (*reverseEndian) reverseFourByteWord(NSAVC);

  /*  Store NAMNF, the number of fixed atoms	     */
  (*NAMNF) = *((int *) (hdrbuf + 36));
  if (*reverseEndian) reverseFourByteWord(NAMNF);

  /*  Read in the timestep, DELTA			*/
  /*  Note: DELTA is stored as a double with X-PLOR but */
  /*  as a float with CHARMm				*/
  if ((*charmm) & DCD_IS_CHARMM) {
    float ftmp;
    ftmp = *((float *)(hdrbuf+40));
    if (*reverseEndian) {
      reverseFourByteWord(((int *)&ftmp));
    }
    *DELTA = (double)ftmp;
  }
  else {
    (*DELTA) = *((double *) (hdrbuf + 40));
    if (*reverseEndian) reverseEightByteWord(DELTA);
  }

  /*  Get the end size of the first block			*/
  ret_val = READ(fd, &input_integer, sizeof(int));
  CHECK_FREAD(ret_val, "reading second 84 from dcd file");
  CHECK_FEOF(ret_val, "reading second 84 from dcd file");
  if (*reverseEndian) reverseFourByteWord(&input_integer);

  if (input_integer != 84) {
    return(DCD_BADFORMAT);
  }

  /*  Read in the size of the next block			*/
  ret_val = READ(fd, &input_integer, sizeof(int));
  CHECK_FREAD(ret_val, "reading size of title block");
  CHECK_FEOF(ret_val, "reading size of title block");
  if (*reverseEndian) reverseFourByteWord(&input_integer);

  if ( ((input_integer-4)%80) == 0) {
    /*  Read NTITLE, the number of 80 characeter    */
    /*  title strings there are			*/
    ret_val = READ(fd, &NTITLE, sizeof(int));
    CHECK_FREAD(ret_val, "reading NTITLE");
    CHECK_FEOF(ret_val, "reading NTITLE");
    if (*reverseEndian) reverseFourByteWord(&NTITLE);

    for (i=0; i<NTITLE; i++) {
      fseek(fd, 80, SEEK_CUR);
      CHECK_FEOF(ret_val, "reading TITLE");
    }

    /*  Get the ending size for this block		*/
    ret_val = READ(fd, &input_integer, sizeof(int));

    CHECK_FREAD(ret_val, "reading size of title block");
    CHECK_FEOF(ret_val, "reading size of title block");
  }
  else {
    return(DCD_BADFORMAT);
  }

  /*  Read in an 4				*/
  ret_val = READ(fd, &input_integer, sizeof(int));
  CHECK_FREAD(ret_val, "reading an 4");
  CHECK_FEOF(ret_val, "reading an 4");
  if (*reverseEndian) reverseFourByteWord(&input_integer);

  if (input_integer != 4) {
    return(DCD_BADFORMAT);
  }

  /*  Read in the number of atoms			*/
  ret_val = READ(fd, N, sizeof(int));
  CHECK_FREAD(ret_val, "reading number of atoms");
  CHECK_FEOF(ret_val, "reading number of atoms");
  if (*reverseEndian) reverseFourByteWord(N);

  /*  Read in an 4				*/
  ret_val = READ(fd, &input_integer, sizeof(int));
  CHECK_FREAD(ret_val, "reading an 4");
  CHECK_FEOF(ret_val, "reading an 4");
  if (*reverseEndian) reverseFourByteWord(&input_integer);

  if (input_integer != 4) {
    return(DCD_BADFORMAT);
  }

  *FREEINDEXES = NULL;
  *fixedcoords = NULL;
  if (*NAMNF != 0) {
    (*FREEINDEXES) = (int *) calloc(((*N)-(*NAMNF)), sizeof(int));
    if (*FREEINDEXES == NULL)
      return(DCD_BADMALLOC);
    *fixedcoords = (float *)calloc((*N)*4 - (*NAMNF), sizeof(float));
	
    /*  Read in an size				*/
    ret_val = READ(fd, &input_integer, sizeof(int));
    CHECK_FREAD(ret_val, "reading size of index array");
    CHECK_FEOF(ret_val, "reading size of index array");
    if (*reverseEndian) reverseFourByteWord(&input_integer);

    if (input_integer != ((*N)-(*NAMNF))*4) {
      return(DCD_BADFORMAT);
    }
		
    ret_val = READ(fd, (*FREEINDEXES), ((*N)-(*NAMNF))*sizeof(int));
    CHECK_FREAD(ret_val, "reading size of index array");
    CHECK_FEOF(ret_val, "reading size of index array");
    if (*reverseEndian)
    {
      for (input_integer =0; input_integer < ((*N)-(*NAMNF)); input_integer++)
      {
        reverseFourByteWord(FREEINDEXES[input_integer]);
      }
    }

    ret_val = READ(fd, &input_integer, sizeof(int));
    CHECK_FREAD(ret_val, "reading size of index array");
    CHECK_FEOF(ret_val, "reading size of index array");
    if (*reverseEndian) reverseFourByteWord(&input_integer);

    if (input_integer != ((*N)-(*NAMNF))*4) {
      return(DCD_BADFORMAT);
    }
  }

  return 0;
}

static int read_charmm_extrablock(FILE *fd, int charmm, int reverseEndian,
                                  float *unitcell) {

  int input_integer; /* XXX should be int32 */
  int i;
  if ((charmm & DCD_IS_CHARMM) &&
      (charmm & DCD_HAS_EXTRA_BLOCK)) {
    /* Leading integer must be 48 */
    if (fread(&input_integer, sizeof(int), 1, fd) != 1) {
      return DCD_BADREAD; 
    }
    if (reverseEndian) reverseFourByteWord(&input_integer);
    if (input_integer == 48) {
      double tmp[6];
      if (fread(tmp, 48, 1, fd) != 1) return DCD_BADREAD;
      if (reverseEndian) 
        for (i=0; i<6; i++) 
          reverseEightByteWord(tmp+i);
      for (i=0; i<6; i++) unitcell[i] = (float)tmp[i];
    } else {
      /* just skip it */
      if (fseek(fd, input_integer, SEEK_CUR)) return DCD_BADREAD;
    }
    if (fread(&input_integer, sizeof(int), 1, fd) != 1) return DCD_BADREAD; 
  } 

  return 0;
}

static int read_nonfixed_atoms(FILE *fd, int N, int reverseEndian, 
                               float *pos) {

  int i;
  int input_integer; /* XXX should be int32 */

  /* Read leading integer */
  if (fread(&input_integer, sizeof(int), 1, fd) != 1) {
    return DCD_BADREAD;
  }
  if (reverseEndian) reverseFourByteWord(&input_integer);
  if (input_integer != 4*N) return DCD_BADFORMAT;

  /* Read atom coordinates */
  if (fread(pos, 4*N, 1, fd) != 1)
    return DCD_BADREAD;
  if (reverseEndian)
    for (i=0; i<N; i++) 
      reverseFourByteWord(((int *)pos) + i);
  
  /* Read trailing integer */
  if (fread(&input_integer, sizeof(int), 1, fd) != 1) return DCD_BADREAD;
  if (reverseEndian) reverseFourByteWord(&input_integer);
  if (input_integer != 4*N) return DCD_BADFORMAT;
  
  return 0;
}


static int read_fixed_atoms(FILE *fd, int N, int num_free, const int *indexes,
                            int reverseEndian, const float *fixedcoords, 
                            float *freeatoms, float *pos) {

  int i;
  int input_integer; /* XXX should be int32 */
  
  /* Read leading integer */
  if (fread(&input_integer, sizeof(int), 1, fd) != 1) {
    return DCD_BADREAD;
  }
  if (reverseEndian) reverseFourByteWord(&input_integer);
  if (input_integer != 4*num_free) return DCD_BADFORMAT;
  
  /* Read free atom coordinates */
  if (fread(freeatoms, 4*num_free, 1, fd) != 1) 
    return DCD_BADREAD;
  if (reverseEndian) {
    for (i=0; i<num_free; i++)
      reverseFourByteWord(((int *)freeatoms)+i);
  }

  /* Copy fixed and free atom coordinates into position buffer */
  memcpy(pos, fixedcoords, 4*N);
  for (i=0; i<num_free; i++)
    pos[indexes[i]-1] = freeatoms[i];

  /* Read trailing integer */ 
  if (fread(&input_integer, sizeof(int), 1, fd) != 1) return DCD_BADREAD;
  if (reverseEndian) reverseFourByteWord(&input_integer);
  if (input_integer != 4*num_free) return DCD_BADFORMAT;
  return 0;
}
  
static int read_charmm_4dim(FILE *fd, int charmm, int reverseEndian) {
  /* If this is a CHARMm file and contains a 4th dimension block,
     we must skip it to avoid problems */
  int input_integer; /* XXX should be int32 */
  if ((charmm & DCD_IS_CHARMM) &&
    (charmm & DCD_HAS_4DIMS)) {
    if (fread(&input_integer, sizeof(int), 1, fd) != 1) return DCD_BADREAD;  
    if (reverseEndian) reverseFourByteWord(&input_integer);
    if (fseek(fd, input_integer, SEEK_CUR)) return DCD_BADREAD;
    if (fread(&input_integer, sizeof(int), 1, fd) != 1) return DCD_BADREAD;  
  }
  return 0;
}

static int read_dcdstep(FILE * fd, int N, float *X, float *Y, float *Z, 
                 float *unitcell, int num_fixed,
		 int first, int *indexes, float *fixedcoords, 
                 int reverseEndian, int charmm)
{
  int ret_val;		/*  Return value from read		*/

  if ( (num_fixed==0) || first) {
    ret_val = read_charmm_extrablock(fd, charmm, reverseEndian, unitcell);
    if (ret_val) return ret_val;

    ret_val = read_nonfixed_atoms(fd, N, reverseEndian, X);
    if (ret_val) return ret_val;
    ret_val = read_nonfixed_atoms(fd, N, reverseEndian, Y);
    if (ret_val) return ret_val;
    ret_val = read_nonfixed_atoms(fd, N, reverseEndian, Z);
    if (ret_val) return ret_val;

    /* Copy fixed atom coordinates into fixedcoords array if we haven't 
     * already.  We just copy all atoms. 
     */
    if (num_fixed && first) {
      memcpy(fixedcoords, X, N*sizeof(float));
      memcpy(fixedcoords+N, Y, N*sizeof(float));
      memcpy(fixedcoords+2*N, Z, N*sizeof(float));
    }
    ret_val = read_charmm_4dim(fd, charmm, reverseEndian);
    if (ret_val) return ret_val;
  } else {
    ret_val = read_charmm_extrablock(fd, charmm, reverseEndian, unitcell);
    if (ret_val) return ret_val;
    ret_val = read_fixed_atoms(fd, N, N-num_fixed, indexes, reverseEndian,
                               fixedcoords, fixedcoords+3*N, X);
    if (ret_val) return ret_val;
    ret_val = read_fixed_atoms(fd, N, N-num_fixed, indexes, reverseEndian,
                               fixedcoords+N, fixedcoords+3*N, Y);
    if (ret_val) return ret_val;
    ret_val = read_fixed_atoms(fd, N, N-num_fixed, indexes, reverseEndian,
                               fixedcoords+2*N, fixedcoords+3*N, Z);
    if (ret_val) return ret_val;
    ret_val = read_charmm_4dim(fd, charmm, reverseEndian);
    if (ret_val) return ret_val;
  }

  return 0;
}

static int skip_dcdstep(FILE *fd, int natoms, int nfixed, int charmm) {
  /* Skip charmm extra block */
  if ((charmm & DCD_IS_CHARMM) &&
     (charmm & DCD_HAS_EXTRA_BLOCK)) {
    if (fseek(fd, 4 + 48 + 4, SEEK_CUR))
      return DCD_BADEOF;
  }
  /* For each atom set, seek past an int, the free atoms, and another int. */
  if (fseek(fd, 3 * (2 + natoms - nfixed) * 4, SEEK_CUR))
    return DCD_BADEOF;
  /* Assume that charmm 4th dim is the same size as the other three. */
  if ((charmm & DCD_IS_CHARMM) &&
    (charmm & DCD_HAS_4DIMS)) {
    if (fseek(fd, (2 + natoms - nfixed) * 4, SEEK_CUR)) 
      return DCD_BADEOF;
  }

  return 0;
}

#define NFILE_POS 8L
#define NSTEP_POS 20L

static int write_dcdstep(FILE * fd, int curframe, int curstep, int N, 
                  const float *X, const float *Y, const float *Z) {
  int out_integer;
  out_integer = N*4;
  WRITE(fd, &out_integer, sizeof(int));
  WRITE(fd, X, out_integer);
  WRITE(fd, &out_integer, sizeof(int));
  WRITE(fd, &out_integer, sizeof(int));
  WRITE(fd, Y, out_integer);
  WRITE(fd, &out_integer, sizeof(int));
  WRITE(fd, &out_integer, sizeof(int));
  WRITE(fd, Z, out_integer);
  WRITE(fd, &out_integer, sizeof(int));

  fseek(fd, NFILE_POS, SEEK_SET);
  WRITE(fd, &curframe, sizeof(int));
  fseek(fd, NSTEP_POS, SEEK_SET);
  WRITE(fd, &curstep, sizeof(int)); 
  fseek(fd, 0, SEEK_END);

  return 0;
}

static int write_dcdheader(FILE * fd, const char *remarks, int N, 
                    int ISTART, int NSAVC, double DELTA) {
  int	out_integer;
  char	title_string[200];
  time_t  cur_time;
  struct  tm *tmbuf;
  char    time_str[81];

  out_integer = 84;
  WRITE(fd, (char *) & out_integer, sizeof(int));
  strcpy(title_string, "CORD");
  WRITE(fd, title_string, 4);
  out_integer=0;
  WRITE(fd, (char *) &out_integer, sizeof(int)); /* No frames written yet */
  WRITE(fd, (char *) &ISTART, sizeof(int));      /* Starting timestep */
  WRITE(fd, (char *) &NSAVC, sizeof(int));       /* Timesteps between frames */
  WRITE(fd, (char *) &out_integer, sizeof(int)); /* No timesteps written yet */
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &DELTA, sizeof(double));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  WRITE(fd, (char *) &out_integer, sizeof(int));
  out_integer = 84;
  WRITE(fd, (char *) & out_integer, sizeof(int));

  out_integer = 164;
  WRITE(fd, (char *) & out_integer, sizeof(int));
  out_integer = 2;
  WRITE(fd, (char *) & out_integer, sizeof(int));

  strncpy(title_string, remarks, 80);
  title_string[79] = '\0';
  WRITE(fd, title_string, 80);

  cur_time=time(NULL);
  tmbuf=localtime(&cur_time);
  strftime(time_str, 80, "REMARKS Created %d %B, %Y at %R", tmbuf);

  WRITE(fd, time_str, 80);
  out_integer = 164;
  WRITE(fd, (char *) & out_integer, sizeof(int));
  out_integer = 4;
  WRITE(fd, (char *) & out_integer, sizeof(int));
  out_integer = N;
  WRITE(fd, (char *) & out_integer, sizeof(int));
  out_integer = 4;
  WRITE(fd, (char *) & out_integer, sizeof(int));

  return 0;
}

static void close_dcd_read(int *indexes, float *fixedcoords) {
  free(indexes);
  free(fixedcoords);
}

#endif

