/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: mapplugin.C,v $
 *      $Author: eamon $       $Locker:  $             $State: Exp $
 *      $Revision: 1.3 $       $Date: 2003/08/20 16:09:21 $
 *
 ***************************************************************************/

/* 
 * Grid Map File format plugin
 *
 * More info for this format can be found at
 * <http://www.scripps.edu/pub/olson-web/gmm/autodock/ad305/
 *  Using_AutoDock_305.21.html#pgfId=75765>
 * 
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <string.h>

#include "molfile_plugin.h"

#define LINESIZE 85

typedef struct {
  FILE *fd;
  int nsets;
  molfile_volumetric_t *vol;
} gridmap_t;


static void *open_map_read(const char *filepath, const char *filetype,
    int *natoms) {
  FILE *fd;
  gridmap_t *map;
  char inbuf[LINESIZE];

  float spacing, midX, midY, midZ;
  int xsize, ysize, zsize;
  
  fd = fopen(filepath, "rb");
  if (!fd) 
    return NULL;

  /* Skip the header */
  if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL)) 
    return NULL;
  if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL)) 
    return NULL;
  if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL)) 
    return NULL;

  /* Space between grid points */
  if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL)) 
    return NULL;
  if (sscanf(inbuf, "SPACING %f", &spacing) != 1)
    return NULL;

  /* Grid size in grid units */
  if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL)) 
    return NULL;
  if (sscanf(inbuf, "NELEMENTS %d %d %d", &xsize, &ysize, &zsize) != 3)
    return NULL;

  /* XXX - I don't know why this is necessary */
  xsize++;
  ysize++;
  zsize++;

  /* Center of the cell */
  if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL)) 
    return NULL;
  if (sscanf(inbuf, "CENTER %f %f %f", &midX, &midY, &midZ) != 3)
    return NULL;

  /* Allocate and initialize the map structure */
  map = new gridmap_t;
  map->fd = fd;
  map->vol = NULL;
  *natoms = MOLFILE_NUMATOMS_NONE;
  map->nsets = 1; /* this file contains only one data set */

  map->vol = new molfile_volumetric_t[1];
  strcpy(map->vol[0].dataname, "Grid Map File");

  /* <midX, midY, midZ> is the middle point of the grid. */
  map->vol[0].origin[0] = -0.5*(xsize+1.0)* spacing  + midX;
  map->vol[0].origin[1] = -0.5*(ysize+1.0)* spacing  + midY;
  map->vol[0].origin[2] = -0.5*(zsize+1.0)* spacing  + midZ;

  map->vol[0].xaxis[0] = xsize * spacing;
  map->vol[0].xaxis[1] = 0;
  map->vol[0].xaxis[2] = 0;

  map->vol[0].yaxis[0] = 0;
  map->vol[0].yaxis[1] = ysize * spacing;
  map->vol[0].yaxis[2] = 0;
  
  map->vol[0].zaxis[0] = 0;
  map->vol[0].zaxis[1] = 0;
  map->vol[0].zaxis[2] = zsize * spacing;

  map->vol[0].xsize = xsize;
  map->vol[0].ysize = ysize;
  map->vol[0].zsize = zsize;

  map->vol[0].has_color = 0;

  return map;
}

static int read_map_metadata(void *v, int *nsets, 
  molfile_volumetric_t **metadata) {
  gridmap_t *map = (gridmap_t *)v;
  *nsets = map->nsets; 
  *metadata = map->vol;  

  return MOLFILE_SUCCESS;
}

static int read_map_data(void *v, int set, float *datablock,
                         float *colorblock) {
  gridmap_t *map = (gridmap_t *)v;
  FILE *fd = map->fd;
  float *cellIndex;
  char inbuf[LINESIZE];
  int count, ndata;

  cellIndex = datablock;
  count = 0;
  ndata = map->vol[0].xsize * map->vol[0].ysize * map->vol[0].zsize;

  /* Read the densities. Order for file is x fast, y medium, z slow */
  while (count < ndata) {
    if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL)) {
      return MOLFILE_ERROR;
    }

    *cellIndex = atof(inbuf);

    cellIndex++;
    count++;
  }

  return MOLFILE_SUCCESS;
}

static void close_map_read(void *v) {
  gridmap_t *map = (gridmap_t *)v;

  fclose(map->fd);
  if (map->vol != NULL)
    delete [] map->vol; 
  delete map;
}

/*
 * Initialization stuff here
 */
static molfile_plugin_t plugin = {
  vmdplugin_ABIVERSION,   /* ABI version */
  MOLFILE_PLUGIN_TYPE, 	  /* plugin type */
  "map",                  /* file format description */
  "Eamon Caddigan",       /* author(s) */
  0,                      /* major version */
  4,                      /* minor version */
  VMDPLUGIN_THREADSAFE,   /* is reentrant */
  "map"                   /* filename extension */
};

int VMDPLUGIN_init(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_fini(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  plugin.open_file_read = open_map_read;
  plugin.read_volumetric_metadata = read_map_metadata;
  plugin.read_volumetric_data = read_map_data;
  plugin.close_file_read = close_map_read;
  (*cb)(v, (vmdplugin_t *)&plugin);
  return VMDPLUGIN_SUCCESS;
}

