/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: graspplugin.C,v $
 *      $Author: johns $       $Locker:  $             $State: Exp $
 *      $Revision: 1.7 $       $Date: 2003/06/12 19:39:21 $
 *
 ***************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>

#include "molfile_plugin.h"

// swap ndata 4-byte words
static void swap4(char *data, int ndata) {
  int i;
  char *dataptr;
  char b0, b1;

  dataptr = data;
  for (i=0; i<ndata; i++) {
    b0 = dataptr[0];
    b1 = dataptr[1];
    dataptr[0] = dataptr[3];
    dataptr[1] = dataptr[2];
    dataptr[2] = b1;
    dataptr[3] = b0;
    dataptr += 4;
  }
}

static void swap2(short *data, int ndata) {
  int i;
  char *dataptr;
  char b0;

  dataptr = (char *)data;
  for (i=0; i<ndata; i++) {
    b0 = dataptr[0];
    dataptr[0] = dataptr[1];
    dataptr[1] = b0;
    dataptr += 2;
  }
}

static int is_little_endian(void) {
  int x=1;
    return *((char *)&x);
}   

static void to_native(void *v, int ndata) {
  if (is_little_endian()) 
    swap4((char *)v, ndata);
}

typedef struct {
  FILE *fd;
  molfile_graphics_t *graphics;
} grasp_t;

static void *open_file_read(const char *filepath, const char *filetype,
    int *natoms) {
  FILE *fd;
  grasp_t *grasp;
  
  fd = fopen(filepath, "rb");
  if (!fd) 
    return NULL;
  grasp = new grasp_t;
  grasp->fd = fd;
  grasp->graphics = NULL;
  *natoms = 0;
  return grasp;
}

static int read_rawgraphics(void *v, int *nelem, 
    const molfile_graphics_t **data) {

  grasp_t *grasp = (grasp_t *)v;
	FILE *infile = grasp->fd;

   // Reverse engineering is your friend, and combined with FORTRAN
   // code, voila!

   // od -c shows the header starts off:
   // \0  \0  \0   P   f   o   r   m   a   t   =   2
   // and according to ungrasp, this is a 1 for grasp versions 1.0
   // and 1.1, and 2 for grasp version 1.2
   // Also, the header lines are of length 80 characters + 4 header chars
   // + 4 trailer chars
   // The 4 bytes at the beginning/end are standard Fortran array trash
	//
   char trash[4];
#define TRASH fread(trash, 4, 1, infile)
   char line[81];

   // FIRST LINE OF HEADER; contains format type
   TRASH; fread(line, 1, 80, infile); TRASH;
   // make sure it says 'format='
   if (strncmp(line, "format=", 7) != 0) {
     fprintf(stderr, "First few characters of file don't look like a GRASP file\n");
     return MOLFILE_ERROR;
   }
   
   // next char should be a 0 or 1
   char gfiletype = line[7];
   if (gfiletype == '1') {
     gfiletype = 1;
   } else if (gfiletype == '2') {
     gfiletype = 2;
   } else {
     fprintf(stderr, "GRASP file is in format %c, but only '1' or '2' is supported\n", gfiletype);
     return MOLFILE_ERROR;
   }

   // SECOND LINE: contains "vertices,accessibles,normals,triangles"
   TRASH; fread(line, 1, 80, infile); TRASH;

   // THIRD LINE: contains (0 or more of)?
   //  "potentials,curvature,distances,gproperty,g2property,vertexcolor
   
   char line3[81];
   TRASH; fread(line3, 1, 80, infile); TRASH;
   line3[80] = 0;

   // FOURTH LINE stores vertices, triangles, gridsize, lattice spacing
   int nvert, ntriangles, gridsize;
   float lattice;
   TRASH; fread(line, 1, 80, infile); TRASH;
   sscanf(line, "%d%d%d%f", &nvert, &ntriangles, &gridsize, &lattice);

   // FIFTH LINE stores the center (x,y,z) position
   float center[3];
   TRASH; fread(line, 1, 80, infile); TRASH;
   sscanf(line, "%f%f%f", center, center+1, center+2);
   // Should do some minor sanity checking ...  But I won't.

   // Create space for things
   float *vertex = new float[3 * nvert];
   if (!vertex) {
     fprintf(stderr, "Cannot allocate space for %d verticies.\n", nvert);
     return MOLFILE_ERROR;
   }
   float *access = new float[3 * nvert];
   if (!access) {
     delete [] vertex;
     fprintf(stderr, "Cannot allocate space for %d accessibilities.\n", nvert);
     return MOLFILE_ERROR;
   }
   float *normal = new float[3 * nvert];
   if (!normal) {   // My name's Abbie, Abbie Normal ...
     delete [] access;
     delete [] vertex;
     fprintf(stderr, "Cannot allocate space for %d normals.\n", nvert);
     return MOLFILE_ERROR;
   }

   // ungrasp says:
   //    if (filetype.eq.1) then integer*2
   //    if (filetype.eq.2) then integer*4
   int *triangle = new int[3 * ntriangles];
   if (!triangle) {
     delete [] normal;  // Oh, for smart pointers ...
     delete [] access;
     delete [] vertex;
     fprintf(stderr, "Cannot allocate space for %d triangles.\n", ntriangles);
     return MOLFILE_ERROR;
   }

   // And read them in.  Who needs error checking?
   TRASH; fread(vertex, 3 * sizeof(float), nvert, infile); TRASH;
   TRASH; fread(access, 3 * sizeof(float), nvert, infile); TRASH;
   TRASH; fread(normal, 3 * sizeof(float), nvert, infile); TRASH;
   to_native(vertex, 3*nvert);
   to_native(access, 3*nvert);
   to_native(normal, 3*nvert);

   if (gfiletype == 2) {
     TRASH; fread(triangle, 3 * sizeof(int), ntriangles, infile); TRASH;
	 to_native(triangle, 3*ntriangles);
   } else {
     // do it the slow way (converting from short to int)
     int i;
     short tmp[3];
     TRASH;
     for (i=0; i<ntriangles; i++) {
       fread(tmp, sizeof(short), 3, infile);
	   if (is_little_endian()) swap2(tmp, 3);
       triangle[3*i+0] = tmp[0];
       triangle[3*i+1] = tmp[1];
       triangle[3*i+2] = tmp[2];
     }
     TRASH;
   }

   grasp->graphics = new molfile_graphics_t[2*ntriangles+1];
   grasp->graphics[0].type = MOLFILE_COLOR;
   grasp->graphics[0].data[0] = grasp->graphics[0].data[1] = 
     grasp->graphics[0].data[2] = 0.5;

   // And draw things
   int vert1, vert2, vert3;

   for (int tri_count = 0; tri_count < ntriangles; tri_count++) {
     vert1 = triangle[3*tri_count+0] - 1;  // from 1-based FORTRAN
     vert2 = triangle[3*tri_count+1] - 1;  // to 0-based C++
     vert3 = triangle[3*tri_count+2] - 1;

     grasp->graphics[2*tri_count+1].type = MOLFILE_TRINORM;
     grasp->graphics[2*tri_count+2].type = MOLFILE_NORMS;

     float *tridata = grasp->graphics[2*tri_count+1].data;
     float *normdata = grasp->graphics[2*tri_count+2].data;
     memcpy(tridata, vertex+3*vert1, 3*sizeof(float));
     memcpy(tridata+3, vertex+3*vert2, 3*sizeof(float));
     memcpy(tridata+6, vertex+3*vert3, 3*sizeof(float));
     memcpy(normdata, normal+3*vert1, 3*sizeof(float));
     memcpy(normdata+3, normal+3*vert2, 3*sizeof(float));
     memcpy(normdata+6, normal+3*vert3, 3*sizeof(float));
   } 

   *nelem = 2*ntriangles + 1;
   *data = grasp->graphics;

   delete [] triangle;
   delete [] normal;
   delete [] access;
   delete [] vertex;

   return MOLFILE_SUCCESS;
}

static void close_file_read(void *v) {
  grasp_t *grasp = (grasp_t *)v;
  fclose(grasp->fd);
  delete [] grasp->graphics;
  delete grasp;
}


/*
 * Initialization stuff here
 */
static molfile_plugin_t plugin = {
  vmdplugin_ABIVERSION,   // ABI version
  MOLFILE_PLUGIN_TYPE, 	  // type of plugin
  "grasp",                // name of plugin
  "Justin Gullingsrud",   // author
  0,                      // major version
  1,                      // minor version
  VMDPLUGIN_THREADSAFE,   // is reentrant
  "grasp"                 // filename extension 
};

int VMDPLUGIN_init(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_fini(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  plugin.open_file_read = open_file_read;
  plugin.read_rawgraphics = read_rawgraphics;
  plugin.close_file_read = close_file_read;
  (*cb)(v, (vmdplugin_t *)&plugin);
  return VMDPLUGIN_SUCCESS;
}


