/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: dsn6plugin.C,v $
 *      $Author: eamon $       $Locker:  $             $State: Exp $
 *      $Revision: 1.8 $       $Date: 2003/06/19 19:45:55 $
 *
 ***************************************************************************/

/* 
 * DSN6 format electron density maps.
 *
 * More info for format can be found at 
 * <http://www.uoxray.uoregon.edu/tnt/manual/node104.html>
 * 
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <string.h>

#if defined(_AIX)
#include <strings.h>
#endif

#ifdef WIN32
#define strcasecmp stricmp
#endif

#include "molfile_plugin.h"

static void swap2(void * data, int nitems)
{
  int i;
  char * dataptr = (char *) data;
  char tmp;

  for (i = 0; i < nitems-1; i += 2)
  {
    tmp = dataptr[i];
    dataptr[i] = dataptr[i+1];
    dataptr[i+1] = tmp;
  }
}

typedef struct {
  FILE *fd;
  int nsets;
  float prod, plus;
  molfile_volumetric_t *vol;
} dsn6_t;


static void *open_dsn6_read(const char *filepath, const char *filetype,
    int *natoms) {
  FILE *fd;
  dsn6_t *dsn6;
  short fileHeader[19];
  // File header data:
  float org_x, org_y, org_z, ext_x, ext_y, ext_z, grid_x, grid_y, grid_z,
        cell_x, cell_y, cell_z, cell_alpha, cell_beta, cell_gamma, 
        alpha, beta, gamma;
  
  fd = fopen(filepath, "rb");
  if (!fd) 
    return NULL;

  // Read the header into a 19-element int array. The integers are stored
  // according to the endianness of the machine used to write the file, so
  // swapping may be necessary.

  fread(fileHeader, sizeof(short), 19, fd);
  
  // Check byte-order, swapping if necessary
  if (fileHeader[18] == 25600)
    swap2(fileHeader, 19*sizeof(short));
  else if (fileHeader[18] != 100)
    return NULL;
  // else fileHeader[18] is 100, byte-order is fine

  org_x = fileHeader[0];
  org_y = fileHeader[1];
  org_z = fileHeader[2];

  ext_x = fileHeader[3];
  ext_y = fileHeader[4];
  ext_z = fileHeader[5];

  grid_x = fileHeader[6];
  grid_y = fileHeader[7];
  grid_z = fileHeader[8];

  cell_x = (float) fileHeader[9] / fileHeader[17];
  cell_y = (float) fileHeader[10] / fileHeader[17];
  cell_z = (float) fileHeader[11] / fileHeader[17];

  cell_alpha = (float) fileHeader[12] / fileHeader[17];
  cell_beta = (float) fileHeader[13] / fileHeader[17];
  cell_gamma = (float) fileHeader[14] / fileHeader[17];

  // Allocate and initialize the dsn6 structure
  dsn6 = new dsn6_t;
  dsn6->fd = fd;
  dsn6->vol = NULL;
  *natoms = MOLFILE_NUMATOMS_NONE;
  dsn6->nsets = 1; // this file contains only one data set

  dsn6->prod = (float) fileHeader[15] / fileHeader[18];
  dsn6->plus = fileHeader[16];

  dsn6->vol = new molfile_volumetric_t[1];
  strcpy(dsn6->vol[0].dataname, "DSN6 Electron Density Map");

  dsn6->vol[0].origin[0] = (org_x / grid_x) * cell_x;
  dsn6->vol[0].origin[1] = (org_y / grid_y) * cell_y;
  dsn6->vol[0].origin[2] = (org_z / grid_z) * cell_z;

  // Calculate non-orthogonal unit cell coordinates
  alpha = 3.14159265358979323846 * cell_alpha / 180.0;
  beta = 3.14159265358979323846 * cell_beta / 180.0;
  gamma = 3.14159265358979323846 * cell_gamma / 180.0;

  dsn6->vol[0].xaxis[0] = (ext_x / grid_x) * cell_x;
  dsn6->vol[0].xaxis[1] = 0;
  dsn6->vol[0].xaxis[2] = 0;

  dsn6->vol[0].yaxis[0] = cos(gamma) * (ext_x / grid_x) * cell_x;
  dsn6->vol[0].yaxis[1] = sin(gamma) * (ext_y / grid_y) * cell_y;
  dsn6->vol[0].yaxis[2] = 0;

  float z1, z2, z3;
  z1 = cos(beta);
  z2 = (cos(alpha) - cos(beta)*cos(gamma)) / sin(gamma);
  z3 = sqrt(1.0 - z1*z1 - z2*z2);
  dsn6->vol[0].zaxis[0] = z1 * (ext_x / grid_x) * cell_x;
  dsn6->vol[0].zaxis[1] = z2 * (ext_y / grid_y) * cell_y;
  dsn6->vol[0].zaxis[2] = z3 * (ext_z / grid_z) * cell_z;

  dsn6->vol[0].xsize = (int) ext_x;
  dsn6->vol[0].ysize = (int) ext_y;
  dsn6->vol[0].zsize = (int) ext_z;

  dsn6->vol[0].has_color = 0;

  return dsn6;
}

static int read_dsn6_metadata(void *v, int *nsets, 
  molfile_volumetric_t **metadata) {
  dsn6_t *dsn6 = (dsn6_t *)v;
  *nsets = dsn6->nsets; 
  *metadata = dsn6->vol;  

  return MOLFILE_SUCCESS;
}

static int read_dsn6_data(void *v, int set, float *datablock,
                         float *colorblock) {
  dsn6_t *dsn6 = (dsn6_t *)v;
  float * cell = datablock;
  unsigned char brick[512];
  unsigned char* brickPtr = NULL;
  int xsize, ysize, zsize, xysize, xbrix, ybrix, zbrix, cellIndex;
  int x, y, z, xbrik, ybrik, zbrik;
  FILE * fd = dsn6->fd;
  float div, plus;

  // Read 512-byte "bricks" of data. Each brick contains data for 8*8*8 
  // gridpoints.
  fseek(fd, 512, SEEK_SET);

  div = 1.0 / dsn6->prod;
  plus = dsn6->plus;

  xsize = dsn6->vol[0].xsize;
  ysize = dsn6->vol[0].ysize;
  zsize = dsn6->vol[0].zsize;
  xysize = xsize * ysize;

  xbrix = (int) ceil((float) xsize / 8.0);
  ybrix = (int) ceil((float) ysize / 8.0);
  zbrix = (int) ceil((float) zsize / 8.0);

  cellIndex = 0;
  for (zbrik = 0; zbrik < zbrix; zbrik++)
  {
    for (ybrik = 0; ybrik < ybrix; ybrik++)
    {
      for (xbrik = 0; xbrik < xbrix; xbrik++)
      {
        // Read the next brick into the buffer and swap its bytes.
        if (feof(fd) || ferror(fd))
          return MOLFILE_ERROR;

        fread(brick, sizeof(char), 512, fd);
        swap2(brick, 512*sizeof(char));
        brickPtr = brick;

        for (z = 0; z < 8; z++)
        {
          if ((z + zbrik*8) >= zsize)
          {
            cellIndex += (8 - z) * xysize;
            break;
          }

          for (y = 0; y < 8; y++)
          {
            if ((y + ybrik*8) >= ysize)
            {
              cellIndex += (8 - y) * xsize;
              brickPtr += (8 - y) * 8;
              break;
            }

            for (x = 0; x < 8; x++)
            {
              if ((x + xbrik*8) >= xsize)
              {
                cellIndex += 8 - x;
                brickPtr += 8 - x;
                break;
              }

              // cell[(x+xbrik*8) + (y+ybrik*8)*xsize + (z+zbrik*8)*xysize] =
              // div * ((float) brick[x + 8*y + 64*z] - plus)
              cell[cellIndex] = div * ((float) *brickPtr - plus);

              brickPtr++;
              cellIndex++;
            } // end for(x)
           
            cellIndex += xsize - 8;
          } // end for(y)
         
          cellIndex += xysize - 8*xsize;
        } // end for(z)
      
        cellIndex += 8 - 8*xysize; 
      } // end for(xbrik)

      cellIndex += 8 * (xsize - xbrix);
    } // end for(ybrik)

    cellIndex += 8 * (xysize - xsize*ybrik);
  } // end for(zbrik)
 
  return MOLFILE_SUCCESS;
}

static void close_dsn6_read(void *v) {
  dsn6_t *dsn6 = (dsn6_t *)v;

  fclose(dsn6->fd);
  if (dsn6->vol != NULL)
    delete [] dsn6->vol; 
  delete dsn6;
}

/*
 * Initialization stuff here
 */
static molfile_plugin_t plugin = {
  vmdplugin_ABIVERSION,   // ABI version
  MOLFILE_PLUGIN_TYPE, 	  // plugin type
  "DSN6",                 // file format description
  "Eamon Caddigan",       // author(s)
  0,                      // major version
  1,                      // minor version
  VMDPLUGIN_THREADSAFE,   // is reentrant
  "ds6"                   // filename extension
};

int VMDPLUGIN_init(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_fini(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  plugin.open_file_read = open_dsn6_read;
  plugin.read_volumetric_metadata = read_dsn6_metadata;
  plugin.read_volumetric_data = read_dsn6_data;
  plugin.close_file_read = close_dsn6_read;
  (*cb)(v, (vmdplugin_t *)&plugin);
  return VMDPLUGIN_SUCCESS;
}

