/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: cubeplugin.C,v $
 *      $Author: eamon $       $Locker:  $             $State: Exp $
 *      $Revision: 1.8 $       $Date: 2003/08/20 18:17:14 $
 *
 ***************************************************************************/

//
// Plugin reader for Gaussian "cube" files
//
// Gaussian "cube" file format described here:
//   http://www.gaussian.com/00000430.htm
//

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "molfile_plugin.h"

/* periodic table of elements for translation of ordinal to atom type */
static const char *pte[] = { 
    "X",  "H",  "He", "Li", "Be", "B",  "C",  "N",  "O",  "F",  "Ne",
    "Na", "Mg", "Al", "Si", "P" , "S",  "Cl", "Ar", "K",  "Ca", "Sc",
    "Ti", "V",  "Cr", "Mn", "Fe", "Co", "Ni", "Cu", "Zn", "Ga", "Ge", 
    "As", "Se", "Br", "Kr", "Rb", "Sr", "Y",  "Zr", "Nb", "Mo", "Tc",
    "Ru", "Rh", "Pd", "Ag", "Cd", "In", "Sn", "Sb", "Te", "I",  "Xe",
    "Cs", "Ba", "La", "Ce", "Pr", "Nd", "Pm", "Sm", "Eu", "Gd", "Tb",
    "Dy", "Ho", "Er", "Tm", "Yb", "Lu", "Hf", "Ta", "W",  "Re", "Os",
    "Ir", "Pt", "Au", "Hg", "Tl", "Pb", "Bi", "Po", "At", "Rn", "Fr",
    "Ra", "Ac", "Th", "Pa", "U",  "Np", "Pu", "Am", "Cm", "Bk", "Cf",
    "Es", "Fm", "Md", "No", "Lr"
 };
static const int nr_pte = sizeof(pte) / sizeof(char *);
static const float bohr = 0.529177249;

typedef struct {
  FILE *fd;              // file descriptor
  int nsets;             // number of volume datasets
  int numatoms;          // number of atoms
  bool coord;            // has coordinate data
  long crdpos, datapos;  // seek offsets for coords and data
  char *file_name;       // original filename 
  float *datacache;      // temporary cache of orbital data prior to conversion
  molfile_volumetric_t *vol; // volume data
} cube_t;

static void eatline(FILE * fd) {
  char readbuf[1025];
  fgets(readbuf, 1024, fd);    // go on to next line
}  

static void close_cube_read(void *v);

static void *open_cube_read(const char *filepath, const char *filetype,
    int *natoms) {
  FILE *fd;
  cube_t *cube;
  int xsize, ysize, zsize;
  
  fd = fopen(filepath, "rb");
  if (!fd) 
    return NULL;

  cube = new cube_t;
  cube->fd = fd;
  cube->vol = NULL;
  cube->coord = false;
  cube->file_name = strdup(filepath);
  cube->datacache = NULL;
  molfile_volumetric_t voltmpl; // base information for all data sets.

  // read in cube file header information
  char readbuf[256]; 
  fgets(readbuf, 256, cube->fd);    // go on to next line

  // identify this file, and read title string into dataset info
  strcpy(voltmpl.dataname, "Gaussian Cube: ");
  strncat(voltmpl.dataname, readbuf, 240);      // 240 is max space left after
                                                //   "Gaussian Cube: "
  eatline(cube->fd);          // skip second header line 

  // read in number of atoms
  if (fscanf(cube->fd, "%d", &cube->numatoms) != 1) {
    close_cube_read(cube);
    return NULL;
  }

  if (cube->numatoms > 0) {   // density cube file
    cube->nsets = 1;          // this cube file contains only one data set
  } else {
    // cube file with orbitals => multiple densities.
    cube->numatoms = - cube->numatoms;
    cube->nsets = 0;          // we don't know yet how many data sets.
  }
  *natoms = cube->numatoms;

  // read in cube origin
  if ( fscanf(cube->fd, "%f %f %f", 
         &voltmpl.origin[0], 
         &voltmpl.origin[1], 
         &voltmpl.origin[2]) != 3 ) {
    close_cube_read(cube);
    return NULL;
  }

  // read in cube axes and sizes
  if ((fscanf(cube->fd, "%d", &xsize) != 1) ||
      (fscanf(cube->fd, "%f %f %f", 
         &voltmpl.xaxis[0], 
         &voltmpl.xaxis[1], 
         &voltmpl.xaxis[2]) != 3)) {
    close_cube_read(cube);
    return NULL;
  }

  if ((fscanf(cube->fd, "%d", &ysize) != 1) ||
      (fscanf(cube->fd, "%f %f %f", 
         &voltmpl.yaxis[0], 
         &voltmpl.yaxis[1], 
         &voltmpl.yaxis[2]) != 3)) {
    close_cube_read(cube);
    return NULL;
  }

  if ((fscanf(cube->fd, "%d", &zsize) != 1) ||
      (fscanf(cube->fd, "%f %f %f", 
         &voltmpl.zaxis[0], 
         &voltmpl.zaxis[1], 
         &voltmpl.zaxis[2]) != 3)) {
    close_cube_read(cube);
    return NULL;
  }

  // calculate number of samples in each dimension
  voltmpl.xsize = xsize;
  voltmpl.ysize = ysize;
  voltmpl.zsize = zsize;
  voltmpl.has_color = 0;

  eatline(cube->fd);     // skip remaining end of line characters

  // all dimensional units are always in Bohr
  // so scale axes and origin correctly.
  // NOTE: the angstroms are only allowed in input.
  voltmpl.origin[0] *= bohr; 
  voltmpl.origin[1] *= bohr;
  voltmpl.origin[2] *= bohr;

  voltmpl.xaxis[0] *= bohr * xsize; 
  voltmpl.xaxis[1] *= bohr * xsize; 
  voltmpl.xaxis[2] *= bohr * xsize; 

  voltmpl.yaxis[0] *= bohr * ysize; 
  voltmpl.yaxis[1] *= bohr * ysize; 
  voltmpl.yaxis[2] *= bohr * ysize; 

  voltmpl.zaxis[0] *= bohr * zsize; 
  voltmpl.zaxis[1] *= bohr * zsize; 
  voltmpl.zaxis[2] *= bohr * zsize; 

  cube->crdpos = ftell(cube->fd); // and record beginning of coordinates
  // XXX fseek()/ftell() are incompatible with 64-bit LFS I/O implementations, 
  // hope we don't read any files >= 2GB...

  if (cube->nsets >0) { 
    int i;

    // density cube file, copy voltmpl into the cube struct.
    cube->vol = new molfile_volumetric_t[1];
    memcpy(cube->vol, &voltmpl, sizeof(voltmpl));

    // skip over coordinates to find the start of volumetric data
    for (i=0; i < cube->numatoms; i++) {
      eatline(cube->fd);
    }

    cube->datapos = ftell(cube->fd); // and record beginning of data
    // XXX fseek()/ftell() are incompatible with 64-bit LFS I/O, 
    // hope we don't read any files >= 2GB...
  } else {              
    int i;

    // orbital cube file. we now have to read the orbitals section
    // skip over coordinates
    for (i=0; i < cube->numatoms; i++) {
      eatline(cube->fd);
    }
      
    fscanf(cube->fd, "%d", &cube->nsets);
    fprintf(stderr, "\ncubeplugin) found %d orbitals\n", cube->nsets);
    cube->vol = new molfile_volumetric_t[cube->nsets];
    for (i=0; i < cube->nsets; ++i) {
      int orb;
      fscanf(cube->fd, "%d", &orb);
      memcpy(&cube->vol[i], &voltmpl, sizeof(voltmpl));
      sprintf(cube->vol[i].dataname, "Gaussian Cube: Orbital %d", orb);
    }
      
    eatline(cube->fd);        // gobble up rest of line
    cube->datapos = ftell(cube->fd); // and record beginning of data
    // XXX fseek()/ftell() are incompatible with 64-bit LFS I/O, 
    // hope we don't read any files >= 2GB...
  }

  return cube;
}

  
static int read_cube_structure(void *v, int *optflags, molfile_atom_t *atoms) {
  int i, j;
  char *k;
  molfile_atom_t *atom;

  cube_t *cube = (cube_t *)v;

  // go to coordinates
  fseek(cube->fd, cube->crdpos, SEEK_SET);
  // XXX fseek()/ftell() are incompatible with 64-bit LFS I/O implementations, 
  // hope we don't read any files >= 2GB...

  *optflags = MOLFILE_NOOPTIONS; /* no optional data */

  for(i=0;i<cube->numatoms;i++) {
    int idx;
    float chrg, coord;
    char fbuffer[1024];

    atom = atoms + i;

    k = fgets(fbuffer, 1024, cube->fd);
    j=sscanf(fbuffer, "%d %f %f %f %f", &idx, &chrg, &coord, &coord, &coord);
    if (k == NULL) {
      fprintf(stderr, "cube structure) missing atom(s) in "
              "file '%s'\n",cube->file_name);
      fprintf(stderr, "cube structure) expecting '%d' atoms,"
              " found only '%d'\n",cube->numatoms,i+1);
      return MOLFILE_ERROR;
    } else if (j < 5) {
      fprintf(stderr, "cube structure) missing type or coordinate(s)"
              " in file '%s' for atom '%d'\n",cube->file_name,i+1);
      return MOLFILE_ERROR;
    }

    // assign atom symbol to number. flag unknown or dummy atoms with X.
    if (idx < nr_pte) {
        strncpy(atom->name, pte[idx], sizeof(atom->name));
    } else {
        strncpy(atom->name, pte[0], sizeof(atom->name));
    }
    strncpy(atom->type, atom->name, sizeof(atom->type));
    strcpy(atom->resname, "");
    atom->resid = 1;
    strcpy(atom->chain, "");
    strcpy(atom->segid, "");
    /* skip to the end of line */
  }

  return MOLFILE_SUCCESS;
}


static int read_cube_timestep(void *v, int natoms, molfile_timestep_t *ts) {
  int i, j;
  char fbuffer[1024];
  float x, y, z;
  char *k;
  
  cube_t *cube = (cube_t *)v;

  // there is only one set of coordinates
  if (cube->coord) return MOLFILE_ERROR;
  cube->coord = true;

  // jump to coordinate position
  fseek(cube->fd, cube->crdpos, SEEK_SET);
  // XXX fseek()/ftell() are incompatible with 64-bit LFS I/O implementations, 
  // hope we don't read any files >= 2GB...
 
  /* read the coordinates */
  for (i=0; i<cube->numatoms; i++) {
    k = fgets(fbuffer, 1024, cube->fd);
    j = sscanf(fbuffer, "%*d %*f %f %f %f", &x, &y, &z);
    
    if (k == NULL) {
      return MOLFILE_ERROR;
    } else if (j < 3) {
      fprintf(stderr, "cube timestep) missing type or coordinate(s) in file '%s' for atom '%d'\n",cube->file_name,i+1);
      return MOLFILE_ERROR;
    } else if (j>=3) {
      if (ts != NULL) { 
        // Only save coords if we're given a timestep pointer, 
        // otherwise assume that VMD wants us to skip past it.
        // All coordinates are in Bohr, so they must be converted.
        ts->coords[3*i  ] = bohr * x;
        ts->coords[3*i+1] = bohr * y;
        ts->coords[3*i+2] = bohr * z;
      }
    } else {
      break;
    }
  }
  
  return MOLFILE_SUCCESS;
}

static int read_cube_metadata(void *v, int *nsets, 
  molfile_volumetric_t **metadata) {
  cube_t *cube = (cube_t *)v;
  *nsets = cube->nsets; 
  *metadata = cube->vol;  

  return MOLFILE_SUCCESS;
}

static int read_cube_data(void *v, int set, float *datablock, float *colorblock) {
  cube_t *cube = (cube_t *)v;

  fprintf(stderr, "cubeplugin) trying to read cube data set %d\n", set);

  int xsize = cube->vol[set].xsize; 
  int ysize = cube->vol[set].ysize;
  int zsize = cube->vol[set].zsize;
  int xysize = xsize*ysize;
  int nsize = cube->nsets;
  int nzsize = nsize*zsize;
  int nyzsize = nsize*zsize*ysize;
  int x, y, z;

  // go to data
  fseek(cube->fd, cube->datapos, SEEK_SET);
  // XXX fseek()/ftell() are incompatible with 64-bit LFS I/O implementations, 
  // hope we don't read any files >= 2GB...

  // read the data values in 
  if (cube->nsets == 1) { // density cube file
    for (x=0; x<xsize; x++) {
      for (y=0; y<ysize; y++) {
        for (z=0; z<zsize; z++) {
          if (fscanf(cube->fd, "%f", &datablock[z*xysize + y*xsize + x]) != 1) {
              return MOLFILE_ERROR;
          }
        }
      } 
    }
  } else {
    // XXX we may wish to examine this strategy for alternatives that provide
    // the same performance but without the extra copy, but it makes sense 
    // for now.  

    // Since the orbital cube file stores the data orb1(a1,b1,c1), orb2(a1,b1,c1), 
    // ... orbn(a1,b1,c1), orb1(a1,b1,c2), orb2(a1,a1,c2), ..., orbn(ai,bj,ck)
    // we have to cache the whole data set of have any kind of reasonable performance.
    // otherwise we would have to read (and parse!) the whole file over and over again.
    // this way we have to do it only once at the temporary expense of some memory.
    if (cube->datacache == NULL) {
      int points = xsize*ysize*zsize * nsize; // number of grid cells * nsets
      int i;

      // let people know what is going on.
      fprintf(stderr, "\ncubeplugin) creating %d MByte cube orbital cache.\n", 
              (points*sizeof(float)) / 1048576);
      cube->datacache = new float[points];
            
      for (i=0; i < points; ++i) {
        if (fscanf(cube->fd, "%f", &cube->datacache[i]) != 1) {
          return MOLFILE_ERROR;
        }

        // print an ascii progress bar so impatient people do not get scared.
        if ((i % (1048576/sizeof(float))) == 0) {  // one dot per MB.
          fprintf(stderr, "."); 
        }
      }
    }
      
    for (x=0; x<xsize; x++) {
      for (y=0; y<ysize; y++) {
        for (z=0; z<zsize; z++) {
          datablock[z*xysize + y*xsize + x] = cube->datacache[x*nyzsize + y*nzsize + z*nsize + set];
        }
      }
    }
  }

  return MOLFILE_SUCCESS;
}

static void close_cube_read(void *v) {
  cube_t *cube = (cube_t *)v;

  fclose(cube->fd);
  if (cube->vol) {
    delete[] cube->vol; 
  }
  free(cube->file_name);
  if (cube->datacache) { 
    fprintf(stderr, "\ncubeplugin) freeing cube orbital cache.\n");
    delete[] cube->datacache; 
  }  

  delete cube;
}

/*
 * Initialization stuff here
 */
static molfile_plugin_t plugin = {
  vmdplugin_ABIVERSION,   // ABI version
  MOLFILE_PLUGIN_TYPE, 	  // plugin type
  "cube",                 // file format description
  "Axel Kohlmeyer, John E. Stone", // author(s)
  0,                      // major version
  4,                      // minor version
  VMDPLUGIN_THREADSAFE,   // is reentrant
  "cube",                 // filename extension
  open_cube_read,               
  read_cube_structure,
  0,                      // read_bonds
  read_cube_timestep,
  close_cube_read,
  0,                      // open_file_write
  0,                      // write_structure
  0,                      // write_timestep
  0,                      // close_file_write
  read_cube_metadata,
  read_cube_data,
  0                       // read_rawgraphics
};

int VMDPLUGIN_init(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_fini(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  (*cb)(v, (vmdplugin_t *)&plugin);
  return VMDPLUGIN_SUCCESS;
}



#ifdef TEST_PLUGIN

int main(int argc, char *argv[]) {
  int natoms;
  void *v;
  int i, nsets, set;
  molfile_volumetric_t * meta;

  while (--argc) {
    ++argv;
    v = open_cube_read(*argv, "cube", &natoms);
    if (!v) {
      fprintf(stderr, "open_cube_read failed for file %s\n", *argv);
      return 1;
    }
    fprintf(stderr, "open_cube_read succeeded for file %s\n", *argv);

    // try loading the EDM metadata now
    if (read_cube_metadata(v, &nsets, &meta)) {
      return 1; // failed to load cube file
    }
    fprintf(stderr, "read_cube_metadata succeeded for file %s\n", *argv);

    for (set=0; set<nsets; set++) {
      printf("Loading volume set: %d\n", set);   
      
      int elements = meta[set].xsize * meta[set].ysize * meta[set].zsize;
      printf("   Grid Elements: %d\n", elements);
      printf(" Grid dimensions: X: %d Y: %d Z: %d\n", 
             meta[set].xsize, meta[set].ysize, meta[set].zsize);

      float * voldata = (float *) malloc(sizeof(float) * elements);
      float * coldata = NULL;

      if (meta[set].has_color) {
        coldata = (float *) malloc(sizeof(float) * elements * 3);
      }

      // try loading the data sets now
      if (read_cube_data(v, set, voldata, coldata)) {
        return 1; // failed to load cube file
      }

      printf("First 6 elements:\n   ");
      for (i=0; i<6; i++) {
        printf("%g, ", voldata[i]);
      }
      printf("\n"); 

      printf("Last 6 elements:\n   ");
      for (i=elements - 6; i<elements; i++) {
        printf("%g, ", voldata[i]);
      }
      printf("\n"); 
    }

    close_cube_read(v);
  }
  return 0;
}

#endif




