/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: crdplugin.c,v $
 *      $Author: eamon $       $Locker:  $             $State: Exp $
 *      $Revision: 1.21 $       $Date: 2003/07/09 15:24:59 $
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "molfile_plugin.h"

typedef struct {
  FILE *file;
  int has_box;
  int numatoms;
} crddata;
 
static void *open_crd_read(const char *filename, const char *filetype, 
    int *natoms) {
 
  FILE *fd;
  crddata *data;
 
  fd = fopen(filename, "rb");
  if (!fd) return NULL;
  
  /* first line is title, so skip past it */
  while (getc(fd) != '\n');

  /* 
   * CRD's don't store the number of atoms in the timestep, so we assume that
   * the application will determine this for us.  
   */
  data = (crddata *)malloc(sizeof(crddata));
  data->file = fd;
  *natoms = MOLFILE_NUMATOMS_UNKNOWN;
  /* filetype "crd" has no box; filetype "crdbox" does. */
  data->has_box = strcmp(filetype, "crd"); 
  return data;
}

/*
 * CRD files with box info are indistinguishable from regular CRD's.  
 * We regard CRD's with box info as a different file format.
 * CRD's don't tell how many atoms there are in each frame.  We therefore
 * rely on the numatoms field in the molfile_timestep_t parameter.
 */
static int read_crd_timestep(void *mydata, int natoms, molfile_timestep_t *ts) {
  crddata *crd = (crddata *)mydata;
  int i, j;
  float x, y, z;
  float a, b, c;

  for (i=0; i<natoms; i++) {
    j = fscanf(crd->file, "%f %f %f", &x, &y, &z);
    if (j == EOF) {
      return MOLFILE_ERROR;
    } else if (j <= 0) {
      fprintf(stderr, "Problem reading CRD file\n");
      return MOLFILE_ERROR;
    }

    /* only save coords if we're given a valid ts pointer */ 
    /* otherwise assume that VMD wants us to skip it..    */
    if (ts != NULL) {
      ts->coords[3*i  ] = x;
      ts->coords[3*i+1] = y;
      ts->coords[3*i+2] = z;
    }
  }

  /* Read the box info. */
  if (crd->has_box) {
    j = fscanf(crd->file, "%f %f %f", &a, &b, &c);
    /* the next line was added by Justin, but doesn't appear to work */
    /* with files I've actually loaded, so I've disabled it for now. */
    /* j = fscanf(crd->file, "%f %f %f", &x, &y, &z);                */
    if (j == EOF) {
      printf("EOF in box\n");
      return MOLFILE_ERROR;
    } else if (j <= 0) {
      printf("Problem reading box part of CRD file, scanf returned %d\n",j);
      return MOLFILE_ERROR;
    }
    ts->A = a;
    ts->B = b;
    ts->C = c;
    ts->alpha = 90.0;
    ts->beta = 90.0;
    ts->gamma = 90.0;
  }

  return MOLFILE_SUCCESS;
}
    
static void close_crd_read(void *mydata) {
  crddata *crd = (crddata *)mydata;
  fclose(crd->file);
  free(crd);
}

static void *open_crd_write(const char *path, const char *filetype,
    int natoms) {
  crddata *crd;
  FILE *fd;

  fd = fopen(path, "wb");
  if (!fd) {
    fprintf(stderr, "Could not open file %s for writing\n", path);
    return NULL;
  }
  fprintf(fd, "TITLE : Created by VMD with %d atoms\n", natoms);
  
  crd = (crddata *)malloc(sizeof(crddata));
  crd->file = fd;
  crd->numatoms = natoms;
  crd->has_box = strcmp(filetype, "crd"); 
  return crd;
}    
  
static int write_crd_timestep(void *v, const molfile_timestep_t *ts) {
  crddata *crd = (crddata *)v;
  int i;
  const int ndata = crd->numatoms * 3;
  for (i=0; i<ndata; i++) {
    fprintf(crd->file, "%8.3f", ts->coords[i]);
    if (i % 10 == 0) fprintf(crd->file, "\n"); 
  }
  if (crd->has_box) {
    fprintf (crd->file, "\n0.000 0.000 0.000\n");
  }

  return MOLFILE_SUCCESS;
}

static void close_crd_write(void *v) {
  crddata *crd = (crddata *)v;
  fclose(crd->file);
  free(crd);
}

/* registration stuff */
    
static molfile_plugin_t crdplugin = {
  vmdplugin_ABIVERSION,                         /* ABI version */
  MOLFILE_PLUGIN_TYPE,                          /* type */
  "crd",                                        /* name */
  "Justin Gullingsrud, John E. Stone",          /* author */
  0,                                            /* major version */
  2,                                            /* minor version */
  VMDPLUGIN_THREADSAFE,                         /* is reentrant */
  "crd",                                        /* filename extension */
  open_crd_read,
  0,
  0,
  read_crd_timestep,
  close_crd_read,
  open_crd_write,
  0,
  write_crd_timestep,
  close_crd_write
};

static molfile_plugin_t crdboxplugin = {
  vmdplugin_ABIVERSION,                         /* ABI version */
  MOLFILE_PLUGIN_TYPE,                          /* type */
  "crdbox",                                     /* name */
  "Justin Gullingsrud, John E. Stone",          /* author */
  0,                                            /* major version */
  3,                                            /* minor version */
  VMDPLUGIN_THREADSAFE,                         /* is reentrant */
  "crd",                                        /* filename extension */
  open_crd_read,
  0,
  0,
  read_crd_timestep,
  close_crd_read,
  open_crd_write,
  0,
  write_crd_timestep,
  close_crd_write
};

int VMDPLUGIN_init(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_fini(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  (*cb)(v, (vmdplugin_t *)&crdplugin);
  (*cb)(v, (vmdplugin_t *)&crdboxplugin);
  return VMDPLUGIN_SUCCESS;
}

