# contactmap.tcl  -- VMD script to compare 2 molecules, in the style of a contact map
#
# Copyright (c) 2002 The Board of Trustees of the University of Illinois
#
# Barry Isralewitz  barryi@ks.uiuc.edu    
# vmd@ks.uiuc.edu
#
# $Id: contactmap.tcl,v 1.21 2003/05/16 20:36:16 johns Exp $
#
#
# To try out:
# 0) start vmd
# 1) mol pdbload 1tit
# 2) source contactmap.tcl; contactmap
# 3) select "fit all", then "Calculate: Calc res-res Dists"
# 4) move over contact map with button 2 (middle button) pressed
#    to see pair selected
# 5) After changing moelcule/typed selection, re-select MoleculeA
#    popup menui, then re-select "Calculate: Calc res-res Dists".
# 6) Move mouse over contact map with middle button pressed to see the vert and horz axis highlights
# 7) caveat: only aware of molec# changes after MoleculeA is changed.  To change both, change B, then A.
#   Note: fun to try with slightly misaligned molecs, try two copies
#    slightly moved/shifted from each other.

package provide contactmap 1.0

#######################
#create the namespace
#######################
proc contactmap {} {
  return [::contactmap::startContactmap]
}
namespace eval ::contactmap {
  namespace export aligntool
  variable fit_scalex 1
  variable fit_scaley 1
}


####################
#define the procs
####################

proc ::contactmap::canvasScrollY {args} { 
  variable w
  
  eval $w.can yview $args
  eval $w.vertScale yview $args 
}     
proc ::contactmap::canvasScrollX {args} { 
  variable w
  eval $w.can xview $args
  eval $w.horzScale xview $args 
  return
}

proc ::contactmap::dataCanvasScrollX {args} {
  #synchronizes scollbar _and_ scale window to data movement
  variable w
  variable scalex
  variable fit_scalex
  variable xcanwindowmax
  variable xcol
  variable firstData
  variable dataWidth
  variable numHoriz
  variable xScaling
  #note -- not needed, runs fine without, isnt even called!
  puts "in dataCanvasScrollX, xScaling= $xScaling"
  if {$xScaling == 1} {return}

  #lock out access to this function
  set xScaling 1
  puts "set xScaling to $xScaling"
  set v [$w.can xview]
  set frac [expr [lindex $v 1] - [lindex $v 0] ] 
  #set frac [expr $start-$end ] 
  #asking for trouble, likely many runs till settles
  set fit_scalex [expr $frac * ( (0.0 + $xcanwindowmax - $xcol($firstData) ) / ($dataWidth * (2 + $numHoriz) ) ) ]
  puts "fit_scalex= $fit_scalex, scalex= $scalex"
  if {$fit_scalex != $scalex} {
    set fit_scalex [expr $frac * ( (0.0 + $xcanwindowmax - $xcol($firstData) ) / ($dataWidth * (2 + $numHoriz) ) ) ]
    set scalex $fit_scalex
    redraw name func op
  }
  set xScaling 0
}
proc ::contactmap::dataCanvasScrollY {args} {
  #synchronizes scollbar _and_ scale window to data movement
  variable w
  variable scaley
  variable fit_scaley
  variable ycanwindowmax
  variable ytopmargin
  variable ybottommargin
  variable ybox
  variable dataValANum
  variable yScaling
  puts "in dataCanvasScrollY, yScaling= $yScaling"
  if {$yScaling == 1} {return}

  #lock out access to this function
  set yScaling 1

  puts "set yScaling to $yScaling"
  set v [$w.can yview]
  set frac [expr [lindex $v 1] - [lindex $v 0] ] 
  #set frac [expr $start-$end ] 
  #asking for trouble, likely many runs till settles
  if {$dataValANum > 0} {
    set fit_scaley  [expr $frac * ((0.0 + $ycanwindowmax - $ytopmargin - $ybottommargin) / ($ybox * ($dataValANum + 1) ) ) ]
    puts "fit_scaley= $fit_scaley, scaley= $scaley"
    if {$fit_scaley != $scaley} {
      set scaley $fit_scaley
      redraw name func op 
    }
  }
  set yScaling 0
}


proc ::contactmap::lookupCode {resname} {
  variable codes

  set result ""
  if {[catch { set result $codes($resname) } ]} {
    set result $resname
  } else {
    set result " $result "
  }
  return $result
}

proc ::contactmap::stopZoomSeq {} {
  menu contactmap off
}

proc ::contactmap::chooseColor {field intensity} {
  variable dataName
  set field_color_type 4 
  #hack to sefault to struct field type coloring
  if {$dataName($field) != "struct"} {
    if {$intensity < 0} {set intensity 0}
    if {$intensity > 255} {set intensity 255}
    set intensity [expr int($intensity)]
    #set field_color_type $field 
    #check color mapping
    set field_color_type 3 
  }
  #super hacky here
  switch -exact $field_color_type {      
    #temporaily diable so greyscale color  only
    3333 {   
      set red $intensity
      set green [expr 255 - $intensity]
      set blue 150 
    }
    4 {
      #the field_color_type hack sends all structs to here 
      if {
          [catch {
            switch $intensity {
              
              B {set red 180; set green 180; set blue 0}
              C {set red 255; set green 255; set blue 255}
              E {set red 255; set green 255; set blue 100}
              T {set red 70; set green 150; set blue 150}
              G {set red 255; set green 160; set blue 255}
              H {set red 225; set green 130; set blue 225}
              I {set red 225; set green 20; set blue 20}
              default {set red 100; set green 100; set blue 100}
            }
          } ] 
        } else { #badly formatted file, intensity may be a number
          set red 0; set green 0; set blue 0 
        } 
    }
    default {
      #set red [expr 200 - int (200.0 * ($intensity / 255.0) )]
      #set green $red
      #set red 140; set blue 90; set green 90;
      set red $intensity
      set green $intensity
      set blue $intensity
    }
  }
  #convert red blue green 0 - 255 to hex
  set hexred     [format "%02x" $red]
  set hexgreen   [format "%02x" $green]
  set hexblue    [format "%02x" $blue]
  set hexcols [list $hexred $hexgreen $hexblue]

  return $hexcols
}


proc ::contactmap::redraw {name func op} { 
  
  variable x1 
  variable y1 
  variable so
  variable w 
  variable monoFont
  variable xcanwindowmax 
  variable ycanwindowmax 
  variable xcanmax 
  variable ycanmax
  variable ybox 
  variable xsize 
  variable ysize 
  variable resnamelist 
  variable structlist 
  variable betalist 
  variable sel 
  variable canvasnew 
  variable scalex 
  variable scaley 
  variable dataValA 
  variable dataValANum 
  variable dataName 
  variable dataNameLast 
  variable ytopmargin 
  variable ybottommargin 
  variable vertTextSkip   
  variable xcolbond_rad 
  variable bond_res 
  variable rep 
  variable xcol 
  variable vertTextRight
  variable vertHighLeft
  variable vertHighRight
  variable amino_code_toggle 
  variable dataWidth 
  variable dataMargin 
  variable firstData 
  variable dataMin 
  variable dataMax 
  variable xPosScaleVal
  variable usableMolLoaded
  variable rectCreated
  variable prevScalex
  variable prevScaley
  variable numHoriz
  
  if { ($usableMolLoaded) && ($dataValANum >=0 ) } {
    set ysize [expr $ytopmargin+ $ybottommargin + ($scaley *  $ybox * ($dataValANum + 1) )]  
    set xsize [expr  $xcol($firstData) +  ($scalex *  $dataWidth * (2 + $numHoriz) ) ]
    set ycanmax(data) $ysize
    set xcanmax(data) $xsize
    if {$ycanmax(data) < $ycanwindowmax} {
      set ycanmax(data) $ycanwindowmax
    }
    
    
    if {$xcanmax(data) < $xcanwindowmax} {
      set xcanmax(data) $xcanwindowmax
    }
    
    $w.can configure -scrollregion "0 0 $xcanmax(data) $ycanmax(data)"
    $w.vertScale configure -scrollregion "0 0 $xcanmax(vert) $ycanmax(data)"
    $w.horzScale configure -scrollregion "0 0 $xcanmax(data) $ycanmax(horz)"
    drawVertScale
    drawHorzScale
    set fieldLast  $dataNameLast
    #puts "fieldLast is $dataNameLast"
    #temporary, until put multi-cols in
    
    #draw data on can
    #loop over all data fields
    
    if {! $rectCreated} {
      #this until separate data and scale highlighting
      $w.vertScale delete yScalable
      $w.can delete dataScalable
      #puts "drawing rects, scalex is $scalex"
      #hack here -- for now skip B-field stuff, so minimal stuff drawn
      puts "setting min/max, firstData= $firstData" 
      for {set field  $firstData}  {$field <= $fieldLast} {incr field} {
        set xPosFieldLeft [expr int  ( $xcol($firstData) + ($scalex * $dataWidth * ($field - $firstData)  ) ) ]
        set xPosFieldRight [expr $xPosFieldLeft + ($scalex * $dataWidth)] 
        #now draw data rectangles
        #puts "drawing field $field at xPosField $xPosField" 
        set y 0.0
        set intensity 0
        for {set i 0} {$i<$dataValANum} {incr i} { 
          set val $dataValA($field,$i)
          if {$val != "null"} {
            #calculate color and create rectange
            set ypos [expr $ytopmargin + ($scaley * $y)]
            #should Prescan  to find range of values!  
            #this should be some per-request-method range / also allow this to be adjusted
            #set intensity except if field 4 (indexed struct)
            if {$dataName($field) != "struct"} {
              ##if { ( ($field != 4)  ) } open brace here 
              set range [expr $dataMax($field) - $dataMin($field)]
              if { ($range > 0)  && ([string is double $val] )} {
                set intensity  [expr int (255 * ( ($val - $dataMin($field) ) / $range)) ]
              }
              
              
              
              set hexcols [chooseColor $field $intensity]
            } else {
              #horrifyingly, sends string for data, tcl is typeless
              set hexcols [chooseColor $field $val ]
            }
            foreach {hexred hexgreen hexblue} $hexcols {} 
            
            
            #draw data rectangle
            $w.can create rectangle  [expr $xPosFieldLeft] [expr $ypos ] [expr $xPosFieldRight]  [expr $ypos + ($scaley * $ybox)]  -fill "\#${hexred}${hexgreen}${hexblue}" -outline "" -tags dataScalable
          }
          
          set y [expr $y + $ybox]
        }
      } 

      drawVertHighlight 
    }  else {

      #$w.can scale dataRect $xcol($firstdata) $ytopmargin 1 $scaley
      #$w.can scale dataScalable $xcol($firstData) [expr $ytopmargin] 1 [expr $scaley / $prevScaley ]

      $w.can scale dataScalable $xcol($firstData) [expr $ytopmargin] [expr $scalex / $prevScalex]  [expr $scaley / $prevScaley ]
      #now for datarect
      $w.vertScale scale yScalable 0 [expr $ytopmargin] 1  [expr $scaley / $prevScaley ]

    } 
    
    set rectCreated 1
    set prevScaley $scaley
    set prevScalex $scalex
  }

  return
}



proc ::contactmap::makecanvas {} {

  variable xcanmax 
  variable ycanmax
  variable w
  variable xsize
  variable ysize 
  variable xcanwindowmax 
  variable ycanwindowmax
  set xcanmax(data) $xsize 
  set ycanmax(data) $ysize
  
  
  #make main canvas

  canvas $w.spacer1 -width [expr $xcanmax(vert)+20] -height [expr $ycanmax(horz)+ 40] -bg #C0C0E0
  canvas $w.spacer2 -width [expr $xcanmax(vert)+20] -height [expr $ycanmax(horz)+40] -bg #C0C0E0
  canvas $w.can -width [expr $xcanwindowmax] -height $ycanwindowmax -bg #E9E9D9 -xscrollcommand "$w.xs set" -yscrollcommand "$w.ys set" -scrollregion  "0 0 $xcanmax(data) $ycanmax(data)" 
  #uncomment to enable auto-resizing
  #canvas $w.can -width [expr $xcanwindowmax] -height $ycanwindowmax -bg #E9E9D9 -xscrollcommand [namespace code dataCanvasScrollX] -yscrollcommand [namespace code dataCanvasScrollY]  -scrollregion  "0 0 $xcanmax(data) $ycanmax(data)" 
  canvas $w.vertScale -width $xcanmax(vert) -height $ycanwindowmax -bg #C0D0C0 -yscrollcommand "$w.ys set" -scrollregion "0 0 $xcanmax(vert) $ycanmax(data)" 

  canvas $w.horzScale -width $xcanwindowmax -height  $ycanmax(horz) -scrollregion "0 0 $xcanmax(data) $ycanmax(horz)" -bg #A9A9A9 -xscrollcommand "$w.xs set"
  #pack the horizontal (x) scrollbar
  pack $w.spacer1 -in $w.cfr -side left  -anchor e  
  pack $w.spacer2 -in $w.cfr -side bottom -anchor s  
  pack $w.can  -in $w.cfr -side left -anchor sw -expand yes -fill both
  #vertical scale/labels
  place $w.vertScale -in $w.can -relheight 1.0 -relx 0.0 -rely 0.5 -bordermode outside -anchor e
  #now place the vertical (y) scrollbar
  place $w.ys -in $w.vertScale -relheight 1.0 -relx 0.0 -rely 0.5 -bordermode outside -anchor e
  # horizontal scale/labels
  place $w.horzScale -in $w.can -relwidth 1.0 -relx 0.5 -rely 1.0 -bordermode outside -anchor n
  #now place the horizontal (x) scrollbar
  place $w.xs -in $w.horzScale -relwidth 1.0 -relx 0.5 -rely 1.0 -bordermode outside -anchor n
  # may need to specify B1-presses shift/nonshift separately...
  bind $w.can <ButtonPress-1>  [namespace code {getStartedMarquee %x %y 0 data}]
  bind $w.can <Shift-ButtonPress-1>  [namespace code {getStartedMarquee %x %y 1 data}]
  bind $w.can <ButtonPress-2>  [namespace code {showPair %x %y 0 data}]
  bind $w.can <B1-Motion>  [namespace code {keepMovingMarquee %x %y data}]
  bind $w.can <B2-Motion>  [namespace code {showPair %x %y 0 data}]
  bind $w.can <ButtonRelease-1> [namespace code {letGoMarquee %x %y data}]

  bind $w.vertScale <ButtonPress-1>  [namespace code {getStartedMarquee %x %y 0 vert}]
  bind $w.vertScale <Shift-ButtonPress-1>  [namespace code {getStartedMarquee %x %y 1 vert}]
  bind $w.vertScale <ButtonPress-2>  [namespace code {showPair%x %y 0 vert}]
  bind $w.vertScale <B1-Motion>  [namespace code {keepMovingMarquee %x %y vert}]
  bind $w.vertScale <B2-Motion>  [namespace code {showPair %x %y 0 vert}]
  bind $w.vertScale <ButtonRelease-1> [namespace code {letGoMarquee %x %y vert}]

  bind $w.horzScale <ButtonPress-1>  [namespace code {getStartedMarquee %x %y 0 horz}]
  bind $w.horzScale <Shift-ButtonPress-1>  [namespace code {getStartedMarquee %x %y 1 horz}]
  bind $w.horzScale <ButtonPress-2>  [namespace code {showPair %x %y 0 horz}]
  bind $w.horzScale <B1-Motion>  [namespace code {keepMovingMarquee %x %y horz}]
  bind $w.horzScale <F12><B2-Motion>  [namespace code {showPair %x %y 0 horz}]
  bind $w.horzScale <ButtonRelease-1> [namespace code {letGoMarquee %x %y horz}]
  
  lower $w.spacer1 $w.cfr
  lower $w.spacer2 $w.cfr

  
  return

  
} 


proc ::contactmap::reconfigureCanvas {} {
  variable xcanmax
  variable ycanmax
  variable w
  variable ysize 
  variable xcanwindowmax 
  variable ycanwindowmax
  variable xcanwindowStarting
  variable xcanwindowmax 
  variable firstData
  variable xcol

  #in future, add to xcanwindowstarting if we widen window
  set xcanwindowmax  $xcanwindowStarting 


  #check if can cause trouble if no mol loaded...
  $w.can configure  -height $ycanwindowmax -width $xcanwindowmax 
  $w.horzScale configure  -height $ycanmax(horz) -scrollregion "0 0 $xcanmax(data) $ycanmax(horz)"

  $w.vertScale configure  -width $xcanmax(vert) -scrollregion "0 0 $xcanmax(vert) $ycanmax(data)" 
  $w.horzScale delete all
  $w.vertScale delete all
  $w.can delete all

}

proc ::contactmap::draw_traj_highlight {xStart xFinish} {

  variable w 
  variable dataValA 
  variable dataValANum 
  variable xcol 
  variable ytopmargin 
  variable ytopmargin 
  variable scaley
  variable ybox  
  variable currentMolA 
  variable rep 
  variable bond_rad 
  variable bond_res
  variable rectCreated

  puts "now in draw_traj_highlight, xStart = $xStart, rectCreated = $rectCreated"
  $w.can delete trajHighlight 
  for {set i 0} {$i<=$dataValANum} {incr i} {
    if  {$dataValA(picked,$i) == 1} {
      set ypos [expr $ytopmargin+ ($scaley * $i *$ybox)]
      
      set red 0 
      set green 0 
      set blue 255 
      #convert red blue green 0 - 255 to hex
      set hexred     [format "%02x" $red]
      set hexgreen   [format "%02x" $green]
      set hexblue    [format "%02x" $blue]
      

      ###draw highlight only if not yet drawn -- if rectCreated is 0, we may just cleared the rects
      ###     to redraw free of accumulated scaling errors
      ###if {($dataValA(pickedId,$i) == "null") || ($rectCreated == 0)} 
      
      #always draw trajBox
      #after prototype, merge this with normal highlight draw method
      set trajBox [$w.can create rectangle  $xStart $ypos $xFinish [expr $ypos + ($scaley * $ybox)]  -fill "\#${hexred}${hexgreen}${hexblue}" -stipple gray25 -outline "" -tags [list dataScalable trajHighlight ] ]
      #puts "trajBox is $trajBox, xStart = $xStart, $xFinish = $xFinish"
      
      #$w.can lower $dataValA(pickedId,$i) vertScaleText 
      
      
      
    }
  }
}

proc ::contactmap::drawVertHighlight  {} {

  variable w 
  variable dataValA 
  variable dataValANum 
  variable xcol 
  variable ytopmargin 
  variable scaley
  variable ybox  
  variable currentMolA 
  variable rep 
  variable bond_rad 
  variable bond_res
  variable rectCreated
  variable vertHighLeft
  variable vertHighRight

  set red 255
  set green 0
  set blue 255
  #convert red blue green 0 - 255 to hex
  set hexred     [format "%02x" $red]
  set hexgreen   [format "%02x" $green]
  set hexblue    [format "%02x" $blue]
  set highlightColorString    "\#${hexred}${hexgreen}${hexblue}" 

  for {set i 0} {$i<=$dataValANum} {incr i} {
    if  {$dataValA(picked,$i) == 1} {
      set ypos [expr $ytopmargin+ ($scaley * $i *$ybox)]
      
      
      #draw highlight only if not yet drawn -- if rectCreated is 0, we may  just cleared the rects
      #     to redraw free of accumulated scaling errors
      if {($dataValA(pickedId,$i) == "null") || ($rectCreated == 0)} {

        set dataValA(pickedId,$i)  [$w.vertScale create rectangle  $vertHighLeft $ypos $vertHighRight [expr $ypos + ($scaley * $ybox)]  -fill $highlightColorString -outline "" -tags yScalable]
        
        
        $w.vertScale lower $dataValA(pickedId,$i) vertScaleText 
        
      }
      
    }
  }

  set ll [makeSelText dataValA  $dataValANum 1]

  
  if {($rep($currentMolA) != "null")} {

    if { [expr [molinfo $currentMolA get numreps] -1] >= $rep($currentMolA) } {

      mol modselect $rep($currentMolA) $currentMolA $ll
    } else {
      createHighlight  rep currentMolA  $ll 11  
    }
  } else {
    createHighlight  rep currentMolA $ll 11  
    #mol selection $ll
    #mol modstyle $rep($currentMolA)  $currentMolA Bonds $bond_rad $bond_res
    #mol color ColorID 11 
    #get info about this
  }
  return
}


proc ::contactmap::list_pick {name element op} {
  
  global vmd_pick_atom 
  global vmd_pick_mol 
  global vmd_pick_shift_state  

  variable w 
  variable xcanmax
  variable ycanmax
  variable xcanwindowmax 
  variable ycanwindowmax
  variable ybox
  variable ytopmargin 
  variable ybottommargin 
  variable vertTextSkip 
  variable scaley 
  variable dataValA 
  variable dataValANum 
  variable dataName 
  variable dataNameLast 
  variable bond_rad 
  variable bond_res 
  variable rep 
  variable xcol 
  variable ysize 
  variable currentMolA
  # get the coordinates



  #later deal with top (and rep)  etc. for multi-mol use


  
  if {$vmd_pick_mol == $currentMolA} {
    
    set sel [atomselect $currentMolA "index $vmd_pick_atom"]
    
    set pickedresid [lindex [$sel get {resid}] 0] 
    set pickedchain  [lindex [$sel get {chain}] 0] 
    set pickedresname [lindex  [$sel get {resname}] 0]
    
    
    set pickedOne -1
    for {set i 0} {$i <= $dataValANum} {incr i} {
      
      if {($dataValA(0,$i) == $pickedresid) && ($dataValA(1,$i) == $pickedresname) &&  ($dataValA(2,$i) == $pickedchain)} {
        set pickedOne $i
        
        break
      }
    }
    
    if {$pickedOne >= 0} {
      set ypos [expr $ytopmargin+ ($scaley * $i *$ybox)]
      
      #do bitwise AND to check for shift-key bit
      if {$vmd_pick_shift_state & 1} {
        set shiftPressed 1
      } else {
        set shiftPressed 0
      }
      

      
      if {$shiftPressed == 0 } {
        #delete all from canvas

        for {set i 0} {$i <= $dataValANum} {incr i} {
          set dataValA(picked,$i) 0
          if {$dataValA(pickedId,$i) != "null"} {
            $w.can delete $dataValA(pickedId,$i)
            set dataValA(pickedId,$i) "null"
          }
        }
      }
      
      
      set dataValA(picked,$pickedOne) 1
      
      drawVertHighlight 
      
      #scroll to picked
      set center [expr $ytopmargin + ($ybox * $scaley * $pickedOne) ] 
      set top [expr $center - 0.5 * $ycanwindowmax]
      
      if {$top < 0} {
        set top 0
      }
      set yfrac [expr $top / $ysize]
      $w.can yview moveto $yfrac
    }
    
  }
  return
}


proc ::contactmap::findResData {rPair cMol cMol_name dVal dValNum dHash selectionText} {            
  
  variable rep

  upvar $rPair repPair $cMol currentMol $cMol_name currentMol_name $dVal dataVal $dValNum dataValNum $dHash dataHash 
  set dataValNum -1 
  set currentMol_name [molinfo $currentMol get name]
  set sel [atomselect $currentMol "$selectionText and name CA"]
  #set sel [atomselect $currentMol "resid 1 to 15  and name CA"]
  #below assumes sel retrievals in same order each time, fix this 
  #by changing to one retreival and chopping up result
  set datalist  [$sel get {resid resname chain}]
  puts "Checking sequence info. for molecule $currentMol..."
  
  #clear this dataHash
  catch {unset dataHash}
  #the upvar'd dataHash is set again below, and reappears in the namespace.  
  foreach elem $datalist {
    
    #set picked state to false -- 'picked' is only non-numerical field
    incr dataValNum
    set dataVal(picked,$dataValNum) 0
    set dataVal(pickedId,$dataValNum) "null"
    set theResid [ lindex [split $elem] 0]
    set dataVal(0,$dataValNum) $theResid 
    
    set dataVal(1,$dataValNum) [ lindex [split $elem] 1]
    set dataVal(1code,$dataValNum) [lookupCode $dataVal(1,$dataValNum)]
    set theChain [ lindex [split $elem] 2]
    set dataVal(2,$dataValNum) $theChain 
    #for fast index searching later
    set dataHash($theResid,$theChain) $dataValNum
  }
  #if datalist is length 0 (empty), dataValNum is still -1, 
  #So we must check before each use of dataValNum    
  
  #set the molec. structure so nothing is highlighted yet
  set rep($currentMol) "null"
  set repPair($currentMol) "null"
  
  
  if {$dataValNum <= -1 } {
    #puts "Couldn't find a sequence in this molecule.\n"
    return
  }
  unset datalist
  puts "dataValNum = $dataValNum"
} 

proc ::contactmap::zoomSeqMain {} {
  #------------------------
  #------------------------
  # main code starts here
  #vars initialized a few lines down
  

  #puts "in zoomSeqMain.."
  variable w 
  variable monoFont
  variable eo 
  variable x1 
  variable y1 
  variable startShiftPressed 
  variable startCanvas
  variable vmd_pick_shift_state 
  variable amino_code_toggle 
  variable bond_rad 
  variable bond_res
  variable so 
  variable xcanwindowmax 
  variable ycanwindowmax 
  variable xcanmax 
  variable ycanmax
  variable ybox 
  variable ysize 
  variable resnamelist 
  variable structlist 
  variable betalist 
  variable sel 
  variable canvasnew 
  variable scaley 
  variable dataValA 
  variable dataValB 
  variable dataHashA
  variable dataHashB
  variable rectId
  #dataValANum is -1 if no data present, 
  variable dataValANum 
  variable dataValBNum 
  variable dataName 
  variable dataNameLast 
  variable ytopmargin 
  variable ybottommargin 
  variable xrightmargin
  variable vertTextSkip   
  variable xcolbond_rad 
  variable bond_res 
  variable rep 
  variable xcol 
  variable amino_code_toggle 
  variable dataWidth 
  variable dataMargin 
  variable firstData 
  variable dataMin 
  variable dataMax 
  variable xPosScaleVal
  variable currentMolA
  variable currentMolB
  variable fit_scalex
  variable fit_scaley
  variable usableMolLoaded 
  variable initializedVars
  variable prevScalet
  variable rectCreated
  variable windowShowing
  variable needsDataUpdate 
  variable numHoriz
  variable selTextA
  variable selTextB
  variable pairVal
  variable repPairA
  variable repPairB
  #if there are no mols at all,
  #there certainly aren't any non-graphics mols
  if {[molinfo num] ==0} {
    set usableMolLoaded 0
  }
  

  #Init vars and draw interface
  if {$initializedVars == 0} {
    initVars
    draw_interface
    makecanvas
    set initializedVars 1
    #watch the slider value, tells us when to redraw
    #this sets a trace for ::contactmap::scaley
    
  } else {
    #even if no molecule is present
    reconfigureCanvas
  }   
  
  
  #-----
  #Now load info from the current molecule, must reload for every molecule change
  
  if {$usableMolLoaded} {
    #get info for new mol
    #set needsDataUpdate 0

    set dataNameLast 2
    #The number of dataNames
    
    #lets fill  a (dataNameLast +1 ) x (dataValANum +1) array
    #dataValANum we'll be the number of objects we found with VMD search
    #if doing proteins, liekly all residues, found with 'name CA'
    
    set dataValANum -1
    set dataValBNum -1 
    #if no data is available, dataValANum will remain -1  
    

    
    # set  a new  trace below, only if dataValANum > -1  
    findResData repPairA currentMolA currentMolA_name dataValA dataValANum dataHashA $selTextA
    #puts "now for find b.., dataValBNum= $dataValBNum"
    findResData repPairB currentMolB currentMolB_name dataValB dataValBNum dataHashB $selTextB
    wm title $w "VMD Seq Compare: $currentMolA_name (mol $currentMolA)   $currentMolB_name (mol $currentMolB)"
    #So dataValANum is number of the last dataValA.  It is also #elements -1, 
    
    #numHoriz (and routines that use it)  will eventualy be changed
    # to reflect loaded data, and  multi-frame-data groups
    set numHoriz $dataValBNum 
    
    set fit_scalex [expr (0.0 + $xcanwindowmax - $xcol($firstData) ) / ($dataWidth * (2 + $numHoriz) ) ]
    set fit_scaley [expr (0.0 + $ycanwindowmax - $ytopmargin - $ybottommargin) / ($ybox * ($dataValANum + 1) ) ]
    #since we zero-count.

    set scaley 1.0
    set scalex $fit_scalex 
    puts "Restarting data, scalex = $scalex"
    #this trace only set if dataValANum != -1

    #Other variable-adding methods
    #should not change this number.  We trust $selA to always
    #give dataValANum elems, other methods might not work as well.
    #if need this data, recreate sel.  
    
    #handle if this value is 0 or -1
    
    
    
    
    
    #now lets fill in some data
    
    #new data, so need to redraw rects when time comes
    set rectCreated 0 
    #also set revScaley back to 1 
    set prevScaley scaley
    set prevScalex scalex 
    #fill in betalist (B-factors/temp factors called beta by VMD)
    #incr dataNameLast
    #set betalist [$selA get beta]
    #set dataName($dataNameLast) "B value"
    #jset dataMin($dataNameLast) 0.0
    #set dataMax($dataNameLast) 150.0
    #set i 0
    #foreach elem $betalist {
    #    set dataValA($dataNameLast,$i) $elem
    #    incr i
    #}
    
    ## Now there are 4 dataNames,   current
    ##value of dataNameNum is 3. last is numbered (dataNameLast) = 3
    #unset  betalist ;#done with it


    #fill in traj data with res-res position 
    
    #calcDataDist
    #puts "after initial calcDataDist, dataNameLast= $dataNameLast, dataValA(4,0)= $dataValA(4,0)  dataValA(8,8)= $dataValA(8,8)"
    

    
  }
  #get min/max of some data
  
  #puts "time for first redraw, scales, min/max not calced"
  #redraw first time
  redraw name func ops
  
  #now draw the scales (after the data, we may need to extract min/max 

  #------
  #draw color legends, loop over all data fields
  set fieldLast  $dataNameLast
  #puts "dataName(0) is $dataName(0) dataName(1) is $dataName(1)"
  
  #temporary, until put multi-cols in
  #for {set field $firstData} {$field <= $fieldLast} {incr field} 
  #hack so only display first 2 scales
  if {1==0} {
    puts "now to draw scales"
    for {set field $firstData} {$field <= 4} {incr field} {
      
      set xPosField [expr int ($xcol($firstData) + ($dataWidth * ($field - $firstData) ) )]
      puts "xPosField= $xPosField"
      #print the the title in center of data rectangle width
      $w.horzScale create text [expr int($xPosField + ( ($dataWidth -$dataMargin)/ 2.0) )] 1 -text "$dataName($field)" -width 200 -font $monoFont -justify center -anchor n 
      
      
      
      #make a scale across data rectange width

      set size [expr $dataWidth - $dataMargin]
      if {$dataName($field) != "struct"} {
        set minString [format "%.3g" $dataMin($field)]
        set maxString [format "%.3g" $dataMax($field)]
        $w.horzScale create text [expr $xPosField - 2  ] $xPosScaleVal -text $minString -width 50 -font $monoFont -justify center -anchor nw

        $w.horzScale create text [expr int ($xPosField - $dataMargin + $dataWidth +2 )] $xPosScaleVal -text $maxString -width 50 -font $monoFont -justify center -anchor ne
        
        set range [expr $dataMax($field) - $dataMin($field)]
        #bounds check, should really print error message
        if {$range <= 0} {
          puts "Bad range for field $dataName($field), min= $dataMin($field) max= $dataMax($field), range = $range"
          set $dataMin($field) -100
          set $dataMax($field) 100
          set $range [expr $dataMax($field) - $dataMin($field)]
          puts "Reset range for $dataName($field), new values: min= $dataMin($field) max= $dataMax($field), range = $range"
        }
        
        for {set yrect 0} {$yrect < $size} {incr yrect} {
          
          #draw linear scale
          set val [expr ( ( 0.0+ $yrect  )/ ($size -1)) * 255]
          #puts "val = $val , range = $range"
          set hexcols [chooseColor $field $val]         
          
          set hexred [lindex $hexcols 0]
          set hexgreen [lindex $hexcols 1]
          set hexblue [lindex $hexcols 2]
          

          $w.horzScale create rectangle [expr $xPosField + $yrect] 15 [expr $xPosField + $yrect] 30 -fill  "\#${hexred}${hexgreen}${hexblue}" -outline ""
        }
      } else {
        
        set prevNameIndex -1
        for {set yrect 0} {$yrect < $size} {incr yrect} {
          set names [list T E B H G I C "other"]
          
          set nameIndex [expr int ([expr [llength $names] -1]  * ($yrect+0.0)/$size)]
          set curName [lindex $names  $nameIndex]
          
          if {$nameIndex != $prevNameIndex} {
            #set line to black
            set hexred 0
            set hexgreen 0
            set hexblue 0
            
            #draw text
            $w.horzScale create text [expr int ($xPosField + $yrect+ 3)] $xPosScaleVal -text $curName -width 20 -font $monoFont -justify left -anchor nw
          } else {
            
            
            
            set hexcols [chooseColor $field $curName]
            
            set hexred [lindex $hexcols 0]
            set hexgreen [lindex $hexcols 1]
            set hexblue [lindex $hexcols 2]
          }

          $w.horzScale create rectangle [expr $xPosField + $yrect] 15 [expr $xPosField + $yrect] 30 -fill  "\#${hexred}${hexgreen}${hexblue}" -outline ""
          set prevNameIndex $nameIndex
        }
        set hexred 0
        set hexgreen 0
        set hexblue 0
        $w.horzScale create rectangle [expr $xPosField + $yrect] 15 [expr $xPosField + $size] 30 -fill  "\#${hexred}${hexgreen}${hexblue}" -outline ""
      }
    }  
    #done with color legends
    #-------
    
  } 
  

  return
}


proc ::contactmap::molChooseMenu {name function op} {
  variable w

  variable usableMolLoaded
  variable currentMolA
  variable currentMolB
  variable prevMol
  variable nullMolString
  $w.molA.menu delete 0 end
  $w.molB.menu delete 0 end



  set molList ""
  foreach mm [molinfo list] {
    if {[molinfo $mm get filetype] != "graphics"} {
      lappend molList $mm
      #add a radiobutton, but control via commands, not trace,
      #since if this used a trace, the trace's callback
      #would delete that trace var, causing app to crash.
      #variable and value only for easy button lighting
      #$w.molA.menu add radiobutton -variable [namespace current]::currentMolA -value $mm -label "$mm [molinfo $mm get name]" -command [namespace code "molChoose name function op"]
      $w.molA.menu add radiobutton -variable [namespace current]::currentMolA -value $mm -label "$mm [molinfo $mm get name]"
      $w.molB.menu add radiobutton -variable [namespace current]::currentMolB -value $mm -label "$mm [molinfo $mm get name]"
    }
  }

  #set if any non-Graphics molecule is loaded
  if {$molList == ""} {
    set usableMolLoaded  0
    if {$prevMol != $nullMolString} {
      set currentMolA $nullMolString
      set currentMolB $nullMolString
    }
  } else {

    #deal with first (or from-no mol state) mol load
    # and, deal with deletion of currentMolA, if mols present
    # by setting the current mol to whatever top is
    if {($usableMolLoaded == 0) || [lsearch -exact $molList $currentMolA]== -1 } {
      set usableMolLoaded 1
      set currentMolA [molinfo top]
      set currentMolB [molinfo top]
    }

  }


  
  
  return
}

proc ::contactmap::setScaling {} {
  variable w
  variable trajMin
  variable trajMax 
  #extract first part of molecule name to print here?
  puts "Enter bottom value of scale..."
  set trajMin  [gets stdin]
  puts "Enter top value of scale..."
  set trajMax [gets stdin]
  puts "trajMin = $trajMin, trajMax= $trajMax" 
  return
}

proc ::contactmap::printCanvas {} {
  variable w
  #extract first part of molecule name to print here?
  set filename "VMD_Contactmap_Window.ps"
  set filename [tk_getSaveFile -initialfile $filename -title "VMD Contactmap Print" -parent $w -filetypes [list {{Postscript Files} {.ps}} {{All files} {*} }] ]
  if {$filename != ""} {
    $w.can postscript -file $filename
  }
  
  return
}





proc ::contactmap::getStartedMarquee {x y shiftState whichCanvas} {

  variable w 
  variable x1 
  variable y1 
  variable so
  variable str 
  variable eo 
  variable g 
  variable startCanvas 
  variable startShiftPressed
  variable xcanmax
  variable ycanmax
  variable usableMolLoaded

  
  
  if {$usableMolLoaded} {

    #calculate offset for canvas scroll
    set startShiftPressed $shiftState  
    set startCanvas $whichCanvas 
    #get actual name of canvas
    switch -exact $startCanvas {
      data {set drawCan can}
      vert {set drawCan vertScale}
      horz {set drawCan horzScale}
      default {puts "problem with finding canvas..., startCanvas= >$startCanvas<"} 
    }   
    set x [expr $x + $xcanmax($startCanvas) * [lindex [$w.$drawCan xview] 0]] 
    set y [expr $y + $ycanmax($startCanvas) * [lindex [$w.$drawCan yview] 0]] 
    
    set x1 $x
    set y1 $y
    

    puts "getStartedMarquee x= $x  y= $y, startCanvas= $startCanvas" 
    #Might have other canvas tools in future..   
    # Otherwise, start drawing rectangle for selection marquee
    
    set so [$w.$drawCan create rectangle $x $y $x $y -fill {} -outline red]
    set eo $so
  } 
  return
}


proc ::contactmap::molChoose {name function op} {

  variable scaley
  variable w
  variable currentMolA
  variable prevMol
  variable nullMolString
  variable rep 
  variable usableMolLoaded
  variable needsDataUpdate
  variable windowShowing
  


  #this does complete restart
  #can do this more gently...
  
  #trace vdelete scaley w [namespace code redraw]
  #trace vdelete ::vmd_pick_atom w  [namespace code list_pick] 
  
  #if there's a mol loaded, and there was an actual non-graphic mol last
  #time, and if there has been a selection, and thus a struct highlight
  #rep made, delete the highlight rep.
  if {($usableMolLoaded)  && ($prevMol != $nullMolString) && ($rep($prevMol) != "null")} {



    #catch this since currently is exposed to user, so 
    #switching/reselecting  molecules can fix problems.
    ##puts "About to delete rep=$rep($prevMol) for prevMol= $prevMol"
    #determine if this mol exists...
    if  {[lsearch -exact [molinfo list] $prevMol] != -1}  {
      #determine if this rep exists (may have been deleted by user)
      if { [expr [molinfo $prevMol get numreps] -1] >= $rep($prevMol) } { 
        
        mol delrep $rep($prevMol) $prevMol 
      }
    }
    
  }

  set prevMol $currentMolA

  #can get here when window is not displayed if:
  #   molecule is loaded, other molecule delete via Molecule GUI form.
  # So, we'll only redraw (and possible make a length (wallclock) call
  # to STRIDE) if sequence window is showing
  
  set needsDataUpdate 1


  if {$windowShowing} {
    set needsDataUpdate 0
    #set this immediately, so other  calls can see this
    
    [namespace current]::zoomSeqMain
  }


  
  #reload/redraw stuff, settings (this may elim. need for above lines...)
  
  
  #change molecule choice and redraw if needed (visible && change) here...
  #change title of window as well
  ##wm title $w "VMD Contactmap  $currentMolA_name (mol $currentMolA) "
  
  #reload sutff (this may elim. need for above lines...)

  return
}
proc ::contactmap::keepMovingMarquee {x y whichCanvas} {

  variable x1 
  variable y1 
  variable so 
  variable w 
  variable xcanmax 
  variable ycanmax
  variable startCanvas
  variable usableMolLoaded
  #get actual name of canbas
  switch -exact $startCanvas {
    data {set drawCan can}
    vert {set drawCan vertScale}
    horz {set drawCan horzScale}
    default {puts "problem with finding canvas (moving marquee)..., startCanvas= $startCanvas"}
  } 

  
  if {$usableMolLoaded} {

    #next two lines for debeugging only
    set windowx $x
    set windowy $y 
    #calculate offset for canvas scroll
    set x [expr $x + $xcanmax($startCanvas) * [lindex [$w.$drawCan xview] 0]] 
    set y [expr $y + $ycanmax($startCanvas) * [lindex [$w.$drawCan yview] 0]] 
    
    
    
    
    $w.$drawCan coords $so $x1 $y1 $x $y
  }
  return
}

proc ::contactmap::letGoMarquee {x y whichCanvas} {


  variable x1 
  variable y1 
  variable startShiftPressed 
  variable startCanvas
  variable so 
  variable eo 
  variable w 
  variable xcanmax
  variable ycanmax
  variable ySelStart 
  variable ySelFinish 
  variable ybox 
  variable ytopmargin 
  variable ybottommargin 
  variable vertTextSkip 
  variable scalex 
  variable scaley 
  variable dataValA 
  variable dataValANum 
  variable dataName 
  variable dataNameLast 
  variable bond_rad 
  variable bond_res 
  variable rep 
  variable xcol
  variable currentMolA
  variable usableMolLoaded
  variable firstData
  variable dataWidth 
  variable ycanwindowmax  
  variable numHoriz
  variable firstStructField
  #set actual name of canvas
  switch -exact $startCanvas {
    data {set drawCan can}
    vert {set drawCan vertScale}
    horz {set drawCan horzScale}
    default {puts "problem with finding canvas (moving marquee)..., startCanvas= $startCanvas"}
  }

  if {$usableMolLoaded} {
    #calculate offset for canvas scroll
    set x [expr $x + $xcanmax(data) * [lindex [$w.$drawCan xview] 0]] 
    set y [expr $y + $ycanmax(data) * [lindex [$w.$drawCan yview] 0]] 

    #compute the frame at xSelStart
    if {$x1 < $x} {
      set xSelStart $x1
      set xSelFinish $x
    }  else {
      set xSelStart $x
      set xSelFinish $x1
    }
    puts "xSelStart is $xSelStart xSelFinish is $xSelStart" 
    
    #in initVars we hardcode firstStructField to be 4
    #later, there may be many field-groups that can be stretched 
    set xPosStructStart [expr int ($xcol($firstData) + ($dataWidth * ($firstStructField - $firstData) ) )] 

    set selStartHoriz [expr  int (($xSelStart - $xcol($firstData))/ ($dataWidth * $scalex)) - 1 ]
    set selFinishHoriz [expr int( ($xSelFinish - $xcol($firstData))/ ($dataWidth * $scalex) ) - 1]
    if { $selFinishHoriz> $numHoriz} {
      set selFinishHoriz $numHoriz
    }
    puts "selected frames $selStartHoriz to   $selFinishHoriz"

    if {$y1 < $y} {
      set ySelStart $y1
      set ySelFinish $y}  else {
        
        set ySelStart $y
        set ySelFinish $y1
      }
    
    set startObject [expr 0.0 + ((0.0 + $ySelStart - $ytopmargin) / ($scaley * $ybox))]
    set finishObject [expr 0.0 + ((0.0 + $ySelFinish - $ytopmargin) / ($scaley * $ybox))]
    
    
    
    if {$startShiftPressed == 1} {
      set singleSel 0
    } else {
      set singleSel 1
    }
    
    if {$startObject < 0} {set startObject 0}
    if {$finishObject < 0} {set finishObject 0}
    if {$startObject > $dataValANum} {set startObject   $dataValANum }
    if {$finishObject > $dataValANum} {set finishObject $dataValANum }
    set startObject [expr int($startObject)]
    set finishObject [expr int($finishObject)]
    
    
    
    #clear all if click/click-drag, don't clear if shift-click, shift-click-drag
    
    if {$singleSel == 1} {
      
      for {set i 0} {$i <= $dataValANum} {incr i} {
        set dataValA(picked,$i) 0
        if {$dataValA(pickedId,$i) != "null"} {
          
          $w.vertScale delete $dataValA(pickedId,$i)
          set dataValA(pickedId,$i) "null"
        }
      }

    } else {
      
      #just leave alone  
    }
    
    
    
    
    #set flags for selection
    for {set i $startObject} {$i <= $finishObject} {incr i} {
      set dataValA(picked,$i) 1
    }
    
    
    
    set field 0
    #note that the column will be 0, but the data will be from picked
    
    drawVertHighlight 
    
    
    puts "now to delete outline, eo= $eo" 
    $w.$drawCan delete $eo
    #now that highlight changed, can animate
    #if single selection in frame area, animate, then jump to that frame
    

  }
  return
}

proc ::contactmap::showall { do_redraw} {



  variable scalex 
  variable scaley 
  variable fit_scalex
  variable fit_scaley
  variable usableMolLoaded
  variable rectCreated 
  variable userScalex
  variable userScaley

  #only redraw once...
  if {$usableMolLoaded} {
    if {$do_redraw == 1} {
      set rectCreated 0
    }  
    
    set scalex $fit_scalex        
    set scaley $fit_scaley
    set userScalex 1.0
    set userScaley $fit_scaley 

    redraw name func ops
  }

  return
}


proc ::contactmap::every_res {} {

  variable usableMolLoaded
  variable rectCreated
  variable fit_scalex
  #this forces redraw, to cure any scaling floating point errors
  #that have crept in 
  set rectCreated 0

  variable scaley
  variable scalex

  if {$usableMolLoaded} {
    #redraw, set x and y  at once
    set scalex $fit_scalex 
    set userScalex 1.000 
    set scaley 1.0
    redraw name func ops
  }
  
  return
}


proc ::contactmap::resname_toggle {} {

  variable w 
  variable amino_code_toggle
  variable usableMolLoaded
  
  if {$usableMolLoaded} {


    if {$amino_code_toggle == 0} {
      set amino_code_toggle 1
      $w.resname_toggle configure -text "3-letter code"
    } else {
      set amino_code_toggle 0
      $w.resname_toggle configure -text "1-letter code"
    }
    
    redraw name function op
  }
  return
}




proc ::contactmap::initVars {} {        

  variable usableMolLoaded 0
  variable windowShowing 0
  variable needsDataUpdate 0
  variable dataNameLast 2
  variable dataValANum -1
  variable eo 0
  variable x1 0 
  variable y1 0
  variable startCanvas ""
  variable startShiftPressed 0
  variable vmd_pick_shift_state 0
  variable amino_code_toggle 0
  variable bond_rad 0.5
  variable bond_res 10
  variable so ""
  variable nullMolString ""
  variable currentMolA $nullMolString
  variable currentMolB $nullMolString
  variable prevMol $nullMolString

  variable  userScalex 1
  variable  userScaley 1
  variable  scalex 1
  variable  scaley 1
  variable prevScalex 1
  variable prevScaley 1
  
  variable ytopmargin 5
  variable ybottommargin 10
  variable xrightmargin 8

  #variable xcanwindowStarting 780 
  variable xcanwindowStarting 685 
  variable ycanwindowStarting 574 

  variable numHoriz 1
  variable xcanwindowmax  $xcanwindowStarting
  variable ycanwindowmax $ycanwindowStarting 
  variable xcanmax
  set xcanmax(data) 610
  set xcanmax(vert) 95
  set xcanmax(horz) $xcanmax(data)
  #make this sensible!
  variable ycanmax
  set ycanmax(data) 400
  set ycanmax(vert) $ycanmax(data) 
  set ycanmax(horz) 46 
  variable codes
  variable trajMin -180
  variable trajMax 180
  variable selTextA "all"
  variable selTextB "all"
  variable pairTextA ""
  variable pairText ""
  variable xScaling 0 
  variable yScaling 0 
  #hard coded, should change
  variable firstStructField 4
  variable alignIndex
  set alignIndex(num) 0

  array set codes {ALA A ARG R ASN N ASP D ASX B CYS C GLN Q GLU E
    GLX Z GLY G HIS H ILE I LEU L LYS K MET M PHE F PRO P SER S
    THR T TRP W TYR Y VAL V}
  
  
  

  #tests if rects for current mol have been created (should extend 
  #so memorize all rectIds in 3dim array, and track num mols-long 
  #vector of rectCreated. Would hide rects for non-disped molec,
  #and remember to delete the data when molec deleted.
  
  variable rectCreated 0

  #the box height
  variable ybox 15.0
  #text skip doesn't need to be same as ybox (e.g. if bigger numbers than boxes in 1.0 scale)
  variable vertTextSkip $ybox

  
  # For vertical scale appearance
  variable vertHighLeft 2
  variable vertHighRight 100
  variable vertTextRight 96
  #The first 3 fields, 0 to 2 are printed all together, they are text
  variable xcol
  set xcol(0) 10.0

  variable dataWidth 85
  variable dataMargin 0
  variable xPosScaleVal 32
  #so rectangge of data is drawn at width $dataWidth - $dataMargin (horizontal measures)
  #
  #residu name data is in umbered entires lowerthan 3
  variable firstData 3
  #puts "firstData is $firstData"
  #column that multi-col data first  appears in

  #old setting from when vertscale and data were on same canvas
  #set xcol($firstData)  96 
  set xcol($firstData)  1 
  #The 4th field (field 3) is the "first data field"
  #we use same data structure for labels and data, but now draw in separate canvases 
  
  # the names for  three fields of data 
  
  #just for self-doc
  # dataValA(picked,n) set if the elem is picked
  # dataValA(pickedId,n) contains the canvas Id of the elem's highlight rectangle
  

  variable dataName

  set dataName(picked) "picked" 
  set dataName(pickedId) "pickedId"
  #not included in count of # datanames
  
  set dataName(0) "resid"
  set dataName(1) "resname"
  set dataName(1code) "res-code"
  set dataName(2) "chain"
  ###set dataName(3) "check error.." 
  
  
}


proc ::contactmap::Show {} {
  variable windowShowing
  variable needsDataUpdate
  set windowShowing 1

  
  if {$needsDataUpdate} {
    set needsDataUpdate 0
    #set immmediately, so other binding callbacks will see
    [namespace current]::zoomSeqMain
  }

}

proc ::contactmap::Hide {} {
  variable windowShowing 
  set windowShowing 0

}



proc ::contactmap::createHighlight { theRep theCurrentMol seltext col} {
  upvar $theRep rep $theCurrentMol currentMol 
  variable bond_rad
  variable bond_res
  #draw first selection, as first residue 
  set rep($currentMol) [molinfo $currentMol get numreps]
  mol selection $seltext
  mol material Opaque
  mol color ColorID $col 
  mol addrep $currentMol
  mol modstyle $rep($currentMol)  $currentMol  Bonds $bond_rad $bond_res
}



proc ::contactmap::draw_interface {} {
  variable w 

  variable eo 
  variable x1  
  variable y1 
  variable startCanvas
  variable startShiftPressed 
  variable vmd_pick_shift_state 
  variable amino_code_toggle 
  variable bond_rad 
  variable bond_res
  variable so 
  variable xcanwindowmax 
  variable ycanwindowmax 
  variable xcanmax 
  variable ycanmax
  variable ybox 
  variable xsize 
  variable ysize 
  variable resnamelist 
  variable structlist 
  variable betalist 
  variable sel 
  variable canvasnew 
  variable userScalex
  variable userScaley
  variable scalex 
  variable scaley 
  variable dataValA 
  variable dataValANum 
  variable dataName 
  variable dataNameLast 
  variable ytopmargin 
  variable ybottommargin 
  variable vertTextSkip   
  variable xcolbond_rad 
  variable bond_res 
  variable rep 
  variable repPairA
  variable repPairB
  variable xcol 
  variable amino_code_toggle 
  variable dataWidth 
  variable dataMargin 
  variable firstData 
  variable dataMin 
  variable dataMax 
  variable xPosScaleVal
  variable currentMolA
  variable fit_scalex
  variable fit_scaley
  variable usableMolLoaded 
  variable numHoriz 
  variable selTextA
  variable selTextB
  variable pairTextA
  variable pairTextB
  variable pairVal

  frame $w.menubar -height 30 -relief raised -bd 2
  pack $w.menubar -in $w -side top -anchor nw -padx 1 -fill x

  #frame $w.fr -width 700 -height 810 -bg #FFFFFF -bd 2 ;#main frame

  #pack $w.fr

  label $w.txtlab -text "Zoom "

  frame $w.panl -width 150 -height [expr $ycanwindowmax + 80] -bg #C0C0D0 -relief raised -bd 1 
  frame $w.cfr -width 350 -height [expr $ycanwindowmax + 85] -borderwidth 1  -bg #D96060 -relief raised -bd 3
  pack $w.panl -in $w -side left -padx 2  -fill y
  #pack $w.cfr -in $w.fr -side left -padx 2 -expand yes -fill both 
  pack $w.cfr -in $w -side left -padx 2 -expand yes -fill both 

  scale $w.panl.zoomlevel -from 0.01 -to 2.01 -length 150 -sliderlength 30  -resolution 0.01 -tickinterval 0.5 -repeatinterval 30 -showvalue true -variable [namespace current]::userScaley -command [namespace code userScaleyChanged] 

  scale $w.zoomBothlevel -orient horizontal -from 0.001 -to 4.000 -length 120 -sliderlength 30  -resolution 0.001 -tickinterval 3.999 -repeatinterval 30 -showvalue true -variable [namespace current]::userScaleBoth -command [namespace code userScaleBothChanged] 
  scale $w.zoomXlevel -orient horizontal -from 0.001 -to 40.000 -length 120 -sliderlength 30  -resolution 0.001 -tickinterval 39.999 -repeatinterval 30 -showvalue true -variable [namespace current]::userScalex -command [namespace code userScalexChanged] 

  #pack $w.panl $w.cfr -in $w.fr -side left -padx 2

  button $w.showall  -text "fit all" -command [namespace code {showall 0}]
  button $w.every_res  -text "every residue" -command [namespace code every_res]
  button $w.resname_toggle  -text "1-letter code" -command [namespace code resname_toggle]

  #draw canvas
  
  #trace for molecule choosing popup menu 
  trace variable ::vmd_initialize_structure w  [namespace code molChooseMenu]
  
  menubutton $w.molA -relief raised -bd 2 -textvariable [namespace current]::currentMolA -direction flush -menu $w.molA.menu
  menubutton $w.molB -relief raised -bd 2 -textvariable [namespace current]::currentMolB -direction flush -menu $w.molB.menu
  menu $w.molA.menu -tearoff no
  menu $w.molB.menu -tearoff no


  molChooseMenu name function op
  

  entry $w.molAPairtext -text "" -textvariable [namespace current]::pairTextA
  entry $w.molBPairtext -text "" -textvariable [namespace current]::pairTextB
  entry $w.molASeltext -text "all" -textvariable [namespace current]::selTextA
  entry $w.molBSeltext -text "all" -textvariable [namespace current]::selTextB
  entry $w.pairval -text "" -textvariable [namespace current]::pairVal
  label $w.molLabA -text "MoleculeA:"
  label $w.molLabB -text "MoleculeB:"

  scrollbar $w.ys -command [namespace code {canvasScrollY}]
  
  scrollbar $w.xs -orient horizontal -command [namespace code {canvasScrollX}]



  #fill the  top menu
  menubutton $w.menubar.file -text File -underline 0 -menu $w.menubar.file.menu
  menubutton $w.menubar.calculate -text Calculate -underline 0 -menu $w.menubar.calculate.menu

  pack $w.menubar.file  $w.menubar.calculate -side left


  menubutton $w.menubar.help -text Help -underline 0 -menu $w.menubar.help.menu
  menu $w.menubar.help.menu -tearoff no

  $w.menubar.help.menu add command -label "Contactmap Help" -command "vmd_open_url [string trimright [vmdinfo www] /]/plugins/contactmap"
  $w.menubar.help.menu add command -label "Structure codes..." -command  [namespace code {tk_messageBox -parent $w  -type ok -message "Secondary Structure Codes\n\nT        Turn\nE        Extended conformation\nB        Isolated bridge\nH        Alpha helix\nG        3-10 helix\nI         Pi-helix\nC        Coil (none of the above)\n" } ]

  pack $w.menubar.help -side right 
  
  #File menu
  menu $w.menubar.file.menu -tearoff no
  $w.menubar.file.menu add command -label "Print to file..." -command [namespace code {printCanvas} ] 
  $w.menubar.file.menu add command -label "Load data file..." -command [namespace code {loadDataFile ""}  ] 
  $w.menubar.file.menu add command -label "Write data file..." -command [namespace code {writeDataFile ""}  ] 
#  $w.menubar.file.menu add command -label "Close Window" -command [namespace code stopZoomSeq] 
  
  #Calculate menu
  
  menu $w.menubar.calculate.menu  -tearoff no

  
  $w.menubar.calculate.menu add command -label "Clear data"  -command  [namespace code clearData] 
  $w.menubar.calculate.menu add command -label "Calc. res-res Dists"  -command [namespace code {calcDataDist; showall 1}] 
  
  
  #Second level menu for highlightColor 

  

  
  #the w.can object made here
  set ysize [expr $ytopmargin+ $ybottommargin + ($scaley *  $ybox * ($dataValANum + 1))]    
  set xsize [expr  $xcol($firstData) +  ($scalex *  $dataWidth * ( $numHoriz) ) ]



  place  $w.zoomXlevel -in $w.panl  -bordermode inside -rely 0.0 -y 30  -relx 0.5 -anchor n
  place  $w.zoomBothlevel -in $w.panl -bordermode inside -rely 0.0 -y 70 -relx 0.5 -anchor n 

  
  place $w.molLabA -in $w.zoomBothlevel  -bordermode outside -rely 1.3 -relx 0.4 -anchor n
  place $w.molASeltext -in $w.molLabA -width 120 -bordermode outside -rely 1.0 -relx 0.5 -anchor n
  place $w.molAPairtext -in $w.molASeltext -width 90 -bordermode outside -rely 1.0 -relx 0.5 -anchor n
  place $w.molLabB -in $w.molAPairtext -bordermode outside -rely 1.2 -relx .5 -anchor n
  place $w.molBSeltext -in $w.molLabB -width 120 -bordermode outside -rely 1.0 -relx 0.5 -anchor n
  place $w.molBPairtext -in $w.molBSeltext -width 90 -bordermode outside -rely 1.0 -relx 0.5 -anchor n
  place $w.pairval -in $w.molBPairtext -width 70 -bordermode outside -rely 1.3 -relx 0.5 -anchor n
  place $w.molA -in $w.molLabA -bordermode outside -rely 0.5 -relx 1.0 -anchor w
  place $w.molB -in $w.molLabB -bordermode outside -rely 0.5 -relx 1.0 -anchor w
  #place $w.panl.zoomlevel -in $w.molBSeltext -bordermode outside -rely 1.8 -relx .5 -anchor n 
  place $w.panl.zoomlevel -in $w.panl -rely 0.5 -relx .5 -anchor n 

  place $w.txtlab -in $w.panl.zoomlevel  -bordermode outside -rely 0.0 -relx 0.5 -anchor s
  place $w.showall -in $w.panl.zoomlevel  -bordermode outside -rely 1.0 -relx 0.5 -anchor n
  place $w.every_res  -in $w.showall -bordermode outside -rely 1.0 -relx 0.5 -anchor n
  place $w.resname_toggle -in $w.every_res -bordermode outside -rely 1.0 -y 40 -relx 0.5 -anchor n

  #done with interface elements     

  #ask window manager for size of window

  #turn traces  on (initialize_struct trace comes later)
  #trace variable userScalex w  [namespace code redraw]
  #trace variable userScaley w  [namespace code redraw]
  trace variable ::vmd_pick_atom w [namespace code list_pick]
  trace variable currentMolA w [namespace code molChoose]

}

proc  ::contactmap::showPair {x y shiftState whichCanvas} {
  variable xcol
  variable firstData
  variable dataWidth
  variable scalex
  variable scaley
  variable dataValA
  variable dataValANum
  variable dataValB
  variable dataValBNum
  variable currentMolA
  variable currentMolB
  variable numHoriz
  variable xcanmax
  variable ycanmax
  variable ytopmargin
  variable ybox
  variable repPairA
  variable repPairB
  variable w
  variable pairTextA
  variable pairTextB
  variable pairVal
  #should use same code as marquee checks!
  #maybe store top, and restore it afterwards
  set x [expr $x + $xcanmax(data) * [lindex [$w.can xview] 0] ]
  set y [expr $y + $ycanmax(data) * [lindex [$w.can yview] 0] ]
  set cursorXnum [expr  int (($x - $xcol($firstData))/ ($dataWidth * $scalex)) - 1 ]
  set cursorYnum [expr int (0.0 + ((0.0 + $y - $ytopmargin) / ($scaley * $ybox)))]
  
  if {$cursorXnum> $numHoriz}  {
    set cursorXnum $numHoriz
  }
  
  if {$cursorXnum< 0 } { 
    set cursorXnum 0
  } 
  
  if {$cursorYnum>$dataValANum} {
    set cursorYnum $dataValANum
  }
  if {$cursorYnum < 0} {
    set cursorYnum 0 
  }

  set residA $dataValA(0,$cursorYnum)
  set chainA $dataValA(2,$cursorYnum)
  set residB $dataValB(0,$cursorXnum)
  set chainB $dataValB(2,$cursorXnum)
  if {[catch {set val  $dataValA([expr $firstData+$cursorXnum],$cursorYnum)}] } { 
    set pairVal "" 
  } else {
    set pairVal $val
  }
  set seltextA "resid $residA and chain $chainA"
  set pairTextA "r $residA  ch $chainA"
  set seltextB "resid $residB and chain $chainB"
  set pairTextB "r $residB  ch $chainB"
  
  #puts "$cursorXnum $cursorYnum chainA= $chainA residA= $residA   chainB= $chainB residB= $residB" 

  #highlight resisdues in the graph

  #show the two involved residues in GL window
  if {($repPairA($currentMolA) != "null")} {

    if { [expr [molinfo $currentMolA get numreps] -1] >= $repPairA($currentMolA) } {

      mol modselect $repPairA($currentMolA) $currentMolA $seltextA
    } else {
      createHighlight  repPairA currentMolA $seltextA 7 
    }
  } else {
    createHighlight repPairA currentMolA $seltextA 7 
  }
  
  if {($repPairB($currentMolB) != "null")} {

    if { [expr [molinfo $currentMolB get numreps] -1] >= $repPairB($currentMolB) } {

      mol modselect $repPairB($currentMolB) $currentMolB $seltextB
    } else {
      createHighlight  repPairB currentMolB $seltextB 5 
    }
  } else {
    createHighlight repPairB currentMolB $seltextB 5 
  }
  
  return
  
  
  
}


proc ::contactmap::writeDataFile {filename} {
  variable w
  variable dataName
  variable dataValA
  variable dataValANum
  variable currentMolA
  variable firstStructField
  variable numHoriz

  if {$filename == ""  } {
    set filename [tk_getSaveFile -initialfile $filename -title "Save Trajectory Da ta file" -parent $w -filetypes [list { {.dat files} {.dat} } { {Text files} {.txt}} {{All files} {*} }] ]
    set writeDataFile [open $filename w]
    puts $writeDataFile "# VMD sequence data"
    puts $writeDataFile "# CREATOR= $::tcl_platform(user)"
    puts $writeDataFile "# DATE= [clock format [clock seconds]]"
    puts $writeDataFile "# TITLE= [molinfo $currentMolA get name]"
    puts $writeDataFile "# NUM_FRAMES= $numHoriz "
    puts $writeDataFile "# FIELD= $dataName($firstStructField) "
    set endStructs [expr $firstStructField + $numHoriz]
    for {set field $firstStructField} {$field <= $endStructs} {incr field} {
      for {set i 0} {$i<=$dataValANum} {incr i} {

        set val $dataValA($field,$i)
        set resid $dataValA(0,$i)
        set chain $dataValA(2,$i)
        set frame [expr $field - $firstStructField]
        puts $writeDataFile "$resid $chain CA $frame $val"
      }
    }
    close $writeDataFile
  }
  return
}


proc ::contactmap::getValsFromAlign {dv r ch co dataset alignPos} {
  variable alignIndex
  upvar $dv dataVal $r res $ch chain $co code 
  set index $alignIndex($dataset,$alignPos)
  if {$index  == "+" }  {
    set res "+++"
    set chain "++"
    set code "+"
  } else {
    set res $dataVal(0,$index)
    set chain $dataVal(2,$index)
    set code [string index $dataVal(1code,$index) 1]
  }
}

proc ::contactmap::checkPairs {} {
  #just a debugging tool
  variable dataValA
  variable dataHashA
  variable dataValANum
  variable dataValB
  variable dataHashB
  variable dataValBNum
  variable alignIndex
  set chainA ""; set resA ""; set codeA ""
  set chainB ""; set resB ""; set codeB ""
  if {1==0} {
    for {set i 0} {$i < $alignIndex(num)} {incr i} {
      catch {getValsFromAlign dataValA resA chainA codeA 0 $i } 
      catch {getValsFromAlign dataValB resB chainB codeB 1 $i  }
      puts "A: $resA $chainA $codeA B: $resB $chainB $codeB"
    }
  }
  set width 40 
  set spacing 10
  set pos 0
  while {$pos < $alignIndex(num) } {
    set align0 "0:"
    set align1 "1:"
    catch {getValsFromAlign dataValA resA chainA codeA 0 [expr  $pos] } 
    catch {getValsFromAlign dataValB resB chainB codeB 1 [expr  $pos] }
    puts "align: $pos to [expr $pos + $width - 1]    A: $resA $codeA $chainA   B: $resB $codeB $chainB"
    for {set i 0 } {$i < $width} {incr i} {
      #for debugging, when catching getVals errors
      set codeA "*"; set codeB "*" 
      catch { getValsFromAlign dataValA resA chainA codeA 0 [expr $i+ $pos]  }
      catch { getValsFromAlign dataValB resB chainB codeB 1 [expr $i + $pos] }
      append align0 $codeA
      append align1 $codeB 
      if {[expr (($pos+$i+1.0)/$spacing) ==  int (($pos+$i+1.0)/$spacing) ]} {
        append align0 " "; append align1 " "
      }
      
    } 
    puts $align0 
    puts "$align1\n"
    set pos [expr $pos + $width]
  }
}     


proc ::contactmap::calcDataDist {} {
  variable w
  variable dataNameLast
  variable dataName
  variable dataValA
  variable dataValANum
  variable currentMolA
  variable dataValB
  variable dataValBNum
  variable currentMolB
  variable firstTrajField
  variable numHoriz
  variable dataMin
  variable dataMax
  variable firstData 
  variable selTextA 
  variable selTextB
  puts "in calcDataDist, dataValANum= $dataValANum, dataValBNum= $dataValBNum"
  set selA [atomselect $currentMolA "$selTextA and name CA"]
  set selB [atomselect $currentMolB "$selTextB and name CA"]
  set coorListA [$selA get [list x y z] ]
  set coordListB [$selB get [list x y z] ]
  unset selA
  unset selB
  for {set indexA 0} {$indexA <= $dataValANum} {incr indexA} {

    set aCoord [lindex $coorListA $indexA]
    for {set indexB 0} {$indexB <= $dataValBNum} {incr indexB} {
      set bCoord [lindex $coordListB $indexB] 
      set diff [veclength [vecsub $bCoord $aCoord] ]
      set dataName([expr $indexB+$firstData]) "x-val"
      set dataValA([expr $indexB+$firstData],$indexA) $diff
      #puts "aCoord= $aCoord, bCoord= $bCoord, diff= $diff, dataValA([expr $indexB+$firstData],$indexA)= $dataValA([expr $indexB+$firstData],$indexA)"
      #puts "curDataName is $curDataName"
    }  
    set dataMin($indexA) 0 
    set dataMax($indexA) 10 
  }
  unset coorListA
  unset coordListB
  set dataNameLast $dataValBNum
  puts "in calcDataDist, dataNameLast is $dataNameLast"
  return
}

proc ::contactmap::loadDataFile {filename} {

  variable w
  variable dataValA
  variable dataHashA
  variable dataValANum
  variable dataName
  variable firstStructField
  variable rectCreated 


  variable dataName
  
  if {$filename == ""  } {
    set filename [tk_getOpenFile -initialfile $filename -title "Open Trajectory Data file" -parent $w -filetypes [list { {.dat files} {.dat} } { {Text files} {.txt}} {{All files} {*} }] ]

  } 
  set dataFile [open $filename r]
  #get file lines into an array
  set commonName ""
  set fileLines ""
  while {! [eof $dataFile] } {
    gets $dataFile curLine
    if { (! [regexp "^#" $curLine] ) && ($curLine != "" ) } {
      
      lappend fileLines $curLine
    } else {
      if { [regexp "^# FIELD=" $curLine] } { 
        set commonName [lindex [split $curLine " "] 2]
        puts "Loading file, field name is $commonName"
      } 
    }
  }
  #done with the file close it 
  close $dataFile
  #set frameList ""
  #data-containing frames
  foreach line $fileLines {
    #puts "the line is >$line<"
    foreach {resid chain atom frame val} [split $line " "] {}
    #puts "resid= $resid chain= $chain atom= $atom frame= $frame val= $val" 
    lappend frameList $frame
  } 
  #puts "framelist is $frameList"

  set frameList [lsort -unique -increasing -integer $frameList]
  set minFrame [lindex $frameList 0]
  set maxFrame [lindex $frameList end]
  puts "frameList is $frameList"
  #  no lkonger find frame list, since catching errors on frame assignment
  #has same effect.  Could still 
  #assign values in a new Group
  # (temporarlily, to hard-coded fields, if still in hacky version)
  puts "now check fileLines:\n"
  foreach line $fileLines {
    #puts "assigning data, the line is >$line<"
    foreach {resid chain atom frame val} [split $line " "] {}
    
    
    #this assumes consecutive frames, should use frameList somewhere
    # if we really want proper reverse lookup
    if { [ catch {set fieldForFrame [expr $firstStructField + $frame ]} ] } {
      set fieldForFrame -2
      puts "couldn't read frame text \"$frame\""
    }

    #now do lookup via dataHashA to find index in dataValA 
    if {[catch {set theIndex $dataHashA($resid,$chain)} ]} {
      puts "failed to find data for resid=$resid, chain=$chain"
    } else {
      if { [catch {set dataValA($fieldForFrame,$theIndex) $val} ]} {
        puts "didn't find data for frame $frame, field= $fieldForFrame, index= $theIndex, new_val= $val"
      } else {
        set dataName($fieldForFrame) $commonName
        #puts "succesfully assigned dataValA($fieldForFrame,$theIndex) as $dataValA($fieldForFrame,$theIndex)" 
      }
    }
  }  
  

  #now delete the list of data lines, no longer needed
  unset fileLines

  #redraw the data rects
  showall 1  

  return
}

proc ::contactmap::clearData {} {
  variable w
  variable dataValA
  variable dataValANum
  variable firstStructField
  variable numHoriz
  variable usableMolLoaded
  variable rectCreated

  puts "clearing 2D data..."
  set endStructs [expr $firstStructField + $numHoriz]
  for {set field $firstStructField} {$field <= $endStructs} {incr field} {
    for {set i 0} {$i<=$dataValANum} {incr i} {

      set  dataValA($field,$i) "null"
      # for the special struct case, the 0 shold give default color
      #puts "dataValA($field,$i) is now $dataValA($field,$i)"
      #set resid $dataValA(0,$i)
      #set chain $dataValA(2,$i)
      #set frame [expr $field - $firstStructField]
      #puts $writeDataFile "$resid $chain CA $frame $val"
      
    }
  }
  #redraw the data rects
  showall 1
  return
}
proc  ::contactmap::userScaleBothChanged {val} {
  variable userScalex
  variable userScaley
  variable userScaleBoth
  variable scaley
  variable fit_scaley
  variable scalex
  variable fit_scalex
  set scalex [expr $userScaleBoth * $fit_scalex]
  set scaley [expr $userScaleBoth * $fit_scaley]
  set userScaleX  $userScaleBoth
  set userScaleY $userScaleBoth
  redraw name func op
  #puts "redrawn, userScaleBoth= $userScaleBoth, scalex= $scalex, userScalex= $userScalex, scaley= $scaley, userScaley= $userScaley"
  return
}
proc  ::contactmap::userScalexChanged {val} {
  variable userScalex
  variable scalex
  variable fit_scalex
  set scalex [expr $userScalex * $fit_scalex]
  redraw name func op
  #puts "redrawn, scalex= $scalex, userScalex= $userScalex"
  return
}
proc ::contactmap::userScaleyChanged {val} {
  variable userScaley
  variable scaley
  variable fit_scaley
  #until working ok, still do direct mapping
  set scaley $userScaley 
  redraw name func op
  return
}



proc ::contactmap::drawVertScale {} {
  variable w
  variable ytopmargin
  variable scaley
  variable ybox
  variable dataValANum
  variable dataValA
  variable vertTextSkip
  variable verTextLeft
  variable vertTextRight
  variable amino_code_toggle
  variable monoFont

  $w.vertScale delete vertScaleText 

  
  #when adding new column, add to this list (maybe adjustable later)
  #The picked fields 
  
  #Add the text...
  set field 0      

  #note that the column will be 0, but the data will be from picked
  
  
  set yDataEnd [expr $ytopmargin + ($scaley * $ybox * ($dataValANum +1))]
  set y 0.0

  set yposPrev  -10000.0

  #Add the text to vertScale...
  set field 0       



  #we want text to appear in center of the dataRect we are labeling
  set vertOffset [expr $scaley * $ybox / 2.0]

  #don't do $dataValANum, its done at end, to ensure always print last 
  for {set i 0} {$i <= $dataValANum} {incr i} {
    set ypos [expr $ytopmargin + ($scaley * $y) + $vertOffset]
    if { ( ($ypos - $yposPrev) >= $vertTextSkip) && ( ( $i == $dataValANum) || ( ($yDataEnd - $ypos) > $vertTextSkip) ) } {
      if {$amino_code_toggle == 0} {
        set res_string $dataValA(1,$i)} else {
          set res_string $dataValA(1code,$i)
        }
      


      #for speed, we use vertScaleText instead of $dataName($field)
      $w.vertScale create text $vertTextRight $ypos -text "$dataValA(0,$i) $res_string $dataValA(2,$i)" -width 200 -font $monoFont -justify right -anchor e -tags vertScaleText 

      set yposPrev  $ypos
    }        
    set y [expr $y + $vertTextSkip]
  } 
}


proc ::contactmap::drawHorzScale {} {
  variable w
  variable amino_code_toggle
  variable ytopmargin
  variable scalex
  variable dataValBNum
  variable dataValB
  variable monoFont
  variable firstData
  variable xcol
  variable dataWidth
  $w.horzScale delete horzScaleText 
  
  #when adding new column, add to this list (maybe adjustable later)
  #The picked fields 
  
  #Add the text...

  #note that the column will be 0, but the data will be from picked
  
  #we want text to appear in center of the dataRect we are labeling
  #ensure minimal horizaontal spacing
  if {$amino_code_toggle == 0} {
    set horzSpacing 30 
  } else {
    set horzSpacing 20
  } 

  
  set horzDataTextSkip [expr $dataWidth]
  set scaledHorzDataTextSkip [expr $scalex * $dataWidth]
  set scaledHorzDataOffset [expr $scalex * $dataWidth / 2.0]
  set ypos 40 
  set xStart [expr ($xcol($firstData) + $scalex * $dataWidth)]
  set xDataEnd  [expr int ($xStart +  $scalex * ($dataWidth * ($dataValBNum) ) )] 
  set x 0 


  #don't do $dataValBNum, its done at end, to ensure always print last 

  #numbers are scaled for 1.0 until xpos
  #this is tied to data fields, which is produced from frames upong
  #first drawing. Should really agreee with writeDataFile, which currently uses frames, not fields
  set xposPrev -1000 
  #there is B data in $firstData, traj data starts at firstData+1
  for {set i 0} {$i <= $dataValBNum} {incr i} {
    set xpos [expr int ($xStart + ($scalex * $x) + $scaledHorzDataOffset)]
    if { ( ($xpos - $xposPrev) >= $horzSpacing) && ( ( $i == $dataValBNum) || ( ($xDataEnd - $xpos) > $horzSpacing) ) } {

      if {$amino_code_toggle == 0} {
        set res_string $dataValB(1,$i)} else {
          set res_string $dataValB(1code,$i)
        }
      


      #for speed, we use vertScaleText instead of $dataName($field)
      $w.horzScale create text $xpos $ypos -text "$dataValB(0,$i)\n$res_string\n$dataValB(2,$i)" -width 30 -font $monoFont -justify center -anchor s -tags horzScaleText 

      set xposPrev  $xpos
    }        
    set x [expr $x + $horzDataTextSkip]
    
  } 

  
}
proc ::contactmap::makeSelText {dval dataValNum pickedonly } {
  upvar $dval dataVal  

  #make selection string to display in VMD 
  set selectText  "" 
  set prevChain "Empty" 


  #Cannot be held by chain  

  for {set i 0} {$i <= $dataValNum} {incr i} {
    if { ($pickedonly == 0) || ($dataVal(picked,$i) == 1  )} {
      if { [string compare $prevChain $dataVal(2,$i)] != 0} {
        #chain is new or has changed
        append selectText ") or (chain $dataVal(2,$i)  and resid $dataVal(0,$i)"
      } else {
        append selectText " $dataVal(0,$i)"
      }
      set prevChain $dataVal(2,$i)
    }
  }  
  append selectText ")"
  set selectText [string trimleft $selectText ") or " ]

  #check for the state when mol first loaded
  if {$selectText ==""} {
    set selectText "none"
  } 
} 
proc contactmap::assignIndex {r c dHash} {
  upvar $r resnum $c chain $dHash dataHash
  #note that this does not yet care about segname, it should... 
  if {$chain=="none" } {set chain "X"}
  set index "+"
  if {$resnum != "-"} {
    #now do lookup via dataHash to find index in dataVal 
    if {[catch {set theIndex $dataHash($resnum,$chain)} ]} {
      puts "failed to find data for resnum=$resnum, chain=$chain"
      set index "+"
    } else {
      set index $theIndex
      puts "index= $index for dataVal($resnum,$chain)"
    }
  }
  return $index
}

#############################################
# end of the proc definitions
############################################






####################################################
# Execution starts here. 
####################################################

#####################################################
# set traces and some binidngs, then call zoomSeqMain
#####################################################
proc ::contactmap::startContactmap {} { 
  
  ####################################################
  # Create the window, in withdrawn form,
  # when script is sourced (at VMD startup)
  ####################################################
  variable  w  .vmd_contactmap_Window
  set windowError 0
  set errMsg ""
  #if contactmap has already been started, just deiconify window
  if { [winfo exists $w] } {
    wm deiconify $w 
    return
  }

  if { [catch {toplevel $w -visual truecolor} errMsg] } {
    puts "Info) Contact map can't find trucolor visual, will use default visual.\nInfo)   (Error reported was: $errMsg)" 
    if { [catch {toplevel $w } errMsg ]} {
      puts "Info) Default visual failed, contact map cannot be created. \nInfo)   (Error reported was: $errMsg)"   
      set windowError 1
    }
  }
  if {$windowError == 0} { 
    #don't withdraw, not under vmd menu control during testing
    wm title $w "VMD Contactmap"
    wm resizable $w 1 1 

    variable monoFont

    variable initializedVars 0
    variable needsDataUpdate 0 

    #overkill for debugging, should only need to delete once....
    trace vdelete currentMolA w [namespace code molChoose]
    trace vdelete currentMolA w [namespace code molChoose]
    trace vdelete ::vmd_pick_atom w  [namespace code list_pick] 
    trace vdelete ::vmd_pick_atom w  [namespace code list_pick] 
    trace vdelete ::vmd_initialize_structure w  [namespace code molChooseMenu]
    trace vdelete ::vmd_initialize_structure w  [namespace code molChooseMenu]

    bind $w <Map> "+[namespace code Show]"
    bind $w <Unmap> "+[namespace code Hide]"
    #specify monospaced font, 12 pixels wide
    #font create tkFixedMulti -family Courier -size -12
    #for test run tkFixed was made by normal sequence window
    set monoFont tkFixed
    #slight var clear, takes vare of biggest space use
    catch {unset dataValA}
    catch {unset dataValB}

    #call to set up, after this, all is driven by trace and bind callbacks
    zoomSeqMain
  }
  return $w
}

