##
## Finder Tool 1.0 
##
## A script to find programs. The function ::FinderTool::find can be used by
## other extensions to find needed executables.
##
## Author: Eamon Caddigan
##         eamon@ks.uiuc.edu
##         vmd@ks.uiuc.edu
##
## $Id: apbsrun.tcl,v 1.7 2003/09/23 19:09:24 eamon Exp $
##

##
## Example code to add this plugin to the VMD extensions menu:
##
#  if { [catch {package require apbsrun} msg] } {
#    puts "VMD APBSRun package could not be loaded:\n$msg"
#  } elseif { [catch {menu tk register "apbsrun" apbsrun} msg] } {
#    puts "VMD APBSRun could not be started:\n$msg"
#  }


## Tell Tcl that we're a package and any dependencies we may have
package provide apbsrun 1.0

namespace eval ::APBSRun:: {
  namespace export apbsrun

  # window handles
  variable main_win      ;# handle to main window
  variable w             ;# handle to elec-option-editing window 

  # global settings
  variable file_list     ;# the listbox displaying the file names
  variable elec_list     ;# the listbox displaying the elec statements
  variable elec_vals     ;# all of the options for each elec statement
  variable elec_temp     ;# temporarily stores elec options while editing
}
  
##
## Main routine
## Create the window and initialize data structures
##
proc ::APBSRun::apbsrun {} {
  variable main_win
  variable file_list
  variable elec_list
  variable elec_temp

  # If already initialized, just turn on
  if { [winfo exists .apbsrun] } {
    wm deiconify $main_win
    return
  }

  set main_win [toplevel ".apbsrun"]
  wm title $main_win "APBS Tool" 
  wm resizable $main_win yes yes

  # Initialize array so first call to unset doesn't fail
  array set elec_temp {}

  # Listboxes get focused when clicked
  #bind Listbox <Button-1> {+ focus %W}

  ##
  ## make the menu bar
  ## 
  frame $main_win.menubar -relief raised -bd 2 ;# frame for menubar
  pack $main_win.menubar -padx 1 -fill x -side top

  menubutton $main_win.menubar.help -text Help -underline 0 \
    -menu $main_win.menubar.help.menu
  pack $main_win.menubar.help -side right

  ##
  ## help menu
  ##
  menu $main_win.menubar.help.menu -tearoff no
  $main_win.menubar.help.menu add command -label "About" \
    -command {tk_messageBox -type ok -title "About apbsrun" \
              -message "GUI for initiating APBS runs."}

  #
  # Main window
  #
  frame $main_win.file

  frame $main_win.file.list
  label $main_win.file.list.filelabel -anchor w -text "Files: "
  set file_list [listbox $main_win.file.list.file_names -relief sunken -bd 2 \
    -height 5]
  pack $main_win.file.list.filelabel $main_win.file.list.file_names \
    -side top -anchor w -fill x

  frame $main_win.file.buttons
  button $main_win.file.buttons.addfile -text "Add File" -width 8 \
    -command ::APBSRun::file_add
  button $main_win.file.buttons.edititem -text "Edit" -width 8 \
    -command ::APBSRun::file_edit
  button $main_win.file.buttons.delitem -text "Delete" -width 8 \
    -command ::APBSRun::file_del
  pack $main_win.file.buttons.addfile \
    $main_win.file.buttons.edititem $main_win.file.buttons.delitem \
    -side left -anchor w

  pack $main_win.file.list $main_win.file.buttons \
    -side top -anchor w -fill x

  frame $main_win.elec

  frame $main_win.elec.list
  label $main_win.elec.list.label -anchor w -text "ELEC statements: "
  set elec_list [listbox $main_win.elec.list.names -relief sunken -bd 2 \
    -height 5]
  pack $main_win.elec.list.label $main_win.elec.list.names \
    -side top -anchor w -fill x

  frame $main_win.elec.buttons
  button $main_win.elec.buttons.addelec -text "Add ELEC" -width 8 \
    -command ::APBSRun::elec_add
  button $main_win.elec.buttons.edititem -text "Edit" -width 8 \
    -command ::APBSRun::elec_edit
  button $main_win.elec.buttons.delitem -text "Delete" -width 8 \
    -command ::APBSRun::elec_del
  pack $main_win.elec.buttons.addelec \
    $main_win.elec.buttons.edititem $main_win.elec.buttons.delitem \
    -side left -anchor w

  pack $main_win.elec.list $main_win.elec.buttons \
    -side top -anchor w -fill x

  pack $main_win.file $main_win.elec \
    -side top -anchor n -pady 10 -padx 10 -fill x
}

proc ::APBSRun::file_add {} {
  variable file_list
  set indexnum [$file_list index end]
  set elementname [concat foo $indexnum]
  $file_list insert end $elementname
}

proc ::APBSRun::file_edit {} {
  variable file_list
  
}
  
proc ::APBSRun::file_del {} {
  variable file_list
  $file_list delete active
}

proc ::APBSRun::elec_add {} {
  variable file_list
  variable elec_list
  variable elec_vals
  variable elec_temp
  set index [$elec_list size]
 
  if {[$file_list index end] == 0} {
    tk_messageBox -type ok -icon warning \
      -message "You must add files before creating an ELEC statement."
    return
  }
 
  # Clear the temporary array, and reinitialize it as empty
#  unset elec_temp
#  array set elec_temp {}
  array unset elec_temp

  # Launch the elec-editing window and wait for it to finish
  ::APBSRun::elecmenu $index
  tkwait window $::APBSRun::w

  # Copy the contents of elec_temp into elec_vals
  array set elec_vals [array get elec_temp]

  # Add an item to the ELEC listbox
  $elec_list insert end "$index ($elec_vals($index,mol))"

#  puts [array get elec_temp]
}

proc ::APBSRun::elec_edit {} {
}

proc ::APBSRun::elec_del {} {
  variable elec_list
  variable elec_vals

#  set index [$elec_list index active]
  set index [string index [$elec_list get active] 0]

  # Remove the entry from the listbox
  $elec_list delete active

#  puts "Before:"
#  puts [array get elec_vals]

  # Remove the data from elec_vals
  array unset elec_vals "$index,*"

#  puts "After:"
#  puts [array get elec_vals]
#  puts ""
}

proc ::APBSRun::elecmenu {index} {
  variable w
  variable file_list
  variable elec_temp

  # If already initialized, just turn on
  if { [winfo exists .elec_win] } {
    wm deiconify $w
    return
  }

  set w [toplevel ".elec_win"]
  wm title $w "ELEC values" 
  wm resizable $w yes yes


  #
  # left frame for selecting ELEC keyword and keyword-specific options
  #
  frame $w.l

#  set descriptions(mg-auto) "Automatically configured multigrid focusing calculations."
#  set descriptions(mg-para) "Automatically configured multigrid parallel focusing calculations"
#  set descriptions(mg-manual) "Manually configured multigrid calculations."
#  set descriptions(mg-dummy) "Same as mg-manual, but simply sets up the problem and skips the solver step."

  frame $w.l.calc_type
  frame $w.l.calc_type.menu
  label $w.l.calc_type.menu.label -text "Type of Calculation: "
  tk_optionMenu $w.l.calc_type.menu.type ::APBSRun::elec_temp($index,calc_type) \
    mg-auto mg-para mg-manual mg-dummy
  pack $w.l.calc_type.menu.label $w.l.calc_type.menu.type \
    -side left -anchor w
  # XXX - this doesn't work
  label $w.l.calc_type.label -text "XXX - fixme" ;#variable descriptions($calc_type)
  pack $w.l.calc_type.menu $w.l.calc_type.label \
    -side top -anchor w

  # nlev
  frame $w.l.nlev
  label $w.l.nlev.label -text "Number of levels: "
  entry $w.l.nlev.entry -width 4 -textvariable ::APBSRun::elec_temp($index,nlev)
  pack $w.l.nlev.label $w.l.nlev.entry \
    -side left -anchor w

  # grid or glen
  frame $w.l.glen
  label $w.l.glen.label -text "Grid Size: "
  frame $w.l.glen.buttons
  radiobutton $w.l.glen.buttons.grid -text "Mesh grid spacing" \
    -variable ::APBSRun::elec_temp($index,grid_selected) -value "grid"
  radiobutton $w.l.glen.buttons.glen -text "Mesh lengths" \
    -variable ::APBSRun::elec_temp($index,grid_selected) -value "glen"
  pack $w.l.glen.buttons.grid $w.l.glen.buttons.glen \
    -side top -anchor w
  frame $w.l.glen.dims
  label $w.l.glen.dims.xlabel -text "x: "
  entry $w.l.glen.dims.xvalue -width 4 -textvariable ::APBSRun::elec_temp($index,grid_x)
  label $w.l.glen.dims.ylabel -text " y: "
  entry $w.l.glen.dims.yvalue -width 4 -textvariable ::APBSRun::elec_temp($index,grid_y)
  label $w.l.glen.dims.zlabel -text " z: "
  entry $w.l.glen.dims.zvalue -width 4 -textvariable ::APBSRun::elec_temp($index,grid_z)
  pack $w.l.glen.dims.xlabel $w.l.glen.dims.xvalue $w.l.glen.dims.ylabel \
    $w.l.glen.dims.yvalue $w.l.glen.dims.zlabel $w.l.glen.dims.zvalue \
    -side left
  pack $w.l.glen.label $w.l.glen.buttons $w.l.glen.dims \
    -side top -anchor w

  # gcent
  frame $w.l.gcent
  label $w.l.gcent.label -text "Grid Center: "
  frame $w.l.gcent.molid
  radiobutton $w.l.gcent.molid.button -text "Center on molecule: " \
    -variable ::APBSRun::elec_temp($index,gcent_method) -value "molid"
  eval tk_optionMenu $w.l.gcent.molid.id ::APBSRun::elec_temp($index,gcent_mol) \
    [$file_list get 0 end]
  pack $w.l.gcent.molid.button $w.l.gcent.molid.id \
    -side left
  radiobutton $w.l.gcent.coordopt -text "Center on coordinates: " \
    -variable ::APBSRun::elec_temp($index,gcent_method) -value "coord"
  frame $w.l.gcent.coord
  label $w.l.gcent.coord.xlabel -text "x: "
  entry $w.l.gcent.coord.xvalue -width 4 -textvariable ::APBSRun::elec_temp($index,gcent_x)
  label $w.l.gcent.coord.ylabel -text " y: "
  entry $w.l.gcent.coord.yvalue -width 4 -textvariable ::APBSRun::elec_temp($index,gcent_y)
  label $w.l.gcent.coord.zlabel -text " z: "
  entry $w.l.gcent.coord.zvalue -width 4 -textvariable ::APBSRun::elec_temp($index,gcent_z)
  pack $w.l.gcent.coord.xlabel $w.l.gcent.coord.xvalue \
    $w.l.gcent.coord.ylabel $w.l.gcent.coord.yvalue $w.l.gcent.coord.zlabel \
    $w.l.gcent.coord.zvalue \
    -side left
  pack $w.l.gcent.label $w.l.gcent.molid $w.l.gcent.coordopt $w.l.gcent.coord \
    -side top -anchor w

  # srfm
  frame $w.l.srfm
  frame $w.l.srfm.top
  label $w.l.srfm.top.label -text "Surface definition: "
  tk_optionMenu $w.l.srfm.top.menu ::APBSRun::elec_temp($index,srfm) \
    0 1 2
  pack $w.l.srfm.top.label $w.l.srfm.top.menu \
    -side left -anchor w
  label $w.l.srfm.description -text "XXX - fixme"
  pack $w.l.srfm.top $w.l.srfm.description \
    -side top -anchor w

  #
  # pack up the left frame
  #
  pack $w.l.calc_type $w.l.nlev $w.l.glen $w.l.gcent $w.l.srfm \
    -side top -pady 10 -padx 10 -fill x


  #
  # right frame for options used by all ELEC keyworks
  #
  frame $w.r

  # dime
  frame $w.r.dime
  label $w.r.dime.label -text "Number of gridpoints: "
  frame $w.r.dime.coord
  label $w.r.dime.coord.xlabel -text "x: "
  entry $w.r.dime.coord.xentry -width 4 -textvar ::APBSRun::elec_temp($index,dime_x)
  label $w.r.dime.coord.ylabel -text " y: "
  entry $w.r.dime.coord.yentry -width 4 -textvar ::APBSRun::elec_temp($index,dime_y)
  label $w.r.dime.coord.zlabel -text " z: "
  entry $w.r.dime.coord.zentry -width 4 -textvar ::APBSRun::elec_temp($index,dime_z)
  pack $w.r.dime.coord.xlabel $w.r.dime.coord.xentry $w.r.dime.coord.ylabel \
    $w.r.dime.coord.yentry $w.r.dime.coord.zlabel $w.r.dime.coord.zentry \
    -side left
  pack $w.r.dime.label $w.r.dime.coord \
    -side top -anchor w

  # mol
  frame $w.r.mol
  label $w.r.mol.label -text "Molecule: "
  eval tk_optionMenu $w.r.mol.id ::APBSRun::elec_temp($index,mol) \
    [$file_list get 0 end]
  pack $w.r.mol.label $w.r.mol.id \
    -side left -anchor w

  # lpbe or npbe
  frame $w.r.pbe
  label $w.r.pbe.label -text "PBE: "
  frame $w.r.pbe.type
  radiobutton $w.r.pbe.type.lpbe -text "linearized" \
    -variable ::APBSRun::elec_temp($index,pbe) -value "lpbe"
  radiobutton $w.r.pbe.type.npbe -text "nonlinear" \
    -variable ::APBSRun::elec_temp($index,pbe) -value "npbe"
  pack $w.r.pbe.type.lpbe $w.r.pbe.type.npbe \
    -side left -anchor w
  pack $w.r.pbe.label $w.r.pbe.type \
    -side top -anchor w

  # bcfl
  frame $w.r.bcfl
  label $w.r.bcfl.label -text "Boundary condition: "
  radiobutton $w.r.bcfl.0 -text "Zero boundary conditions" \
    -variable ::APBSRun::elec_temp($index,bcfl) -value 0
  radiobutton $w.r.bcfl.1 -text "Single ion for molecule" \
    -variable ::APBSRun::elec_temp($index,bcfl) -value 1
  radiobutton $w.r.bcfl.2 -text "Single ion for each ion" \
    -variable ::APBSRun::elec_temp($index,bcfl) -value 2
  radiobutton $w.r.bcfl.4 -text "Solution from previous calculation" \
    -variable ::APBSRun::elec_temp($index,bcfl) -value 4
  pack $w.r.bcfl.label $w.r.bcfl.0 $w.r.bcfl.1 $w.r.bcfl.2 $w.r.bcfl.4 \
    -side top -anchor w

  # ion (optional)
  # XXX - TODO

  # pdie
  # sdie
  frame $w.r.diel
  label $w.r.diel.label -text "Dielectric constants: "
  frame $w.r.diel.vals
  label $w.r.diel.vals.plabel -text "solute: "
  entry $w.r.diel.vals.pval -width 6 -textvar ::APBSRun::elec_temp($index,pdie)
  label $w.r.diel.vals.slabel -text " solvent: "
  entry $w.r.diel.vals.sval -width 6 -textvar ::APBSRun::elec_temp($index,sdie)
  pack $w.r.diel.vals.plabel $w.r.diel.vals.pval \
    $w.r.diel.vals.slabel $w.r.diel.vals.sval \
    -side left -anchor w
  pack $w.r.diel.label $w.r.diel.vals \
    -side top -anchor w

  # chgm
  frame $w.r.chgm
  label $w.r.chgm.label -text "Charge discretization: "
  radiobutton $w.r.chgm.0 -text "Trilinear hat-function" \
    -variable ::APBSRun::elec_temp($index,chgm) -value 0
  radiobutton $w.r.chgm.1 -text "Cubic B-spline" \
    -variable ::APBSRun::elec_temp($index,chgm) -value 1
  pack $w.r.chgm.label $w.r.chgm.0 $w.r.chgm.1 \
    -side top -anchor w

  # usemap (optional)
  # XXX - TODO

  # srad
  frame $w.r.srad
  label $w.r.srad.label -text "Solvent radius: "
  entry $w.r.srad.value -width 4 -textvar ::APBSRun::elec_temp($index,srad)
  pack $w.r.srad.label $w.r.srad.value \
    -side left -anchor w

  # swin
  frame $w.r.swin
  label $w.r.swin.label -text "Spline window: "
  entry $w.r.swin.value -width 4 -textvar ::APBSRun::elec_temp($index,swin)
  pack $w.r.swin.label $w.r.swin.value \
    -side left -anchor w

  # temp
  frame $w.r.temp
  label $w.r.temp.label -text "System temperature (K): "
  entry $w.r.temp.value -width 4 -textvar ::APBSRun::elec_temp($index,temp)
  pack $w.r.temp.label $w.r.temp.value \
    -side left -anchor w

  # gamma
  frame $w.r.gamma
  label $w.r.gamma.label -text "Surface tension: "
  entry $w.r.gamma.value -width 4 -textvar ::APBSRun::elec_temp($index,gamma)
  pack $w.r.gamma.label $w.r.gamma.value \
    -side left -anchor w

  # calcenergy
  frame $w.r.calcenergy
  frame $w.r.calcenergy.top
  label $w.r.calcenergy.top.label -text "Electrostatic energy output: "
  tk_optionMenu $w.r.calcenergy.top.menu ::APBSRun::elec_temp($index,calcenergy) \
    0 1 2
  pack $w.r.calcenergy.top.label $w.r.calcenergy.top.menu \
    -side left -anchor w
  label $w.r.calcenergy.description -text "XXX - fixme"
  pack $w.r.calcenergy.top $w.r.calcenergy.description \
    -side top -anchor w

  # calcforce
  frame $w.r.calcforce
  frame $w.r.calcforce.top
  label $w.r.calcforce.top.label -text "Electrostatic force output: "
  tk_optionMenu $w.r.calcforce.top.menu ::APBSRun::elec_temp($index,calcforce) \
    0 1 2
  pack $w.r.calcforce.top.label $w.r.calcforce.top.menu \
    -side left -anchor w
  label $w.r.calcforce.description -text "XXX - fixme"
  pack $w.r.calcforce.top $w.r.calcforce.description \
    -side top -anchor w

  # write (optional)
  # XXX - TODO

  # writemat (optional)
  # XXX - TODO

  frame $w.r.okaycancel
  button $w.r.okaycancel.okay -text OK -width 6
  button $w.r.okaycancel.cancel -text Cancel -width 6
  pack $w.r.okaycancel.okay $w.r.okaycancel.cancel \
    -side left -anchor e -fill x

  #
  # pack the right frame
  #
  pack $w.r.mol $w.r.dime $w.r.pbe $w.r.bcfl $w.r.diel $w.r.chgm $w.r.srad \
       $w.r.swin $w.r.temp $w.r.gamma $w.r.calcenergy $w.r.calcforce \
       $w.r.okaycancel \
       -side top -pady 10 -padx 10 -fill x

  ## 
  ## pack up the main frame
  ##
  pack $w.l $w.r \
   -side left -anchor n -pady 10 -padx 10 -fill x
}


##
## Call the search routine and update the GUI
##
proc ::APBSRun::find_gui {} {
  variable program
  variable location

  set programpath [::APBSRun::find $program]

  if {[string length $programpath] > 0} {
    set location $programpath
  } else {
    set location "$program: Command not found."
  }
}


##
## Find the executable, prompting the user if necessary
##
proc ::APBSRun::find { execname } {
  global env
  set execpath ""

  # XXX - The hard part: make this work on non-Unix platforms
  if {[info exists env(PATH)]} {
    set pathlist [split $env(PATH) :]

    # Search the path for the program
    foreach directory $pathlist {
      if {[file executable [file join $directory $execname]] > 0} {
        set execpath [file join $directory $execname]
        break
      }
    }
  }

  # If the file isn't found in the path, prompt the user for its location
  if {[string length $execpath] == 0} {
    set answer [tk_messageBox -type yesno -title "Program not found" \
                -message " Could not find a program named `$execname'. \n\
                           Would you like to locate it manually?"]
    while {$answer} {
      set answer no
      # XXX - not sure if getOpenFile is best here
      set execpath [tk_getOpenFile -initialfile $execname \
                    -title "Please select a program"]  
      if {[string length $execpath] > 0 && \
          [file executable [file join $execpath]] == 0} {
        set answer [tk_messageBox -type yesno -title "Warning" \
                    -message " Warning, `$execpath' is not executable. \n\
                               Would you like to change the selection?"]
      }
    }
  }

  return $execpath
}


# This gets called by VMD the first time the menu is opened.
proc apbsrun_tk_cb {} {
  variable foobar
  # Don't destroy the main window, because we want to register the window
  # with VMD and keep reusing it.  The window gets iconified instead of
  # destroyed when closed for any reason.
  #set foobar [catch {destroy $::APBSRun::w  }]  ;# destroy any old windows

  ::APBSRun::apbsrun   ;# start the tool 
  return $APBSRun::main_win
}

# WISH stuff
apbsrun_tk_cb
