/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: dxplugin.C,v $
 *      $Author: eamon $       $Locker:  $             $State: Exp $
 *      $Revision: 1.4 $       $Date: 2003/08/06 16:22:42 $
 *
 ***************************************************************************/

/* 
 * DX potential maps
 *
 * Format of the file is:
 * # Comments
 * .
 * .
 * .
 * object 1 class gridpositions counts xn yn zn
 * origin xorg yorg zorg
 * delta xdel 0 0
 * delta 0 ydel 0
 * delta 0 0 zdel
 * object 2 class gridconnections counts xn yn zn
 * object 3 class array type double rank 0 items [ xn*yn*zn ] data follows
 * f1 f2 f3
 * f4 f5 f6
 * .
 * .
 * .
 *
 * Where xn, yn, and zn are the number of data points along each axis;
 * xorg, yorg, and zorg is the origin of the grid, in angstroms;
 * xdel, ydel, and zdel are the scaling factors to convert grid units to
 * angstroms.
 * Grid data follows, with three values per line, ordered z fast, y medium,
 * and x slow.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <string.h>

#if defined(_AIX)
#include <strings.h>
#endif

#ifdef WIN32
#define strcasecmp  stricmp
#define strncasecmp strnicmp
#endif

#include "molfile_plugin.h"

#define LINESIZE 85


typedef struct {
  FILE *fd;
  int nsets;
  molfile_volumetric_t *vol;
} dx_t;


static void *open_dx_read(const char *filepath, const char *filetype,
    int *natoms) {
  FILE *fd;
  dx_t *dx;
  char inbuf[LINESIZE];
  int xsize, ysize, zsize;
  float xorig, yorig, zorig, xdelta, ydelta, zdelta;
  
  fd = fopen(filepath, "rb");
  if (!fd) 
    return NULL;

  /* skip comments */
  do {
    if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL)) 
      return NULL;
  } while (inbuf[0] == '#');

  /* get the number of grid points */
  if (sscanf(inbuf, "object 1 class gridpositions counts %i %i %i", &zsize, &ysize, &xsize) != 3)
    return NULL;

  /* get the cell origin */
  if ( feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL) ||
       (sscanf(inbuf, "origin %e %e %e", &xorig, &yorig, &zorig) != 3) )
    return NULL;

  /* get the cell dimensions */
  if ( feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL) ||
       (sscanf(inbuf, "delta %e %*e %*e", &zdelta) != 1) )
    return NULL;

  if ( feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL) ||
       (sscanf(inbuf, "delta %*e %e %*e", &ydelta) != 1) )
    return NULL;

  if ( feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL) ||
       (sscanf(inbuf, "delta %*e %*e %e", &xdelta) != 1) )
    return NULL;

  /* skip the last two lines of the header */
  if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL)) 
    return NULL;
  if (feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL))
    return NULL;
 
  /* allocate and initialize the dx structure */
  dx = new dx_t;
  dx->fd = fd;
  dx->vol = NULL;
  *natoms = MOLFILE_NUMATOMS_NONE;
  dx->nsets = 1; /* this file contains only one data set */

  dx->vol = new molfile_volumetric_t[1];
  strcpy(dx->vol[0].dataname, "dx electron potential map");

  /* <midx, midy, midz> is the middle point of the grid. */
  dx->vol[0].origin[0] = xorig;
  dx->vol[0].origin[1] = yorig;
  dx->vol[0].origin[2] = zorig;

  dx->vol[0].xaxis[0] = xdelta * xsize;
  dx->vol[0].xaxis[1] = 0;
  dx->vol[0].xaxis[2] = 0;

  dx->vol[0].yaxis[0] = 0;
  dx->vol[0].yaxis[1] = ydelta * ysize;
  dx->vol[0].yaxis[2] = 0;
  
  dx->vol[0].zaxis[0] = 0;
  dx->vol[0].zaxis[1] = 0;
  dx->vol[0].zaxis[2] = zdelta * zsize;

  dx->vol[0].xsize = xsize;
  dx->vol[0].ysize = ysize;
  dx->vol[0].zsize = zsize;

  dx->vol[0].has_color = 0;

  return dx;
}

static int read_dx_metadata(void *v, int *nsets, 
  molfile_volumetric_t **metadata) {
  dx_t *dx = (dx_t *)v;
  *nsets = dx->nsets; 
  *metadata = dx->vol;  

  return MOLFILE_SUCCESS;
}

static int read_dx_data(void *v, int set, float *datablock,
                         float *colorblock) {
  dx_t *dx = (dx_t *)v;
  FILE *fd = dx->fd;
  char inbuf[LINESIZE];
  float grid[3];
  int x, y, z, xsize, ysize, zsize, xysize, count, total, i;

  xsize = dx->vol[0].xsize;
  ysize = dx->vol[0].ysize;
  zsize = dx->vol[0].zsize;
  xysize = xsize * ysize;
  total = xysize * zsize;

  /* Read the values from the file */
  x = y = z = 0;
  for (count = 0; count < total/3; count++) {
    if ( feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL) )
      return MOLFILE_ERROR;
    if ( sscanf(inbuf, "%e %e %e", &grid[0], &grid[1], &grid[2]) != 3 )
      return MOLFILE_ERROR;

    for (i = 0; i < 3; i++) { 
      datablock[x + y*xsize + z*xysize] = grid[i];
      z++;
      if (z >= zsize) {
        z = 0;
        y++;
        if (y >= ysize) {
          y = 0;
          x++;
        }
      }
    }
  }

  if ((total%3) != 0) {
    if ( feof(fd) || ferror(fd) || (fgets(inbuf, LINESIZE, fd) == NULL) )
      return MOLFILE_ERROR;

    count = sscanf(inbuf, "%e %e %e", &grid[0], &grid[1], &grid[2]);
    if (count != (total%3))
      return MOLFILE_ERROR;

    for (i = 0; i < count; i++) {
      datablock[x + y*xsize + z*xysize] = grid[i];
      z++;
    }
  }

  return MOLFILE_SUCCESS;
}

static void close_dx_read(void *v) {
  dx_t *dx = (dx_t *)v;

  fclose(dx->fd);
  if (dx->vol != NULL)
    delete [] dx->vol; 
  delete dx;
}

/*
 * Initialization stuff here
 */
static molfile_plugin_t plugin = {
  vmdplugin_ABIVERSION,   /* ABI version */
  MOLFILE_PLUGIN_TYPE, 	  /* plugin type */
  "dx",                   /* file format description */
  "Eamon Caddigan",       /* author(s) */
  0,                      /* major version */
  1,                      /* minor version */
  VMDPLUGIN_THREADSAFE,   /* is reentrant */
  "dx"                    /* filename extension */
};

int VMDPLUGIN_init(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_fini(void) { return VMDPLUGIN_SUCCESS; }
int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  plugin.open_file_read = open_dx_read;
  plugin.read_volumetric_metadata = read_dx_metadata;
  plugin.read_volumetric_data = read_dx_data;
  plugin.close_file_read = close_dx_read;
  (*cb)(v, (vmdplugin_t *)&plugin);
  return VMDPLUGIN_SUCCESS;
}

