/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: psfplugin.c,v $
 *      $Author: johns $       $Locker:  $             $State: Exp $
 *      $Revision: 1.19 $       $Date: 2003/08/18 21:00:13 $
 *
 ***************************************************************************/

#include "molfile_plugin.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#define PSF_RECORD_LENGTH   80

typedef struct {
  FILE *fd;
  int numatoms;
  int charmmfmt;
  int *from, *to;
} psfdata;


/* Open the .psf file and skip past the remarks to the first data section.
 * Returns the file pointer, or NULL if error.  Also puts the number of
 * atoms in the molecule into the given integer.  
 */
static FILE *open_psf_file(const char *fname, int *natom) {
  char inbuf[PSF_RECORD_LENGTH+2];
  FILE *f;

  if (!fname)
    return NULL;

  if ((f = fopen(fname, "r")) == NULL)
    return NULL;

  *natom = MOLFILE_NUMATOMS_NONE;

  /* read lines until a line with NATOM and without REMARKS appears    */
  do {
    if (inbuf != fgets(inbuf, PSF_RECORD_LENGTH+1, f)) {
      /* EOF encountered with no NATOM line found ==> error, return null */
      *natom = MOLFILE_NUMATOMS_NONE;
      fclose(f);
      return NULL;
    }

    if (strlen(inbuf) > 0) {
      if (!strstr(inbuf, "REMARKS")) {
        if (strstr(inbuf, "NATOM")) {
          *natom = atoi(inbuf);
        }
      } 
    }
  } while (*natom == MOLFILE_NUMATOMS_NONE);

  return f;
}

/* Read in the next atom info into the given storage areas; this assumes
   that file has already been moved to the beginning of the atom records.
   Returns the serial number of the atom. If there is an error, returns -1.*/
static int get_psf_atom(FILE *f, char *name, char *atype, char *resname,
   char *segname, int *resid, float *q, float *m) {
  char inbuf[PSF_RECORD_LENGTH+2];
  int i,num;

  if (inbuf != fgets(inbuf, PSF_RECORD_LENGTH+1, f)) {
    return(-1); /* failed to read in an atom */
  }

  if (strlen(inbuf) < 50) {
    fprintf(stderr, "Line too short in psf file: \n%s\n", inbuf);
    return -1;
  }

  num = atoi(inbuf); /* atom index */

  strncpy(segname, inbuf+9, 4); 
  segname[4] = '\0';
  strncpy(resname, inbuf+19, 4);  
  resname[4] = '\0';
  strncpy(name, inbuf+24, 4);  
  name[4] = '\0';
  strncpy(atype, inbuf+29, 4); 
  atype[4] = '\0';

  /* null terminate any extraneous spaces */
  for (i=3; i >= 0; i--) {
    if (segname[i] == ' ')   
      segname[i] = '\0';
    if (resname[i] == ' ')   
      resname[i] = '\0';
    if (name[i] == ' ')      
      name[i] = '\0';
    if (atype[i] == ' ')     
      atype[i] = '\0';
  }

  *resid = atoi(inbuf+13);
  *q = (float) atof(inbuf+34);
  *m = (float) atof(inbuf+50);

  return num;
}

/* Read in the beginning of the bond information, but don't read in the
   bonds.  Returns the number of bonds in molecule.  If error, returns (-1). */
static int start_psf_bonds(FILE *f) {
  char inbuf[PSF_RECORD_LENGTH+2];
  int nbond = -1;

  /* keep reading the next line until a line with NBOND appears */
  do {
    if (inbuf != fgets(inbuf, PSF_RECORD_LENGTH+1, f)) {
      /* EOF encountered with no NBOND line found ==> error, return (-1) */
      return (-1);
    }
    if (strlen(inbuf) > 0 && strstr(inbuf,"NBOND"))
      nbond = atoi(inbuf);
  } while (nbond == -1);

  return nbond;
}


/* Read in the bond info into the given integer arrays, one for 'from' and
   one for 'to' atoms; remember that .psf files use 1-based indices,
   not 0-based.  Returns 1 if all nbond bonds found; 0 otherwise.  */
static int get_psf_bonds(FILE *f, int nbond, int fromAtom[], int toAtom[]) {
  char *bondptr=NULL;
  char inbuf[PSF_RECORD_LENGTH+2];
  int i=0;
  size_t minlinesize;

  while (i < nbond) {
    if ((i % 4) == 0) {
      /* must read next line */
      if (!fgets(inbuf, PSF_RECORD_LENGTH+2, f)) {
        /* early EOF encountered */
        break;
      }
      /* Check that there is enough space in the line we are about to read */
      if (nbond-i >= 4) {
        minlinesize = 16*4;
      } else {
        minlinesize = 16*(nbond-i);
      }
      if (strlen(inbuf) < minlinesize) {
        fprintf(stderr, "Bonds line too short in psf file: \n%s\n", inbuf);
        break;
      }
      bondptr = inbuf;
    }
    if ((fromAtom[i] = atoi(bondptr)) < 1)
      break;
    bondptr += 8;
    if ((toAtom[i] = atoi(bondptr)) < 1)
      break;
    bondptr += 8;
    i++;
  }

  return (i == nbond);
}

/*
 * API functions
 */

static void *open_psf_read(const char *path, const char *filetype, 
    int *natoms) {
  FILE *fd;
  psfdata *psf;
  assert(path);
  
  if (!(fd = open_psf_file(path, natoms))) {
    fprintf(stderr, "Couldn't open psf file %s\n", path);
    return NULL;
  } 
  psf = (psfdata *) malloc(sizeof(psfdata));
  memset(psf, 0, sizeof(psfdata));
  psf->numatoms = *natoms;
  psf->fd = fd;
  psf->charmmfmt = 0; /* initialize to off for now */

  return psf;
}

static int read_psf(void *v, int *optflags, molfile_atom_t *atoms) {
  psfdata *psf = (psfdata *)v;
  int i;
  assert(psf);
  assert(psf->fd);
  
  /* we read in the optional mass and charge data */
  *optflags = MOLFILE_MASS | MOLFILE_CHARGE;

  for (i=0; i<psf->numatoms; i++) {
    molfile_atom_t *atom = atoms+i; 
    if (get_psf_atom(psf->fd, atom->name, atom->type, 
                     atom->resname, atom->segid, 
                     &atom->resid, &atom->charge, &atom->mass) < 0) {
      fprintf(stderr, "couldn't read atom %d\n", i);
      fclose(psf->fd);
      psf->fd = NULL;
      return MOLFILE_ERROR;
    }
    atom->chain[0] = atom->segid[0];
    atom->chain[1] = '\0';
  }

  return MOLFILE_SUCCESS;
}

static int read_bonds(void *v, int *nbonds, int **fromptr, int **toptr) {
  psfdata *psf = (psfdata *)v;

  /* now read bond data */
  *nbonds = start_psf_bonds(psf->fd);

  if (*nbonds > 0) {
    psf->from = (int *) malloc(*nbonds*sizeof(int));
    psf->to = (int *) malloc(*nbonds*sizeof(int));

    if (!get_psf_bonds(psf->fd, *nbonds, psf->from, psf->to)) {
      fclose(psf->fd);
      psf->fd = NULL;
      return MOLFILE_ERROR;
    }
    *fromptr = psf->from;
    *toptr = psf->to;
  } else {
    printf("psfplugin) WARNING: no bonds defined in PSF file.\n");
  }

  return MOLFILE_SUCCESS;
}

static void close_psf_read(void *mydata) {
  psfdata *psf = (psfdata *)mydata;
  if (psf) {
    if (psf->fd) fclose(psf->fd);
    if (psf->from) free(psf->from);
    if (psf->to) free(psf->to);
    free(psf);
  }
}  


static void *open_psf_write(const char *path, const char *filetype,
    int natoms) {
  FILE *fd;
  psfdata *psf;
  assert(path);
  fd = fopen(path, "w");
  if (!fd) {
    fprintf(stderr, "Unable to open file %s for writing\n", path);
    return NULL;
  }
  psf = (psfdata *) malloc(sizeof(psfdata));
  psf->fd = fd; 
  psf->numatoms = natoms;
  psf->charmmfmt = 0; /* initialize to off for now */

  return psf;
}

static int write_psf_structure(void *v, int optflags,
    const molfile_atom_t *atoms) {
  psfdata *psf = (psfdata *)v;
  const molfile_atom_t *atom;
  int i;

  printf("psfplugin) WARNING: PSF file is incomplete, no bonds, angles, dihedrals,\n");
  printf("           or impropers will be written. \n");

  fprintf(psf->fd, "PSF\n\n%8d !NTITLE\n", 1);

  if (psf->charmmfmt) {
    fprintf(psf->fd," REMARKS %s\n","VMD generated structure charmm psf file");

    printf("psfplugin) WARNING: Charmm format PSF file is incomplete, atom type ID\n");
    printf("           codes have been emitted as '0'. \n");
  } else {
    fprintf(psf->fd," REMARKS %s\n","VMD generated structure x-plor psf file");
  }
  fprintf(psf->fd, "\n");

  /* write out total number of atoms */
  fprintf(psf->fd, "%8d !NATOM\n", psf->numatoms);

  /* write out all of the atom records */
  for (i=0; i<psf->numatoms; i++) {
    const char *atomname; 
    atom = &atoms[i];
    atomname = atom->name;

    /* skip any leading space characters given to us by VMD */ 
    while (*atomname == ' ')
      atomname++;

    if (psf->charmmfmt) {
      /* XXX replace hard-coded 0 with proper atom type ID code */
      fprintf(psf->fd, "%8d %-4s %-4d %-4s %-4s %4d %10.6f     %9.4f  %10d\n",
              i+1, atom->segid, atom->resid, atom->resname,
              atomname, /* atom->typeid */ 0, atom->charge, atom->mass, 0);
    } else {
      fprintf(psf->fd, "%8d %-4s %-4d %-4s %-4s %-4s %10.6f     %9.4f  %10d\n",
              i+1, atom->segid, atom->resid, atom->resname,
              atomname, atom->type, atom->charge, atom->mass, 0);
    }
  } 
  fprintf(psf->fd, "\n");

  fprintf(psf->fd, "%8d !NBOND: bonds\n", 0);
  /* XXX put in code to emit bonds here */
  fprintf(psf->fd, "\n\n");

  fprintf(psf->fd, "%8d !NTHETA: angles\n", 0);
  /* XXX put in code to emit angles here */
  fprintf(psf->fd, "\n\n");

  fprintf(psf->fd, "%8d !NPHI: dihedrals\n", 0);
  /* XXX put in code to emit dihedrals here */
  fprintf(psf->fd, "\n\n");

  fprintf(psf->fd, "%8d !NIMPHI: impropers\n", 0);
  /* XXX put in code to emit impropers here */
  fprintf(psf->fd, "\n\n");

  fprintf(psf->fd, "%8d !NDON: donors\n", 0);
  fprintf(psf->fd, "\n\n");

  fprintf(psf->fd, "%8d !NACC: acceptors\n", 0);
  fprintf(psf->fd, "\n\n");

  fprintf(psf->fd, "%8d !NNB\n\n", 0);
  /* Pad with zeros, one for every atom */
  {
    int i, fullrows;
    fullrows = psf->numatoms/8;
    for (i=0; i<fullrows; ++i)
      fprintf(psf->fd, "%8d%8d%8d%8d%8d%8d%8d%8d\n", 0, 0, 0, 0, 0, 0, 0, 0);
    for (i=psf->numatoms - fullrows*8; i; --i)
      fprintf(psf->fd, "%8d", 0);
  }
  fprintf(psf->fd, "\n\n");

  fprintf(psf->fd, "%8d %7d !NGRP\n%8d%8d%8d\n", 1, 0, 0, 0, 0);
  fprintf(psf->fd, "\n");

  return MOLFILE_SUCCESS;
}

static void close_psf_write(void *v) {
  psfdata *psf = (psfdata *)v;
  fclose(psf->fd);
  free(psf);
}


/*
 * Initialization stuff down here
 */

static molfile_plugin_t plugin = {
  vmdplugin_ABIVERSION,         /* ABI version */
  MOLFILE_PLUGIN_TYPE,		/* type */
  "psf",			/* name */
  "Justin Gullingsrud",		/* author */
  0,				/* major version */
  2,				/* minor version */
  VMDPLUGIN_THREADSAFE,         /* is_reentrant  */
  "psf",                        /* filename extension */
  open_psf_read,		/* open_file_read     */
  read_psf, 			/* read_structure     */
  read_bonds,                   /* read bond list     */
  0, 				/* read_next_timestep */
  close_psf_read,   		/* close_file_read    */
  open_psf_write,               /* open file for writing  */
  write_psf_structure,          /* write structure        */
  0,                            /* write timestep         */
  close_psf_write               /* close file for writing */
};

int VMDPLUGIN_init() {
  return VMDPLUGIN_SUCCESS;
}

int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  (*cb)(v, (vmdplugin_t *)&plugin);
  return VMDPLUGIN_SUCCESS;
}

int VMDPLUGIN_fini() {
  return VMDPLUGIN_SUCCESS;
}
