##
## Finder Tool 1.0 
##
## A script to find programs. The function ::FinderTool::find can be used by
## other extensions to find needed executables.
##
## Author: Eamon Caddigan
##         eamon@ks.uiuc.edu
##         vmd@ks.uiuc.edu
##
## $Id: exectool.tcl,v 1.6 2003/09/09 23:05:35 eamon Exp $
##

##
## Example code to add this plugin to the VMD extensions menu:
##
#  if { [catch {package require exectool} msg] } {
#    puts "VMD FinderTool package could not be loaded:\n$msg"
#  } elseif { [catch {menu tk register "exectool" exectool} msg] } {
#    puts "VMD FinderTool could not be started:\n$msg"
#  }


## Tell Tcl that we're a package and any dependencies we may have
package provide exectool 1.0

namespace eval ::FinderTool:: {
  namespace export exectool

  # window handles
  variable w                         ;# handle to main window

  # global settings for the program to find and its location
  variable program  ""
  variable location ""
}
  
##
## Main routine
## Create the window and initialize data structures
##
proc ::FinderTool::exectool {} {
  variable w
  variable program

  # If already initialized, just turn on
  if { [winfo exists .exectool] } {
    wm deiconify $w
    return
  }

  set w [toplevel ".exectool"]
  wm title $w "Program Finder" 
  wm resizable $w no no

  set normalfont [font create -family helvetica -weight normal]
  set boldfont [font create -family helvetica -weight bold]

  ##
  ## make the menu bar
  ## 
  frame $w.menubar -relief raised -bd 2 ;# frame for menubar
  pack $w.menubar -padx 1 -fill x -side top

  menubutton $w.menubar.help -text Help -underline 0 -menu $w.menubar.help.menu
  pack $w.menubar.help -side right

  ##
  ## help menu
  ##
  menu $w.menubar.help.menu -tearoff no
  $w.menubar.help.menu add command -label "About" \
    -command {tk_messageBox -type ok -title "About exectool" \
              -message "A simple tool used to find programs."}

  frame $w.inputframe
  label $w.inputframe.label -font $normalfont -text "Program name: "
  entry $w.inputframe.entry -font $normalfont -width 16 -relief sunken -bd 2 \
    -textvariable "::FinderTool::program"
  button $w.inputframe.search -font $normalfont -text "Locate" \
    -command "::FinderTool::find_gui"
  pack $w.inputframe.label $w.inputframe.entry $w.inputframe.search \
    -side left -anchor w

  frame $w.outputframe
  label $w.outputframe.resultlabel -font $boldfont \
    -textvariable ::FinderTool::location
  pack $w.outputframe.resultlabel \
    -side left -anchor w

  ## 
  ## pack up the main frame
  ##
  pack \
       $w.inputframe $w.outputframe \
       -side top -pady 10 -padx 10 -fill x
}


##
## Call the search routine and update the GUI
##
proc ::FinderTool::find_gui {} {
  variable program
  variable location

  set programpath [::FinderTool::find $program]

  if {[string length $programpath] > 0} {
    set location $programpath
  } else {
    set location "$program: Command not found."
  }
}


##
## Find the executable, prompting the user if necessary
##
proc ::FinderTool::find { execname } {
  global env
  global tk_version
  set execpath ""

  if {[info exists env(PATH)]} {
    switch [vmdinfo arch] {
      WIN32 {
        set pathlist [split $env(PATH) \;]
      }
      default {
        set pathlist [split $env(PATH) :]
      }
    }

    # Search the path for the program
    foreach directory $pathlist {
      if {[file executable [file join $directory $execname]] > 0} {
        set execpath [file join $directory $execname]
        break
      }
    }
  }

  # If the file isn't found in the path and Tk is available, prompt the user 
  # for its location
  if {[string length $execpath] == 0 && \
      [info exists tk_version] > 0} {
    set answer [tk_messageBox -type yesno -title "Program not found" \
                -message " Could not find a program named `$execname'. \n\
                           Would you like to locate it manually?"]
    while {$answer} {
      set answer no
      # XXX - not sure if getOpenFile is best here
      set execpath [tk_getOpenFile -initialfile $execname \
                    -title "Please select a program"]  
      if {[string length $execpath] > 0 && \
          [file executable [file join $execpath]] == 0} {
        set answer [tk_messageBox -type yesno -title "Warning" \
                    -message " Warning, `$execpath' is not executable. \n\
                               Would you like to change the selection?"]
      }
    }
  }

  return $execpath
}


# This gets called by VMD the first time the menu is opened.
proc exectool_tk_cb {} {
  variable foobar
  # Don't destroy the main window, because we want to register the window
  # with VMD and keep reusing it.  The window gets iconified instead of
  # destroyed when closed for any reason.
  #set foobar [catch {destroy $::FinderTool::w  }]  ;# destroy any old windows

  ::FinderTool::exectool   ;# start the tool 
  return $FinderTool::w
}

