/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2002 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: molfile_plugin.h,v $
 *      $Author: johns $       $Locker:  $             $State: Exp $
 *      $Revision: 1.15 $       $Date: 2003/08/12 19:34:05 $
 *
 ***************************************************************************/

/** @file 
 * API for C extensions to define a way to load structure, coordinate,
 * trajectory, and volumetric data files  
 */ 

#ifndef MOL_FILE_PLUGIN_H
#define MOL_FILE_PLUGIN_H

#include "vmdplugin.h"

/**
 * Define a common plugin type to be used when registering the plugin.
 */
#define MOLFILE_PLUGIN_TYPE "mol file reader"

/**
 * File converter plugins use the same API  but register under a different
 * type so that regular file readers can have priority.
 */
#define MOLFILE_CONVERTER_PLUGIN_TYPE "mol file converter"

/* File plugin symbolic constants for better code readability */
#define MOLFILE_EOF              -1   /**< end of file                  */
#define MOLFILE_ERROR            -1   /**< error reading/opening a file */
#define MOLFILE_SUCCESS           0   /**< succeeded in reading file    */
#define MOLFILE_NUMATOMS_UNKNOWN -1   /**< unknown number of atoms      */
#define MOLFILE_NUMATOMS_NONE     0   /**< no atoms in this file type   */

/* 
 * Define a structure for specifying atoms in a molecular structure.  The 
 * first six are required, the rest are optional and their presence is 
 * indicating by setting the corresponding bit in optsflag.  When omitted,
 * the application (for read_structure) or plugin
 * for write_structure) must be able to supply default values if the missing
 * parameters are part of its internal data structure.
 * Note that it is not possible to specify coordinates with this structure.
 * This is intentional; all coordinate I/O is done with the read_timestep and 
 * write_timestep functions. 
 */

/*@{*/
/** Plugin optional data field availability flag */
#define MOLFILE_NOOPTIONS 0
#define MOLFILE_INSERTION 0x01
#define MOLFILE_OCCUPANCY 0x02
#define MOLFILE_BFACTOR   0x04
#define MOLFILE_MASS      0x08
#define MOLFILE_CHARGE    0x10
#define MOLFILE_RADIUS    0x20
/*@}*/

typedef struct {
  char name[8];       /**< required atom name string      */
  char type[8];       /**< required atom type string      */
  char resname[8];    /**< required residue name string   */
  int resid;          /**< required integer residue ID    */
  char chain[8];      /**< required chain name string     */
  char segid[8];      /**< required segment name string   */

  /* rest are optional; use optflags to specify what's present */
  char insertion[8];  /**< optional insertion code string */
  float occupancy;    /**< optional occupancy value       */
  float bfactor;      /**< optional B-factor value        */
  float mass;         /**< optional mass value            */
  float charge;       /**< optional charge value          */
  float radius;       /**< optional radius value          */
} molfile_atom_t;

/*
 * Timestep information
 */ 
typedef struct {
  float *coords;  /**< space for coordinates of all atoms, arranged xyzxyzxyz */

  /*@{*/   
  /**
   * Unit cell specification of the form A, B, C, alpha, beta, gamma.
   * A, B, and C are the lengths of the vectors.  alpha is angle between A and
   * B, beta between A and C, and gamma between B and C.
   */ 
  float A, B, C, alpha, beta, gamma; 
  /*@}*/   
} molfile_timestep_t;

/**
 * Metadata for volumetric datasets, read initially and used for subsequent
 * memory allocations and file loading.  
 */
typedef struct {
  char dataname[256]; /**< name of volumetric data set                    */
  float origin[3];    /**< origin: origin of volume (x=0, y=0, z=0 corner */
  float xaxis[3];     /**< direction (and length) for X axis              */ 
  float yaxis[3];     /**< direction (and length) for Y axis              */
  float zaxis[3];     /**< direction (and length) for Z axis              */
  int xsize;          /**< number of grid cells along the X axis          */
  int ysize;          /**< number of grid cells along the Y axis          */
  int zsize;          /**< number of grid cells along the Z axis          */
  int has_color;      /**< flag indicating presence of voxel color data   */
} molfile_volumetric_t;

/**
 *  Enumeration of all of the supported graphics objects that can be read
 *  from graphics file reader plugins.
 */
enum molfile_graphics_type {
  MOLFILE_POINT,  MOLFILE_TRIANGLE, MOLFILE_TRINORM, MOLFILE_NORMS, 
  MOLFILE_LINE,   MOLFILE_CYLINDER, MOLFILE_CAPCYL,  MOLFILE_CONE,    
  MOLFILE_SPHERE, MOLFILE_TEXT,     MOLFILE_COLOR,   MOLFILE_TRICOLOR
};

/**
 *  Individual graphics object/element data
 */ 
typedef struct {
  int type;             /* One of molfile_graphics_type */
  int style;            /* A general style parameter    */
  float size;           /* A general size parameter     */
  float data[9];        /* All data for the element     */
} molfile_graphics_t;

/*
 * Types for raw graphics elements stored in files.  Data for each type
 * should be stored by the plugin as follows:

type        data                                     style       size
----        ----                                     -----       ----
point       x, y, z                                              pixel size
triangle    x1,y1,z1,x2,y2,z2,x3,y3,z3                 
trinorm     x1,y1,z1,x2,y2,z2,x3,y3,z3                 
            the next array element must be NORMS
tricolor    x1,y1,z1,x2,y2,z2,x3,y3,z3                 
            the next array elements must be NORMS
            the following element must be COLOR, with three RGB triples
norms       x1,y1,z1,x2,y2,z2,x3,y3,z3                 
line        x1,y1,z1,x2,y2,z2                        0=solid     pixel width
                                                     1=stippled
cylinder    x1,y1,z1,x2,y2,z2                        resolution  radius
capcyl      x1,y1,z1,x2,y2,z2                        resolution  radius
sphere      x1,y1,z1                                 resolution  radius
text        x, y, z, up to 24 bytes of text                      pixel size
color       r, g, b
*/

/**
 * Main file reader API.  Any function in this struct may be NULL
 * if not implemented by the plugin; the application checks this to determine
 * what functionality is present in the plugin. 
 */ 
typedef struct {
  /**
   * Required header 
   */
  vmdplugin_HEAD

  /**
   * Filename extension for this file type.  May be NULL if no filename 
   * extension exists and/or is known.
   */
  const char *filename_extension;

  /**
   * Try to open the file for reading.  Return an opaque handle, or NULL on
   * failure. Set the number of atoms; if the number of atoms cannot be 
   * determined, set natoms to MOLFILE_NUMATOMS_UNKNOWN. 
   * Filetype should be the name under which this plugin was registered;
   * this is provided so that plugins can provide the same function pointer
   * to handle multiple file types.
   */
  void *(* open_file_read)(const char *filepath, const char *filetype, 
      int *natoms);
  
  /**
   * Read molecular structure from the given file handle.  atoms is allocated
   * by the caller and points to space for natoms.
   * On success, place atom information in the passed-in pointer.  
   * optflags specifies which optional fields in the atoms will be set by
   * the plugin.
   */
  int (*read_structure)(void *, int *optflags, molfile_atom_t *atoms);

  /**
   * Read bond information for the molecule.  On success the arrays from
   * and to should point to the (one-based) indices of bonded atoms.
   * These arrays must be freed by the plugin in the close_file_read function.
   * This function can be called only after read_structure().  Return 0 on
   * success.
   */
  int (*read_bonds)(void *, int *nbonds, int **from, int **to);

  /**
   * Read the next timestep from the file.  Return 0 on success, or -1
   * on EOF.  If the molfile_timestep_t argument is NULL, then the frame should
   * be skipped.  Otherwise, the application must prepare molfile_timestep_t
   * by allocating space in coords for the corresponding number of 
   * coordinates.  The natoms parameter
   * is present because some coordinate file formats (like CRD) cannot
   * determine for themselves how many atoms are in each timestep; the 
   * application must therefore obtain this information from somewhere else
   * and provide it to the plugin.
   */
  int (* read_next_timestep)(void *, int natoms, molfile_timestep_t *);

  /**
   * Close the file and release all data.  The handle cannot be reused.
   */
  void (* close_file_read)(void *);
   
  /**
   * Open a coordinate file for writing using the given header information.
   * Return an opaque handle, or NULL on failure.  The application must
   * specify the number of atoms to be written. 
   * filetype should be the name under which this plugin was registered.
   */
  void *(* open_file_write)(const char *filepath, const char *filetype, 
      int natoms);
  
  /**
   * Write structure information.  Return success.
   */
  int (* write_structure)(void *, int optflags, const molfile_atom_t *atoms);

  /**
   * Write a timestep to the coordinate file.  Return 0 on success.   If the
   * file contains structure information in each timestep (like a multi-entry
   * PDB, it will have to cache the information from the initial calls from
   * write_structure.
   */
  int (* write_timestep)(void *, const molfile_timestep_t *);
  
  /**
   * Close the file and release all data.  The handle cannot be reused.
   */
  void (* close_file_write)(void *);

  /**
   * Retrieve metadata pertaining to volumetric datasets in this file.
   * Set nsets to the number of volumetric data sets, and set *metadata
   * to point to an array of molfile_volumetric_t.  The array is owned by
   * the plugin and should be freed by close_file_read().  The application
   * may call this function any number of times.
   */
  int (* read_volumetric_metadata)(void *, int *nsets, 
        molfile_volumetric_t **metadata);

  /** 
   * Read the specified volumetric data set into the space pointed to by 
   * datablock.  The set is specified with a zero-based index.  The space 
   * allocated for the datablock must be equal to
   * xsize * ysize * zsize.  No space will be allocated for colorblock 
   * unless has_color is nonzero; in that case, colorblock should be
   * filled in with three RGB floats per datapoint.
   */
  int (* read_volumetric_data)(void *, int set, float *datablock, 
        float *colorblock);

  /**
   * Read raw graphics data stored in this file.   Return the number of data
   * elements and the data itself as an array of molfile_graphics_t in the 
   * pointer provided by the application.  The plugin is responsible for 
   * freeing the data when the file is closed.
   */
  int (* read_rawgraphics)(void *, int *nelem, 
      const molfile_graphics_t **data);

} molfile_plugin_t;

#endif

